﻿#!/usr/bin/env python3
"""
kaiFarma Local - Entry Point
Smart Loader App Layer v0.2.0

This is the main entry point for the kaiFarma local application.
It starts both the backend API and the frontend Dash app.

PORTS (Local App):
- Backend: 9000 (diferente de Docker 8000)
- Frontend: 9050 (diferente de Docker 8050)

Esto permite ejecutar la app local simultaneamente con Docker (dev cloud).
Ver docs/DEV_ENVIRONMENT.md para mas detalles.
"""
import os
import sys
import signal
import threading
import webbrowser
import time

# =============================================================================
# CONFIGURACION DE PUERTOS - APP LOCAL
# =============================================================================
# Usamos puertos diferentes a Docker para poder ejecutar ambos simultaneamente:
# - Docker (Cloud/Dev): 8000/8050
# - Local (Farmacia):   9000/9050
BACKEND_PORT = 9000
FRONTEND_PORT = 9050

# Global flag for graceful shutdown
_shutdown_requested = False
_uvicorn_server = None

# Setup paths for imports
# Structure: app_dir/backend/app/... and app_dir/frontend/...
app_dir = os.path.dirname(os.path.abspath(__file__))
backend_dir = os.path.join(app_dir, 'backend')
frontend_dir = os.path.join(app_dir, 'frontend')

# Backend expects: from app.core.xxx (so backend/ must be in path)
# Frontend expects: from callbacks.xxx (so frontend/ must be in path)
sys.path.insert(0, backend_dir)
sys.path.insert(0, frontend_dir)

# Set environment for local mode
os.environ['KAIFARMA_LOCAL'] = 'true'

# CRITICO: Configurar BACKEND_URL para que frontend apunte al puerto correcto
os.environ['BACKEND_URL'] = f'http://127.0.0.1:{BACKEND_PORT}'

# Database path relative to data directory
data_dir = os.environ.get('KAIFARMA_DATA_DIR', os.path.join(os.path.dirname(app_dir), 'data'))
os.makedirs(data_dir, exist_ok=True)
os.environ['DATABASE_URL'] = f'sqlite:///{os.path.join(data_dir, "kaifarma.db")}'

def signal_handler(signum, frame):
    """Handle SIGINT/SIGTERM for graceful shutdown."""
    global _shutdown_requested, _uvicorn_server
    print("\nShutdown signal received, stopping servers...")
    _shutdown_requested = True
    if _uvicorn_server is not None:
        _uvicorn_server.should_exit = True
    # Exit with code 130 (128 + SIGINT) to signal interrupt to parent
    sys.exit(130)

def start_backend():
    """Start the FastAPI backend server."""
    global _uvicorn_server
    import uvicorn
    # Ensure backend_dir is first in path for 'app' package resolution
    # (frontend/app.py would otherwise shadow backend/app/ package)
    if frontend_dir in sys.path:
        sys.path.remove(frontend_dir)
    if backend_dir not in sys.path:
        sys.path.insert(0, backend_dir)

    from app.main import app

    # Create server with shutdown capability
    config = uvicorn.Config(
        app,
        host="127.0.0.1",
        port=BACKEND_PORT,
        log_level="info"
    )
    _uvicorn_server = uvicorn.Server(config)
    _uvicorn_server.run()

def start_frontend():
    """Start the Dash frontend."""
    global _shutdown_requested
    # Change to frontend dir for relative imports in callbacks
    os.chdir(frontend_dir)
    # CRITICAL: Remove backend from path to avoid 'app' package conflict
    # backend/app/ package would shadow frontend/app.py module
    while backend_dir in sys.path:
        sys.path.remove(backend_dir)
    # Frontend needs its directory first in path
    if frontend_dir not in sys.path:
        sys.path.insert(0, frontend_dir)
    # CRITICAL: Clear cached 'app' module from backend import
    # Otherwise Python returns the cached backend/app package instead of frontend/app.py
    if 'app' in sys.modules:
        del sys.modules['app']

    from app import app as dash_app

    # Use threaded=False to allow proper signal handling
    dash_app.run(
        host="127.0.0.1",
        port=FRONTEND_PORT,
        debug=False,
        threaded=False,
        use_reloader=False
    )

def open_browser():
    """Open browser after a short delay."""
    time.sleep(3)
    if not _shutdown_requested:
        webbrowser.open(f'http://localhost:{FRONTEND_PORT}')

if __name__ == '__main__':
    # Register signal handlers for graceful shutdown
    signal.signal(signal.SIGINT, signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    print(f"Starting kaiFarma Local v0.2.0...")
    print(f"Data directory: {data_dir}")
    print(f"Backend: http://127.0.0.1:{BACKEND_PORT}")
    print(f"Frontend: http://127.0.0.1:{FRONTEND_PORT}")
    print("Press Ctrl+C to stop")

    # Start backend in a thread
    backend_thread = threading.Thread(target=start_backend, daemon=True)
    backend_thread.start()

    # Give backend time to start
    time.sleep(2)

    # Open browser in a thread
    browser_thread = threading.Thread(target=open_browser, daemon=True)
    browser_thread.start()

    try:
        # Start frontend in main thread (blocking)
        start_frontend()
    except KeyboardInterrupt:
        print("\nInterrupted by user, shutting down...")
        sys.exit(130)
