#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Script de validación de callbacks de Dash.
Verifica que todos los IDs referenciados en callbacks existen en los layouts.

Uso:
    python frontend/utils/validate_callbacks.py

Retorna:
    0 si todo está correcto
    1 si hay IDs faltantes
"""

import os
import re
import sys
from pathlib import Path
from typing import Dict, List, Set, Tuple

# REGLA #11: Excepciones lícitas para Inputs broadcast
# Estos Inputs pueden ser escuchados por múltiples callbacks sin conflicto
# Ver: docs/CALLBACK_DUPLICATES_ANALYSIS.md para justificación completa
BROADCAST_EXCEPTIONS = {
    # Stores de Estado Global (broadcast readers)
    "url",  # Navegación global leída por múltiples subsistemas
    "viewport-store",  # Dimensiones de viewport para responsive design
    "auth-state",  # Estado de autenticación global
    "auth-ready",  # Issue #344: Señal de auth listo (broadcast para múltiples callbacks)
    "sidebar-state-store",  # Estado del sidebar compartido
    "catalog-status-store",  # Estado del catálogo CIMA
    "catalog-progress-store",  # Progreso de sincronización CIMA
    "auth-tokens-store",  # Tokens JWT compartidos
    "catalog-panel-unified-store",  # Estado unificado del panel de catálogo
    "context-store",  # Contexto de análisis farmacéutico
    "partners-selection-store",  # Selección de laboratorios partners
    # Intervals de Actualización (broadcast triggers)
    "catalog-initial-load-interval",  # Carga inicial de catálogo
    "catalog-real-progress-interval",  # Progreso de sincronización en tiempo real
    "dashboard-update-interval",  # Actualización del dashboard
    "admin-stats-interval",  # Estadísticas de admin
    "health-refresh-interval",  # Métricas de salud del sistema
    # Stores de Análisis Complejos (multiple readers)
    "analysis-store",  # Resultados de análisis farmacéutico compartidos
    "homepage-data-store",  # Datos del dashboard compartidos
    # Sistema de toasts centralizado (patrón original)
    "toast-trigger-store",
    "auth-guard-interval",  # Auth guard puede ser leído por múltiples callbacks
    # Excepciones REGLA #11 (Issue #258 - patrones válidos)
    "expand-all-homogeneous-btn",  # Broadcast: 3 responsabilidades independientes (matrix, UI, data)
    "maintenance-tools-panel-operation-store",  # Store pattern: 1 writer → multiple readers
    # Prescription module broadcasts (Issue #441, #483)
    "prescription-overview-store",  # Store pattern: 1 writer → multiple readers (dropdowns + UI)
    "prescription-date-range-store",  # Issue #483: triggers data loading + filters initialization
    "evolution-chart-type",  # UI control affecting multiple outputs (chart + trigger store)
    # Generics date range (refactoring Issue)
    "generics-date-range-store",  # Store pattern: date filter triggers context + storage callbacks
    # Tab components (Issue #471) - múltiples callbacks cargan secciones independientes
    "ventalibre-tabs",  # Tab switch triggers: analisis, inventory (Issue #488: Gestión removed)
    "prescription-tabs",  # Tab switch triggers: overview, evolution, products, inventory
    # Ventalibre data store - uses modified_timestamp pattern (1 writer → multiple readers)
    "ventalibre-data-store",  # Context treemap + other categories section (both use modified_timestamp)
    # Issue #494: Product search - different props (search_value → options, value → selection)
    "ventalibre-product-search",  # Multi-prop pattern: 2 different properties, not duplicate Input
    # Issue #489: Prescription filters button triggers multiple independent data loads
    "prescription-apply-filters-btn",  # Broadcast: loads prescription data + seasonality data
    # Date range pickers - trigger multiple independent data loads (Issue #537)
    "prescription-date-range",  # Triggers: data_loading, filters, products, seasonality
    "ventalibre-date-range",  # Triggers: brands, data_loading, l2_drilldown
    # Seasonality ATC filter - triggers multiple independent chart loads
    "seasonality-atc-filter",  # Triggers: seasonality, stockout_risk, anomalies, badges
}


class CallbackValidator:
    def __init__(self, frontend_dir: str = "frontend"):
        self.frontend_dir = Path(frontend_dir)
        self.callback_ids: Dict[str, List[str]] = {}  # ID -> archivos donde se usa
        self.layout_ids: Dict[str, List[str]] = {}  # ID -> archivos donde se define

        # REGLA #11: Rastrear Inputs para detectar duplicados
        # input_signature -> (archivo, función, línea)
        self.callback_inputs: Dict[str, List[Tuple[str, str, int]]] = {}

        # IDs que se crean dinámicamente y no están en layouts estáticos
        self.known_dynamic_ids = {
            # Auth components
            "auth-check",
            "logout-button",
            # Backend URL (usado como store global)
            "backend-url",
            # Dynamic components
            "danger-operation-btn",
            "maintenance-tool-btn",
            "flag-switch",
            # Dynamic modals and stores
            "admin-danger-zone-modal",
            "admin-danger-zone-modal-cancel",
            "admin-danger-zone-modal-confirm",
            "admin-danger-zone-modal-impacts",
            "admin-danger-zone-modal-operation-info",
            "admin-danger-zone-modal-prerequisites",
            "admin-danger-zone-modal-text-confirmation",
            "admin-danger-zone-modal-title",
            "admin-danger-zone-operation-store",
            "admin-danger-zone-content",
            # Maintenance tools
            "maintenance-tools-panel-confirmation-modal",
            "maintenance-tools-panel-confirmation-modal-cancel",
            "maintenance-tools-panel-confirmation-modal-confirm",
            "maintenance-tools-panel-confirmation-modal-message",
            "maintenance-tools-panel-confirmation-modal-title",
            "maintenance-tools-panel-operation-store",
            # Catalog real progress
            "catalog-real-progress-display",
            "catalog-real-progress-interval",
            "catalog-sync-details",
            "view-sync-details-btn",
            # Database config
            "check-db-btn",
            # Generic analysis components
            "detail-content",
            "detail-row",
            "expand-btn",
            "expand-icon",
            "matrix-header",
            "select-homogeneous-btn",  # Issue #346: Botón de selección de conjunto homogéneo
            # System status unified
            "homepage-unified-status-config-store",
            "homepage-unified-status-content",
            "homepage-unified-status-data-store",
            "homepage-unified-status-interval",
            # Admin sync operations
            "conflict-resolution-strategy",
            "enable-scheduled-sync-switch",
            "resolve-conflicts-button",
            "scheduled-syncs-panel",
            "sync-conflicts-resolution",
            "sync-history-display",
            "sync-schedule-dropdown",
            # Upload components (Issue #287 - pattern matching dismiss button)
            "dismiss-upload-summary",
            "upload-summary-card",
            "delete-upload",  # Pattern matching delete button
            # User management pattern-matching buttons (Issue #348 FASE 3.1)
            "edit-user-btn",
            "delete-user-btn",
            "restore-user-btn",
            # Prescription pattern-matching containers (Issue #441)
            "category-products-container",
            "ingredient-products-container",
            # Prescription waterfall category chips (Issue #454)
            "waterfall-category-chip",
            # VentaLibre feedback button (Issue #461 - pattern matching)
            "ventalibre-feedback-btn",
            # Duplicates management pattern-matching buttons (Issue #477)
            "duplicate-merge-btn",
            "duplicate-reject-btn",
            "duplicate-skip-btn",
            # Classification NECESIDAD pattern-matching buttons (Issue #449)
            "classification-approve-btn",
            "classification-outlier-btn",
            "classification-skip-btn",
            "classification-category-dropdown",
            # Keywords management pattern-matching buttons (Issue #449)
            "keywords-edit-btn",
            "keywords-delete-btn",
            # Category Aliases pattern-matching buttons (Issue #459)
            "aliases-edit-btn",
            "aliases-toggle-btn",
            "aliases-delete-btn",
            # Brand Aliases pattern-matching buttons
            "brand-aliases-edit-btn",
            "brand-aliases-toggle-btn",
            "brand-aliases-delete-btn",
            "brand-aliases-reprocess-btn",
            # VentaLibre filters (creados por componentes reutilizables en layout)
            # DateRangeFilter y EmployeeFilter crean estos IDs dinámicamente
            "ventalibre-date-range",
            "ventalibre-employee-filter",
            "ventalibre-employee-tooltip",
            # Prescription Seasonality graphs (Issue #489)
            # Creados por funciones componente con default IDs
            "seasonality-hourly-heatmap",
            "seasonality-monthly-index-chart",
            # Issue #498: STL Decomposition + Forecast charts
            "seasonality-decomposition-chart",
            "seasonality-forecast-chart",
            # Issue #506: Insight Engine pattern-matching buttons
            "insight-action-btn",
            "insight-deeplink",
            "insight-dismiss-btn",
            "insight-snooze-btn",
            "ventalibre-insight-filter",
            # Issue #521: Curated Groups pattern-matching buttons
            "curated-groups-delete-btn",
            "curated-groups-edit-btn",
            "curated-groups-members-btn",
            "curated-groups-toggle-btn",
            "curated-members-remove-btn",
        }

    def extract_callback_ids(self, file_path: Path) -> Set[str]:
        """Extrae todos los IDs usados en callbacks de un archivo."""
        try:
            with open(file_path, "r", encoding="utf-8") as f:
                content = f.read()
        except:
            return set()

        # Eliminar bloques de código comentados con triple comillas
        # Remover strings multi-línea que contienen callbacks comentados
        import re

        # Patrón para encontrar bloques entre ''' o """
        content = re.sub(r"'''[\s\S]*?'''", "", content)
        content = re.sub(r'"""[\s\S]*?"""', "", content)

        # Filtrar líneas comentadas con #
        lines = content.split("\n")
        active_lines = []
        for line in lines:
            stripped = line.strip()
            if not stripped.startswith("#"):
                # También quitar comentarios al final de la línea
                if "#" in line and not ('"#"' in line or "'#'" in line):
                    line = line[: line.index("#")]
                active_lines.append(line)

        content = "\n".join(active_lines)

        # Buscar Output, Input, State con sus IDs
        patterns = [
            r'Output\s*\(\s*["\']([^"\']+)["\']',
            r'Input\s*\(\s*["\']([^"\']+)["\']',
            r'State\s*\(\s*["\']([^"\']+)["\']',
            # También buscar con diccionarios para IDs tipo pattern-matching
            r'Output\s*\(\s*\{\s*["\']type["\']\s*:\s*["\']([^"\']+)["\']',
            r'Input\s*\(\s*\{\s*["\']type["\']\s*:\s*["\']([^"\']+)["\']',
            r'State\s*\(\s*\{\s*["\']type["\']\s*:\s*["\']([^"\']+)["\']',
        ]

        ids = set()
        for pattern in patterns:
            matches = re.findall(pattern, content)
            ids.update(matches)

        # Guardar qué archivo usa cada ID
        for id in ids:
            if id not in self.callback_ids:
                self.callback_ids[id] = []
            self.callback_ids[id].append(str(file_path))

        return ids

    def extract_callback_inputs(self, file_path: Path) -> None:
        """
        Extrae Inputs de callbacks y los registra para detectar duplicados.
        REGLA #11: Un Input solo debe ser escuchado por UN callback.
        """
        try:
            with open(file_path, "r", encoding="utf-8") as f:
                content = f.read()
        except:
            return

        # CRITICAL FIX (Issue #291): Eliminar docstrings ANTES de parsear callbacks
        # Los ejemplos en docstrings NO son callbacks reales
        content = re.sub(r'"""[\s\S]*?"""', "", content)  # Eliminar docstrings """ """
        content = re.sub(r"'''[\s\S]*?'''", "", content)  # Eliminar docstrings ''' '''

        lines = content.split("\n")

        # Buscar definiciones de callbacks
        current_callback = None
        current_line_num = 0
        in_callback_decorator = False
        callback_inputs = []

        for i, line in enumerate(lines, 1):
            stripped = line.strip()

            # Ignorar líneas comentadas con #
            if stripped.startswith("#"):
                continue

            # Detectar inicio de callback
            if "@app.callback" in line or "@callback" in line:
                in_callback_decorator = True
                current_line_num = i
                callback_inputs = set()  # Use set to deduplicate Inputs from same component
                continue

            # Si estamos en un decorator de callback, buscar Inputs
            if in_callback_decorator:
                # Buscar Input(...) en la línea
                input_matches = re.findall(r'Input\s*\(\s*["\']([^"\']+)["\']', line)
                for input_id in input_matches:
                    callback_inputs.add(input_id)  # Use add() for set

                # Detectar fin del decorator (función def)
                if stripped.startswith("def "):
                    # Extraer nombre de función
                    func_match = re.match(r"def\s+(\w+)", stripped)
                    if func_match:
                        current_callback = func_match.group(1)

                        # Registrar cada Input de este callback
                        for input_id in callback_inputs:
                            # Crear firma única del Input
                            input_signature = f"Input('{input_id}')"

                            if input_signature not in self.callback_inputs:
                                self.callback_inputs[input_signature] = []

                            self.callback_inputs[input_signature].append(
                                (str(file_path), current_callback, current_line_num)
                            )

                    in_callback_decorator = False

    def extract_layout_ids(self, file_path: Path) -> Set[str]:
        """Extrae todos los IDs definidos en layouts de un archivo."""
        try:
            with open(file_path, "r", encoding="utf-8") as f:
                content = f.read()
        except:
            return set()

        # Buscar definiciones de IDs
        patterns = [
            r'id\s*=\s*["\']([^"\']+)["\']',
            r'id\s*:\s*["\']([^"\']+)["\']',  # Para diccionarios
            # Para IDs en stores y otros componentes
            r'Store\s*\(\s*id\s*=\s*["\']([^"\']+)["\']',
            r'Interval\s*\(\s*id\s*=\s*["\']([^"\']+)["\']',
        ]

        ids = set()
        for pattern in patterns:
            matches = re.findall(pattern, content)
            ids.update(matches)

        # Guardar qué archivo define cada ID
        for id in ids:
            if id not in self.layout_ids:
                self.layout_ids[id] = []
            self.layout_ids[id].append(str(file_path))

        return ids

    def scan_project(self) -> Tuple[Set[str], Set[str]]:
        """Escanea todo el proyecto y recopila IDs."""
        all_callback_ids = set()
        all_layout_ids = set()

        print("[SCAN] Escaneando archivos Python...")

        for file_path in self.frontend_dir.rglob("*.py"):
            rel_path = file_path.relative_to(self.frontend_dir)

            # Excluir el propio archivo de validación
            if "validate_callbacks.py" in str(file_path):
                continue

            # Buscar callbacks
            with open(file_path, "r", encoding="utf-8") as f:
                content = f.read()
                if "@app.callback" in content or "@callback" in content or "Output(" in content:
                    # REGLA #11: Extraer IDs y detectar Inputs duplicados
                    ids = self.extract_callback_ids(file_path)
                    if ids:
                        all_callback_ids.update(ids)
                        print(f"  [FILE] {rel_path}: {len(ids)} IDs en callbacks")

                    self.extract_callback_inputs(file_path)

            # Buscar layouts
            ids = self.extract_layout_ids(file_path)
            if ids:
                all_layout_ids.update(ids)

        return all_callback_ids, all_layout_ids

    def validate_duplicate_inputs(self) -> bool:
        """
        Valida REGLA #11: Un Input solo debe ser escuchado por UN callback.
        Retorna True si no hay duplicados, False si hay errores.

        Excepciones lícitas documentadas en: docs/CALLBACK_DUPLICATES_ANALYSIS.md
        """
        print("\n[REGLA #11] Validando Inputs duplicados...")
        print("-" * 60)

        duplicates_found = False

        for input_sig, callbacks in self.callback_inputs.items():
            if len(callbacks) > 1:
                # Extraer ID del Input
                input_id = input_sig.split("'")[1]

                # Si está en excepciones, solo advertir (no error)
                if input_id in BROADCAST_EXCEPTIONS:
                    print(f"\n  [INFO] Input duplicado (excepción broadcast): {input_sig}")
                    print("         Callbacks que escuchan este Input:")
                    for file_path, func_name, line_num in callbacks:
                        rel_path = Path(file_path).name
                        print(f"           - {rel_path}:{line_num} -> {func_name}()")
                    continue

                # Error: Input duplicado detectado
                duplicates_found = True
                print(f"\n  [ERROR] Input duplicado detectado: {input_sig}")
                print("          REGLA #11: Un Input debe ser escuchado por UN SOLO callback")
                print("")
                print("          Callbacks que escuchan este Input:")

                for file_path, func_name, line_num in callbacks:
                    rel_path = Path(file_path).name
                    print(f"            - {rel_path}:{line_num} -> {func_name}()")

                print("")
                print("          Soluciones:")
                print("             1. Combinar ambos callbacks en uno solo con múltiples outputs")
                print("             2. Usar Store intermedio para desacoplar lógica")
                print("             3. Ver CLAUDE.md REGLA #11 (lineas 256-326)")

        if not duplicates_found:
            print("  [OK] No se encontraron Inputs duplicados")

        return not duplicates_found

    def validate(self) -> bool:
        """Ejecuta la validación completa."""
        print("\n" + "=" * 60)
        print("VALIDADOR DE CALLBACKS DE DASH")
        print("=" * 60 + "\n")

        callback_ids, layout_ids = self.scan_project()

        print("\n[RESUMEN]")
        print(f"  - IDs en callbacks: {len(callback_ids)}")
        print(f"  - IDs en layouts: {len(layout_ids)}")

        # Encontrar IDs faltantes
        missing_ids = callback_ids - layout_ids

        # Filtrar IDs dinámicos conocidos
        truly_missing_ids = missing_ids - self.known_dynamic_ids
        dynamic_ids_found = missing_ids & self.known_dynamic_ids

        # Encontrar IDs no usados (opcional, informativo)
        unused_ids = layout_ids - callback_ids

        # IDs especiales que podemos ignorar (pattern matching, etc.)
        ignore_patterns = ["match", "all", "type", "index", "toast-item"]
        truly_missing_ids = {id for id in truly_missing_ids if id not in ignore_patterns}

        # REGLA #11: Validar Inputs duplicados
        duplicate_inputs_ok = self.validate_duplicate_inputs()

        success = True

        # Mostrar información sobre IDs dinámicos encontrados (informativo)
        if dynamic_ids_found:
            print(f"\n[INFO] IDs dinámicos conocidos encontrados: {len(dynamic_ids_found)}")
            print("-" * 60)
            for id in sorted(dynamic_ids_found):
                print(f"  [OK] {id} (componente dinamico)")

        # Solo mostrar errores para IDs realmente problemáticos
        if truly_missing_ids:
            print("\n[ERROR] IDs problemáticos en callbacks (NO en layouts NI dinámicos):")
            print("-" * 60)
            for id in sorted(truly_missing_ids):
                print(f"\n  [X] ID: '{id}'")
                print("     Usado en:")
                for file in self.callback_ids.get(id, []):
                    print(f"       - {file}")
            success = False
        else:
            print("\n[OK] Todos los IDs de callbacks estan correctamente referenciados")
            if dynamic_ids_found:
                print(f"     INFO: {len(dynamic_ids_found)} IDs dinamicos identificados correctamente")

        # Combinar resultados de validación
        if not duplicate_inputs_ok:
            success = False

        if unused_ids and len(unused_ids) < 20:  # Solo mostrar si no son demasiados
            print("\n[ADVERTENCIA] IDs definidos pero no usados en callbacks:")
            print("-" * 60)
            for id in sorted(list(unused_ids)[:10]):
                print(f"  - {id}")
            if len(unused_ids) > 10:
                print(f"  ... y {len(unused_ids) - 10} mas")

        print("\n" + "=" * 60)
        if success:
            print("[OK] VALIDACION EXITOSA")
        else:
            print("[ERROR] VALIDACION FALLIDA - Corregir errores antes de continuar")
        print("=" * 60 + "\n")

        return success

    def generate_report(self, output_file: str = "callback_report.md"):
        """Genera un reporte detallado en Markdown."""
        with open(output_file, "w", encoding="utf-8") as f:
            f.write("# Reporte de Validación de Callbacks\n\n")
            f.write(f"## IDs en Callbacks ({len(self.callback_ids)})\n\n")

            for id, files in sorted(self.callback_ids.items()):
                f.write(f"### `{id}`\n")
                f.write("Usado en:\n")
                for file in files:
                    f.write(f"- {file}\n")
                f.write("\n")

            f.write(f"\n## IDs en Layouts ({len(self.layout_ids)})\n\n")

            for id, files in sorted(self.layout_ids.items()):
                f.write(f"### `{id}`\n")
                f.write("Definido en:\n")
                for file in files:
                    f.write(f"- {file}\n")
                f.write("\n")


def main():
    """Función principal."""
    # Determinar el directorio frontend
    if os.path.exists("frontend"):
        frontend_dir = "frontend"
    elif os.path.exists("../frontend"):
        frontend_dir = "../frontend"
    else:
        print("[ERROR] No se encuentra el directorio frontend")
        sys.exit(1)

    validator = CallbackValidator(frontend_dir)

    # Agregar opción para generar reporte
    if len(sys.argv) > 1 and sys.argv[1] == "--report":
        validator.scan_project()
        validator.generate_report()
        print("[INFO] Reporte generado: callback_report.md")
        return

    # Ejecutar validación
    if validator.validate():
        sys.exit(0)
    else:
        sys.exit(1)


if __name__ == "__main__":
    main()
