# frontend/utils/token_refresh_service.py
"""
Token Refresh Service - Renovación automática de JWT tokens

Este módulo implementa la renovación automática de access tokens ANTES de que expiren,
usando el refresh token almacenado en sesión.

Flujo:
1. El auth_guard verifica cada 5-8s si el token necesita renovarse
2. Si quedan < 3 minutos para expirar, intenta renovarlo
3. Si el refresh falla, procede con logout (token refresh expirado)
4. Si el refresh tiene éxito, actualiza solo el access_token

Issue #187: Prevenir logout automático cada 15 minutos
"""

import logging
from datetime import datetime, timezone
from typing import Dict, Optional

import jwt
import requests

from utils.config import BACKEND_URL

logger = logging.getLogger(__name__)

# Constantes de configuración
TOKEN_REFRESH_THRESHOLD_SECONDS = 180  # 3 minutos antes de expirar
TOKEN_EXPIRY_BUFFER_SECONDS = 60  # Buffer adicional de 60s para compensar latencia


def get_token_expiry(token: str) -> Optional[datetime]:
    """
    Obtiene la fecha de expiración de un JWT token.

    Args:
        token: JWT token a analizar

    Returns:
        datetime con la fecha de expiración en UTC, o None si el token es inválido
    """
    try:
        # Decodificar sin verificar signature (solo necesitamos el exp)
        payload = jwt.decode(token, options={"verify_signature": False})
        exp_timestamp = payload.get("exp")

        if exp_timestamp:
            return datetime.fromtimestamp(exp_timestamp, tz=timezone.utc)

        return None
    except Exception as e:
        logger.debug(f"[TOKEN_REFRESH] Error decoding token: {e}")
        return None


def should_refresh_token(token: str) -> bool:
    """
    Determina si un token necesita ser renovado.

    Un token necesita renovarse si:
    - Está próximo a expirar (< 3 minutos restantes)
    - Es válido pero con poco tiempo restante

    Args:
        token: Access token JWT a verificar

    Returns:
        True si el token debe renovarse
    """
    if not token:
        return False

    expiry = get_token_expiry(token)

    if not expiry:
        logger.debug("[TOKEN_REFRESH] Cannot determine token expiry")
        return False

    now = datetime.now(timezone.utc)
    time_until_expiry = (expiry - now).total_seconds()

    # Logs de debugging para monitoreo
    logger.debug(
        f"[TOKEN_REFRESH] Token expires in {time_until_expiry:.0f}s "
        f"(threshold: {TOKEN_REFRESH_THRESHOLD_SECONDS}s)"
    )

    # Renovar si quedan menos de 3 minutos Y el token no ha expirado aún
    should_refresh = 0 < time_until_expiry <= TOKEN_REFRESH_THRESHOLD_SECONDS

    if should_refresh:
        logger.info(
            f"[TOKEN_REFRESH] Token will expire in {time_until_expiry:.0f}s, "
            "triggering proactive refresh"
        )

    return should_refresh


def refresh_access_token(refresh_token: str) -> Optional[Dict[str, str]]:
    """
    Renueva el access token usando el refresh token.

    Llama al endpoint POST /api/v1/auth/refresh del backend.

    Args:
        refresh_token: Refresh token JWT válido (duración: 7 días)

    Returns:
        Dict con nuevo access_token y token_type, o None si falló
        Ejemplo: {'access_token': 'eyJ...', 'token_type': 'bearer'}
    """
    if not refresh_token:
        logger.warning("[TOKEN_REFRESH] No refresh token provided")
        return None

    try:
        logger.info("[TOKEN_REFRESH] Attempting to refresh access token")

        response = requests.post(
            f"{BACKEND_URL}/api/v1/auth/refresh",
            json={"refresh_token": refresh_token},
            timeout=10,  # Timeout corto (10s) para no bloquear auth_guard
        )

        if response.status_code == 200:
            data = response.json()
            new_access_token = data.get("access_token")

            if new_access_token:
                logger.info("[TOKEN_REFRESH] Successfully refreshed access token")
                return {"access_token": new_access_token, "token_type": data.get("token_type", "bearer")}
            else:
                logger.error("[TOKEN_REFRESH] Backend response missing access_token")
                return None

        elif response.status_code == 401:
            # Refresh token inválido o expirado
            logger.warning("[TOKEN_REFRESH] Refresh token invalid or expired (401)")
            return None

        else:
            logger.error(f"[TOKEN_REFRESH] Refresh failed with status {response.status_code}")
            return None

    except requests.RequestException as e:
        logger.error(f"[TOKEN_REFRESH] Network error during refresh: {e}")
        return None
    except Exception as e:
        logger.error(f"[TOKEN_REFRESH] Unexpected error during refresh: {e}")
        return None


def is_token_expired(token: str) -> bool:
    """
    Verifica si un token ya ha expirado.

    Args:
        token: JWT token a verificar

    Returns:
        True si el token expiró, False si aún es válido o no se puede determinar
    """
    expiry = get_token_expiry(token)

    if not expiry:
        return False

    now = datetime.now(timezone.utc)
    is_expired = now >= expiry

    if is_expired:
        time_expired = (now - expiry).total_seconds()
        logger.debug(f"[TOKEN_REFRESH] Token expired {time_expired:.0f}s ago")

    return is_expired
