"""
Style Helpers - Sistema de Diseño xFarma

Funciones auxiliares para facilitar el uso de design tokens en componentes.
Simplifica la aplicación de spacing, colores, sombras y estilos de texto.

Uso:
    from utils.style_helpers import spacing, color, shadow, text_style

    style = {
        "padding": spacing(m=True, l=True),  # padding: 16px 24px
        "color": color("primary"),
        "boxShadow": shadow("md")
    }
"""

import warnings

from styles.design_tokens import COLORS, SHADOWS, SPACING, TYPOGRAPHY


def spacing(top=None, right=None, bottom=None, left=None, **kwargs):
    """
    Helper para generar valores de spacing usando design tokens.

    Args:
        top: Spacing superior (nombre de token: "xs", "s", "m", "l", "xl", "2xl", "3xl")
        right: Spacing derecho
        bottom: Spacing inferior
        left: Spacing izquierdo
        **kwargs: Shortcuts para spacing uniforme
            - xs=True: Aplica "xs" a todos los lados
            - s=True: Aplica "s" a todos los lados
            - m=True: Aplica "m" a todos los lados
            - l=True: Aplica "l" a todos los lados
            - xl=True: Aplica "xl" a todos los lados

    Returns:
        str: Valor de spacing en formato CSS (ej: "16px 24px")

    Ejemplos:
        # Spacing uniforme
        spacing(m=True)  # "16px"

        # Spacing vertical y horizontal
        spacing(top="l", bottom="l")  # "24px 0px"

        # Spacing completo
        spacing(top="l", right="m", bottom="s", left="m")  # "24px 16px 8px 16px"

        # Usar en un componente
        style = {"padding": spacing(m=True, l=True)}  # "16px 24px"
    """
    # Procesar shortcuts para spacing uniforme
    for size_key in ["xs", "s", "m", "l", "xl", "2xl", "3xl"]:
        if kwargs.get(size_key):
            return SPACING[size_key]

    # Construir spacing con valores individuales
    values = []
    for side in [top, right, bottom, left]:
        if side is None:
            values.append("0")
        elif side in SPACING:
            values.append(SPACING[side])
        else:
            # Validar que sea un valor CSS válido
            if isinstance(side, str) and any(side.endswith(unit) for unit in ["px", "rem", "em", "%", "vh", "vw"]):
                values.append(side)  # Permitir valores CSS válidos custom
            else:
                warnings.warn(
                    f"Invalid spacing value '{side}'. Expected a SPACING token ('xs', 's', 'm', 'l', 'xl', '2xl', '3xl') "
                    f"or a valid CSS value (ending with px, rem, em, %, vh, vw). Using '0' as fallback.",
                    UserWarning,
                    stacklevel=2,
                )
                values.append("0")

    # Simplificar la salida según los valores
    if len(set(values)) == 1:
        # Todos los valores son iguales
        return values[0]
    elif top == bottom and right == left:
        # Vertical y horizontal iguales
        return f"{values[0]} {values[1]}"
    elif right == left:
        # Derecha e izquierda iguales
        return f"{values[0]} {values[1]} {values[2]}"
    else:
        # Todos diferentes
        return " ".join(values)


def color(name, fallback=None):
    """
    Helper para obtener colores del sistema de diseño.

    Args:
        name: Nombre del color (token de COLORS)
        fallback: Color de respaldo si no se encuentra el token

    Returns:
        str: Valor hexadecimal del color

    Ejemplos:
        color("primary")  # "#2C3E50"
        color("success")  # "#18BC9C"
        color("custom", fallback="#FF0000")  # "#FF0000"
    """
    return COLORS.get(name, fallback or COLORS["text_primary"])


def shadow(size="sm"):
    """
    Helper para obtener sombras del sistema de diseño.

    Args:
        size: Tamaño de la sombra ("none", "sm", "md", "lg", "xl")

    Returns:
        str: Valor CSS de box-shadow

    Ejemplos:
        shadow("sm")  # Sombra pequeña
        shadow("lg")  # Sombra grande
        shadow("none")  # Sin sombra
    """
    return SHADOWS.get(size, SHADOWS["sm"])


def text_style(variant="body"):
    """
    Helper para obtener estilos de texto del sistema de diseño.

    Args:
        variant: Variante de tipografía ("h1", "h2", "h3", "h4", "h5", "body", "body_small", "caption")

    Returns:
        dict: Diccionario con estilos de texto (fontSize, fontWeight, lineHeight)

    Ejemplos:
        # Aplicar estilo de h3
        style = {
            **text_style("h3"),
            "color": color("primary")
        }

        # Estilo de body pequeño
        style = text_style("body_small")  # {"fontSize": "0.875rem", ...}
    """
    if variant not in TYPOGRAPHY:
        variant = "body"  # Fallback a body por defecto

    typography = TYPOGRAPHY[variant]
    return {
        "fontSize": typography["size"],
        "fontWeight": typography["weight"],
        "lineHeight": typography["line_height"],
    }


def merge_styles(*style_dicts):
    """
    Helper para combinar múltiples diccionarios de estilos.

    Args:
        *style_dicts: Diccionarios de estilos a combinar (los últimos sobrescriben los primeros)

    Returns:
        dict: Diccionario de estilos combinado

    Ejemplos:
        # Combinar estilos base con personalizados
        base = {"padding": spacing(m=True), "color": color("primary")}
        custom = {"backgroundColor": color("bg_secondary")}
        final = merge_styles(base, custom)
    """
    result = {}
    for style_dict in style_dicts:
        if style_dict:
            result.update(style_dict)
    return result


def responsive_spacing(mobile="s", tablet="m", desktop="l"):
    """
    Helper para generar spacing responsive (para uso con media queries).

    Args:
        mobile: Spacing para móvil
        tablet: Spacing para tablet
        desktop: Spacing para desktop

    Returns:
        dict: Diccionario con valores para diferentes breakpoints

    Ejemplos:
        spacing_values = responsive_spacing(mobile="s", tablet="m", desktop="l")
        # {"mobile": "8px", "tablet": "16px", "desktop": "24px"}
    """
    return {
        "mobile": SPACING.get(mobile, SPACING["s"]),
        "tablet": SPACING.get(tablet, SPACING["m"]),
        "desktop": SPACING.get(desktop, SPACING["l"]),
    }
