"""
Ribbon Chart Builder para xFarma.

Construye gráficos de cintas (ribbon chart) estilo Power BI con Plotly.
Las cintas muestran cómo cambian los rankings y valores de categorías a lo largo del tiempo.

Características:
- Barras apiladas en cada período, ordenadas por valor (mayor arriba, menor abajo)
- Cintas suavizadas (interpolación cúbica 3t²-2t³) que conectan categorías entre períodos
- Las cintas se cruzan cuando cambia el ranking

Basado en: docs/instrucciones_ribbon_chart.md
"""

import logging
from typing import Dict, List, Optional, Tuple

import numpy as np
import plotly.graph_objects as go

from styles.design_tokens import COLORS
from utils.constants import format_category_name
from utils.helpers import format_currency

logger = logging.getLogger(__name__)

# Paleta de colores para categorías (14 colores distintos)
CATEGORY_COLORS = [
    "#636EFA",  # Azul
    "#EF553B",  # Rojo
    "#00CC96",  # Verde
    "#AB63FA",  # Púrpura
    "#FFA15A",  # Naranja
    "#19D3F3",  # Cyan
    "#FF6692",  # Rosa
    "#B6E880",  # Lima
    "#FF97FF",  # Magenta
    "#FECB52",  # Amarillo
    "#1F77B4",  # Azul oscuro
    "#D62728",  # Rojo oscuro
    "#2CA02C",  # Verde oscuro
    "#9467BD",  # Púrpura oscuro
]


def hex_to_rgba(hex_color: str, alpha: float = 1.0) -> str:
    """Convierte color hex a rgba string."""
    hex_color = hex_color.lstrip("#")
    r, g, b = tuple(int(hex_color[i : i + 2], 16) for i in (0, 2, 4))
    return f"rgba({r},{g},{b},{alpha})"


def build_ribbon_chart(
    data_by_category: Dict[str, Dict],
    sorted_periods: List[str],
    x_labels: List[str],
    top_n: int = 8,
    chart_title: str = "Evolución de Ventas por Categoría",
) -> go.Figure:
    """
    Construye ribbon chart real según instrucciones.

    Args:
        data_by_category: {category: {"periods": [...], "sales": [...]}}
        sorted_periods: Períodos ordenados cronológicamente
        x_labels: Etiquetas legibles para eje X
        top_n: Número de categorías a mostrar
        chart_title: Título del gráfico

    Returns:
        go.Figure con ribbon chart
    """
    fig = go.Figure()

    if not data_by_category or not sorted_periods:
        logger.warning("[build_ribbon_chart] No data provided")
        fig.update_layout(
            title={"text": "Sin datos disponibles", "x": 0.5},
            xaxis={"visible": False},
            yaxis={"visible": False},
            height=400,
        )
        return fig

    # === PASO 1: Calcular posiciones acumuladas por período ===
    cumulative_data = {}
    category_colors = {}
    all_categories = set()

    for i, period in enumerate(sorted_periods):
        # Obtener ventas por categoría para este período
        period_data = []
        for cat, data in data_by_category.items():
            periods_list = data.get("periods", [])
            sales_list = data.get("sales", [])

            # Crear diccionario de período -> ventas
            sales_dict = dict(zip(periods_list, sales_list))
            sales = sales_dict.get(period, 0)

            if sales > 0:
                period_data.append((cat, sales))
                all_categories.add(cat)

        # Ordenar por ventas (mayor arriba)
        period_data.sort(key=lambda x: x[1], reverse=True)

        # Tomar top N
        period_data = period_data[:top_n]

        if not period_data:
            cumulative_data[period] = {
                "categories": [],
                "sales": [],
                "cumsum": [],
            }
            continue

        # Calcular cumsum (posiciones acumuladas desde abajo)
        cumsum = [0]
        for cat, sales in period_data[:-1]:
            cumsum.append(cumsum[-1] + sales)

        cumulative_data[period] = {
            "categories": [p[0] for p in period_data],
            "sales": [p[1] for p in period_data],
            "cumsum": cumsum,
        }

    # Asignar colores consistentes a categorías
    sorted_categories = sorted(all_categories)
    for idx, cat in enumerate(sorted_categories):
        category_colors[cat] = CATEGORY_COLORS[idx % len(CATEGORY_COLORS)]

    # === PASO 2: Dibujar cintas entre períodos consecutivos ===
    bar_width = 0.15  # Ancho de las barras

    for period_idx in range(len(sorted_periods) - 1):
        period1 = sorted_periods[period_idx]
        period2 = sorted_periods[period_idx + 1]

        data1 = cumulative_data.get(period1, {"categories": []})
        data2 = cumulative_data.get(period2, {"categories": []})

        if not data1["categories"] or not data2["categories"]:
            continue

        # Para cada categoría en período 1
        for cat_idx, cat in enumerate(data1["categories"]):
            if cat not in data2["categories"]:
                continue  # Categoría no aparece en período 2

            # Posiciones en período 1
            x1 = period_idx
            y1_bottom = data1["cumsum"][cat_idx]
            y1_top = y1_bottom + data1["sales"][cat_idx]

            # Posiciones en período 2
            cat_idx_2 = data2["categories"].index(cat)
            x2 = period_idx + 1
            y2_bottom = data2["cumsum"][cat_idx_2]
            y2_top = y2_bottom + data2["sales"][cat_idx_2]

            color = category_colors.get(cat, CATEGORY_COLORS[0])

            # Crear cinta suavizada (polígono)
            n_points = 50
            t = np.linspace(0, 1, n_points)

            # Interpolación cúbica (ease in-out): 3t² - 2t³
            ease = 3 * t**2 - 2 * t**3

            # Curva de X: desde borde derecho barra 1 hasta borde izquierdo barra 2
            x_curve = (x1 + bar_width) + ease * ((x2 - bar_width) - (x1 + bar_width))

            # Curvas de Y con interpolación suavizada
            y_top_curve = y1_top + ease * (y2_top - y1_top)
            y_bottom_curve = y1_bottom + ease * (y2_bottom - y1_bottom)

            # Crear polígono cerrado
            x_polygon = np.concatenate([x_curve, x_curve[::-1]])
            y_polygon = np.concatenate([y_top_curve, y_bottom_curve[::-1]])

            fig.add_trace(
                go.Scatter(
                    x=x_polygon,
                    y=y_polygon,
                    fill="toself",
                    fillcolor=hex_to_rgba(color, 0.5),  # Semitransparente
                    line=dict(color=color, width=0.5),
                    mode="lines",
                    name=cat,
                    showlegend=False,
                    hoverinfo="skip",
                )
            )

    # === PASO 3: Dibujar barras en cada período ===
    # Diccionario para tracking de leyenda
    legend_added = set()

    # Calcular total por período para tooltip
    period_totals = {}
    for period in sorted_periods:
        data = cumulative_data.get(period, {"sales": []})
        period_totals[period] = sum(data.get("sales", []))

    for period_idx, period in enumerate(sorted_periods):
        data = cumulative_data.get(period, {"categories": []})
        x = period_idx
        period_total = period_totals.get(period, 0)

        for cat_idx, cat in enumerate(data["categories"]):
            y_bottom = data["cumsum"][cat_idx]
            y_top = y_bottom + data["sales"][cat_idx]
            color = category_colors.get(cat, CATEGORY_COLORS[0])
            sales_value = data["sales"][cat_idx]
            # Nombre formateado para tooltip (ej: "MEDICAMENTOS" → "Medicamentos")
            cat_display_name = format_category_name(cat)
            # Formato español para ventas (1.234,56 €)
            sales_formatted = format_currency(sales_value)
            period_total_formatted = format_currency(period_total)

            # Barra como polígono (rectángulo)
            # Issue #444: Añadir customdata con período para drill-down
            period_label = x_labels[period_idx] if period_idx < len(x_labels) else period
            fig.add_trace(
                go.Scatter(
                    x=[
                        x - bar_width,
                        x + bar_width,
                        x + bar_width,
                        x - bar_width,
                        x - bar_width,
                    ],
                    y=[y_bottom, y_bottom, y_top, y_top, y_bottom],
                    fill="toself",
                    fillcolor=color,
                    line=dict(color="white", width=1.5),
                    mode="lines",
                    name=cat_display_name,  # Nombre formateado en leyenda
                    showlegend=(cat not in legend_added),
                    legendgroup=cat,
                    # Issue #444: customdata para drill-down (5 puntos del polígono)
                    customdata=[period] * 5,
                    # Tooltip simplificado - sin usar el nombre de la serie como título
                    hovertemplate=f"<b>{cat_display_name}</b><br>Período: {period_label}<br>Ventas: {sales_formatted}<br>Total período: {period_total_formatted}<br>Ranking: #{cat_idx + 1}<extra></extra>",
                )
            )
            legend_added.add(cat)

    # === PASO 4: Configurar layout ===
    # Ocultar leyenda si hay muchas categorías (evita superposición)
    num_categories = len(all_categories)
    show_legend = num_categories <= 8

    fig.update_layout(
        title={"text": chart_title, "x": 0.5, "font": {"size": 14}},
        xaxis=dict(
            tickmode="array",
            tickvals=list(range(len(sorted_periods))),
            ticktext=x_labels,
            title="Período",
        ),
        yaxis=dict(
            title="Ventas (€)",
            tickformat=",",
        ),
        height=450,
        margin=dict(t=40, b=40, l=80, r=40),
        hovermode="closest",
        showlegend=show_legend,
        hoverlabel=dict(
            bgcolor="white",
            font_size=12,
            font_family="Arial",
        ),
    )

    # Configurar leyenda solo si se muestra
    if show_legend:
        fig.update_layout(
            legend=dict(
                title=dict(text=""),
                orientation="h",
                yanchor="top",
                y=-0.08,
                xanchor="center",
                x=0.5,
                font=dict(size=10),
                itemwidth=30,
            )
        )

    logger.info(
        f"[build_ribbon_chart] Built chart with {len(all_categories)} categories, "
        f"{len(sorted_periods)} periods, top_n={top_n}"
    )

    return fig
