# frontend/utils/request_coordinator.py
"""
Coordinador de Requests para optimizar rate limiting en Render.
Implementa cache inteligente, backoff exponencial y coordinación entre componentes.

PROBLEMA: xFarma en Render excede 100 requests/minuto debido a polling excesivo.
SOLUCIÓN: Centralizar y optimizar todas las llamadas API del frontend.
"""

import hashlib
import json
import logging
import os
import threading
import time
from collections import defaultdict
from datetime import datetime, timedelta
from threading import Lock
from typing import Any, Dict, Optional, Tuple

import requests

logger = logging.getLogger(__name__)


class APICache:
    """Cache inteligente con TTL para requests API"""

    def __init__(self):
        self.cache: Dict[str, Dict[str, Any]] = {}
        self.lock = Lock()

    def _generate_key(self, url: str, params: Optional[Dict] = None) -> str:
        """Genera clave única para la request"""
        content = f"{url}:{json.dumps(params, sort_keys=True) if params else ''}"
        return hashlib.md5(content.encode()).hexdigest()

    def get(self, url: str, params: Optional[Dict] = None, ttl_seconds: int = 30) -> Optional[Any]:
        """Obtiene datos del cache si están frescos"""
        key = self._generate_key(url, params)

        with self.lock:
            if key in self.cache:
                entry = self.cache[key]
                if datetime.now() < entry["expires_at"]:
                    logger.debug(f"[CACHE_HIT] {url}")
                    return entry["data"]
                else:
                    del self.cache[key]
                    logger.debug(f"[CACHE_EXPIRED] {url}")

        return None

    def set(self, url: str, data: Any, params: Optional[Dict] = None, ttl_seconds: int = 30):
        """Almacena datos en cache con TTL"""
        key = self._generate_key(url, params)
        expires_at = datetime.now() + timedelta(seconds=ttl_seconds)

        with self.lock:
            self.cache[key] = {"data": data, "expires_at": expires_at, "created_at": datetime.now()}
            logger.debug(f"[CACHE_SET] {url} (TTL: {ttl_seconds}s)")

    def clear_expired(self):
        """Limpia entradas expiradas del cache"""
        now = datetime.now()
        with self.lock:
            expired_keys = [key for key, entry in self.cache.items() if now >= entry["expires_at"]]
            for key in expired_keys:
                del self.cache[key]

            if expired_keys:
                logger.debug(f"[CACHE_CLEANUP] Removed {len(expired_keys)} expired entries")


class RateLimitBackoff:
    """
    Manejo de backoff exponencial para rate limiting con soporte por endpoint.

    Issue #191: Backoff diferenciado por tipo de error:
    - 422 (validation): NO backoff (es bug de código, no sobrecarga)
    - 401/403 (auth): Backoff agresivo (600s)
    - 500/503/timeout: Backoff moderado (30s max)
    """

    @staticmethod
    def _sanitize_endpoint(endpoint: str) -> str:
        """Remueve query params sensibles de endpoints para logging"""
        return endpoint.split("?")[0]

    def __init__(self):
        # Backoff específico por endpoint
        self.failure_counts: Dict[str, int] = {}
        self.last_failures: Dict[str, float] = {}
        self.endpoint_locks = defaultdict(threading.Lock)  # Lock por endpoint
        self.base_delay = 2  # 2 segundos base
        self.lock = Lock()  # Lock global solo para cleanup

    def should_wait(self, endpoint: str, max_delay: int = 600) -> Tuple[bool, float]:
        """
        Determina si debe esperar antes de hacer una request a un endpoint específico.

        Args:
            endpoint: Endpoint a verificar
            max_delay: Delay máximo en segundos (varía según tipo de error)

        Returns:
            (should_wait, wait_seconds)
        """
        with self.endpoint_locks[endpoint]:  # Lock específico del endpoint
            failure_count = self.failure_counts.get(endpoint, 0)

            if failure_count == 0:
                return False, 0

            # Calcular delay exponencial con max_delay específico
            delay = min(self.base_delay * (2 ** (failure_count - 1)), max_delay)

            last_failure = self.last_failures.get(endpoint)
            if last_failure:
                elapsed = time.time() - last_failure
                remaining = max(0, delay - elapsed)

                if remaining > 0:
                    return True, remaining

            return False, 0

    def record_success(self, endpoint: str):
        """Registra request exitosa - resetea contador del endpoint"""
        with self.endpoint_locks[endpoint]:  # Lock específico del endpoint
            if endpoint in self.failure_counts:
                del self.failure_counts[endpoint]
            if endpoint in self.last_failures:
                del self.last_failures[endpoint]
            logger.debug(f"[BACKOFF_RESET] Request successful for {self._sanitize_endpoint(endpoint)}")

    def record_failure(self, endpoint: str, max_delay: int = 600):
        """
        Registra fallo de request - incrementa backoff del endpoint.

        Args:
            endpoint: Endpoint que falló
            max_delay: Delay máximo en segundos (varía según tipo de error)
        """
        with self.endpoint_locks[endpoint]:  # Lock específico del endpoint
            self.failure_counts[endpoint] = self.failure_counts.get(endpoint, 0) + 1
            self.last_failures[endpoint] = time.time()
            failure_count = self.failure_counts[endpoint]
            delay = min(self.base_delay * (2 ** (failure_count - 1)), max_delay)
            logger.warning(
                f"[BACKOFF_INCREASE] {self._sanitize_endpoint(endpoint)} failure count: {failure_count}, next delay: {delay}s (max: {max_delay}s)"
            )

    def cleanup_inactive_endpoints(self, max_age_seconds: int = 3600):
        """
        Limpia endpoints sin actividad reciente para prevenir memory leak.

        Args:
            max_age_seconds: Edad máxima en segundos para endpoints inactivos (default: 1 hora)
        """
        with self.lock:  # Lock global para cleanup
            now = time.time()
            inactive = [
                endpoint for endpoint, last_fail in self.last_failures.items() if now - last_fail > max_age_seconds
            ]
            for endpoint in inactive:
                # Limpiar datos del endpoint
                self.failure_counts.pop(endpoint, None)
                self.last_failures.pop(endpoint, None)
                # Limpiar lock del endpoint para liberar memoria
                self.endpoint_locks.pop(endpoint, None)

            if inactive:
                logger.info(f"[BACKOFF_CLEANUP] Removed {len(inactive)} inactive endpoints")


class RequestCoordinator:
    """
    Coordinador principal para optimizar requests API.

    Características:
    - Cache inteligente con TTL configurable
    - Backoff exponencial para rate limiting
    - Deduplicación de requests simultáneas
    - Métricas de performance
    """

    def __init__(self):
        self.cache = APICache()
        self.backoff = RateLimitBackoff()
        self.pending_requests: Dict[str, Any] = {}
        self.metrics = {"total_requests": 0, "cache_hits": 0, "rate_limit_hits": 0, "backoff_waits": 0}
        self.lock = Lock()

        # Issue #187: Rastreo de errores 401 para auth guard
        # Comportamiento "last wins": Si múltiples requests generan 401 simultáneamente,
        # solo se registra el último. Esto es aceptable porque todos los 401s indican
        # el mismo problema (token inválido/expirado) y solo necesitamos detectar UNO
        # para iniciar el proceso de logout y redirección.
        self.last_401_error = None
        self.last_401_timestamp = None

        # Environment validation on startup
        environment = os.getenv("ENVIRONMENT", "production").lower()
        logger.info(f"[REQUEST_COORDINATOR] Initialized - Environment: {environment}")

        if environment not in ["production", "development", "staging", "test"]:
            logger.warning(f"[REQUEST_COORDINATOR] Unknown environment: {environment}, defaulting to production mode")

        logger.info("[REQUEST_COORDINATOR] Using JWT authentication for all requests")

    def get_backend_url(self) -> str:
        """Obtiene URL del backend según el entorno"""
        return os.getenv("BACKEND_URL", "http://localhost:8000")

    def _create_error_response(
        self, error_type: str, message: str, status_code: Optional[int] = None, endpoint: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Crea una respuesta de error estructurada.

        Args:
            error_type: Tipo de error ('timeout', 'connection', 'unauthorized', 'forbidden', 'server', 'validation', 'rate_limit', 'unknown')
            message: Mensaje descriptivo del error
            status_code: Código HTTP si aplica
            endpoint: Endpoint que generó el error

        Returns:
            Dict con estructura de error estandarizada
        """
        error_response = {"error": True, "error_type": error_type, "message": message}

        if status_code is not None:
            error_response["status_code"] = status_code

        if endpoint is not None:
            error_response["endpoint"] = self.backoff._sanitize_endpoint(endpoint)

        error_response["timestamp"] = datetime.now().isoformat()

        return error_response

    def make_request(
        self,
        endpoint: str,
        method: str = "GET",
        params: Optional[Dict] = None,
        data: Optional[Dict] = None,
        cache_ttl: int = 30,
        bypass_cache: bool = False,
        timeout: int = 10,
        return_error_details: bool = False,
        auth_headers: Optional[Dict] = None,
    ) -> Optional[Dict]:
        """
        Hace request optimizada con cache, backoff y coordinación.

        Args:
            endpoint: Endpoint API (ej: "/api/v1/health")
            method: Método HTTP
            params: Query parameters
            data: Request body para POST/PUT
            cache_ttl: TTL del cache en segundos
            bypass_cache: Saltarse cache para requests críticas
            timeout: Timeout en segundos (default: 10s, usar 60s para operaciones pesadas)
            return_error_details: Si True, retorna dict con detalles del error en lugar de None
            auth_headers: Headers de autenticación explícitos (opcional). Si se proveen, se usan en lugar del singleton.
                         Útil en entornos multi-worker donde el singleton puede estar desincronizado.

        Returns:
            Respuesta JSON, None si falla (default), o dict con error estructurado si return_error_details=True

        Note:
            Con return_error_details=True, retorna dict con:
            {"error": True, "error_type": str, "message": str, "status_code": int, "endpoint": str, "timestamp": str}
        """
        self.metrics["total_requests"] += 1

        # Construir URL completa
        base_url = self.get_backend_url()
        url = f"{base_url}{endpoint}"

        # Verificar cache (solo para GET)
        if method == "GET" and not bypass_cache:
            cached_data = self.cache.get(url, params, cache_ttl)
            if cached_data is not None:
                self.metrics["cache_hits"] += 1
                return cached_data

        # Verificar backoff con max_delay por defecto (será ajustado según tipo de error)
        should_wait, wait_time = self.backoff.should_wait(endpoint, max_delay=600)
        if should_wait:
            logger.warning(
                f"[BACKOFF_WAIT] Waiting {wait_time:.1f}s before request to {self.backoff._sanitize_endpoint(endpoint)}"
            )
            self.metrics["backoff_waits"] += 1
            time.sleep(wait_time)
            # Continuar con la request normalmente después del sleep

        try:
            # Construir headers de autenticación con prioridad explícita
            headers = {}

            # PRIORIDAD 1: Usar headers explícitos si se proveen (multi-worker safe)
            # Esto resuelve timing issues en entornos Gunicorn donde el singleton puede estar desincronizado
            if auth_headers:
                headers.update(auth_headers)
                logger.debug(f"[REQUEST_COORDINATOR] Using explicit auth headers for {self.backoff._sanitize_endpoint(endpoint)}")
            else:
                # PRIORIDAD 2: Fallback al singleton auth_context (backward compatible)
                # En entornos single-worker (desarrollo) funciona correctamente
                try:
                    from utils.auth_context import auth_context

                    ctx_headers = auth_context.get_auth_headers()
                    if ctx_headers:
                        headers.update(ctx_headers)
                        logger.debug(f"[REQUEST_COORDINATOR] Using singleton auth headers for {self.backoff._sanitize_endpoint(endpoint)}")
                except (ImportError, AttributeError):
                    # En tests unitarios, auth_context puede no estar disponible
                    pass

            # Hacer request
            if method == "GET":
                response = requests.get(url, params=params, headers=headers, timeout=timeout)
            elif method == "POST":
                response = requests.post(url, json=data, params=params, headers=headers, timeout=timeout)
            elif method == "PUT":
                response = requests.put(url, json=data, params=params, headers=headers, timeout=timeout)
            elif method == "DELETE":
                # DELETE usa params para confirmación, no json body (RFC 7231 compliance)
                response = requests.delete(url, params=params, headers=headers, timeout=timeout)
            else:
                raise ValueError(f"Método HTTP no soportado: {method}")

            # Issue #191: Clasificación de errores con backoff diferenciado
            if not response.ok:
                status_code = response.status_code

                # 401 Unauthorized: Backoff agresivo (600s max)
                if status_code == 401:
                    logger.error(f"[AUTH_ERROR] 401 Unauthorized en {self.backoff._sanitize_endpoint(endpoint)}")
                    logger.error("[AUTH_ERROR] JWT token required - user needs to login")

                    # Issue #187: Registrar error 401 para auth guard
                    with self.lock:
                        self.last_401_error = {
                            "status": 401,
                            "endpoint": endpoint,
                            "timestamp": datetime.now().isoformat(),
                        }
                        self.last_401_timestamp = time.time()

                    self.backoff.record_failure(endpoint, max_delay=600)

                    if return_error_details:
                        return self._create_error_response(
                            error_type="unauthorized",
                            message="No autorizado. Por favor, inicie sesión nuevamente.",
                            status_code=401,
                            endpoint=endpoint,
                        )
                    return None

                # 403 Forbidden: Backoff agresivo (600s max)
                elif status_code == 403:
                    logger.error(f"[FORBIDDEN_ERROR] 403 Forbidden en {self.backoff._sanitize_endpoint(endpoint)}")
                    self.backoff.record_failure(endpoint, max_delay=600)

                    if return_error_details:
                        return self._create_error_response(
                            error_type="forbidden",
                            message="Acceso denegado. No tiene permisos suficientes.",
                            status_code=403,
                            endpoint=endpoint,
                        )
                    return None

                # 422 Unprocessable Entity: NO backoff (es bug de backend, no sobrecarga)
                elif status_code == 422:
                    logger.warning(
                        f"[VALIDATION_ERROR] 422 en {self.backoff._sanitize_endpoint(endpoint)} - Bug de backend, NO se aplica backoff"
                    )
                    # NO llamar a record_failure() - no penalizar

                    if return_error_details:
                        return self._create_error_response(
                            error_type="validation",
                            message="Error de validación en el servidor.",
                            status_code=422,
                            endpoint=endpoint,
                        )
                    return None

                # 429 Too Many Requests: Backoff agresivo con retry-after
                elif status_code == 429:
                    self.metrics["rate_limit_hits"] += 1

                    # Extraer retry-after si está disponible
                    retry_after = int(response.headers.get("Retry-After", "60"))
                    max_delay = max(retry_after, 600)

                    logger.error(
                        f"[RATE_LIMIT] 429 on {self.backoff._sanitize_endpoint(endpoint)}, retry after {retry_after}s"
                    )
                    self.backoff.record_failure(endpoint, max_delay=max_delay)

                    if return_error_details:
                        error_resp = self._create_error_response(
                            error_type="rate_limit",
                            message=f"Demasiadas solicitudes. Intente nuevamente en {retry_after} segundos.",
                            status_code=429,
                            endpoint=endpoint,
                        )
                        error_resp["retry_after"] = retry_after
                        return error_resp
                    return None

                # 500/502/503/504 Server Errors: Backoff moderado (30s max)
                elif status_code in [500, 502, 503, 504]:
                    logger.warning(
                        f"[SERVER_ERROR] {status_code} en {self.backoff._sanitize_endpoint(endpoint)} - Backoff moderado (30s max)"
                    )
                    self.backoff.record_failure(endpoint, max_delay=30)

                    if return_error_details:
                        return self._create_error_response(
                            error_type="server",
                            message="Error del servidor. Por favor, intente nuevamente en unos momentos.",
                            status_code=status_code,
                            endpoint=endpoint,
                        )
                    return None

                # Otros errores: Backoff moderado por defecto
                else:
                    logger.warning(f"[HTTP_ERROR] {status_code} en {self.backoff._sanitize_endpoint(endpoint)}")
                    self.backoff.record_failure(endpoint, max_delay=30)

                    if return_error_details:
                        return self._create_error_response(
                            error_type="unknown",
                            message=f"Error HTTP {status_code}",
                            status_code=status_code,
                            endpoint=endpoint,
                        )
                    return None

            # Parsear respuesta exitosa
            result = response.json()

            # Almacenar en cache (solo GET exitosos)
            if method == "GET" and not bypass_cache:
                self.cache.set(url, result, params, cache_ttl)

            # Registrar éxito - resetea contador del endpoint
            self.backoff.record_success(endpoint)

            return result

        except requests.exceptions.Timeout:
            logger.warning(
                f"[TIMEOUT] Request to {self.backoff._sanitize_endpoint(endpoint)} timed out - Backoff moderado (30s max)"
            )
            self.backoff.record_failure(endpoint, max_delay=30)

            if return_error_details:
                return self._create_error_response(
                    error_type="timeout",
                    message=f"La operación excedió el tiempo límite de {timeout} segundos.",
                    endpoint=endpoint,
                )
            return None

        except requests.exceptions.RequestException as e:
            logger.warning(f"[REQUEST_ERROR] {self.backoff._sanitize_endpoint(endpoint)}: {e}")
            # Errores de conexión: backoff moderado
            self.backoff.record_failure(endpoint, max_delay=30)

            if return_error_details:
                return self._create_error_response(
                    error_type="connection", message=f"Error de conexión: {str(e)}", endpoint=endpoint
                )
            return None

        except ValueError as e:
            logger.warning(f"[JSON_ERROR] {self.backoff._sanitize_endpoint(endpoint)}: {e}")
            # Error de parseo JSON: NO backoff (es bug de backend)

            if return_error_details:
                return self._create_error_response(
                    error_type="validation", message="Error al procesar la respuesta del servidor.", endpoint=endpoint
                )
            return None

    def get_health_status(self, detailed: bool = False) -> Optional[Dict]:
        """Obtiene estado de salud del sistema con cache optimizado"""
        endpoint = "/health" if detailed else "/health/simple"
        cache_ttl = 15 if detailed else 30  # Menos cache para health detallado

        return self.make_request(endpoint, cache_ttl=cache_ttl)

    def get_system_status(self) -> Optional[Dict]:
        """Obtiene estado del sistema con cache extendido"""
        return self.make_request("/api/v1/system/status", cache_ttl=45)

    def get_catalog_status(self) -> Optional[Dict]:
        """Obtiene estado del catálogo con cache de 2 minutos usando endpoint unificado"""
        return self.make_request("/api/v1/system-unified/status-complete", cache_ttl=120)

    def get_dashboard_data(self, pharmacy_id: str) -> Optional[Dict]:
        """Obtiene datos del dashboard con cache de 1 minuto"""
        return self.make_request(f"/api/v1/pharmacies/{pharmacy_id}/dashboard", cache_ttl=60)

    def get_metrics(self) -> Dict[str, Any]:
        """Obtiene métricas del coordinador"""
        cache_hit_rate = (self.metrics["cache_hits"] / max(1, self.metrics["total_requests"])) * 100

        return {
            "total_requests": self.metrics["total_requests"],
            "cache_hits": self.metrics["cache_hits"],
            "cache_hit_rate": f"{cache_hit_rate:.1f}%",
            "rate_limit_hits": self.metrics["rate_limit_hits"],
            "backoff_waits": self.metrics["backoff_waits"],
            "cache_entries": len(self.cache.cache),
        }

    def cleanup_cache(self):
        """Limpia cache expirado"""
        self.cache.clear_expired()

    def get_and_clear_401_error(self) -> Optional[Dict[str, Any]]:
        """
        Obtiene y limpia el último error 401 detectado.
        Usado por auth guard para detectar cuando se necesita redirección a login.

        Returns:
            Dict con info del error 401 si existe, None en caso contrario
        """
        with self.lock:
            if self.last_401_error:
                error = self.last_401_error.copy()
                # Limpiar después de leer
                self.last_401_error = None
                self.last_401_timestamp = None
                return error
            return None


# Instancia global del coordinador
request_coordinator = RequestCoordinator()


# Funciones helper para usar en callbacks
def get_health_status(detailed: bool = False) -> Optional[Dict]:
    """Helper function para obtener health status"""
    return request_coordinator.get_health_status(detailed)


def get_system_status() -> Optional[Dict]:
    """Helper function para obtener system status"""
    return request_coordinator.get_system_status()


def get_catalog_status() -> Optional[Dict]:
    """Helper function para obtener catalog status"""
    return request_coordinator.get_catalog_status()


def get_dashboard_data(pharmacy_id: str) -> Optional[Dict]:
    """Helper function para obtener dashboard data"""
    return request_coordinator.get_dashboard_data(pharmacy_id)


def get_coordinator_metrics() -> Dict[str, Any]:
    """Helper function para obtener métricas"""
    return request_coordinator.get_metrics()


def cleanup_cache():
    """Helper function para limpiar cache"""
    request_coordinator.cleanup_cache()
