# frontend/utils/pharmacy_context.py
"""
Context global para manejo de farmacia del usuario autenticado.
Diseñado para sistema multi-usuario donde cada usuario = una farmacia.

SEGURIDAD (Issue #189):
- En producción: FAIL-FAST si usuario autenticado sin pharmacy_id
- En producción: FAIL-FAST si intenta acceder sin autenticación
- En desarrollo: Permite fallbacks para testing
"""

import functools
import logging
import os
from dataclasses import dataclass
from datetime import datetime
from typing import Callable, List, Optional

from dash import no_update

logger = logging.getLogger(__name__)


@dataclass
class PharmacyInfo:
    """Información de una farmacia"""

    id: str
    name: str
    code: str
    city: str
    sales_records: int
    enriched_records: int
    enrichment_rate: float


class PharmacyContext:
    """
    Context global para manejo de farmacia del usuario autenticado.

    Funcionalidades:
    - Producción: Farmacia basada en usuario autenticado (1 user = 1 pharmacy)
    - Desarrollo: Permite fallbacks para testing sin autenticación

    SEGURIDAD (Issue #189):
    - En producción: FAIL-FAST con ValueError si usuario autenticado sin pharmacy_id
    - En producción: FAIL-FAST con ValueError si intenta acceder sin autenticación
    - En desarrollo: Permite fallbacks para flexibilidad de testing
    """

    def __init__(self):
        self._current_pharmacy: Optional[PharmacyInfo] = None
        self._available_pharmacies: List[PharmacyInfo] = []
        self._last_refresh: Optional[datetime] = None
        self._refresh_interval = 300  # 5 minutos

        # Caché para pharmacy_id (Issue #254, PR #289 - Mejoras)
        self._pharmacy_id_cache: Optional[str] = None
        self._cache_token_hash: Optional[str] = None

    def _is_production(self) -> bool:
        """
        Detecta si estamos en entorno de producción.

        Returns:
            True si ENVIRONMENT != "development"
        """
        environment = os.getenv("ENVIRONMENT", "production").lower()
        return environment != "development"

    def _get_token_hash(self, token: Optional[str]) -> Optional[str]:
        """
        Genera hash simple del token para usar como cache key.

        Args:
            token: JWT token string

        Returns:
            Hash del token o None si no hay token
        """
        if not token:
            return None

        # Hash simple usando hash() built-in de Python
        # Suficiente para cache key (no necesitamos hash criptográfico)
        return str(hash(token))

    def clear_pharmacy_id_cache(self) -> None:
        """
        Limpia el caché de pharmacy_id.

        Llamar cuando:
        - Usuario hace logout
        - Token JWT expira/se refresca
        - Cambio de usuario
        """
        self._pharmacy_id_cache = None
        self._cache_token_hash = None
        logger.debug("Pharmacy ID cache cleared")

    def get_current_pharmacy_id(self) -> str:
        """
        Obtener ID de la farmacia activa con caché inteligente.

        SEGURIDAD (Issue #189):
        - FAIL-FAST en producción y desarrollo si no hay autenticación válida
        - NO permite fallbacks (removido para debugging claro)

        CACHÉ (PR #289 - Mejoras):
        - Cachea pharmacy_id basado en hash del JWT token
        - Se invalida automáticamente cuando cambia el token
        - Reduce parsing del JWT en cada llamada

        Returns:
            pharmacy_id de la farmacia activa

        Raises:
            ValueError: Si no hay autenticación válida con pharmacy_id (producción Y desarrollo)
        """
        is_production = self._is_production()
        from utils.auth import auth_manager

        # 1. Verificar caché si hay token
        try:
            current_token = auth_manager.get_access_token()
            current_token_hash = self._get_token_hash(current_token)

            # Cache hit: mismo token y pharmacy_id cacheado
            if current_token_hash and current_token_hash == self._cache_token_hash and self._pharmacy_id_cache:
                logger.debug(f"✅ Cache hit: pharmacy_id={self._pharmacy_id_cache}")
                return self._pharmacy_id_cache

        except Exception as e:
            # Error obteniendo token - continuar sin caché
            logger.debug(f"Error obteniendo token para caché: {e}")
            current_token_hash = None

        # 2. Cache miss o token cambió - obtener pharmacy_id del usuario autenticado
        if auth_manager.is_authenticated():
            user_info = auth_manager.get_current_user()

            if user_info and user_info.get("pharmacy_id"):
                pharmacy_id = str(user_info["pharmacy_id"])

                # Cachear resultado
                self._pharmacy_id_cache = pharmacy_id
                self._cache_token_hash = current_token_hash

                logger.info(f"✅ Usando pharmacy_id del usuario autenticado: {pharmacy_id} (cached)")
                return pharmacy_id
            else:
                # Usuario autenticado pero sin pharmacy_id
                error_msg = (
                    "SEGURIDAD: Usuario autenticado sin pharmacy_id. "
                    "Cada usuario debe tener una farmacia asignada (1 user = 1 pharmacy). "
                    "Contacte al administrador para asignar una farmacia."
                )
                # ❌ FAIL-FAST siempre (producción Y desarrollo)
                logger.error(f"🚨 {error_msg}")
                raise ValueError(error_msg)
        else:
            # Sin autenticación
            error_msg = (
                "SEGURIDAD: Acceso sin autenticación. "
                "Todos los usuarios deben autenticarse con JWT token. "
                "Por favor, inicie sesión para continuar."
            )
            # ❌ FAIL-FAST siempre (producción Y desarrollo)
            logger.error(f"🚨 {error_msg}")
            raise ValueError(error_msg)

    def get_current_pharmacy_info(self) -> Optional[PharmacyInfo]:
        """Obtener información completa de la farmacia activa"""
        if not self._current_pharmacy or self._needs_refresh():
            self._refresh_pharmacies()

        return self._current_pharmacy

    def get_available_pharmacies(self) -> List[PharmacyInfo]:
        """Obtener todas las farmacias disponibles"""
        if not self._available_pharmacies or self._needs_refresh():
            self._refresh_pharmacies()

        return self._available_pharmacies.copy()

    def set_user_pharmacy(self, user_id: str, pharmacy_id: str) -> bool:
        """
        Establecer la farmacia del usuario autenticado (para futuro con auth).

        Args:
            user_id: ID del usuario autenticado
            pharmacy_id: ID de la farmacia del usuario

        Returns:
            True si el establecimiento fue exitoso
        """
        # TODO: En el futuro, verificar que user_id tiene acceso a pharmacy_id
        if not self._available_pharmacies:
            self._refresh_pharmacies()

        for pharmacy in self._available_pharmacies:
            if pharmacy.id == pharmacy_id:
                self._current_pharmacy = pharmacy
                self._save_to_session(pharmacy_id)
                logger.info(f"Farmacia del usuario {user_id}: {pharmacy.name} ({pharmacy_id})")
                return True

        logger.warning(f"Farmacia no encontrada para usuario {user_id}: {pharmacy_id}")
        return False

    def _refresh_pharmacies(self) -> None:
        """
        Refrescar lista de farmacias desde API.

        Raises:
            Exception: Si no se pueden cargar las farmacias (sin fallback)
        """
        from utils.api_client import api_client

        response = api_client.get_available_pharmacies()
        if response and "pharmacies" in response:
            # Convertir a objetos PharmacyInfo
            self._available_pharmacies = [
                PharmacyInfo(
                    id=p["id"],
                    name=p["name"],
                    code=p["code"],
                    city=p.get("city", ""),
                    sales_records=p["sales_records"],
                    enriched_records=p["enriched_records"],
                    enrichment_rate=p["enrichment_rate"],
                )
                for p in response["pharmacies"]
            ]

            # Establecer farmacia por defecto (temporal, requiere autenticación)
            if self._available_pharmacies and not self._current_pharmacy:
                saved_id = self._load_from_session()

                # Intentar usar la guardada en sesión
                if saved_id:
                    for pharmacy in self._available_pharmacies:
                        if pharmacy.id == saved_id:
                            self._current_pharmacy = pharmacy
                            break

                # Si no hay guardada, usar la primera con más datos (temporal para desarrollo)
                if not self._current_pharmacy:
                    self._current_pharmacy = max(self._available_pharmacies, key=lambda p: p.enriched_records)
                    self._save_to_session(self._current_pharmacy.id)
                    logger.info(f"🚧 DESARROLLO: Usando farmacia por defecto: {self._current_pharmacy.name}")

            self._last_refresh = datetime.now()
            logger.info(
                f"Farmacias cargadas: {len(self._available_pharmacies)}, activa: {self._current_pharmacy.name if self._current_pharmacy else 'None'}"
            )

        else:
            error_msg = "No se pudieron cargar farmacias desde API - backend no disponible"
            logger.error(f"🚨 {error_msg}")
            raise ConnectionError(error_msg)

    def _needs_refresh(self) -> bool:
        """Verificar si necesita refrescar datos"""
        if not self._last_refresh:
            return True
        return (datetime.now() - self._last_refresh).seconds > self._refresh_interval

    def _save_to_session(self, pharmacy_id: str) -> None:
        """Guardar farmacia activa en almacenamiento (para futuro)"""
        # Por ahora solo log, en futuro usar session storage
        logger.debug(f"Guardando farmacia en sesión: {pharmacy_id}")

    def _load_from_session(self) -> Optional[str]:
        """Cargar farmacia activa desde almacenamiento (para futuro)"""
        # Por ahora retorna None, en futuro leer desde session storage
        return None


# Instancia global del contexto
pharmacy_context = PharmacyContext()


# === DECORATOR PARA CALLBACKS ===


def require_pharmacy_id(callback_func: Callable) -> Callable:
    """
    Decorator que inyecta pharmacy_id al callback y maneja errores automáticamente.

    Este decorator elimina código duplicado en callbacks que necesitan obtener
    el pharmacy_id del usuario autenticado (Issue #254, PR #289).

    **Antes (código duplicado en cada callback):**
    ```python
    @app.callback(...)
    def my_callback(...):
        try:
            pharmacy_id = get_current_pharmacy_id()
        except ValueError as e:
            logger.error(f"Error obteniendo pharmacy_id: {str(e)}")
            return {"error": "No se pudo identificar la farmacia"}

        # Lógica del callback usando pharmacy_id
        response = api_client.get(f"/api/v1/endpoint/{pharmacy_id}")
        return response
    ```

    **Después (con decorator):**
    ```python
    @app.callback(...)
    @require_pharmacy_id
    def my_callback(..., pharmacy_id=None):
        # pharmacy_id ya está disponible, validado y seguro
        response = api_client.get(f"/api/v1/endpoint/{pharmacy_id}")
        return response
    ```

    Beneficios:
    - ✅ Elimina ~40 líneas de código duplicado en callbacks
    - ✅ Manejo consistente de errores en todos los callbacks
    - ✅ Logging centralizado con contexto del callback
    - ✅ Retorna no_update automáticamente si falla

    Args:
        callback_func: Función callback de Dash que necesita pharmacy_id

    Returns:
        Función wrapper que inyecta pharmacy_id como parámetro

    Example:
        ```python
        @app.callback(
            Output('data-store', 'data'),
            Input('trigger', 'n_clicks')
        )
        @require_pharmacy_id
        def load_pharmacy_data(n_clicks, pharmacy_id=None):
            # pharmacy_id ya disponible sin try-except
            data = api_client.get(f"/api/v1/data/{pharmacy_id}")
            return data
        ```

    Note:
        - El callback DEBE tener parámetro `pharmacy_id=None` en su firma
        - Si falla, retorna `no_update` (no rompe el callback)
        - Logging automático del error con nombre del callback
    """

    @functools.wraps(callback_func)
    def wrapper(*args, **kwargs):
        try:
            # Obtener pharmacy_id del usuario autenticado
            pharmacy_id = get_current_pharmacy_id()

            # Inyectar pharmacy_id como kwarg al callback
            kwargs["pharmacy_id"] = pharmacy_id

            # Ejecutar callback original con pharmacy_id inyectado
            return callback_func(*args, **kwargs)

        except ValueError as e:
            # Error obteniendo pharmacy_id (usuario no autenticado o sin farmacia)
            logger.error(
                f"Callback '{callback_func.__name__}' falló: Error obteniendo pharmacy_id",
                extra={"callback": callback_func.__name__, "error": str(e), "error_type": "pharmacy_id_not_available"},
            )
            # Retornar no_update para no romper el callback
            return no_update

        except Exception as e:
            # Error inesperado en el callback
            logger.error(
                f"Callback '{callback_func.__name__}' falló: Error inesperado",
                extra={"callback": callback_func.__name__, "error": str(e), "error_type": type(e).__name__},
            )
            return no_update

    return wrapper


# === FUNCIONES DE CONVENIENCIA ===


def get_current_pharmacy_id() -> str:
    """
    Obtiene el pharmacy_id del usuario autenticado desde el JWT token.

    Esta función es el punto central para obtener el pharmacy_id en todos los
    callbacks del frontend, asegurando la arquitectura multi-tenant correcta
    (1 user = 1 pharmacy, ver REGLA #10 en CLAUDE.md).

    **Flujo de Autenticación:**
    1. Usuario hace login → Backend retorna JWT con `pharmacy_id`
    2. Frontend almacena JWT en sesión
    3. Esta función extrae `pharmacy_id` del JWT decodificado
    4. Callbacks usan este ID para hacer requests al backend

    **Arquitectura Multi-Tenant (Issue #254, PR #289):**
    - Cada usuario tiene exactamente UNA farmacia asignada
    - Relación 1:1 enforced por constraint UNIQUE en User.pharmacy_id
    - El pharmacy_id viene del JWT (no es configurable por el usuario)
    - Backend valida que el pharmacy_id del request coincida con el del JWT

    **Seguridad:**
    - En PRODUCCIÓN: FAIL-FAST si no hay autenticación o pharmacy_id
    - En DESARROLLO: Permite fallback a farmacia demo para testing
    - NO confiar solo en esta función: backend SIEMPRE valida permisos

    Returns:
        str: UUID del pharmacy_id asociado al usuario autenticado

    Raises:
        ValueError: Si el usuario no está autenticado (producción)
        ValueError: Si el usuario no tiene pharmacy_id asociado (producción)

    Example:
        ```python
        # En un callback Dash
        @app.callback(...)
        def my_callback(...):
            pharmacy_id = get_current_pharmacy_id()
            response = api_client.get(f"/api/v1/data/{pharmacy_id}")
            return response

        # Con decorator (recomendado)
        @app.callback(...)
        @require_pharmacy_id
        def my_callback(..., pharmacy_id=None):
            # pharmacy_id ya inyectado automáticamente
            response = api_client.get(f"/api/v1/data/{pharmacy_id}")
            return response
        ```

    Note:
        - Thread-safe: Puede ser llamada desde múltiples callbacks simultáneamente
        - CON caché interno: Cachea pharmacy_id basado en hash del JWT token
        - Caché se invalida automáticamente cuando cambia el token
        - Fallback solo en desarrollo: producción siempre falla explícitamente
        - Reemplaza todos los usos de PHARMACY_ID_DEFAULT o IDs hardcodeados

    Performance:
        - Primera llamada: ~5-10ms (parsea JWT y obtiene user_info)
        - Llamadas subsecuentes (cache hit): ~0.1ms (retorna desde caché)
        - Caché se invalida automáticamente al cambiar token (logout, refresh)

    See Also:
        - `require_pharmacy_id`: Decorator para inyectar pharmacy_id en callbacks
        - `clear_pharmacy_id_cache()`: Limpiar caché manualmente
        - `backend/app/api/deps.py:verify_pharmacy_access()`: Validación backend
        - `CLAUDE.md`: REGLA #10 (Relación 1:1 User-Pharmacy)
    """
    return pharmacy_context.get_current_pharmacy_id()


def get_current_pharmacy_info() -> Optional[PharmacyInfo]:
    """Obtener información completa de la farmacia activa"""
    return pharmacy_context.get_current_pharmacy_info()


def get_available_pharmacies() -> List[PharmacyInfo]:
    """Obtener todas las farmacias disponibles"""
    return pharmacy_context.get_available_pharmacies()


def set_user_pharmacy(user_id: str, pharmacy_id: str) -> bool:
    """Establecer farmacia del usuario autenticado (1 user = 1 pharmacy)"""
    return pharmacy_context.set_user_pharmacy(user_id, pharmacy_id)


def clear_pharmacy_id_cache() -> None:
    """
    Limpia el caché de pharmacy_id.

    Llamar esta función cuando:
    - Usuario hace logout
    - Token JWT expira o se refresca
    - Cambio de usuario en la sesión

    Example:
        ```python
        # En el callback de logout
        @app.callback(...)
        def logout_user(...):
            auth_manager.logout()
            clear_pharmacy_id_cache()  # Limpiar caché
            return redirect('/')  # Redirigir a landing
        ```
    """
    pharmacy_context.clear_pharmacy_id_cache()
