"""
Layout Validator - xFarma

Valida que todos los IDs de componentes requeridos por callbacks
existan en el layout ANTES de registrar callbacks.

Previene race conditions y ReferenceErrors en producción.

Issue: /generics muestra ReferenceErrors para componentes faltantes en Render
Root cause: Layout modificado después de asignación, callbacks registrados antes de layout completo
"""

import logging
from typing import Set

from dash.development.base_component import Component

logger = logging.getLogger(__name__)

# IDs críticos que DEBEN estar en el layout global (skeleton)
# Estos IDs son referenciados en callbacks de múltiples páginas
CRITICAL_IDS = [
    # ===================================================================
    # GENERICS PAGE - Componentes críticos para análisis de partners
    # ===================================================================
    "refresh-partners-list-btn",
    "partner-discount-slider",
    "partners-dropdown",
    "context-total-substitutable",
    "homogeneous-matrix-container",
    "sales-drill-down-chart",
    "units-drill-down-chart",
    "partner-references-container",
    "partner-refs-count",
    "partner-refs-sales",
    "partner-refs-search",
    "homogeneous-search-input",
    # ===================================================================
    # UPLOAD PAGE - Componentes de subida y progreso
    # ===================================================================
    "upload-data",
    "upload-progress-interval",
    "upload-progress-container",
    "upload-history-container",
    "file-preview-container",
    "upload-system-status",
    # ===================================================================
    # DASHBOARD PAGE - Componentes de métricas y KPIs
    # ===================================================================
    "dashboard-update-interval",
    "sales-trend-chart",
    "kpis-row",
    # ===================================================================
    # AUTH & GLOBAL - Componentes de autenticación y stores globales
    # ===================================================================
    "auth-state",
    "auth-tokens-store",
    "context-store",
    "laboratory-cache-store",
    "partners-selection-store",
    "analysis-store",
]


def extract_all_ids(component: Component) -> Set[str]:
    """
    Extrae recursivamente todos los IDs de un componente Dash y sus hijos.

    Args:
        component: Componente Dash (puede ser html.Div, dbc.Card, etc.)

    Returns:
        Set de todos los IDs encontrados en el árbol de componentes
    """
    ids = set()

    # Caso base: componente tiene ID
    if hasattr(component, "id") and component.id:
        # Pattern-matching callbacks usan dict como ID (ej: {"type": "btn", "index": 1})
        # Solo agregar IDs tipo string (IDs regulares)
        if isinstance(component.id, str):
            ids.add(component.id)
        # Pattern-matching IDs (dict) se ignoran en esta validación

    # Caso recursivo: componente tiene children
    if hasattr(component, "children"):
        children = component.children

        # Children puede ser: None, Component, lista, o string
        if children is None or isinstance(children, str):
            pass
        elif isinstance(children, Component):
            ids.update(extract_all_ids(children))
        elif isinstance(children, list):
            for child in children:
                if isinstance(child, Component):
                    ids.update(extract_all_ids(child))

    return ids


def validate_layout_completeness(layout: Component) -> None:
    """
    Valida que todos los IDs críticos estén presentes en el layout.

    Esta función DEBE ejecutarse DESPUÉS de construir el layout completo
    pero ANTES de registrar callbacks.

    Raises:
        RuntimeError: Si faltan IDs críticos en el layout

    Example:
        >>> base_layout = html.Div([...])
        >>> base_layout = add_catalog_ux_components(base_layout)
        >>> validate_layout_completeness(base_layout)  # Validar ANTES de asignar
        >>> app.layout = base_layout
        >>> register_all_callbacks(app)  # Registrar DESPUÉS de validar
    """
    logger.info("[LAYOUT_VALIDATOR] Starting layout validation...")

    # Extraer todos los IDs del layout actual
    actual_ids = extract_all_ids(layout)
    logger.info(f"[LAYOUT_VALIDATOR] Found {len(actual_ids)} component IDs in layout")

    # Detectar IDs faltantes
    missing_ids = set(CRITICAL_IDS) - actual_ids

    if missing_ids:
        error_msg = (
            f"\n{'='*70}\n"
            f"❌ LAYOUT VALIDATION FAILED\n"
            f"{'='*70}\n"
            f"Missing {len(missing_ids)} critical component IDs in layout:\n\n"
            f"{chr(10).join(sorted(missing_ids))}\n\n"
            f"FIX: Add these IDs to skeleton in frontend/app.py (líneas 108-410)\n"
            f"or ensure they are added by add_catalog_ux_components()\n"
            f"{'='*70}\n"
        )
        logger.error(error_msg)
        raise RuntimeError(error_msg)

    # Validación exitosa
    logger.info(f"[LAYOUT_VALIDATOR] ✅ Layout validation passed: {len(CRITICAL_IDS)} critical IDs present")
    logger.debug(f"[LAYOUT_VALIDATOR] Present IDs: {sorted(actual_ids)}")


def validate_callback_ids(app) -> None:
    """
    Valida que todos los IDs usados en callbacks existan en el layout.

    Esta es una validación más exhaustiva que se puede ejecutar en tests.

    Args:
        app: Instancia de la aplicación Dash

    Raises:
        AssertionError: Si hay IDs en callbacks que no existen en layout
    """
    # Extraer IDs del layout
    layout_ids = extract_all_ids(app.layout)

    # Extraer IDs de callbacks (todos los Input/Output/State)
    callback_ids = set()
    for callback in app.callback_map.values():
        # Inputs
        if hasattr(callback, "inputs"):
            for inp in callback.inputs:
                callback_ids.add(inp.component_id)
        # Outputs
        if hasattr(callback, "outputs"):
            for out in callback.outputs:
                callback_ids.add(out.component_id)
        # States
        if hasattr(callback, "state"):
            for state in callback.state:
                callback_ids.add(state.component_id)

    # Detectar IDs en callbacks pero no en layout
    missing_in_layout = callback_ids - layout_ids

    if missing_in_layout:
        error_msg = (
            f"\n{'='*70}\n"
            f"❌ CALLBACK VALIDATION FAILED\n"
            f"{'='*70}\n"
            f"IDs used in callbacks but missing in layout:\n\n"
            f"{chr(10).join(sorted(missing_in_layout))}\n\n"
            f"These IDs must be added to the skeleton in frontend/app.py\n"
            f"{'='*70}\n"
        )
        logger.error(error_msg)
        raise AssertionError(error_msg)

    logger.info(f"[LAYOUT_VALIDATOR] ✅ Callback validation passed: {len(callback_ids)} callback IDs present in layout")
