# frontend/utils/laboratory_mapping.py
"""
Utilidad frontend para mapeo de códigos de laboratorio ↔ nombres
Maneja la conversión entre códigos (backend) y nombres (frontend UI)
"""

import logging
import os
import time
from typing import Dict, List



logger = logging.getLogger(__name__)


class LaboratoryMapper:
    """
    Clase para manejar conversión códigos ↔ nombres de laboratorios

    **Uso típico:**

    ```python
    mapper = LaboratoryMapper()

    # Obtener nombres para mostrar en UI
    codes = ["111", "426", "863"]
    names = mapper.codes_to_names(codes)
    # names = {"111": "CINFA S.A.", "426": "NORMON S.A.", ...}

    # Convertir nombres a códigos para enviar al backend
    selected_names = ["CINFA S.A.", "NORMON S.A."]
    codes = mapper.names_to_codes(selected_names)
    # codes = {"CINFA S.A.": "111", "NORMON S.A.": "426"}
    ```

    **IMPORTANTE**: Thread-safe para multi-worker environments (Render production)
    - Cada llamada crea nueva instancia de APIClient
    - No se mantiene estado mutable entre requests
    - Safe para uso en callbacks de Dash multi-proceso
    """

    def __init__(self):
        # CRITICAL FIX: No almacenar APIClient como atributo de instancia
        # Esto evita compartir sesiones entre workers en producción
        # self.api_client = APIClient()  # ELIMINADO - causa HTTP 500 en Render

        # Cache removido - no es thread-safe en multi-worker
        # Los laboratorios se cachean en Redis del backend
        pass

    def _retry_with_backoff(self, func, *args, **kwargs):
        """
        Helper method to retry operations with exponential backoff.
        Particularly important for Render production environment.

        Args:
            func: Function to retry
            *args: Positional arguments for func
            **kwargs: Keyword arguments for func

        Returns:
            Result of func or None if all retries failed
        """
        # Configuración de retry con variables de entorno
        is_render = os.getenv("RENDER") is not None
        max_retries = int(os.getenv("MAPPER_MAX_RETRIES", os.getenv("API_MAX_RETRIES", "3")))
        base_delay = float(os.getenv("MAPPER_BASE_DELAY", os.getenv("API_BASE_RETRY_DELAY", "2" if is_render else "1")))
        retry_multiplier = float(os.getenv("API_RETRY_MULTIPLIER", "2"))

        for attempt in range(max_retries):
            try:
                return func(*args, **kwargs)
            except Exception as e:
                if attempt < max_retries - 1:
                    delay = base_delay * (retry_multiplier**attempt)  # Exponential backoff
                    logger.warning(
                        f"[LABORATORY_MAPPER] Attempt {attempt + 1}/{max_retries} failed: {e}. "
                        f"Retrying in {delay}s..."
                    )
                    time.sleep(delay)
                else:
                    logger.error(f"[LABORATORY_MAPPER] All {max_retries} attempts failed. Last error: {e}")
                    raise

        return None

    def codes_to_names(self, codes: List[str]) -> Dict[str, str]:
        """
        Convierte códigos de laboratorio a nombres para mostrar en UI

        Args:
            codes: Lista de códigos (ej: ["111", "426"])

        Returns:
            Diccionario código → nombre

        Example:
            >>> mapper.codes_to_names(["111", "426"])
            {"111": "CINFA S.A.", "426": "NORMON S.A."}
        """
        if not codes:
            return {}

        def _make_request():
            # ✅ FIX RATE LIMITING: Usar request_coordinator (singleton con mejor auth)
            # en lugar de APIClient() que puede tener token desincronizado
            from utils.request_coordinator import request_coordinator

            # request_coordinator usa params dict directamente (no necesita encoding especial)
            params = {"codes": codes}

            # The laboratory mapping API returns data directly, not wrapped in success/data
            response = request_coordinator.make_request(
                endpoint="/api/v1/laboratory-mapping/codes-to-names",
                method="GET",
                params=params,
                cache_ttl=0,  # No cache aquí (cache persistente maneja esto en callback)
                timeout=30,
                bypass_cache=True,  # Bypass cache del request_coordinator
            )

            # Check if response is a dict (mapping) or empty dict on error
            if isinstance(response, dict) and response:
                logger.info(f"[LABORATORY_MAPPER] Códigos convertidos: {len(response)}")
                return response
            else:
                # Raise exception to trigger retry
                error_msg = "No se pudieron obtener los datos de laboratorios. El servidor puede estar procesando otras solicitudes."
                logger.error(f"[LABORATORY_MAPPER] API returned empty or invalid response: {response}")
                raise ValueError(error_msg)

        try:
            # Use retry wrapper for resilience in production
            return self._retry_with_backoff(_make_request)
        except Exception as e:
            logger.error(f"[LABORATORY_MAPPER] Failed after retries: {e}")
            # Retornar diccionario vacío para mantener la aplicación funcionando
            return {}

    def names_to_codes(self, names: List[str]) -> Dict[str, str]:
        """
        Convierte nombres de laboratorio a códigos para enviar al backend

        Args:
            names: Lista de nombres (ej: ["CINFA S.A.", "NORMON S.A."])

        Returns:
            Diccionario nombre → código

        Note:
            El backend limita requests a 20 nombres por llamada (LaboratoryConfig.MAX_NAMES_PER_REQUEST).
            Esta función implementa batching automático para manejar listas grandes.

        Fix Rate Limiting (Issue #XXX):
            Ahora usa request_coordinator en lugar de APIClient directo para:
            - Mejor manejo de autenticación (singleton global)
            - Prevenir errores 401 por token desincronizado
            - Detectar errores de autenticación y disparar auth guard
        """
        if not names:
            return {}

        # ✅ Validación preventiva: Filtrar nombres vacíos antes de procesar
        valid_names = [name.strip() for name in names if name and name.strip()]

        if not valid_names:
            logger.warning("[LABORATORY_MAPPER] No hay nombres válidos para convertir a códigos")
            return {}

        # BATCHING: Backend limita a 20 nombres por request (MAX_NAMES_PER_REQUEST)
        BATCH_SIZE = 20
        all_mappings = {}

        # Dividir nombres en lotes de 20
        total_batches = (len(valid_names) + BATCH_SIZE - 1) // BATCH_SIZE

        for i in range(0, len(valid_names), BATCH_SIZE):
            batch = valid_names[i:i + BATCH_SIZE]
            batch_num = (i // BATCH_SIZE) + 1

            logger.info(
                f"[LABORATORY_MAPPER] Procesando lote {batch_num}/{total_batches} "
                f"({len(batch)} nombres)"
            )

            def _make_request(current_batch=batch, current_batch_num=batch_num):
                # ✅ FIX RATE LIMITING: Usar request_coordinator (singleton con mejor auth)
                # en lugar de APIClient() que puede tener token desincronizado
                from utils.request_coordinator import request_coordinator

                # request_coordinator usa params dict directamente (no necesita encoding especial)
                params = {"names": current_batch}  # Lote de máximo 20 nombres

                # The laboratory mapping API returns data directly, not wrapped in success/data
                response = request_coordinator.make_request(
                    endpoint="/api/v1/laboratory-mapping/names-to-codes",
                    method="GET",
                    params=params,
                    cache_ttl=0,  # No cache aquí (cache persistente maneja esto en callback)
                    timeout=30,  # Timeout generoso para lotes grandes
                    bypass_cache=True,  # Bypass cache del request_coordinator
                )

                # Check if response is a dict (mapping) or empty dict on error
                if isinstance(response, dict) and response:
                    logger.info(
                        f"[LABORATORY_MAPPER] Lote {current_batch_num}/{total_batches} convertido: "
                        f"{len(response)} mappings"
                    )
                    return response
                else:
                    # Raise exception to trigger retry
                    error_msg = f"No se pudieron obtener los datos de laboratorios (lote {current_batch_num}). El servidor puede estar procesando otras solicitudes."
                    logger.error(f"[LABORATORY_MAPPER] API returned empty or invalid response: {response}")
                    raise ValueError(error_msg)

            try:
                # Use retry wrapper for resilience in production
                batch_mappings = self._retry_with_backoff(_make_request)
                all_mappings.update(batch_mappings)
            except Exception as e:
                logger.error(
                    f"[LABORATORY_MAPPER] Lote {batch_num}/{total_batches} failed after retries: {e}"
                )
                # Continuar con siguiente lote (no fallar toda la operación)

        logger.info(
            f"[LABORATORY_MAPPER] Total nombres convertidos: {len(all_mappings)}/{len(valid_names)}"
        )
        return all_mappings

    def get_generic_laboratories(self) -> Dict[str, str]:
        """
        Obtiene mapeo de laboratorios genéricos principales
        Útil para dropdowns/selectors de partners

        Returns:
            Diccionario código → nombre de laboratorios genéricos
        """

        def _make_request():
            # ✅ FIX RATE LIMITING: Usar request_coordinator (singleton con mejor auth)
            # en lugar de APIClient() que puede tener token desincronizado
            from utils.request_coordinator import request_coordinator

            # The laboratory mapping API returns data directly, not wrapped in success/data
            response = request_coordinator.make_request(
                endpoint="/api/v1/laboratory-mapping/generic-laboratories",
                method="GET",
                cache_ttl=0,  # No cache aquí
                timeout=30,
                bypass_cache=True,  # Bypass cache del request_coordinator
            )

            # Check if response is a dict (mapping) or empty dict on error
            if isinstance(response, dict) and response:
                logger.info(f"[LABORATORY_MAPPER] Laboratorios genéricos: {len(response)}")
                return response
            else:
                # Raise exception to trigger retry
                error_msg = "No se pudieron obtener los datos de laboratorios. El servidor puede estar procesando otras solicitudes."
                logger.error(f"[LABORATORY_MAPPER] API returned empty or invalid response: {response}")
                raise ValueError(error_msg)

        try:
            # Use retry wrapper for resilience in production
            return self._retry_with_backoff(_make_request)
        except Exception as e:
            logger.error(f"[LABORATORY_MAPPER] Failed after retries: {e}")
            # Retornar diccionario vacío para mantener la aplicación funcionando
            return {}

    def get_display_names_for_codes(self, codes: List[str]) -> List[str]:
        """
        Helper para obtener nombres para mostrar desde códigos

        Args:
            codes: Códigos de laboratorio

        Returns:
            Lista de nombres para mostrar (mantiene orden)
        """
        if not codes:
            return []

        mapping = self.codes_to_names(codes)
        return [mapping.get(code, f"Código {code}") for code in codes]

    def get_codes_for_names(self, names: List[str]) -> List[str]:
        """
        Helper para obtener códigos desde nombres seleccionados

        Args:
            names: Nombres de laboratorio seleccionados

        Returns:
            Lista de códigos para enviar al backend
        """
        if not names:
            return []

        mapping = self.names_to_codes(names)
        return [mapping.get(name) for name in names if mapping.get(name)]


# CRITICAL FIX: Cambio de patrón singleton a factory function
# En multi-worker environments (Render production), el singleton global
# causa HTTP 500 errors por compartir sesiones entre procesos.
# Solución: Cada función crea su propia instancia thread-safe.


def _get_mapper() -> LaboratoryMapper:
    """
    Factory function thread-safe para obtener mapper.
    Crea nueva instancia en cada llamada para evitar compartir estado.
    """
    return LaboratoryMapper()


# Funciones de conveniencia para uso directo
def codes_to_display_names(codes: List[str]) -> List[str]:
    """
    Función de conveniencia: códigos → nombres para mostrar
    Thread-safe: Crea nueva instancia para cada llamada.

    Usage:
        display_names = codes_to_display_names(["111", "426"])
        # ["CINFA S.A.", "NORMON S.A."]
    """
    mapper = _get_mapper()
    return mapper.get_display_names_for_codes(codes)


def selected_names_to_codes(names: List[str]) -> List[str]:
    """
    Función de conveniencia: nombres seleccionados → códigos para backend
    Thread-safe: Crea nueva instancia para cada llamada.

    Usage:
        codes = selected_names_to_codes(["CINFA S.A.", "NORMON S.A."])
        # ["111", "426"]
    """
    mapper = _get_mapper()
    return mapper.get_codes_for_names(names)


def get_generic_lab_options() -> List[Dict[str, str]]:
    """
    Función de conveniencia: opciones para dropdown de laboratorios genéricos
    Thread-safe: Crea nueva instancia para cada llamada.

    Returns:
        Lista en formato [{"label": "CINFA S.A.", "value": "111"}, ...]
    """
    mapper = _get_mapper()
    mapping = mapper.get_generic_laboratories()
    return [{"label": name, "value": code} for code, name in mapping.items()]
