# frontend/utils/laboratory_cache_store.py
"""
Cache persistente para códigos de laboratorio usando dcc.Store.

Este módulo resuelve el problema de rate limiting con el endpoint
/api/v1/laboratory-mapping/names-to-codes implementando un cache
que persiste entre callbacks y se comparte en la sesión del usuario.

Issue: Rate limiting excedido (500 req/hora) en producción después de Ctrl+F5
Solución: Cache persistente en dcc.Store con TTL de 7 días (datos estáticos)
"""

import logging
from datetime import datetime, timedelta, timezone
from typing import Dict, List, Optional, Tuple

logger = logging.getLogger(__name__)

# Cache TTL: 7 días (códigos de laboratorio son datos estáticos de catálogo)
# Los nombres → códigos NO cambian frecuentemente (solo con nuevos laboratorios)
# Aumentado de 1h → 7d para prevenir rate limiting en Render después de Ctrl+F5
CACHE_TTL_HOURS = 24 * 7  # 7 días


class LaboratoryCacheStore:
    """
    Cache manager para códigos de laboratorio que persiste en dcc.Store.

    Esta clase gestiona un cache de mapeos nombre→código que:
    1. Persiste entre callbacks usando dcc.Store
    2. Reduce llamadas al API (previene rate limiting)
    3. Tiene TTL de 1 hora para datos frescos
    4. Se invalida automáticamente cuando expira
    """

    @staticmethod
    def get_from_cache(
        names: List[str],
        cache_store: Optional[Dict]
    ) -> Tuple[Dict[str, str], List[str]]:
        """
        Obtiene códigos del cache para nombres dados.

        Args:
            names: Lista de nombres de laboratorio a buscar
            cache_store: Cache actual del dcc.Store

        Returns:
            Tuple[Dict[str, str], List[str]]:
                - Diccionario con mapeos encontrados en cache
                - Lista de nombres NO encontrados en cache
        """
        if not cache_store or not names:
            return {}, names

        # Verificar si el cache ha expirado
        if LaboratoryCacheStore._is_cache_expired(cache_store):
            logger.info("[LAB_CACHE] Cache expirado, invalidando")
            return {}, names

        cached_mappings = cache_store.get("mappings", {})
        found = {}
        not_found = []

        for name in names:
            if name in cached_mappings:
                found[name] = cached_mappings[name]
            else:
                not_found.append(name)

        # Log a nivel INFO para monitoreo en producción (Issue rate limiting 2025-12-01)
        hit_rate = (len(found) / len(names) * 100) if names else 0
        logger.info(
            f"[LAB_CACHE] Cache lookup: {len(found)}/{len(names)} hits ({hit_rate:.0f}%), "
            f"{len(not_found)} API calls needed"
        )

        return found, not_found

    @staticmethod
    def update_cache(
        new_mappings: Dict[str, str],
        current_cache: Optional[Dict]
    ) -> Dict:
        """
        Actualiza el cache con nuevos mapeos.

        Args:
            new_mappings: Nuevos mapeos nombre→código a agregar
            current_cache: Cache actual del dcc.Store

        Returns:
            Cache actualizado para guardar en dcc.Store
        """
        if current_cache and not LaboratoryCacheStore._is_cache_expired(current_cache):
            # Merge con cache existente
            existing_mappings = current_cache.get("mappings", {})
            updated_mappings = {**existing_mappings, **new_mappings}
        else:
            # Cache nuevo o expirado
            updated_mappings = new_mappings

        updated_cache = {
            "mappings": updated_mappings,
            "timestamp": datetime.now(timezone.utc).isoformat(),
            "ttl_hours": CACHE_TTL_HOURS,
            "size": len(updated_mappings)
        }

        logger.info(
            f"[LAB_CACHE] Cache actualizado: {len(new_mappings)} nuevos, "
            f"{updated_cache['size']} total"
        )

        return updated_cache

    @staticmethod
    def _is_cache_expired(cache_store: Dict) -> bool:
        """
        Verifica si el cache ha expirado.

        Args:
            cache_store: Cache del dcc.Store

        Returns:
            True si el cache ha expirado
        """
        if not cache_store or "timestamp" not in cache_store:
            return True

        try:
            cache_time = datetime.fromisoformat(cache_store["timestamp"])
            ttl_hours = cache_store.get("ttl_hours", CACHE_TTL_HOURS)
            expiry_time = cache_time + timedelta(hours=ttl_hours)

            is_expired = datetime.now(timezone.utc) > expiry_time

            if is_expired:
                logger.debug(
                    f"[LAB_CACHE] Cache expirado: creado {cache_time.isoformat()}, "
                    f"TTL {ttl_hours}h"
                )

            return is_expired

        except (ValueError, TypeError) as e:
            logger.error(f"[LAB_CACHE] Error verificando expiración: {e}")
            return True

    @staticmethod
    def get_cache_stats(cache_store: Optional[Dict]) -> Dict[str, any]:
        """
        Obtiene estadísticas del cache para debugging.

        Args:
            cache_store: Cache del dcc.Store

        Returns:
            Diccionario con estadísticas del cache
        """
        if not cache_store:
            return {"status": "empty", "size": 0}

        is_expired = LaboratoryCacheStore._is_cache_expired(cache_store)

        stats = {
            "status": "expired" if is_expired else "valid",
            "size": cache_store.get("size", 0),
            "timestamp": cache_store.get("timestamp", "unknown"),
            "ttl_hours": cache_store.get("ttl_hours", CACHE_TTL_HOURS)
        }

        if not is_expired and "timestamp" in cache_store:
            try:
                cache_time = datetime.fromisoformat(cache_store["timestamp"])
                age_minutes = (datetime.now(timezone.utc) - cache_time).total_seconds() / 60
                stats["age_minutes"] = round(age_minutes, 1)
            except:
                pass

        return stats


def get_codes_with_persistent_cache(
    lab_names: List[str],
    cache_store: Optional[Dict],
    api_fetch_function
) -> Tuple[List[str], Dict]:
    """
    Función helper que combina cache persistente con llamadas API.

    Esta función es el punto de entrada principal para obtener códigos
    de laboratorio con cache persistente.

    Args:
        lab_names: Lista de nombres de laboratorio
        cache_store: Cache actual del dcc.Store
        api_fetch_function: Función para obtener códigos del API

    Returns:
        Tuple[List[str], Dict]:
            - Lista de códigos en el mismo orden que lab_names
            - Cache actualizado para guardar en dcc.Store
    """
    if not lab_names:
        return [], cache_store or {}

    # 1. Buscar en cache
    cached_mappings, missing_names = LaboratoryCacheStore.get_from_cache(
        lab_names, cache_store
    )

    # 2. Si hay nombres faltantes, llamar API
    if missing_names:
        logger.info(
            f"[LAB_CACHE] Llamando API para {len(missing_names)} nombres faltantes"
        )

        try:
            # Llamar función API (debe retornar lista de códigos)
            new_codes = api_fetch_function(missing_names)

            if new_codes and len(new_codes) == len(missing_names):
                # Crear mapeo nombre→código para los nuevos
                new_mappings = {
                    name: code
                    for name, code in zip(missing_names, new_codes)
                    if code  # Solo cachear si hay código válido
                }

                # Combinar con cache existente
                all_mappings = {**cached_mappings, **new_mappings}

                # Actualizar cache store
                updated_cache = LaboratoryCacheStore.update_cache(
                    new_mappings, cache_store
                )
            else:
                logger.warning(
                    f"[LAB_CACHE] API retornó {len(new_codes) if new_codes else 0} "
                    f"códigos para {len(missing_names)} nombres"
                )
                all_mappings = cached_mappings
                updated_cache = cache_store or {}

        except Exception as e:
            logger.error(f"[LAB_CACHE] Error llamando API: {e}")
            # Usar solo lo que está en cache
            all_mappings = cached_mappings
            updated_cache = cache_store or {}
    else:
        # Todo estaba en cache
        logger.info(f"[LAB_CACHE] Todos los {len(lab_names)} nombres en cache")
        all_mappings = cached_mappings
        updated_cache = cache_store or {}

    # 3. Construir lista de códigos en el orden original
    result_codes = []
    for name in lab_names:
        code = all_mappings.get(name)
        if code:
            result_codes.append(code)
        else:
            logger.warning(f"[LAB_CACHE] No se encontró código para: {name}")

    # Log estadísticas con métricas de budget (Issue rate limiting 2025-12-01)
    stats = LaboratoryCacheStore.get_cache_stats(updated_cache)
    api_calls_made = len(missing_names) if missing_names else 0
    logger.info(
        f"[LAB_CACHE] Budget: API calls={api_calls_made}, "
        f"cache_size={stats.get('size', 0)}, "
        f"cache_status={stats.get('status', 'unknown')}, "
        f"cache_age_min={stats.get('age_minutes', 'N/A')}"
    )

    return result_codes, updated_cache
