# frontend/utils/interval_optimizer.py
"""
Optimizador de intervals para componentes Dash
Reduce la frecuencia de requests basándose en el contexto y entorno
"""

import os
from typing import Dict, Optional


class IntervalOptimizer:
    """
    Optimizador inteligente de intervals para reducir carga en producción
    mientras mantiene la experiencia de usuario en desarrollo
    """

    def __init__(self, environment: Optional[str] = None):
        self.environment = environment or os.getenv("ENVIRONMENT", "development")
        self.is_production = self.environment == "production"
        self.is_render = os.getenv("RENDER") == "true"

    def get_optimized_intervals(self) -> Dict[str, int]:
        """
        Obtener intervals optimizados basados en el entorno y contexto

        Returns:
            Dict[str, int]: Intervals en milisegundos por componente
        """
        if self.is_production or self.is_render:
            return self._get_production_intervals()
        else:
            return self._get_development_intervals()

    def _get_development_intervals(self) -> Dict[str, int]:
        """Intervals para desarrollo - más frecuentes para debugging"""
        return {
            # Health checks y status
            "health_check": 45000,  # 45s
            "system_status_banner": 30000,  # 30s
            "system_health_dashboard": 30000,  # 30s
            # Progreso y tasks
            "progress_update": 5000,  # 5s durante sync
            "upload_progress": 2000,  # 2s durante upload
            # Dashboard y datos
            "dashboard_update": 30000,  # 30s
            "context_refresh": 15 * 60 * 1000,  # 15min
            "admin_stats": 30000,  # 30s
            # Catalog y sync
            "catalog_info": 5000,  # 5s durante sync (deshabilitado por defecto)
            "catalog_update": 15000,  # 15s durante sync
            "catalog_initial_load": 500,  # 500ms disparo inmediato
            "auto_update_check": 60 * 60 * 1000,  # 1h
            # System monitoring - Issue #150: Reducido para evitar re-renders excesivos en tablets
            "viewport": 5000,  # 5s para responsive (antes 1s) - Fix performance tablets 768px
            "app_load": 60000,  # 1min carga inicial
            # Auth guard - Issue #187: Detección rápida de 401s para UX óptima
            "auth_guard": 5000,  # 5s verificación de errores 401
        }

    def _get_production_intervals(self) -> Dict[str, int]:
        """Intervals para producción - reducidos para optimizar performance"""
        return {
            # Health checks y status - SIGNIFICATIVAMENTE reducidos
            "health_check": 120000,  # 2min (de 45s)
            "system_status_banner": 60000,  # 1min (de 30s)
            "system_health_dashboard": 90000,  # 1.5min (de 30s)
            # Progreso y tasks - mantener responsividad durante operaciones activas
            "progress_update": 8000,  # 8s (de 5s)
            "upload_progress": 3000,  # 3s (de 2s)
            # Dashboard y datos - reducir significativamente
            "dashboard_update": 120000,  # 2min (de 30s)
            "context_refresh": 30 * 60 * 1000,  # 30min (de 15min)
            "admin_stats": 120000,  # 2min (de 30s)
            # Catalog y sync - reducir durante operaciones pasivas
            "catalog_info": 10000,  # 10s durante sync (de 5s)
            "catalog_update": 30000,  # 30s durante sync (de 15s)
            "catalog_initial_load": 1000,  # 1s disparo inicial
            "auto_update_check": 2 * 60 * 60 * 1000,  # 2h (de 1h)
            # System monitoring - eliminar o reducir drasticamente
            "viewport": 5000,  # 5s (de 1s)
            "app_load": 120000,  # 2min carga inicial (de 1min)
            # Auth guard - Issue #187: Balance entre detección rápida y recursos
            "auth_guard": 8000,  # 8s en producción (más conservador)
        }

    def get_interval(self, component_type: str) -> int:
        """
        Obtener interval optimizado para un componente específico

        Args:
            component_type: Tipo de componente ('health_check', 'dashboard_update', etc.)

        Returns:
            int: Interval en milisegundos
        """
        intervals = self.get_optimized_intervals()
        return intervals.get(component_type, 30000)  # Default: 30s

    def is_component_enabled(self, component_type: str) -> bool:
        """
        Determinar si un componente debe estar habilitado basándose en el entorno

        Args:
            component_type: Tipo de componente

        Returns:
            bool: True si debe estar habilitado
        """
        # Componentes que se deshabilitan en producción para reducir carga
        production_disabled = {
            "viewport",  # Solo necesario para development debugging
        }

        if self.is_production and component_type in production_disabled:
            return False

        return True

    def get_health_check_config(self) -> Dict[str, any]:
        """
        Configuración especial para health checks considerando Render

        Returns:
            Dict con configuración de health checks
        """
        if self.is_render:
            # Render hace health checks cada ~5s desde su balanceador
            # Nuestros intervals internos deben ser menos frecuentes
            return {
                "interval": 180000,  # 3min para evitar duplicar con Render
                "enabled": True,
                "reason": "render_optimization",
            }
        elif self.is_production:
            return {
                "interval": 120000,  # 2min para producción no-Render
                "enabled": True,
                "reason": "production_optimization",
            }
        else:
            return {"interval": 45000, "enabled": True, "reason": "development_monitoring"}  # 45s para desarrollo

    def get_dash_update_config(self) -> Dict[str, any]:
        """
        Configuración especial para _dash-update-component frequency

        Returns:
            Dict con recomendaciones para callbacks Dash
        """
        if self.is_production:
            return {
                "suggested_debounce_ms": 1000,  # 1s debounce en producción
                "batch_updates": True,
                "reason": "reduce_production_noise",
            }
        else:
            return {
                "suggested_debounce_ms": 300,  # 300ms debounce en desarrollo
                "batch_updates": False,
                "reason": "development_responsiveness",
            }


# Instancia global del optimizador
interval_optimizer = IntervalOptimizer()


def get_optimized_interval(component_type: str) -> int:
    """
    Función helper para obtener interval optimizado

    Args:
        component_type: Tipo de componente

    Returns:
        int: Interval en milisegundos
    """
    return interval_optimizer.get_interval(component_type)


def should_enable_component(component_type: str) -> bool:
    """
    Función helper para determinar si un componente debe estar habilitado

    Args:
        component_type: Tipo de componente

    Returns:
        bool: True si debe estar habilitado
    """
    return interval_optimizer.is_component_enabled(component_type)


def get_production_stats() -> Dict[str, any]:
    """
    Obtener estadísticas de optimización para logging

    Returns:
        Dict con estadísticas de optimización aplicada
    """
    dev_intervals = IntervalOptimizer("development").get_optimized_intervals()
    prod_intervals = IntervalOptimizer("production").get_optimized_intervals()

    stats = {
        "environment": interval_optimizer.environment,
        "is_production": interval_optimizer.is_production,
        "is_render": interval_optimizer.is_render,
        "optimizations_applied": {},
    }

    for component, prod_interval in prod_intervals.items():
        dev_interval = dev_intervals.get(component, prod_interval)
        if prod_interval != dev_interval:
            reduction_percentage = ((dev_interval - prod_interval) / dev_interval) * 100
            stats["optimizations_applied"][component] = {
                "dev_interval_ms": dev_interval,
                "prod_interval_ms": prod_interval,
                "reduction_percentage": round(reduction_percentage, 1),
            }

    return stats
