# frontend/utils/helpers.py
"""
Funciones helper para formateo y utilidades comunes.
"""

import json
import locale
import logging
from datetime import date, datetime
from typing import Any, Union, Tuple

from .constants import DATE_FORMATS

logger = logging.getLogger(__name__)

# Configuración global de formato español
DECIMAL_SEPARATOR = ","
THOUSANDS_SEPARATOR = "."

# Meses en español (índice 1-12)
MESES_ESPANOL = {
    1: "Enero", 2: "Febrero", 3: "Marzo", 4: "Abril",
    5: "Mayo", 6: "Junio", 7: "Julio", 8: "Agosto",
    9: "Septiembre", 10: "Octubre", 11: "Noviembre", 12: "Diciembre"
}

# Meses abreviados (3 caracteres)
MESES_ESPANOL_CORTO = {
    1: "Ene", 2: "Feb", 3: "Mar", 4: "Abr",
    5: "May", 6: "Jun", 7: "Jul", 8: "Ago",
    9: "Sep", 10: "Oct", 11: "Nov", 12: "Dic"
}

# Días de la semana en español
DIAS_SEMANA_ESPANOL = {
    0: "Lunes", 1: "Martes", 2: "Miércoles", 3: "Jueves",
    4: "Viernes", 5: "Sábado", 6: "Domingo"
}

# Configurar locale para España (fallback si no está disponible)
try:
    locale.setlocale(locale.LC_ALL, "es_ES.UTF-8")
except locale.Error:
    try:
        locale.setlocale(locale.LC_ALL, "Spanish_Spain.1252")
    except locale.Error:
        # Usar locale por defecto si no hay español disponible
        pass


def format_currency(
    value: Union[int, float, None],
    symbol: str = "€",
    decimals: int = 2,
    show_sign: bool = False
) -> str:
    """
    Formatear valor como moneda española.

    Args:
        value: Valor numérico a formatear
        symbol: Símbolo de moneda (default: €)
        decimals: Número de decimales
        show_sign: Si True, muestra '+' para valores positivos (default: False)

    Returns:
        Cadena formateada como moneda
    """
    if value is None:
        return f"0,00 {symbol}"

    try:
        # Formatear con separadores españoles
        formatted = f"{abs(value):,.{decimals}f}"
        # Cambiar punto decimal por coma y separador de miles
        formatted = formatted.replace(",", "X").replace(".", ",").replace("X", ".")

        # Manejar signo
        if value < 0:
            return f"-{formatted} {symbol}"
        elif show_sign and value > 0:
            return f"+{formatted} {symbol}"
        return f"{formatted} {symbol}"
    except (ValueError, TypeError):
        return f"0,00 {symbol}"


def format_percentage(value: Union[int, float, None], decimals: int = 1) -> str:
    """
    Formatear valor como porcentaje.

    Args:
        value: Valor numérico a formatear (0.15 = 15%)
        decimals: Número de decimales

    Returns:
        Cadena formateada como porcentaje
    """
    if value is None:
        return "0,0%"

    try:
        percentage = value * 100
        formatted = f"{percentage:.{decimals}f}"
        formatted = formatted.replace(".", ",")
        return f"{formatted}%"
    except (ValueError, TypeError):
        return "0,0%"


def format_number(value: Union[int, float, None], decimals: int = 0, show_decimals_if_zero: bool = False) -> str:
    """
    Formatear número con separadores españoles.

    Args:
        value: Valor numérico a formatear
        decimals: Número de decimales
        show_decimals_if_zero: Si mostrar decimales cuando el valor es entero

    Returns:
        Número formateado
    """
    if value is None:
        return "0"

    try:
        # Si es entero y no queremos mostrar decimales
        if decimals == 0 or (not show_decimals_if_zero and value == int(value)):
            formatted = f"{int(value):,}"
        else:
            formatted = f"{value:,.{decimals}f}"

        # Cambiar separadores al formato español
        formatted = formatted.replace(",", "X").replace(".", ",").replace("X", THOUSANDS_SEPARATOR)
        return formatted
    except (ValueError, TypeError):
        return "0"


def format_compact_number(value: Union[int, float, None]) -> str:
    """
    Formatear número de forma compacta (K, M, B).

    Args:
        value: Valor numérico a formatear

    Returns:
        Número formateado de forma compacta
    """
    if value is None or value == 0:
        return "0"

    try:
        abs_value = abs(value)
        sign = "-" if value < 0 else ""

        if abs_value >= 1_000_000_000:
            return f"{sign}{abs_value/1_000_000_000:.1f}B".replace(".", DECIMAL_SEPARATOR)
        elif abs_value >= 1_000_000:
            return f"{sign}{abs_value/1_000_000:.1f}M".replace(".", DECIMAL_SEPARATOR)
        elif abs_value >= 1_000:
            return f"{sign}{abs_value/1_000:.1f}K".replace(".", DECIMAL_SEPARATOR)
        else:
            return format_number(value, decimals=0)
    except (ValueError, TypeError):
        return "0"


def format_date(date_value: Union[datetime, date, str, None], format_type: str = "display") -> str:
    """
    Formatear fecha según el formato especificado.

    Args:
        date_value: Valor de fecha a formatear
        format_type: Tipo de formato (ver DATE_FORMATS)

    Returns:
        Fecha formateada
    """
    if date_value is None:
        return ""

    # Convertir string a datetime si es necesario
    if isinstance(date_value, str):
        try:
            # Intentar varios formatos comunes
            for fmt in ["%Y-%m-%d", "%Y-%m-%dT%H:%M:%S", "%d/%m/%Y"]:
                try:
                    date_value = datetime.strptime(date_value, fmt)
                    break
                except ValueError:
                    continue
            else:
                return date_value  # Retornar string original si no se puede parsear
        except Exception:
            return str(date_value)

    try:
        format_str = DATE_FORMATS.get(format_type, DATE_FORMATS["display"])
        return date_value.strftime(format_str)
    except Exception:
        return str(date_value)


def safe_divide(
    numerator: Union[int, float, None], denominator: Union[int, float, None], default: float = 0.0
) -> float:
    """
    División segura que maneja valores None y división por cero.

    Args:
        numerator: Numerador
        denominator: Denominador
        default: Valor por defecto si la división no es posible

    Returns:
        Resultado de la división o valor por defecto
    """
    if numerator is None or denominator is None:
        return default

    try:
        if denominator == 0:
            return default
        return float(numerator) / float(denominator)
    except (ValueError, TypeError):
        return default


def truncate_text(text: str, max_length: int = 50, suffix: str = "...") -> str:
    """
    Truncar texto si excede la longitud máxima.

    Args:
        text: Texto a truncar
        max_length: Longitud máxima
        suffix: Sufijo para texto truncado

    Returns:
        Texto truncado si es necesario
    """
    if not text or len(text) <= max_length:
        return text

    return text[: max_length - len(suffix)] + suffix


def get_trend_icon(current: Union[int, float], previous: Union[int, float]) -> str:
    """
    Obtener icono de tendencia basado en comparación de valores.

    Args:
        current: Valor actual
        previous: Valor anterior

    Returns:
        Clase CSS de icono FontAwesome
    """
    if current is None or previous is None:
        return "fas fa-minus"

    if current > previous:
        return "fas fa-arrow-up"
    elif current < previous:
        return "fas fa-arrow-down"
    else:
        return "fas fa-minus"


def get_trend_color(current: Union[int, float], previous: Union[int, float]) -> str:
    """
    Obtener color de tendencia basado en comparación de valores.

    Args:
        current: Valor actual
        previous: Valor anterior

    Returns:
        Clase CSS de color Bootstrap
    """
    if current is None or previous is None:
        return "text-muted"

    if current > previous:
        return "text-success"
    elif current < previous:
        return "text-danger"
    else:
        return "text-muted"


def validate_file_extension(filename: str, allowed_extensions: list) -> bool:
    """
    Validar extensión de archivo.

    Args:
        filename: Nombre del archivo
        allowed_extensions: Lista de extensiones permitidas

    Returns:
        True si la extensión es válida
    """
    if not filename:
        return False

    extension = "." + filename.split(".")[-1].lower()
    return extension in [ext.lower() for ext in allowed_extensions]


def generate_page_id(page_name: str) -> str:
    """
    Generar ID único para una página.

    Args:
        page_name: Nombre de la página

    Returns:
        ID único para la página
    """
    return f"page-{page_name.lower().replace(' ', '-')}"


# ============================================================================
# SESSION STORAGE VALIDATION (Post-Review Optimization - Commit 2005504)
# ============================================================================

# Límite recomendado: 4MB (margen de seguridad del límite de 5MB de sessionStorage)
DEFAULT_SESSION_STORAGE_LIMIT_MB = 4.0


def validate_session_storage_size(
    data: Any,
    store_id: str,
    max_size_mb: float = DEFAULT_SESSION_STORAGE_LIMIT_MB,
    strict: bool = False
) -> Tuple[bool, float]:
    """
    Validar que datos no excedan límite de sessionStorage.

    Post-review optimization para commit 2005504 (hotfix multi-worker).
    Previene errores de QuotaExceededError en producción.

    Args:
        data: Datos a guardar en sessionStorage (cualquier tipo JSON-serializable)
        store_id: ID del dcc.Store (para logging)
        max_size_mb: Tamaño máximo permitido en MB (default: 4MB, margen de seguridad)
        strict: Si True, lanza ValueError cuando excede límite (default: False)

    Returns:
        Tuple (is_valid, size_mb)
        - is_valid: True si válido, False si excede límite
        - size_mb: Tamaño en MB de los datos serializados

    Raises:
        ValueError: Si strict=True y se excede límite

    Examples:
        >>> is_valid, size = validate_session_storage_size({"data": [...]}, "context-store")
        >>> if not is_valid:
        >>>     logger.warning(f"Store {store_id} excede límite: {size:.2f}MB")

        >>> # Modo strict (desarrollo)
        >>> validate_session_storage_size(large_data, "test-store", strict=True)
        ValueError: Store test-store excede límite de sessionStorage: 5.2MB > 4.0MB
    """
    try:
        # Serializar datos como JSON (igual que Dash)
        data_json = json.dumps(data)
        size_bytes = len(data_json.encode('utf-8'))
        size_mb = size_bytes / (1024 * 1024)

        # Validar límite
        if size_mb > max_size_mb:
            logger.warning(
                f"[STORAGE_WARNING] {store_id} excede {max_size_mb}MB: {size_mb:.2f}MB. "
                f"Considerar pagination o reducción de datos."
            )

            if strict:
                raise ValueError(
                    f"Store {store_id} excede límite de sessionStorage: "
                    f"{size_mb:.2f}MB > {max_size_mb}MB"
                )

            return False, size_mb

        # Log informativo si está cerca del límite (>80%)
        threshold = max_size_mb * 0.8
        if size_mb > threshold:
            logger.info(
                f"[STORAGE_INFO] {store_id} cerca del límite: "
                f"{size_mb:.2f}MB / {max_size_mb}MB ({size_mb/max_size_mb*100:.1f}%)"
            )

        return True, size_mb

    except (TypeError, ValueError) as e:
        # Error de serialización (no JSON serializable)
        logger.error(
            f"[STORAGE_ERROR] Error validando tamaño de {store_id}: {e}. "
            f"Tipo de dato: {type(data)}"
        )
        # En modo strict, propagar el error
        if strict:
            raise
        # Modo lenient: asumir válido (no bloqueamos el flujo)
        return True, 0.0


def get_storage_recommendations(size_mb: float, store_id: str) -> str:
    """
    Generar recomendaciones de optimización basadas en tamaño de store.

    Helper para sugerir acciones cuando stores se acercan al límite.

    Args:
        size_mb: Tamaño actual en MB
        store_id: ID del store (para recomendaciones específicas)

    Returns:
        Mensaje con recomendaciones específicas o confirmación de tamaño OK

    Examples:
        >>> recommendations = get_storage_recommendations(3.5, "analysis-store")
        >>> print(recommendations)
        "⚠️ ADVERTENCIA: Considerar paginación pronto | 💡 Limitar filas a top 100"

        >>> recommendations = get_storage_recommendations(0.5, "context-store")
        >>> print(recommendations)
        "Tamaño OK - Sin optimizaciones necesarias"
    """
    if size_mb < 1.0:
        return "Tamaño OK - Sin optimizaciones necesarias"

    recommendations = []

    # Severidad basada en tamaño
    if size_mb > 3.0:
        recommendations.append("🔴 CRÍTICO: Implementar paginación URGENTE")
    elif size_mb > 2.0:
        recommendations.append("⚠️ ADVERTENCIA: Considerar paginación pronto")

    # Recomendaciones específicas por tipo de store
    store_id_lower = store_id.lower()

    if "analysis" in store_id_lower:
        recommendations.append("💡 Limitar filas a top 100 (usar pagination)")

    if "drill" in store_id_lower:
        recommendations.append("💡 Reducir detalles de drill-down (solo campos necesarios)")

    if "context" in store_id_lower:
        recommendations.append("💡 Almacenar solo IDs, no objetos completos")

    if "partners" in store_id_lower:
        recommendations.append("💡 Cachear solo datos esenciales (nombre + ID)")

    if "homogeneous" in store_id_lower:
        recommendations.append("💡 Paginar conjuntos homogéneos (mostrar 20 a la vez)")

    return " | ".join(recommendations) if recommendations else "✅ Tamaño aceptable"


def normalize_utf8_string(text: str) -> str:
    """
    Normalizar string UTF-8 para matching consistente.

    Soluciona problemas de encoding donde "TECNIMEDE ESPAÑA" podría tener
    diferentes representaciones binarias en PostgreSQL vs Python vs JavaScript.

    Issue detectado en Render: Al deseleccionar "VIATRIS" se borra "TECNIMEDE"
    por mismatch de encoding en dropdown multi-select de Dash.

    Args:
        text: String a normalizar (nombre de laboratorio, producto, etc.)

    Returns:
        String normalizado en forma NFC (Canonical Decomposition seguido de Canonical Composition)

    Examples:
        >>> normalize_utf8_string("TECNIMEDE ESPAÑA")  # Desde BD
        "TECNIMEDE ESPAÑA"
        >>> normalize_utf8_string("TECNIMEDE ESPAÑA")  # Desde dropdown
        "TECNIMEDE ESPAÑA"  # Ahora matcha consistentemente

    Technical Details:
        - NFC (Normalization Form C): Forma canónica recomendada para la mayoría de aplicaciones
        - Convierte "Ñ" a una única representación binaria (evita variantes decomposed vs precomposed)
        - Compatible con PostgreSQL, JavaScript (navegador), y Python
    """
    import unicodedata

    if not text or not isinstance(text, str):
        return text

    # Normalizar a forma NFC (precomposed, más compatible con APIs y UIs)
    normalized = unicodedata.normalize('NFC', text)

    # Strip espacios adicionales (por si acaso)
    normalized = normalized.strip()

    return normalized


# ============================================================================
# LOCALIZACIÓN ESPAÑOLA - FECHAS Y MESES
# ============================================================================


def format_month_year(date_value: Union[datetime, date], short: bool = True) -> str:
    """
    Formatear fecha como mes y año en español.

    Args:
        date_value: Fecha a formatear
        short: Si True usa abreviatura (Ene), si False usa completo (Enero)

    Returns:
        Cadena formateada: "Ene 24" o "Enero 2024"

    Examples:
        >>> format_month_year(datetime(2024, 1, 15))
        "Ene 24"
        >>> format_month_year(datetime(2024, 10, 1), short=False)
        "Octubre 2024"
    """
    if date_value is None:
        return ""

    try:
        month = date_value.month
        year = date_value.year

        if short:
            month_name = MESES_ESPANOL_CORTO.get(month, str(month))
            return f"{month_name} {str(year)[-2:]}"
        else:
            month_name = MESES_ESPANOL.get(month, str(month))
            return f"{month_name} {year}"
    except (AttributeError, TypeError):
        return str(date_value)


def format_date_spanish(
    date_value: Union[datetime, date, str, None],
    include_year: bool = True,
    short_month: bool = True
) -> str:
    """
    Formatear fecha completa en español.

    Args:
        date_value: Fecha a formatear
        include_year: Si incluir el año
        short_month: Si usar mes abreviado

    Returns:
        Fecha formateada: "15 Ene 2024" o "15 de Enero de 2024"

    Examples:
        >>> format_date_spanish(datetime(2024, 10, 1))
        "1 Oct 2024"
        >>> format_date_spanish(datetime(2024, 10, 1), short_month=False)
        "1 de Octubre de 2024"
    """
    if date_value is None:
        return ""

    # Convertir string a datetime si es necesario
    if isinstance(date_value, str):
        try:
            for fmt in ["%Y-%m-%d", "%Y-%m-%dT%H:%M:%S", "%d/%m/%Y"]:
                try:
                    date_value = datetime.strptime(date_value, fmt)
                    break
                except ValueError:
                    continue
            else:
                return date_value
        except Exception:
            return str(date_value)

    try:
        day = date_value.day
        month = date_value.month
        year = date_value.year

        if short_month:
            month_name = MESES_ESPANOL_CORTO.get(month, str(month))
            if include_year:
                return f"{day} {month_name} {year}"
            return f"{day} {month_name}"
        else:
            month_name = MESES_ESPANOL.get(month, str(month))
            if include_year:
                return f"{day} de {month_name} de {year}"
            return f"{day} de {month_name}"
    except (AttributeError, TypeError):
        return str(date_value)


def get_month_name(month: int, short: bool = False) -> str:
    """
    Obtener nombre del mes en español.

    Args:
        month: Número del mes (1-12)
        short: Si usar abreviatura

    Returns:
        Nombre del mes en español

    Examples:
        >>> get_month_name(10)
        "Octubre"
        >>> get_month_name(10, short=True)
        "Oct"
    """
    if short:
        return MESES_ESPANOL_CORTO.get(month, str(month))
    return MESES_ESPANOL.get(month, str(month))


def generate_plotly_spanish_month_ticks(dates: list) -> Tuple[list, list]:
    """
    Genera tickvals y ticktext en español para gráficos Plotly con fechas.

    Workaround para Plotly que no soporta locale de D3 para español.
    https://stackoverflow.com/questions/32361113/python-plotly-dates-in-other-languages

    Args:
        dates: Lista de fechas del gráfico (datetime, date, o strings ISO)

    Returns:
        Tuple (tickvals, ticktext) para usar en xaxis de Plotly:
        - tickvals: Lista de fechas ISO (primer día de cada mes)
        - ticktext: Lista de nombres de mes en español ("Ene 24", "Feb 24", etc.)

    Examples:
        >>> from datetime import datetime
        >>> dates = [datetime(2024, 10, 1), datetime(2024, 11, 15)]
        >>> tickvals, ticktext = generate_plotly_spanish_month_ticks(dates)
        >>> tickvals
        ['2024-10-01', '2024-11-01']
        >>> ticktext
        ['Oct 24', 'Nov 24']

    Usage in Plotly:
        >>> fig.update_layout(
        >>>     xaxis={
        >>>         'tickvals': tickvals,
        >>>         'ticktext': ticktext
        >>>     }
        >>> )
    """
    if not dates:
        return [], []

    parsed_dates = []
    for d in dates:
        if isinstance(d, str):
            try:
                parsed_dates.append(datetime.fromisoformat(d.replace('Z', '+00:00')))
            except ValueError:
                try:
                    parsed_dates.append(datetime.strptime(d, "%Y-%m-%d"))
                except ValueError:
                    continue
        elif hasattr(d, 'month'):
            parsed_dates.append(d)

    if not parsed_dates:
        return [], []

    # Generar ticks mensuales únicos
    seen_months = set()
    tickvals = []
    ticktext = []

    for d in sorted(parsed_dates):
        month_key = (d.year, d.month)
        if month_key not in seen_months:
            seen_months.add(month_key)
            # Usar el primer día del mes como tickval
            tick_date = datetime(d.year, d.month, 1)
            tickvals.append(tick_date.strftime("%Y-%m-%d"))
            # Usar nombre de mes en español
            month_name = MESES_ESPANOL_CORTO.get(d.month, str(d.month))
            ticktext.append(f"{month_name} {str(d.year)[-2:]}")

    return tickvals, ticktext
