# frontend/utils/generics_helpers.py
"""
Helper functions and types for generics analysis.

This module contains:
- TypedDict definitions for type safety
- Cache classes for BD-first pattern
- Helper functions for data processing
- API data fetching functions

Extracted from frontend/callbacks/generics.py for better organization.
"""

import logging
from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional, Tuple, TypedDict

from utils.api_client import api_client
from utils.laboratory_mapping import selected_names_to_codes
from utils.laboratory_cache_store import get_codes_with_persistent_cache
from utils.pharmacy_context import get_current_pharmacy_id

logger = logging.getLogger(__name__)


# ============================================================================
# TYPE DEFINITIONS
# ============================================================================
class PartnersFetchResult(TypedDict):
    """Type-safe result from BD-first partner fetch operations."""
    partners: List[str]
    success: bool
    error_message: Optional[str]
    source: str  # 'database', 'cache', or 'error'


# ============================================================================
# SHORT-LIVED CACHE FOR BD-FIRST PATTERN
# ============================================================================
class PartnersCache:
    """
    Short-lived in-memory cache for partners (2-5 seconds TTL).

    Reduces DB load for high-frequency callbacks while maintaining freshness.
    Cache is per-pharmacy and expires automatically.

    Note: This is NOT a replacement for dcc.Store persistence.
          It's a performance optimization for the BD-first pattern.
    """
    def __init__(self):
        self._cache: Dict[str, Tuple[List[str], datetime]] = {}
        self._default_ttl_seconds = 2  # Conservative TTL

    def get(self, pharmacy_id: str, ttl_seconds: Optional[int] = None) -> Optional[List[str]]:
        """Get cached partners if not expired."""
        if pharmacy_id not in self._cache:
            return None

        partners, timestamp = self._cache[pharmacy_id]
        ttl = ttl_seconds if ttl_seconds is not None else self._default_ttl_seconds

        if datetime.now() - timestamp > timedelta(seconds=ttl):
            # Expired, remove from cache
            del self._cache[pharmacy_id]
            return None

        return partners

    def set(self, pharmacy_id: str, partners: List[str]) -> None:
        """Cache partners with current timestamp."""
        self._cache[pharmacy_id] = (partners, datetime.now())

    def invalidate(self, pharmacy_id: str) -> None:
        """Manually invalidate cache for pharmacy (e.g., after selection change)."""
        if pharmacy_id in self._cache:
            del self._cache[pharmacy_id]

    def clear(self) -> None:
        """Clear entire cache (e.g., for testing)."""
        self._cache.clear()


# Global cache instance for BD-first pattern
_partners_cache = PartnersCache()


# ============================================================================
# CONSTANTS
# ============================================================================
PARTNER_COLOR = "#0d6efd"  # Azul para partners
NO_PARTNER_COLOR = "#ffc107"  # Amarillo para no-partners

# DISCOUNT CALCULATION CONSTANTS (Issue #339)
DEFAULT_DISCOUNT_RATE = 0.10  # 10% - ALINEADO CON BACKEND (partner_analysis_service.py línea 315)

# CONSTANTS FOR HOMOGENEOUS EXPANSION (Issue #263 - Fix #6)
ICON_EXPANDED = "fas fa-minus-square"
ICON_COLLAPSED = "fas fa-plus-square"
ICON_EXPAND_ALL = "fas fa-expand-arrows-alt"
ICON_COLLAPSE_ALL = "fas fa-compress-arrows-alt"


# ============================================================================
# LABORATORY CODE CACHING (P1 Fix - Rate Limiting)
# ============================================================================
def get_codes_with_cache(lab_names: List[str], laboratory_cache_store: Optional[Dict]) -> Tuple[List[str], Dict]:
    """
    Obtiene códigos de laboratorios usando cache persistente en dcc.Store.

    HOTFIX CRÍTICO: Ahora usa cache persistente que sobrevive entre callbacks
    y reduce drásticamente las llamadas al API (previene rate limiting).

    Esta función implementa el patrón de cache-first para reducir llamadas
    al endpoint /api/v1/laboratory-mapping/names-to-codes que tiene un límite
    de 500 requests/hora.

    Args:
        lab_names: Lista de nombres de laboratorios a convertir
        laboratory_cache_store: Cache persistente del dcc.Store (session storage)

    Returns:
        Tuple[List[str], Dict]:
            - Lista de códigos de laboratorio (strings)
            - Cache actualizado para guardar en dcc.Store

    Note:
        El cache tiene TTL de 7 días (datos estáticos de catálogo).
        Reduce llamadas de 400+/hora a ~1/semana (99.9% reducción).
        Previene rate limiting en Render después de Ctrl+F5 + re-login.

    Ejemplo:
        >>> lab_names = ["CINFA", "KERN PHARMA", "NORMON"]
        >>> cache_store = {"mappings": {"CINFA": "111"}, "timestamp": "2025-10-31T09:00:00"}
        >>> codes, updated_cache = get_codes_with_cache(lab_names, cache_store)
        >>> # Solo llama API para KERN PHARMA y NORMON
        >>> print(codes)  # ["111", "426", "644"]
    """
    if not lab_names:
        return [], laboratory_cache_store or {}

    # Usar la función helper con cache persistente
    codes, updated_cache = get_codes_with_persistent_cache(
        lab_names,
        laboratory_cache_store,
        selected_names_to_codes  # Función API existente
    )

    logger.info(
        f"[LAB_CACHE] get_codes_with_cache: {len(codes)} códigos obtenidos "
        f"para {len(lab_names)} nombres (cache size: {updated_cache.get('size', 0)})"
    )

    return codes, updated_cache


def get_homogeneous_detail_from_store(analysis_data: Optional[Dict], homogeneous_code: str) -> Optional[Dict]:
    """
    Extraer detalle de conjunto homogéneo desde analysis-store.

    Evita llamadas API redundantes reutilizando datos ya calculados con partners actuales.

    Args:
        analysis_data: Datos del analysis-store
        homogeneous_code: Código del conjunto homogéneo a buscar

    Returns:
        Dict con group_summary (NO incluye products_detail porque no está en /partner-dynamic)
        None si no encontrado

    Issue #346: Normalización de códigos para manejar "1160" vs "1160.0"
    NOTA: HomogeneousGroupDetail (schema backend) NO incluye productos individuales.
          Para obtener productos, usar endpoint /homogeneous-detail/{pharmacy_id}/{code}
    """
    if not analysis_data or "homogeneous_groups_detail" not in analysis_data:
        return None

    # Normalizar código de búsqueda (remover decimales .0)
    search_code = str(homogeneous_code).rstrip('.0') if homogeneous_code else None

    if not search_code:
        return None

    groups = analysis_data["homogeneous_groups_detail"]
    for group in groups:
        # Normalizar código del grupo (remover decimales .0)
        group_code = str(group.get("homogeneous_code", "")).rstrip('.0')

        if group_code == search_code:
            # Retornar solo group_summary (datos agregados disponibles)
            # NO products_detail porque no está en HomogeneousGroupDetail del backend
            return {
                "homogeneous_code": group.get("homogeneous_code"),
                "homogeneous_name": group.get("homogeneous_name", "Sin nombre"),
                "total_units": group.get("total_units", 0),
                "total_revenue": group.get("total_revenue", 0),
                "partner_units": group.get("partner_units", 0),
                "opportunity_units": group.get("opportunity_units", 0),
                "savings_base": group.get("savings_base", 0),
                "partner_penetration": group.get("partner_penetration", 0),
            }

    return None


def simplify_lab_name(lab_name: str) -> str:
    """
    Simplificar nombre de laboratorio según regla:
    - Primera palabra del nombre
    - EXCEPTO si es "laboratorio" o "laboratorios", entonces tomar 2 palabras
    """
    if not lab_name:
        return ""

    words = lab_name.split()
    if len(words) == 1:
        return lab_name

    first_word_lower = words[0].lower()
    if first_word_lower in ["laboratorio", "laboratorios"]:
        return " ".join(words[:2]) if len(words) >= 2 else lab_name

    return words[0]


def has_placeholder(content: Any) -> bool:
    """
    Check if content is a placeholder that needs loading.

    Helper function to avoid redundant placeholder detection logic (Issue #263 - Fix #4).

    Args:
        content: Content to check for placeholder status

    Returns:
        True if content is empty or contains placeholder text
    """
    if not content:
        return True
    content_str = str(content)
    return "Haz click" in content_str or "Loading" in content_str


def persist_partners_selection(pharmacy_id: str, selected_partners: List[str], available_partners: List[str]) -> bool:
    """
    Persistir selección de partners en base de datos (Issue #303 - Mejora #3).

    Helper function to centralize BD persistence logic and improve testability.

    Args:
        pharmacy_id: UUID de la farmacia
        selected_partners: Lista de laboratorios seleccionados
        available_partners: Lista de todos los laboratorios disponibles

    Returns:
        bool: True si se persistió correctamente, False si falló

    Example:
        >>> success = persist_partners_selection(
        ...     pharmacy_id="abc-123",
        ...     selected_partners=["Lab A", "Lab B"],
        ...     available_partners=["Lab A", "Lab B", "Lab C"]
        ... )
        >>> assert success == True
    """
    try:
        # Construir payload con selections
        selections = []
        for lab_name in available_partners:
            selections.append({"laboratory_name": lab_name, "is_selected": lab_name in selected_partners})

        if not selections:
            logger.warning("⚠️ No hay selections para persistir (available_partners vacío)")
            return False

        logger.info(
            f"[persist_partners] Persistiendo {len(selected_partners)} seleccionados "
            f"de {len(available_partners)} disponibles para farmacia {pharmacy_id}"
        )

        # ✅ AGREGAR logs detallados (Fix Problema #3)
        response = api_client.put(
            f"/api/v1/pharmacy-partners/{pharmacy_id}/selection",
            json={"selections": selections}
        )

        if response:
            logger.info(
                f"✅ [PERSISTENCIA_BD] Selección guardada correctamente:\n"
                f"   - pharmacy_id: {pharmacy_id}\n"
                f"   - seleccionados: {len(selected_partners)}/{len(available_partners)}\n"
                f"   - partners: {selected_partners[:3]}{'...' if len(selected_partners) > 3 else ''}\n"
                f"   - timestamp: {response.get('timestamp', 'N/A')}"
            )
            return True
        else:
            logger.error(
                f"❌ [PERSISTENCIA_BD] FALLO AL GUARDAR - Respuesta vacía del backend\n"
                f"   - pharmacy_id: {pharmacy_id}\n"
                f"   - seleccionados: {len(selected_partners)}/{len(available_partners)}\n"
                f"   - endpoint: PUT /api/v1/pharmacy-partners/{pharmacy_id}/selection\n"
                f"   - IMPACTO: Selección NO se guardará entre sesiones"
            )
            return False

    except Exception as e:
        logger.error(
            f"❌ Error persistiendo selección de partners: {str(e)}\n"
            f"   - pharmacy_id: {pharmacy_id}\n"
            f"   - selected: {len(selected_partners)}\n"
            f"   - available: {len(available_partners)}\n"
            f"   - error type: {type(e).__name__}"
        )
        return False


def get_selected_partners_from_db(use_cache: bool = True, cache_ttl_seconds: int = 2) -> PartnersFetchResult:
    """
    🆕 BD-FIRST PATTERN: Leer partners seleccionados SIEMPRE desde base de datos.

    ✨ CODE REVIEW IMPROVEMENT: Ahora con caching de corto plazo y type safety.

    Esta función garantiza que el análisis use la BD como fuente de verdad única,
    independiente del estado del store o race conditions entre callbacks.

    Args:
        use_cache: Si True, usa cache de corto plazo para reducir DB load (default: True)
        cache_ttl_seconds: TTL del cache en segundos (default: 2s, rango recomendado: 2-5s)

    Returns:
        PartnersFetchResult: TypedDict con:
            - partners: Lista de nombres de laboratorios seleccionados (puede estar vacía)
            - success: True si lectura exitosa, False si error
            - error_message: Mensaje de error si success=False, None otherwise
            - source: 'cache', 'database', o 'error'

    Example:
        >>> result = get_selected_partners_from_db()
        >>> if result["success"]:
        ...     partners = result["partners"]
        ...     # Usar partners para análisis
        >>> else:
        ...     logger.error(f"Error: {result['error_message']}")

    Notes:
        - BD es SIEMPRE la fuente de verdad (Issue #345)
        - Cache de 2s reduce DB load sin comprometer freshness
        - Lista vacía [] es válida (usuario puede deseleccionar todos)
        - Elimina dependencia de stores/race conditions
        - Logs detallados para debugging
    """
    try:
        # Obtener pharmacy_id
        try:
            pharmacy_id = get_current_pharmacy_id()
        except ValueError as e:
            error_msg = f"Error obteniendo pharmacy_id: {str(e)}"
            logger.error(f"[BD_FIRST] {error_msg}")
            return PartnersFetchResult(
                partners=[],
                success=False,
                error_message=error_msg,
                source="error"
            )

        # 🔄 CHECK CACHE FIRST (si está habilitado)
        if use_cache:
            cached_partners = _partners_cache.get(pharmacy_id, ttl_seconds=cache_ttl_seconds)
            if cached_partners is not None:
                logger.debug(
                    f"[BD_FIRST] ⚡ Using cached partners (age < {cache_ttl_seconds}s):\n"
                    f"   - Pharmacy: {pharmacy_id}\n"
                    f"   - Partners: {len(cached_partners)} cached\n"
                    f"   - Source: SHORT-LIVED CACHE (performance optimization)"
                )
                return PartnersFetchResult(
                    partners=cached_partners,
                    success=True,
                    error_message=None,
                    source="cache"
                )

        # 📊 LOGGING: Antes de llamar API (cache miss or disabled)
        logger.info(f"[BD_FIRST] Reading partners from database for pharmacy {pharmacy_id}")

        # Llamar endpoint de partners seleccionados
        # Endpoint con auto-inicialización si vacío (REGLA #12)
        response = api_client.get(f"/api/v1/pharmacy-partners/{pharmacy_id}/selected")

        if not response:
            error_msg = "Empty response from /selected endpoint"
            logger.error(f"[BD_FIRST] {error_msg}")
            return PartnersFetchResult(
                partners=[],
                success=False,
                error_message=error_msg,
                source="error"
            )

        # Extraer partners seleccionados
        # ⚠️ FIX: El endpoint /selected devuelve "partners_detail" (no "partners")
        partners_data = response.get("partners_detail", [])

        if not isinstance(partners_data, list):
            error_msg = f"Invalid format: expected list, received {type(partners_data)}"
            logger.error(f"[BD_FIRST] {error_msg}")
            return PartnersFetchResult(
                partners=[],
                success=False,
                error_message=error_msg,
                source="error"
            )

        # Extraer nombres de laboratorios (solo los que tienen is_selected=True)
        # Nota: El endpoint /selected ya filtra por is_selected=True
        selected_partners = [p["laboratory_name"] for p in partners_data if isinstance(p, dict) and "laboratory_name" in p]

        # 💾 STORE IN CACHE (si está habilitado)
        if use_cache:
            _partners_cache.set(pharmacy_id, selected_partners)

        # 📊 LOGGING: Resultado exitoso
        logger.info(
            f"✅ [BD_FIRST] Partners read from database:\n"
            f"   - Total selected: {len(selected_partners)}\n"
            f"   - Partners: {selected_partners[:3]}{'...' if len(selected_partners) > 3 else ''}\n"
            f"   - Source: DATABASE (fresh data)\n"
            f"   - Cached: {use_cache} (TTL={cache_ttl_seconds}s)\n"
            f"   - Guarantee: Race-condition immune"
        )

        return PartnersFetchResult(
            partners=selected_partners,
            success=True,
            error_message=None,
            source="database"
        )

    except Exception as e:
        error_msg = f"Exception reading partners: {str(e)} (type: {type(e).__name__})"
        logger.error(f"❌ [BD_FIRST] {error_msg}")
        return PartnersFetchResult(
            partners=[],
            success=False,
            error_message=error_msg,
            source="error"
        )


def create_expand_button_content(all_expanded: bool) -> List[Any]:
    """
    Create button content for expand/collapse all button.

    Helper function to avoid button content duplication (Issue #263 - Fix #7).

    Args:
        all_expanded: Whether all rows are currently expanded

    Returns:
        List of button content elements [icon, text]
    """
    from dash import html

    if all_expanded:
        return [html.I(className=ICON_COLLAPSE_ALL, style={"marginRight": "0.25rem"}), "Contraer Todo"]
    return [html.I(className=ICON_EXPAND_ALL, style={"marginRight": "0.25rem"}), "Expandir Todo"]


def get_temporal_breakdown_data(
    level: str,
    path: List[str],
    partners_store: dict,
    codes_cache: dict = None,
    homogeneous_code: Optional[str] = None,
    employee_names: Optional[List[str]] = None
) -> Tuple[List[dict], dict]:
    """
    Obtener datos de breakdown temporal desde el endpoint correspondiente.

    🆕 CODE REVIEW IMPROVEMENT: Ahora usa BD-first pattern.

    Args:
        level: Nivel de agregación temporal (quarter, month, week)
        path: Path de drill-down
        partners_store: IGNORED - kept for backwards compatibility but value ignored
        codes_cache: Cache de códigos de laboratorio (opcional)
        homogeneous_code: Código de conjunto homogéneo para filtrar (opcional, Issue #346)
        employee_names: Lista de nombres de empleados para filtrar (opcional, Issue #428)

    Returns:
        Tuple[List[dict], dict]: (temporal_data, updated_cache)

    Note:
        partners_store parameter is maintained for backwards compatibility but its
        value is IGNORED. Partners are always read from DB (BD-first pattern).
    """
    try:
        # 🆕 BD-FIRST PATTERN: Read partners from DB (ignore store)
        logger.debug("[get_temporal_breakdown_data] Reading partners from DB (BD-first)")
        result = get_selected_partners_from_db()

        if not result["success"]:
            logger.error(f"[get_temporal_breakdown_data] Failed to load partners: {result['error_message']}")
            return [], codes_cache or {}

        selected_partners = result["partners"]

        if not selected_partners:
            logger.debug("[get_temporal_breakdown_data] No partners selected, returning empty data")
            return [], codes_cache or {}

        # Obtener pharmacy_id dinámicamente
        try:
            pharmacy_id = get_current_pharmacy_id()
        except ValueError as e:
            logger.error(f"Error obteniendo pharmacy_id: {str(e)}")
            return [], codes_cache or {}

        # 🔧 NUEVA ARQUITECTURA: Convertir nombres a códigos (P1 Fix - Cache-first)
        codes_cache = codes_cache or {}  # Nit: simplified initialization
        selected_partner_codes, codes_cache = get_codes_with_cache(
            selected_partners,  # ✅ From DB, not store
            codes_cache
        )
        if not selected_partner_codes:
            selected_partner_codes = ["111", "426", "644"]  # Fallback

        # FastAPI List[str] requires multiple query params, not comma-separated
        params = {
            "level": level,
            "period_months": 12,
            "partner_codes": selected_partner_codes,  # Pass as list for proper multi-param handling
        }

        # ✅ NUEVO: Agregar homogeneous_code si se proporciona (Issue #346)
        if homogeneous_code:
            params["homogeneous_code"] = homogeneous_code
            logger.info(f"[TEMPORAL_BREAKDOWN] Filtering by homogeneous_code: {homogeneous_code}")

        # ✅ NUEVO: Agregar employee_names si se proporciona (Issue #428)
        if employee_names:
            params["employee_names"] = employee_names
            logger.info(f"[TEMPORAL_BREAKDOWN] Filtering by employees: {employee_names}")

        # Añadir path para drill-down específico
        if path:
            params["drill_path"] = ",".join(path)

        data = api_client.get(f"/api/v1/analysis/temporal-breakdown/{pharmacy_id}", params=params)

        if data:
            return data.get("temporal_data", []), codes_cache
        else:
            logger.error("Error obteniendo breakdown temporal: respuesta vacía")
            return [], codes_cache

    except Exception as e:
        logger.error(f"Excepción obteniendo breakdown temporal: {str(e)}")
        return [], codes_cache or {}
