# frontend/utils/error_handlers.py
"""
Utilidades para manejo estandarizado de errores en callbacks Dash.

Este módulo proporciona funciones para retornar valores de error consistentes
según los tipos de Output de cada callback (Issue #254, PR #289 - Mejoras).
"""

from typing import Any, List, Tuple, Union

from components.common import create_alert
from dash import html, no_update
from utils.helpers import format_currency


def get_auth_error_return(output_types: List[str]) -> Union[Any, Tuple[Any, ...]]:
    """
    Retorna valores de error apropiados cuando falla la autenticación.

    Esta función estandariza los returns de error en callbacks cuando
    get_current_pharmacy_id() falla, evitando inconsistencias entre callbacks.

    **Problema que resuelve:**
    Antes del PR #289, cada callback retornaba diferentes estructuras cuando
    fallaba la autenticación:
    - Algunos: `{"error": "..."}`
    - Otros: `(create_alert(...), "0", format_currency(0))`
    - Otros: `[]`

    **Solución:**
    Mapeo consistente de tipos de Output a valores de error apropiados.

    Args:
        output_types: Lista de tipos de Output del callback
            Tipos soportados:
            - 'dict': Retorna {"error": "auth_error", "message": "..."}
            - 'component': Retorna create_alert con mensaje de error
            - 'str': Retorna "0" o string vacío
            - 'currency': Retorna format_currency(0)
            - 'list': Retorna []
            - 'int': Retorna 0
            - 'float': Retorna 0.0
            - 'bool': Retorna False
            - 'none': Retorna None
            - 'no_update': Retorna no_update

    Returns:
        Valor único o tupla de valores según cantidad de outputs

    Example:
        ```python
        # Callback con múltiples outputs de diferentes tipos
        @app.callback(
            [Output('data-store', 'data'),        # dict
             Output('error-alert', 'children'),   # component
             Output('count-label', 'children')],  # str
            Input('trigger', 'n_clicks')
        )
        def my_callback(n_clicks):
            try:
                pharmacy_id = get_current_pharmacy_id()
            except ValueError:
                # Retornar valores de error estandarizados
                return get_auth_error_return(['dict', 'component', 'str'])

            # Lógica normal del callback
            return data, None, "100"
        ```

    Note:
        - Si solo hay 1 output, retorna valor directo (no tupla)
        - Si hay múltiples outputs, retorna tupla en el orden correcto
        - Usar con decorator @require_pharmacy_id (recomendado) evita necesidad de esto
    """
    # Mapeo de tipos a valores de error
    error_map = {
        "dict": {
            "error": "auth_error",
            "message": "No se pudo identificar la farmacia del usuario. Recargue la página.",
        },
        "component": create_alert("Error de autenticación. Por favor, recargue la página.", "danger"),
        "str": "0",
        "currency": format_currency(0),
        "list": [],
        "int": 0,
        "float": 0.0,
        "bool": False,
        "none": None,
        "no_update": no_update,
    }

    # Construir lista de returns según tipos especificados
    returns = []
    for output_type in output_types:
        error_value = error_map.get(output_type.lower(), None)
        returns.append(error_value)

    # Retornar valor único o tupla según cantidad de outputs
    if len(returns) == 1:
        return returns[0]
    else:
        return tuple(returns)


def get_api_error_return(
    output_types: List[str], error_message: str = "Error al cargar datos. Intente nuevamente."
) -> Union[Any, Tuple[Any, ...]]:
    """
    Retorna valores de error apropiados cuando falla una llamada a la API.

    Similar a get_auth_error_return() pero para errores de API (timeouts,
    errores 500, etc).

    Args:
        output_types: Lista de tipos de Output del callback
        error_message: Mensaje de error personalizado

    Returns:
        Valor único o tupla de valores según cantidad de outputs

    Example:
        ```python
        @app.callback(...)
        def my_callback(...):
            try:
                response = api_client.get(endpoint)
            except RequestException as e:
                return get_api_error_return(
                    ['dict', 'component'],
                    error_message=f"Error al cargar datos: {str(e)}"
                )

            return response, None
        ```
    """
    error_map = {
        "dict": {"error": "api_error", "message": error_message},
        "component": create_alert(error_message, "warning"),
        "str": "--",
        "currency": format_currency(0),
        "list": [],
        "int": 0,
        "float": 0.0,
        "bool": False,
        "none": None,
        "no_update": no_update,
    }

    returns = []
    for output_type in output_types:
        error_value = error_map.get(output_type.lower(), None)
        returns.append(error_value)

    if len(returns) == 1:
        return returns[0]
    else:
        return tuple(returns)


def get_empty_data_return(output_types: List[str]) -> Union[Any, Tuple[Any, ...]]:
    """
    Retorna valores apropiados cuando no hay datos disponibles (caso válido).

    A diferencia de errores, esto representa ausencia válida de datos
    (ej: usuario no tiene ventas en el período seleccionado).

    Args:
        output_types: Lista de tipos de Output del callback

    Returns:
        Valor único o tupla de valores según cantidad de outputs

    Example:
        ```python
        @app.callback(...)
        def my_callback(...):
            data = fetch_sales_data(...)

            if not data or len(data) == 0:
                # No hay datos (no es error, es caso válido)
                return get_empty_data_return(['list', 'component'])

            return data, None
        ```
    """
    empty_map = {
        "dict": {},
        "component": html.Div(
            "No hay datos disponibles para el período seleccionado.", className="text-muted text-center py-3"
        ),
        "str": "--",
        "currency": format_currency(0),
        "list": [],
        "int": 0,
        "float": 0.0,
        "bool": False,
        "none": None,
        "no_update": no_update,
    }

    returns = []
    for output_type in output_types:
        empty_value = empty_map.get(output_type.lower(), None)
        returns.append(empty_value)

    if len(returns) == 1:
        return returns[0]
    else:
        return tuple(returns)


# Shortcuts para casos comunes
def auth_error_dict() -> dict:
    """Shortcut para retornar error de autenticación como dict."""
    return get_auth_error_return(["dict"])


def auth_error_component():
    """Shortcut para retornar error de autenticación como componente Dash."""
    return get_auth_error_return(["component"])


def api_error_dict(message: str = "Error al cargar datos") -> dict:
    """Shortcut para retornar error de API como dict."""
    return get_api_error_return(["dict"], error_message=message)


def api_error_component(message: str = "Error al cargar datos"):
    """Shortcut para retornar error de API como componente Dash."""
    return get_api_error_return(["component"], error_message=message)


if __name__ == "__main__":
    # Tests de demostración
    print("=== Tests de Error Handlers ===\n")

    print("1. Auth error con múltiples outputs:")
    result = get_auth_error_return(["dict", "component", "str"])
    print(f"   Resultado: {type(result)} con {len(result)} elementos\n")

    print("2. Auth error con un solo output:")
    result = get_auth_error_return(["dict"])
    print(f"   Resultado: {result}\n")

    print("3. API error con mensaje personalizado:")
    result = get_api_error_return(["component"], "Timeout al conectar con el servidor")
    print(f"   Resultado: {type(result)}\n")

    print("4. Empty data return:")
    result = get_empty_data_return(["list", "str"])
    print(f"   Resultado: {result}\n")

    print("5. Shortcuts:")
    print(f"   auth_error_dict(): {auth_error_dict()}")
    print(f"   api_error_dict(): {api_error_dict()}")
