"""
Helpers reutilizables para drill-down temporal (quarter → month → fortnight).

Issue #346: Extraído de callbacks/generics.py para evitar duplicación.
Este módulo centraliza toda la lógica de drill-down temporal que antes estaba duplicada
entre diferentes callbacks (temporal-drill-store y homogeneous-drill-store).

Estos helpers permiten implementar navegación temporal drill-down de forma consistente
en cualquier gráfico de la aplicación.

Author: Claude Code
Date: 2025-10-30
"""

from typing import Tuple, Optional, Dict, List
import structlog

logger = structlog.get_logger(__name__)

# Constantes
DRILL_LEVELS = ["quarter", "month", "fortnight"]
DRILL_LEVEL_NAMES = {
    "quarter": "Trimestre",
    "month": "Mes",
    "fortnight": "Quincena"
}


def parse_drill_down_click(click_data: Optional[Dict]) -> Optional[str]:
    """
    Extrae el período clickeado del clickData de Plotly.

    Args:
        click_data: Click data de Plotly graph (formato: {"points": [{"x": "2024 Q1"}]})

    Returns:
        str: Período clickeado (ej: "2024 Q1", "2024-01", "2024-01-01 to 2024-01-15")
        None: Si no hay click válido

    Examples:
        >>> parse_drill_down_click({"points": [{"x": "2024 Q1"}]})
        "2024 Q1"

        >>> parse_drill_down_click(None)
        None

        >>> parse_drill_down_click({})
        None
    """
    if not click_data or "points" not in click_data:
        return None

    try:
        return click_data["points"][0]["x"]
    except (KeyError, IndexError, TypeError):
        return None


def calculate_next_drill_level(
    current_level: str,
    current_path: List[str],
    clicked_period: str
) -> Tuple[Optional[str], Optional[List[str]]]:
    """
    Calcula el siguiente nivel de drill-down basado en el nivel actual.

    Lógica de drill-down:
    - quarter (path=[]) → click → month (path=["2024 Q1"])
    - month (path=["2024 Q1"]) → click → fortnight (path=["2024 Q1", "2024-01"])
    - fortnight (path=["2024 Q1", "2024-01"]) → click → NO-OP (ya estamos en nivel más bajo)

    Args:
        current_level: Nivel actual ('quarter', 'month', 'fortnight')
        current_path: Path actual de drill-down (lista de períodos navegados)
        clicked_period: Período clickeado para agregar al path

    Returns:
        Tuple[new_level, new_path]: Nuevo nivel y path actualizados
        Tuple[None, None]: Si no se puede hacer más drill-down (ya en fortnight)

    Examples:
        >>> calculate_next_drill_level('quarter', [], '2024 Q1')
        ('month', ['2024 Q1'])

        >>> calculate_next_drill_level('month', ['2024 Q1'], '2024-01')
        ('fortnight', ['2024 Q1', '2024-01'])

        >>> calculate_next_drill_level('fortnight', ['2024 Q1', '2024-01'], '2024-01-01 to 2024-01-15')
        (None, None)  # Ya estamos en el nivel más bajo
    """
    # Validar nivel actual
    if current_level not in DRILL_LEVELS:
        logger.warning(f"[DRILL_DOWN] Invalid drill level: {current_level}")
        return None, None

    # Quarter → Month
    if current_level == "quarter" and len(current_path) == 0:
        new_level = "month"
        new_path = [clicked_period]
        logger.debug(f"[DRILL_DOWN] quarter → month: {new_path}")
        return new_level, new_path

    # Month → Fortnight
    elif current_level == "month" and len(current_path) == 1:
        new_level = "fortnight"
        new_path = current_path + [clicked_period]
        logger.debug(f"[DRILL_DOWN] month → fortnight: {new_path}")
        return new_level, new_path

    # Fortnight → NO-OP (ya estamos en el nivel más bajo)
    elif current_level == "fortnight":
        logger.debug("[DRILL_DOWN] Already at lowest level (fortnight)")
        return None, None

    # Inconsistencia: path no coincide con nivel esperado
    else:
        logger.warning(
            f"[DRILL_DOWN] Inconsistent state: level={current_level}, path_length={len(current_path)}"
        )
        return None, None


def calculate_previous_drill_level(current_path: List[str]) -> Tuple[str, List[str], bool]:
    """
    Calcula el nivel anterior de drill-up eliminando el último período del path.

    Lógica de drill-up:
    - fortnight (path=["2024 Q1", "2024-01"]) → month (path=["2024 Q1"])
    - month (path=["2024 Q1"]) → quarter (path=[])
    - quarter (path=[]) → NO-OP (deshabilitar botón drill-up)

    Args:
        current_path: Path actual de drill-down (lista de períodos navegados)

    Returns:
        Tuple[new_level, new_path, is_disabled]:
            - new_level: Nivel anterior según longitud del path reducido
            - new_path: Path sin el último elemento
            - is_disabled: True si llegamos a nivel raíz (quarter), False en otro caso

    Examples:
        >>> calculate_previous_drill_level(['2024 Q1', '2024-01'])
        ('month', ['2024 Q1'], False)  # De fortnight a month

        >>> calculate_previous_drill_level(['2024 Q1'])
        ('quarter', [], True)  # De month a quarter (nivel raíz)

        >>> calculate_previous_drill_level([])
        ('quarter', [], True)  # Ya en raíz, mantener estado
    """
    # Caso especial: ya estamos en raíz (quarter)
    if len(current_path) == 0:
        logger.debug("[DRILL_UP] Already at root level (quarter)")
        return "quarter", [], True  # Deshabilitar botón

    # Eliminar último elemento del path
    new_path = current_path[:-1]

    # Determinar nivel según longitud del nuevo path
    if len(new_path) == 1:
        # De fortnight (path=["Q", "M"]) → month (path=["Q"])
        new_level = "month"
        is_disabled = False  # Aún podemos volver a quarter
        logger.debug(f"[DRILL_UP] fortnight → month: {new_path}")

    elif len(new_path) == 0:
        # De month (path=["Q"]) → quarter (path=[])
        new_level = "quarter"
        is_disabled = True  # Deshabilitar en nivel raíz
        logger.debug("[DRILL_UP] month → quarter (root)")

    else:
        # Caso excepcional: path con longitud > 2 (no debería ocurrir)
        logger.warning(f"[DRILL_UP] Unexpected path length: {len(new_path)}")
        new_level = "quarter"
        is_disabled = True

    return new_level, new_path, is_disabled


def initialize_drill_state() -> Dict[str, any]:
    """
    Inicializa un estado de drill-down en el nivel raíz (quarter).

    Returns:
        Dict con estructura: {"level": "quarter", "path": []}

    Example:
        >>> initialize_drill_state()
        {"level": "quarter", "path": []}
    """
    return {"level": "quarter", "path": []}


def get_drill_level_display_name(level: str) -> str:
    """
    Obtiene el nombre en español del nivel de drill-down.

    Args:
        level: Nivel de drill ('quarter', 'month', 'fortnight')

    Returns:
        str: Nombre en español ('Trimestre', 'Mes', 'Quincena')

    Example:
        >>> get_drill_level_display_name('quarter')
        'Trimestre'
    """
    return DRILL_LEVEL_NAMES.get(level, level.capitalize())


# =============================================================================
# QUARTER HELPERS (Issue #436 - Prescription Dashboard)
# =============================================================================

def get_quarter_from_period(period: str) -> str:
    """
    Convierte período YYYY-MM a formato trimestral YYYY-Q#.

    Args:
        period: Período en formato YYYY-MM (ej: "2025-01")

    Returns:
        str: Trimestre en formato YYYY-Q# (ej: "2025-Q1")

    Examples:
        >>> get_quarter_from_period("2025-01")
        "2025-Q1"
        >>> get_quarter_from_period("2025-06")
        "2025-Q2"
        >>> get_quarter_from_period("2025-12")
        "2025-Q4"
    """
    try:
        year, month = period.split("-")
        quarter = (int(month) - 1) // 3 + 1
        return f"{year}-Q{quarter}"
    except (ValueError, IndexError):
        logger.warning(f"[get_quarter_from_period] Invalid period format: {period}")
        return period


def get_quarter_label(quarter: str) -> str:
    """
    Convierte formato trimestral a etiqueta legible en español.

    Args:
        quarter: Trimestre en formato YYYY-Q# (ej: "2025-Q1")

    Returns:
        str: Etiqueta legible (ej: "T1 2025")

    Examples:
        >>> get_quarter_label("2025-Q1")
        "T1 2025"
        >>> get_quarter_label("2024-Q4")
        "T4 2024"
    """
    try:
        year, q = quarter.split("-Q")
        quarter_names = {"1": "T1", "2": "T2", "3": "T3", "4": "T4"}
        return f"{quarter_names.get(q, f'T{q}')} {year}"
    except (ValueError, IndexError):
        logger.warning(f"[get_quarter_label] Invalid quarter format: {quarter}")
        return quarter


def months_in_quarter(quarter: str) -> List[str]:
    """
    Retorna la lista de meses que pertenecen a un trimestre.

    Args:
        quarter: Trimestre en formato YYYY-Q# (ej: "2025-Q1")

    Returns:
        List[str]: Lista de meses en formato YYYY-MM

    Examples:
        >>> months_in_quarter("2025-Q1")
        ["2025-01", "2025-02", "2025-03"]
        >>> months_in_quarter("2025-Q4")
        ["2025-10", "2025-11", "2025-12"]
    """
    try:
        year, q = quarter.split("-Q")
        q_num = int(q)
        start_month = (q_num - 1) * 3 + 1
        return [f"{year}-{str(m).zfill(2)}" for m in range(start_month, start_month + 3)]
    except (ValueError, IndexError):
        logger.warning(f"[months_in_quarter] Invalid quarter format: {quarter}")
        return []


def validate_drill_state(drill_state: Optional[Dict]) -> bool:
    """
    Valida que el estado de drill-down sea consistente.

    Verifica:
    - Estructura correcta (level y path presentes)
    - Nivel válido (quarter, month, fortnight)
    - Path coherente con nivel (quarter=0, month=1, fortnight=2)

    Args:
        drill_state: Estado de drill-down a validar

    Returns:
        bool: True si el estado es válido, False en caso contrario

    Examples:
        >>> validate_drill_state({"level": "quarter", "path": []})
        True

        >>> validate_drill_state({"level": "month", "path": ["2024 Q1"]})
        True

        >>> validate_drill_state({"level": "quarter", "path": ["2024 Q1"]})
        False  # Inconsistente: quarter debe tener path=[]
    """
    if not drill_state or not isinstance(drill_state, dict):
        return False

    # Verificar estructura
    if "level" not in drill_state or "path" not in drill_state:
        return False

    level = drill_state["level"]
    path = drill_state["path"]

    # Verificar nivel válido
    if level not in DRILL_LEVELS:
        return False

    # Verificar coherencia path vs nivel
    expected_path_length = DRILL_LEVELS.index(level)

    if len(path) != expected_path_length:
        logger.warning(
            f"[DRILL_STATE] Inconsistent state: level={level} expects path length {expected_path_length}, "
            f"but got {len(path)}"
        )
        return False

    return True
