"""
Debug Feature Flags para xFarma Frontend.

Sistema de feature flags para activar/desactivar logging de debugging
sin impactar performance en producción.

Usage:
    from utils.debug_flags import CACHE_DEBUG_ENABLED, log_cache_debug

    # Opción 1: Guard directo
    if CACHE_DEBUG_ENABLED:
        logger.info("[CACHE_DEBUG] ...")

    # Opción 2: Helper function
    log_cache_debug("Mi mensaje de debug")

Configuration:
    Set environment variable CACHE_DEBUG_ENABLED=true to enable cache debugging.
    Default: false in production, true in development (.env.development)
"""

import os
import logging

logger = logging.getLogger(__name__)

# ============================================================================
# FEATURE FLAGS
# ============================================================================

# Cache Debugging: Logs detallados de cache hits/misses/age
# Use case: Debugging de rate limiting, cache invalidation, multi-worker persistence
# Cost: Minimal (~4-6 log statements per callback when enabled)
CACHE_DEBUG_ENABLED = os.getenv("CACHE_DEBUG_ENABLED", "false").lower() in ["true", "1", "yes"]

# ============================================================================
# HELPER FUNCTIONS
# ============================================================================

def log_cache_debug(message: str, level: str = "info"):
    """
    Log cache debug message si feature flag está habilitado.

    Args:
        message: Mensaje a loggear (sin prefix [CACHE_DEBUG])
        level: Nivel de log ("info", "debug", "warning", "error")

    Example:
        log_cache_debug("laboratory-cache-store vacío, inicializando...")
        log_cache_debug("Cache age: 45 min (TTL: 60 min)", level="debug")
    """
    if not CACHE_DEBUG_ENABLED:
        return

    # Agregar prefix automáticamente
    prefixed_message = f"[CACHE_DEBUG] {message}"

    # Log según nivel especificado
    log_method = getattr(logger, level.lower(), logger.info)
    log_method(prefixed_message)


# ============================================================================
# INITIALIZATION LOGGING
# ============================================================================

# Log de inicialización (solo si está habilitado)
# Ayuda a confirmar configuración en startup de la app
if CACHE_DEBUG_ENABLED:
    logger.warning(
        "🔍 [CACHE_DEBUG] Feature flag ENABLED - Cache debugging logs activos. "
        "Set CACHE_DEBUG_ENABLED=false para desactivar en producción."
    )
else:
    logger.debug("[DEBUG_FLAGS] CACHE_DEBUG_ENABLED=false (production mode)")
