# frontend/utils/dash_component_validator.py
"""
Validador estático de componentes Dash para prevenir errores comunes.
Detecta antipatrones ANTES de que causen errores en runtime.

Antipatrones detectados:
1. DASH001: Listas Python como children/label sin wrapper (causa "Objects are not valid as a React child")
2. DASH002: Outputs duplicados en callbacks (viola REGLA #11)
3. DASH003: Uso de callback_context deprecado (Dash 3.x)
4. DASH004: Importaciones circulares (causa errores difíciles de debuggear)
5. DASH005: Atributos HTML inválidos en dbc.* (ej: aria-label no aceptado por dbc.Button)
6. DASH006: Callbacks con prevent_initial_call=False + API calls SIN auth_state (causa timeouts 180s en sesiones no autenticadas)

Uso:
    python frontend/utils/dash_component_validator.py [archivo.py]
    python frontend/utils/dash_component_validator.py --all  # Validar todos los archivos frontend
    python frontend/utils/dash_component_validator.py --all --strict  # Exit 1 si hay errores
"""

import ast
import sys
from dataclasses import dataclass
from pathlib import Path
from typing import Dict, List, Tuple

# Fix Windows console encoding BEFORE any print
if sys.platform == "win32":
    # Reconfigure stdout to use UTF-8
    sys.stdout.reconfigure(encoding="utf-8")


@dataclass
class ValidationIssue:
    """Representa un problema detectado en el código."""

    file: str
    line: int
    column: int
    severity: str  # 'error', 'warning', 'info'
    code: str  # Código del error (ej: 'DASH001')
    message: str
    suggestion: str


class DashComponentValidator(ast.NodeVisitor):
    """
    Visitor de AST que detecta antipatrones en componentes Dash.
    """

    # Componentes Dash que aceptan children/label
    CONTAINER_COMPONENTS = {
        "DropdownMenu",
        "DropdownMenuItem",
        "Button",
        "Badge",
        "Card",
        "CardBody",
        "Alert",
        "Toast",
        "Modal",
        "Tooltip",
        "Div",
        "Span",
        "P",
        "H1",
        "H2",
        "H3",
        "H4",
        "H5",
        "H6",
        "BaseCard",
        "BaseButton",
        "Title",
    }

    # Componentes HTML que SÍ aceptan listas directamente (son wrappers)
    HTML_WRAPPER_COMPONENTS = {
        "Div",
        "Span",
        "P",
        "H1",
        "H2",
        "H3",
        "H4",
        "H5",
        "H6",
        "Section",
        "Article",
        "Header",
        "Footer",
        "Nav",
        "Main",
        "Ul",
        "Ol",
        "Li",
        "Dl",
        "Dt",
        "Dd",
        "Table",
        "Thead",
        "Tbody",
        "Tr",
        "Td",
        "Th",
        "Form",
        "Fieldset",
        "Legend",
    }

    # Parámetros que requieren componentes envueltos
    WRAPPER_REQUIRED_PARAMS = {"children", "label", "title"}

    def __init__(self, filename: str):
        self.filename = filename
        self.issues: List[ValidationIssue] = []
        self.callback_outputs: Dict[str, List[int]] = {}  # ID -> [line_numbers]
        self.current_callback_info: Dict = {}  # Info del callback siendo analizado

    def visit_Call(self, node: ast.Call) -> None:
        """Visita llamadas a funciones (componentes y callbacks)."""

        # Detectar componentes Dash con listas directas
        if self._is_dash_component(node):
            self._check_wrapper_antipattern(node)
            # DASH005: Detectar atributos HTML inválidos en dbc.*
            self._check_invalid_html_attrs(node)

        # Detectar callbacks con outputs duplicados
        if self._is_callback_decorator(node):
            self._check_duplicate_outputs(node)

        # Detectar uso de callback_context deprecado
        if self._is_deprecated_callback_context(node):
            self._add_issue(
                node,
                "error",
                "DASH003",
                "Uso de dash.callback_context deprecado en Dash 3.x",
                'Usar "from dash import ctx" en su lugar',
            )

        self.generic_visit(node)

    def visit_Attribute(self, node: ast.Attribute) -> None:
        """Visita acceso a atributos (ej: dash.callback_context)."""

        # Detectar dash.callback_context
        if isinstance(node.value, ast.Name) and node.value.id == "dash" and node.attr == "callback_context":
            self._add_issue(
                node,
                "error",
                "DASH003",
                "Uso de dash.callback_context deprecado en Dash 3.x",
                'Usar "from dash import ctx" en su lugar',
            )

        self.generic_visit(node)

    def _is_dash_component(self, node: ast.Call) -> bool:
        """Detecta si es una llamada a componente Dash."""
        if isinstance(node.func, ast.Attribute):
            # dbc.Button(...), html.Div(...)
            return node.func.attr in self.CONTAINER_COMPONENTS
        elif isinstance(node.func, ast.Name):
            # Button(...), Div(...)
            return node.func.id in self.CONTAINER_COMPONENTS
        return False

    def _is_html_wrapper(self, node: ast.Call) -> bool:
        """Detecta si es un componente HTML wrapper que acepta listas."""
        if isinstance(node.func, ast.Attribute):
            # html.Div(...), html.Span(...)
            return node.func.attr in self.HTML_WRAPPER_COMPONENTS
        elif isinstance(node.func, ast.Name):
            # Div(...), Span(...)
            return node.func.id in self.HTML_WRAPPER_COMPONENTS
        return False

    def _get_component_name(self, node: ast.Call) -> str:
        """Obtiene el nombre completo del componente (ej: 'dbc.Button', 'html.Div')."""
        if isinstance(node.func, ast.Attribute):
            # dbc.Button(...), html.Div(...)
            if isinstance(node.func.value, ast.Name):
                return f"{node.func.value.id}.{node.func.attr}"
        elif isinstance(node.func, ast.Name):
            # Button(...), Div(...)
            return node.func.id
        return ""

    def _check_wrapper_antipattern(self, node: ast.Call) -> None:
        """
        Detecta listas Python usadas directamente como children/label.

        Antipatrón:
            dbc.Button([html.I(...), "Texto"], ...)  # ❌ Lista directa

        Correcto:
            dbc.Button(html.Div([html.I(...), "Texto"]), ...)  # ✅ Wrapper

        Excepción:
            html.Div([...]) SÍ acepta listas - es un wrapper HTML puro
        """
        # Excluir HTML wrappers (Div, Span, etc.) que SÍ aceptan listas
        if self._is_html_wrapper(node):
            return  # HTML wrappers pueden tener listas directamente

        # Revisar argumentos posicionales (children es el primero)
        if node.args:
            first_arg = node.args[0]
            if isinstance(first_arg, ast.List):
                # Lista directa como children
                if self._list_contains_components_and_strings(first_arg):
                    self._add_issue(
                        node,
                        "error",
                        "DASH001",
                        "Lista Python usada directamente como children sin wrapper",
                        "Envolver en html.Div([...]) o html.Span([...])",
                    )

        # Revisar argumentos nombrados (label=, title=, etc.)
        for keyword in node.keywords:
            if keyword.arg in self.WRAPPER_REQUIRED_PARAMS:
                if isinstance(keyword.value, ast.List):
                    if self._list_contains_components_and_strings(keyword.value):
                        self._add_issue(
                            node,
                            "error",
                            "DASH001",
                            f"Lista Python usada directamente como {keyword.arg}= sin wrapper",
                            f"Usar {keyword.arg}=html.Div([...]) en lugar de {keyword.arg}=[...]",
                        )

    def _list_contains_components_and_strings(self, list_node: ast.List) -> bool:
        """
        Detecta si la lista contiene mezcla de componentes y strings.
        Esto es problemático porque React no puede renderizarlo.
        """
        has_component = False
        has_string = False

        for elem in list_node.elts:
            if isinstance(elem, ast.Call):
                has_component = True
            elif isinstance(elem, ast.Constant) and isinstance(elem.value, str):
                has_string = True

        return has_component and has_string

    def _check_invalid_html_attrs(self, node: ast.Call) -> None:
        """
        Detecta uso de atributos HTML inválidos en componentes dbc.* (DASH005).

        Antipatrón:
            dbc.Button(..., **{"aria-label": "..."})  # ❌ dbc.* NO acepta aria-label

        Correcto:
            dbc.Button(..., title="...")  # ✅ Usar 'title' para accesibilidad
            html.Nav(..., **{"aria-label": "..."})  # ✅ html.* SÍ acepta atributos HTML
        """
        # Lista de atributos HTML que NO son aceptados por dbc.* components
        INVALID_DBC_ATTRS = {
            "aria-label",
            "aria-labelledby",
            "aria-describedby",
            "role",
            "tabindex",
        }

        # Verificar si es un componente dbc.*
        component_name = self._get_component_name(node)
        if not component_name or not component_name.startswith("dbc."):
            return  # Solo validar componentes dbc.*

        # Buscar **{...} en keywords
        for keyword in node.keywords:
            if keyword.arg is None:  # **kwargs
                if isinstance(keyword.value, ast.Dict):
                    # Iterar sobre las claves del diccionario
                    for key in keyword.value.keys:
                        if isinstance(key, ast.Constant):
                            attr_name = key.value
                            if attr_name in INVALID_DBC_ATTRS:
                                suggestion = {
                                    "aria-label": "Usar 'title' en su lugar para accesibilidad",
                                    "aria-labelledby": "Los componentes dbc.* no soportan este atributo",
                                    "aria-describedby": "Los componentes dbc.* no soportan este atributo",
                                    "role": "Los componentes dbc.* ya tienen roles semánticos predefinidos",
                                    "tabindex": "Usar 'disabled' o modificar el orden del DOM",
                                }.get(attr_name, "Consultar documentación de Dash Bootstrap Components")

                                self._add_issue(
                                    node,
                                    "error",
                                    "DASH005",
                                    f"Componente {component_name} no acepta el atributo HTML '{attr_name}'",
                                    suggestion,
                                )

    def _is_callback_decorator(self, node: ast.Call) -> bool:
        """Detecta si es un decorador @app.callback."""
        if isinstance(node.func, ast.Attribute):
            return node.func.attr == "callback"
        elif isinstance(node.func, ast.Name):
            return node.func.id == "callback"
        return False

    def _check_duplicate_outputs(self, node: ast.Call) -> None:
        """
        Detecta outputs duplicados en callbacks (REGLA #11).

        Antipatrón:
            @app.callback(Output('foo', 'children'), Input('bar', 'value'))
            def callback1(): ...

            @app.callback(Output('foo', 'children'), Input('baz', 'value'))  # ❌ Duplicado
            def callback2(): ...

        Excepción válida (Dash oficial):
            @app.callback(
                Output('foo', 'children', allow_duplicate=True),
                Input('bar', 'value')
            )
            def callback1(): ...

            @app.callback(
                Output('foo', 'children', allow_duplicate=True),  # ✅ Permitido
                Input('baz', 'value')
            )
            def callback2(): ...
        """
        outputs = []

        for arg in node.args:
            if self._is_output_call(arg):
                # Verificar si tiene allow_duplicate=True
                if self._has_allow_duplicate(arg):
                    # Output con allow_duplicate=True es válido
                    continue

                # Extraer ID del output
                output_id = self._extract_output_id(arg)
                if output_id:
                    outputs.append(output_id)
            elif isinstance(arg, ast.List):
                # Lista de outputs
                for elem in arg.elts:
                    if self._is_output_call(elem):
                        # Verificar allow_duplicate en cada elemento
                        if self._has_allow_duplicate(elem):
                            continue

                        output_id = self._extract_output_id(elem)
                        if output_id:
                            outputs.append(output_id)

        # Registrar outputs encontrados
        for output_id in outputs:
            if output_id in self.callback_outputs:
                # Output duplicado detectado
                previous_line = self.callback_outputs[output_id][0]
                self._add_issue(
                    node,
                    "error",
                    "DASH002",
                    f"Output duplicado detectado: {output_id}",
                    f"Ya definido en línea {previous_line}. Ver REGLA #11: Un Input → Un Callback. "
                    f"Si necesitas múltiples callbacks para el mismo output, usa allow_duplicate=True",
                )
            else:
                self.callback_outputs[output_id] = [node.lineno]

    def _is_output_call(self, node: ast.AST) -> bool:
        """Detecta si es una llamada a Output()."""
        return isinstance(node, ast.Call) and isinstance(node.func, ast.Name) and node.func.id == "Output"

    def _extract_output_id(self, node: ast.Call) -> str:
        """Extrae el ID de un Output('component-id', 'property')."""
        if len(node.args) >= 2:
            if isinstance(node.args[0], ast.Constant):
                component_id = node.args[0].value
                if isinstance(node.args[1], ast.Constant):
                    prop = node.args[1].value
                    return f"{component_id}.{prop}"
        return None

    def _has_allow_duplicate(self, node: ast.Call) -> bool:
        """
        Verifica si un Output tiene allow_duplicate=True.

        Reconoce patrones según documentación oficial de Dash:
        - Output('id', 'prop', allow_duplicate=True)
        - Output(..., allow_duplicate=True)

        Retorna True si tiene allow_duplicate=True, False en caso contrario.
        """
        if not isinstance(node, ast.Call):
            return False

        # Buscar en argumentos nombrados (keywords)
        for keyword in node.keywords:
            if keyword.arg == "allow_duplicate":
                # Verificar si el valor es True
                if isinstance(keyword.value, ast.Constant):
                    return keyword.value.value is True
                # También aceptar ast.NameConstant para compatibilidad con versiones antiguas
                elif hasattr(ast, "NameConstant") and isinstance(keyword.value, ast.NameConstant):
                    return keyword.value.value is True

        return False

    def _is_deprecated_callback_context(self, node: ast.Call) -> bool:
        """Detecta uso de dash.callback_context (deprecado)."""
        if isinstance(node.func, ast.Attribute):
            if (
                isinstance(node.func.value, ast.Name)
                and node.func.value.id == "dash"
                and node.func.attr == "callback_context"
            ):
                return True
        return False

    def _add_issue(self, node: ast.AST, severity: str, code: str, message: str, suggestion: str) -> None:
        """Registra un problema detectado."""
        issue = ValidationIssue(
            file=self.filename,
            line=node.lineno,
            column=node.col_offset,
            severity=severity,
            code=code,
            message=message,
            suggestion=suggestion,
        )
        self.issues.append(issue)

    def visit_FunctionDef(self, node: ast.FunctionDef) -> None:
        """
        Visita definiciones de funciones para detectar callbacks inseguros (DASH006).

        Detecta callbacks con prevent_initial_call=False que hacen API calls
        sin verificar autenticación (State("auth-state", "data")).
        """
        # Verificar si esta función es un callback decorado
        for decorator in node.decorator_list:
            if self._is_callback_decorator_node(decorator):
                # Analizar el callback
                self._check_unsafe_unauthed_callback(node, decorator)
                break

        self.generic_visit(node)

    def _is_callback_decorator_node(self, decorator: ast.AST) -> bool:
        """Detecta si un decorador es @app.callback o @callback."""
        if isinstance(decorator, ast.Call):
            if isinstance(decorator.func, ast.Attribute):
                return decorator.func.attr == "callback"
            elif isinstance(decorator.func, ast.Name):
                return decorator.func.id == "callback"
        return False

    def _check_unsafe_unauthed_callback(self, func_node: ast.FunctionDef, decorator: ast.Call) -> None:
        """
        Detecta callbacks con prevent_initial_call=False + API calls sin auth_state (DASH006).

        Patrón detectado:
            @app.callback(..., prevent_initial_call=False)
            def my_callback(...):  # Sin State("auth-state", "data")
                api_client.get(...)  # API call sin verificación auth

        Solución:
            @app.callback(..., State("auth-state", "data"), prevent_initial_call=False)
            def my_callback(..., auth_state):
                if not is_user_authenticated(auth_state):
                    raise PreventUpdate
                api_client.get(...)
        """
        # 1. Verificar prevent_initial_call=False
        has_prevent_initial_false = self._has_prevent_initial_call_false(decorator)
        if not has_prevent_initial_false:
            return  # Solo analizar callbacks con prevent_initial_call=False

        # 2. Verificar si tiene State("auth-state", "data")
        has_auth_state = self._has_auth_state_param(decorator)

        # 3. Verificar si hace API calls en el cuerpo de la función
        has_api_calls = self._contains_api_calls(func_node)

        # 4. DASH006: prevent_initial_call=False + API calls - auth_state = VULNERABLE
        if has_api_calls and not has_auth_state:
            self._add_issue(
                func_node,
                "error",
                "DASH006",
                f"Callback '{func_node.name}' con prevent_initial_call=False hace API calls "
                "sin verificar autenticación (auth-state missing)",
                "Agregar State('auth-state', 'data') y verificar autenticación con "
                "is_user_authenticated(auth_state) antes de hacer API calls. "
                "Ver Issue #403 para patrón correcto."
            )

    def _has_prevent_initial_call_false(self, decorator: ast.Call) -> bool:
        """Verifica si el callback tiene prevent_initial_call=False."""
        for keyword in decorator.keywords:
            if keyword.arg == "prevent_initial_call":
                if isinstance(keyword.value, ast.Constant):
                    return keyword.value.value is False
                elif hasattr(ast, "NameConstant") and isinstance(keyword.value, ast.NameConstant):
                    return keyword.value.value is False
        return False  # Default es False en Dash, pero para DASH006 solo flagear explícito

    def _has_auth_state_param(self, decorator: ast.Call) -> bool:
        """
        Verifica si el callback tiene State("auth-state", "data") en sus parámetros.

        Busca en Input/State lists del decorator.
        """
        for arg in decorator.args:
            # Revisar listas de Input/State
            if isinstance(arg, ast.List):
                for elem in arg.elts:
                    if self._is_auth_state_call(elem):
                        return True
            # Revisar State individual
            elif self._is_auth_state_call(arg):
                return True

        return False

    def _is_auth_state_call(self, node: ast.AST) -> bool:
        """Detecta si un nodo es State("auth-state", "data")."""
        if isinstance(node, ast.Call):
            if isinstance(node.func, ast.Name) and node.func.id == "State":
                if len(node.args) >= 2:
                    if isinstance(node.args[0], ast.Constant) and node.args[0].value == "auth-state":
                        if isinstance(node.args[1], ast.Constant) and node.args[1].value == "data":
                            return True
        return False

    def _contains_api_calls(self, func_node: ast.FunctionDef) -> bool:
        """
        Detecta si el cuerpo de la función contiene llamadas HTTP REALES.

        Patrones detectados (SOLO HTTP calls reales):
        - request_coordinator.make_request(...)  ✅ HTTP call real
        - api_client.get/post/put/delete(...)    ✅ HTTP call real
        - requests.get/post/put/delete(...)      ✅ HTTP call real
        - httpx.get/post/put/delete(...)         ✅ HTTP call real

        Patrones EXCLUIDOS (no son HTTP):
        - logger.info/debug(...)         ❌ Logging
        - create_*(), render_*()         ❌ UI helpers
        - format_*(), calculate_*()      ❌ Pure functions
        - cualquier función con "request" en nombre ❌ Helpers
        """
        # HTTP client objects (solo estos hacen HTTP calls reales)
        HTTP_CLIENTS = {"request_coordinator", "api_client", "requests", "httpx", "aiohttp"}

        # Métodos HTTP (solo en contexto de HTTP clients)
        HTTP_METHODS = {"get", "post", "put", "delete", "patch", "make_request", "request"}

        for node in ast.walk(func_node):
            if isinstance(node, ast.Call):
                # Patrón: request_coordinator.make_request()
                if isinstance(node.func, ast.Attribute):
                    if isinstance(node.func.value, ast.Name):
                        client_name = node.func.value.id
                        method_name = node.func.attr

                        # ✅ SOLO HTTP clients conocidos
                        if client_name in HTTP_CLIENTS:
                            # Verificar que usa método HTTP
                            if method_name in HTTP_METHODS:
                                return True

        return False


class CircularImportDetector:
    """
    Detector de importaciones circulares en archivos Python.
    Construye un grafo de dependencias y detecta ciclos.
    """

    def __init__(self, root_dir: Path):
        self.root_dir = root_dir
        self.import_graph: Dict[str, List[str]] = {}
        self.file_imports: Dict[str, List[Tuple[str, int]]] = {}  # file -> [(import, line)]

    def extract_imports(self, filepath: Path) -> List[Tuple[str, int]]:
        """
        Extrae imports de un archivo Python.
        Retorna lista de tuplas (módulo_importado, línea).
        """
        try:
            with open(filepath, "r", encoding="utf-8") as f:
                source = f.read()

            tree = ast.parse(source, filename=str(filepath))
            imports = []

            for node in ast.walk(tree):
                # from X import Y
                if isinstance(node, ast.ImportFrom):
                    if node.module:
                        # Convertir import relativo a ruta absoluta desde root
                        imports.append((node.module, node.lineno))

                # import X
                elif isinstance(node, ast.Import):
                    for alias in node.names:
                        imports.append((alias.name, node.lineno))

            return imports

        except Exception:
            return []

    def build_import_graph(self, files: List[Path]) -> None:
        """Construye grafo de dependencias entre archivos."""
        for filepath in files:
            # Extraer imports del archivo
            imports = self.extract_imports(filepath)
            self.file_imports[str(filepath)] = imports

            # Convertir imports a rutas de archivos
            file_key = self._get_module_name(filepath)
            self.import_graph[file_key] = []

            for import_module, _ in imports:
                # Solo considerar imports internos del proyecto
                if self._is_internal_import(import_module):
                    self.import_graph[file_key].append(import_module)

    def _get_module_name(self, filepath: Path) -> str:
        """Convierte ruta de archivo a nombre de módulo."""
        try:
            # Obtener ruta relativa desde root
            rel_path = filepath.relative_to(self.root_dir)
            # Convertir a nombre de módulo (ej: components/admin.py -> components.admin)
            module_name = str(rel_path.with_suffix("")).replace("\\", ".").replace("/", ".")
            return module_name
        except ValueError:
            return str(filepath)

    def _is_internal_import(self, import_module: str) -> bool:
        """Verifica si el import es interno del proyecto."""
        # Imports internos empiezan con nombres de directorios del proyecto
        internal_prefixes = ["components", "layouts", "callbacks", "utils", "pages"]
        return any(import_module.startswith(prefix) for prefix in internal_prefixes)

    def find_cycles(self) -> List[List[str]]:
        """
        Detecta ciclos en el grafo de importaciones usando DFS.
        Retorna lista de ciclos encontrados.
        """
        cycles = []
        visited = set()
        rec_stack = set()

        def dfs(node: str, path: List[str]) -> None:
            visited.add(node)
            rec_stack.add(node)
            path.append(node)

            # Explorar vecinos
            for neighbor in self.import_graph.get(node, []):
                if neighbor not in visited:
                    dfs(neighbor, path.copy())
                elif neighbor in rec_stack:
                    # Ciclo detectado
                    cycle_start = path.index(neighbor)
                    cycle = path[cycle_start:] + [neighbor]
                    if cycle not in cycles:
                        cycles.append(cycle)

            rec_stack.remove(node)

        # Ejecutar DFS desde cada nodo
        for node in self.import_graph:
            if node not in visited:
                dfs(node, [])

        return cycles

    def detect_circular_imports(self, files: List[Path]) -> List[ValidationIssue]:
        """
        Detecta importaciones circulares en lista de archivos.
        Retorna lista de ValidationIssue con los ciclos encontrados.
        """
        self.build_import_graph(files)
        cycles = self.find_cycles()

        issues = []

        for cycle in cycles:
            # Crear issue para el primer archivo del ciclo
            first_module = cycle[0]
            cycle_str = " → ".join(cycle)

            # Buscar archivo correspondiente al primer módulo
            for filepath, imports in self.file_imports.items():
                module_name = self._get_module_name(Path(filepath))
                if module_name == first_module:
                    # Buscar línea del import que causa el ciclo
                    next_module = cycle[1] if len(cycle) > 1 else cycle[0]
                    line_num = 1

                    for import_module, line in imports:
                        if next_module.startswith(import_module):
                            line_num = line
                            break

                    issue = ValidationIssue(
                        file=filepath,
                        line=line_num,
                        column=0,
                        severity="error",
                        code="DASH004",
                        message=f"Importación circular detectada: {cycle_str}",
                        suggestion="Refactorizar código para eliminar dependencia circular. "
                        "Considerar mover código común a un módulo separado o usar "
                        "imports dinámicos (importlib) si es necesario.",
                    )
                    issues.append(issue)
                    break

        return issues


def validate_file(filepath: Path) -> List[ValidationIssue]:
    """
    Valida un archivo Python en busca de antipatrones Dash.

    Args:
        filepath: Ruta al archivo a validar

    Returns:
        Lista de problemas detectados
    """
    try:
        with open(filepath, "r", encoding="utf-8") as f:
            source = f.read()

        tree = ast.parse(source, filename=str(filepath))
        validator = DashComponentValidator(str(filepath))
        validator.visit(tree)

        return validator.issues

    except SyntaxError as e:
        return [
            ValidationIssue(
                file=str(filepath),
                line=e.lineno or 0,
                column=e.offset or 0,
                severity="error",
                code="SYNTAX",
                message=f"Error de sintaxis: {e.msg}",
                suggestion="Corregir sintaxis del archivo",
            )
        ]
    except Exception as e:
        return [
            ValidationIssue(
                file=str(filepath),
                line=0,
                column=0,
                severity="error",
                code="PARSE_ERROR",
                message=f"Error al parsear archivo: {str(e)}",
                suggestion="Verificar que el archivo es Python válido",
            )
        ]


def validate_frontend(root_dir: Path = None, check_circular_imports: bool = True) -> Dict[str, List[ValidationIssue]]:
    """
    Valida todos los archivos Python del frontend.

    Args:
        root_dir: Directorio raíz del frontend (default: frontend/)
        check_circular_imports: Si True, detecta importaciones circulares (DASH004)

    Returns:
        Diccionario {archivo: [problemas]}
    """
    if root_dir is None:
        root_dir = Path(__file__).parent.parent  # frontend/

    results = {}
    all_files = []

    # Validar archivos en subdirectorios clave
    directories = ["components", "layouts", "pages", "callbacks"]

    for dir_name in directories:
        dir_path = root_dir / dir_name
        if not dir_path.exists():
            continue

        for py_file in dir_path.rglob("*.py"):
            if py_file.name == "__init__.py":
                continue

            all_files.append(py_file)

            # Validación AST individual (DASH001, DASH002, DASH003)
            issues = validate_file(py_file)
            if issues:
                results[str(py_file)] = issues

    # Validación global: Importaciones circulares (DASH004)
    if check_circular_imports and all_files:
        print(f"\n[DASH004] Detectando importaciones circulares en {len(all_files)} archivos...")
        detector = CircularImportDetector(root_dir)
        circular_issues = detector.detect_circular_imports(all_files)

        if circular_issues:
            print(f"  [WARN] {len(circular_issues)} importación(es) circular(es) detectada(s)")
            for issue in circular_issues:
                if issue.file not in results:
                    results[issue.file] = []
                results[issue.file].append(issue)
        else:
            print("  [OK] No se detectaron importaciones circulares")

    return results


def format_issue(issue: ValidationIssue) -> str:
    """Formatea un problema para mostrar en consola."""
    severity_emoji = {"error": "❌", "warning": "⚠️", "info": "ℹ️"}

    emoji = severity_emoji.get(issue.severity, "•")

    return (
        f"{emoji} {issue.file}:{issue.line}:{issue.column}\n"
        f"   [{issue.code}] {issue.message}\n"
        f"   💡 Sugerencia: {issue.suggestion}\n"
    )


def main():
    """Entry point del validador."""
    import argparse

    parser = argparse.ArgumentParser(description="Valida componentes Dash en busca de antipatrones")
    parser.add_argument("files", nargs="*", help="Archivos a validar (vacío = validar todo frontend/)")
    parser.add_argument("--all", action="store_true", help="Validar todos los archivos del frontend")
    parser.add_argument("--strict", action="store_true", help="Salir con código 1 si hay errores (útil para CI/CD)")

    args = parser.parse_args()

    # Determinar qué archivos validar
    if args.all or not args.files:
        print("🔍 Validando todos los archivos del frontend...\n")
        results = validate_frontend()
    else:
        print(f"🔍 Validando {len(args.files)} archivo(s)...\n")
        results = {}
        for filepath in args.files:
            path = Path(filepath)
            if path.exists():
                issues = validate_file(path)
                if issues:
                    results[str(path)] = issues

    # Mostrar resultados
    total_errors = 0
    total_warnings = 0

    if not results:
        print("✅ No se encontraron problemas!\n")
        return 0

    for filepath, issues in results.items():
        print(f"\n📄 {filepath}")
        print("─" * 80)

        for issue in issues:
            print(format_issue(issue))

            if issue.severity == "error":
                total_errors += 1
            elif issue.severity == "warning":
                total_warnings += 1

    # Resumen
    print("\n" + "=" * 80)
    print(f"📊 Resumen: {total_errors} errores, {total_warnings} advertencias")
    print("=" * 80)

    if total_errors > 0:
        print("\n❌ Validación FALLIDA - Corregir errores antes de continuar")
        if args.strict:
            return 1
    else:
        print("\n✅ Validación EXITOSA")

    return 0


if __name__ == "__main__":
    sys.exit(main())
