# frontend/utils/constants.py
"""
Constantes y configuraciones para el frontend de xfarma.
Incluye colores, temas de gráficos y formatos.
"""

# Colores del tema xfarma
COLORS = {
    # Colores principales
    "primary": "#0d6efd",  # Azul principal
    "secondary": "#6c757d",  # Gris
    "success": "#198754",  # Verde
    "danger": "#dc3545",  # Rojo
    "warning": "#ffc107",  # Amarillo
    "info": "#0dcaf0",  # Cyan
    "light": "#f8f9fa",  # Gris claro
    "muted": "#dee2e6",  # Gris Bootstrap para empty states
    "dark": "#212529",  # Negro
    # Colores farmacia específicos
    "medicine": "#28a745",  # Verde medicina
    "generic": "#17a2b8",  # Azul genérico
    "brand": "#6f42c1",  # Púrpura marca
    "otc": "#fd7e14",  # Naranja sin receta
    "prescription": "#dc3545",  # Rojo con receta
    # Gradientes para gráficos
    "chart_primary": ["#0d6efd", "#6610f2", "#6f42c1", "#d63384", "#dc3545"],
    "chart_success": ["#198754", "#20c997", "#0dcaf0", "#0d6efd"],
    "chart_neutral": ["#6c757d", "#495057", "#343a40", "#212529"],
}

# Configuraciones de gráficos
CHART_THEMES = {
    "default": {
        "font_family": "Arial, sans-serif",
        "font_size": 12,
        "font_color": COLORS["dark"],
        "paper_bgcolor": "white",
        "plot_bgcolor": "white",
        "grid_color": COLORS["light"],
        "margin": {"l": 60, "r": 20, "t": 60, "b": 60},
    },
    "kpi_card": {
        "font_family": "Arial, sans-serif",
        "font_size": 14,
        "font_color": COLORS["dark"],
        "paper_bgcolor": "rgba(0,0,0,0)",
        "plot_bgcolor": "rgba(0,0,0,0)",
        "showgrid": False,
        "zeroline": False,
        "showticklabels": False,
        "margin": {"l": 0, "r": 0, "t": 0, "b": 0},
    },
}

# Formatos de fecha y hora
DATE_FORMATS = {
    "display": "%d/%m/%Y",  # 25/12/2024
    "display_with_time": "%d/%m/%Y %H:%M",  # 25/12/2024 14:30
    "api": "%Y-%m-%d",  # 2024-12-25
    "api_with_time": "%Y-%m-%dT%H:%M:%S",  # 2024-12-25T14:30:00
    "month_year": "%m/%Y",  # 12/2024
    "short": "%d/%m",  # 25/12
    "weekday": "%A %d/%m",  # Miércoles 25/12
}

# ⚠️ COMPLETAMENTE DEPRECADO - NO USAR
# Esta constante ha sido eliminada para forzar debugging claro
# Si ves errores, es porque el backend no está corriendo
# Ver: .\scripts\dev.ps1 para iniciar el backend
# PHARMACY_ID_DEFAULT = "11111111-1111-1111-1111-111111111111"  # ELIMINADO


def get_dynamic_pharmacy_id() -> str:
    """
    Obtener pharmacy_id dinámicamente desde la API.

    ⚠️ DEPRECADO: Usar get_current_pharmacy_id() from utils.pharmacy_context

    Raises:
        ValueError: Si no se puede obtener pharmacy_id (backend no disponible)
    """
    from utils.pharmacy_context import get_current_pharmacy_id

    # NO hay fallback - falla explícitamente si no puede obtener ID
    return get_current_pharmacy_id()


# Configuraciones de la aplicación con optimización inteligente de intervals
from .interval_optimizer import get_optimized_interval

APP_CONFIG = {
    "update_intervals": {
        "health_check": get_optimized_interval("health_check"),
        "dashboard": get_optimized_interval("dashboard_update"),
        "charts": get_optimized_interval("dashboard_update"),
    },
    "pagination": {
        "default_page_size": 20,
        "max_page_size": 100,
    },
    "file_upload": {
        "max_size_mb": 50,
        "allowed_extensions": [".csv", ".xlsx", ".xls", ".txt"],
        "chunk_size": 8192,
    },
    "cache": {
        "default_timeout": 300,  # 5 minutos
        "long_timeout": 1800,  # 30 minutos
    },
}

# Mensajes de la interfaz
UI_MESSAGES = {
    "loading": "Cargando datos...",
    "no_data": "No hay datos disponibles",
    "error": "Ha ocurrido un error",
    "success": "Operación completada exitosamente",
    "backend_offline": "Servidor no disponible",
    "backend_online": "Servidor conectado",
    "upload_success": "Archivo subido correctamente",
    "upload_error": "Error al subir archivo",
    "invalid_file": "Formato de archivo no válido",
    "file_too_large": "El archivo es demasiado grande",
}

# Tipos de archivo ERP soportados
ERP_TYPES = {
    "farmatic": "Farmatic",
    "farmanager": "Farmanager",
    "nixfarma": "Nixfarma",
    "unycop": "Unycop",
    "iofwin": "IOFWin",
    "other": "Otro",
}

# Pathnames de páginas (Issue #303 - Mejora #6)
PAGE_PATHS = {
    "home": "/",
    "dashboard": "/dashboard",
    "upload": "/upload",
    "generics": "/generics",
    "catalog": "/catalog",
    "settings": "/settings",
    "admin": "/admin",
}

# Configuraciones de KPIs
KPI_CONFIG = {
    "ventas_totales": {
        "title": "Ventas Totales",
        "icon": "fas fa-euro-sign",
        "color": COLORS["primary"],
        "format": "currency",
    },
    "productos_vendidos": {
        "title": "Productos Vendidos",
        "icon": "fas fa-pills",
        "color": COLORS["success"],
        "format": "number",
    },
    "tickets_promedio": {
        "title": "Ticket Promedio",
        "icon": "fas fa-receipt",
        "color": COLORS["info"],
        "format": "currency",
    },
    "margen_bruto": {
        "title": "Margen Bruto",
        "icon": "fas fa-percentage",
        "color": COLORS["warning"],
        "format": "percentage",
    },
}

# Roles de usuario (Issue #154 - User menu dropdown)
USER_ROLES = {
    "admin": "Administrador",
    "user": "Usuario",
}
# Categorías de prescripción (Issue #400, #432 - 15 categorías xFarma)
# SINCRONIZADO con backend/app/models/enums.py::PrescriptionCategory
PRESCRIPTION_CATEGORIES = [
    # Prescripción especializada (Prioridad 1-2)
    "formulas_magistrales",
    "vacunas_individualizadas",
    # Material sanitario financiado (Prioridad 3-6)
    "tiras_reactivas_glucosa",
    "incontinencia_financiada",
    "efectos_financiados",
    "ortopedia_financiada",
    # Dietoterápicos (Prioridad 7)
    "dietoterapicos",
    # Veterinaria (Prioridad 8)
    "veterinaria",
    # Conjunto homogéneo (Prioridad 9)
    "conjunto_homogeneo",
    # Márgenes especiales RD 823/2008 (Prioridad 10-12)
    "margen_especial_3",  # PVP >= 500€
    "margen_especial_2",  # 200€ <= PVP < 500€
    "margen_especial_1",  # 91.63€ <= PVP < 200€
    # Medicamentos estándar (Prioridad 13)
    "medicamentos",
    # Productos no comercializados (Prioridad 14 - Issue #432)
    "no_comercializado",
    # Productos no financiados (Prioridad 15)
    "uso_humano_no_financiado",
]

# Nombres legibles en español para categorías de prescripción (Issue #436)
# SINCRONIZADO con backend/app/models/enums.py::PrescriptionCategory
# Fuente única de verdad para mapeo clave → nombre visible
PRESCRIPTION_CATEGORY_NAMES = {
    # Prescripción especializada (Prioridad 1-2)
    "formulas_magistrales": "Fórmulas Magistrales",
    "vacunas_individualizadas": "Vacunas Individualizadas",
    # Material sanitario financiado (Prioridad 3-6)
    "tiras_reactivas_glucosa": "Tiras Reactivas de Glucosa",
    "incontinencia_financiada": "Incontinencia Financiada",
    "efectos_financiados": "Efectos Financiados",
    "ortopedia_financiada": "Ortopedia Financiada",
    # Dietoterápicos (Prioridad 7)
    "dietoterapicos": "Dietoterapéuticos",
    # Veterinaria (Prioridad 8)
    "veterinaria": "Veterinaria",
    # Conjunto homogéneo (Prioridad 9)
    "conjunto_homogeneo": "Conjunto Homogéneo",
    # Márgenes especiales RD 823/2008 (Prioridad 10-12)
    "margen_especial_3": "PVL ≥ 500€",
    "margen_especial_2": "200€ ≤ PVL < 500€",
    "margen_especial_1": "91,63€ ≤ PVL < 200€",
    # Medicamentos estándar (Prioridad 13)
    "medicamentos": "Medicamentos",
    # Productos no comercializados (Prioridad 14 - Issue #432)
    "no_comercializado": "No Comercializado",
    # Productos no financiados (Prioridad 15)
    "uso_humano_no_financiado": "Uso Humano No Financiado",
}

# Mapeo inverso: nombre visible → clave (auto-generado)
# Usado para obtener clave de categoría desde label de click en gráficos
CATEGORY_NAME_TO_KEY = {v: k for k, v in PRESCRIPTION_CATEGORY_NAMES.items()}

# Issue #441: Códigos ATC Nivel 1 (Grupos Anatómicos Principales)
# Referencia: WHO ATC/DDD Index - https://www.whocc.no/atc_ddd_index/
ATC_LEVEL_1_CODES = [
    {"code": "A", "name": "Tracto alimentario y metabolismo"},
    {"code": "B", "name": "Sangre y órganos hematopoyéticos"},
    {"code": "C", "name": "Sistema cardiovascular"},
    {"code": "D", "name": "Dermatológicos"},
    {"code": "G", "name": "Sistema genitourinario"},
    {"code": "H", "name": "Hormonas sistémicas"},
    {"code": "J", "name": "Antiinfecciosos sistémicos"},
    {"code": "L", "name": "Antineoplásicos e inmunomoduladores"},
    {"code": "M", "name": "Sistema musculoesquelético"},
    {"code": "N", "name": "Sistema nervioso"},
    {"code": "P", "name": "Antiparasitarios e insecticidas"},
    {"code": "R", "name": "Sistema respiratorio"},
    {"code": "S", "name": "Órganos de los sentidos"},
    {"code": "V", "name": "Varios"},
]


def format_category_name(category_key: str) -> str:
    """
    Convierte clave de categoría a nombre legible en español.

    Fuente única de verdad para el mapeo de categorías de prescripción.
    SINCRONIZADO con backend/app/models/enums.py::PrescriptionCategory (15 categorías)

    Args:
        category_key: Clave de categoría (ej: "formulas_magistrales")

    Returns:
        Nombre legible (ej: "Fórmulas Magistrales")

    Examples:
        >>> format_category_name("conjunto_homogeneo")
        "Conjunto Homogéneo"
        >>> format_category_name("unknown_category")
        "Unknown Category"
    """
    return PRESCRIPTION_CATEGORY_NAMES.get(
        category_key,
        category_key.replace("_", " ").title()
    )
