# frontend/utils/config.py
"""
Configuración centralizada del frontend xFarma.
Evita duplicación de configuración de URLs y variables de entorno.
"""

import logging
import os
from typing import Any, Dict
from urllib.parse import urlparse

logger = logging.getLogger(__name__)


def get_backend_url() -> str:
    """
    Obtener URL del backend según el entorno con validación.

    Prioridad:
    1. BACKEND_URL (si está configurada explícitamente)
    2. RENDER_EXTERNAL_URL (en Render.com)
    3. http://localhost:8000 (desarrollo local)

    Returns:
        URL del backend validada y normalizada

    Raises:
        ValueError: Si la URL es malformada o inválida
    """
    # Determinar URL según prioridad
    if "BACKEND_URL" in os.environ:
        url = os.getenv("BACKEND_URL")
        logger.info(f"Usando BACKEND_URL explícito: {url}")
    elif "RENDER_EXTERNAL_URL" in os.environ:
        url = os.getenv("RENDER_EXTERNAL_URL")
        logger.info(f"Usando RENDER_EXTERNAL_URL como fallback: {url}")
    else:
        url = "http://localhost:8000"
        logger.info(f"Usando localhost por defecto (desarrollo): {url}")

    # Validar formato de URL
    try:
        parsed = urlparse(url)

        # Verificar que tiene scheme y netloc
        if not parsed.scheme or not parsed.netloc:
            raise ValueError(f"URL inválida (falta scheme o netloc): {url}")

        # Verificar que el scheme es http o https
        if parsed.scheme not in ["http", "https"]:
            raise ValueError(f"URL inválida (scheme debe ser http/https): {url}")

        # Normalizar: eliminar trailing slash
        normalized_url = f"{parsed.scheme}://{parsed.netloc}"
        if parsed.path and parsed.path != "/":
            normalized_url += parsed.path.rstrip("/")

        # Si solo hay '/', eliminar
        if normalized_url.endswith("/") and parsed.path == "/":
            normalized_url = normalized_url.rstrip("/")

        return normalized_url

    except (ValueError, AttributeError, TypeError) as e:
        raise ValueError(f"URL inválida para backend: {url}. Error: {str(e)}")


# Configuración de backend URL (centralizada)
# Prioridad: BACKEND_URL > RENDER_EXTERNAL_URL > localhost:8000
# Se inicializa al importar el módulo
BACKEND_URL = get_backend_url()

# IMPORTANTE: BACKEND_URL se inicializa al importar el módulo.
# Si cambias variables de entorno en runtime, necesitas reimportar el módulo
# o llamar a get_backend_url() directamente para obtener el nuevo valor.


def get_app_config() -> Dict[str, Any]:
    """
    Obtener configuración completa de la aplicación.

    Returns:
        Diccionario con toda la configuración
    """
    return {
        "backend_url": get_backend_url(),  # Usar función centralizada
        "environment": os.getenv("ENVIRONMENT", "development"),
        "debug_mode": os.getenv("ENVIRONMENT", "development") == "development",
        "frontend_host": os.getenv("FRONTEND_HOST", "0.0.0.0"),
        "frontend_port": int(os.getenv("FRONTEND_PORT", 8050)),
        "secret_key": os.getenv("SECRET_KEY", "xfarma-dev-secret-2024"),
    }


def is_development() -> bool:
    """
    Verificar si estamos en modo desarrollo.

    Returns:
        True si es desarrollo
    """
    return os.getenv("ENVIRONMENT", "development") == "development"


def get_version_info() -> Dict[str, str]:
    """
    Obtener información de versión de la aplicación.

    Returns:
        Diccionario con información de versión
    """
    return {
        "version": "1.0.0",
        "build": "MVP-2024.12",
        "release_date": "2024-12-25",
        "environment": os.getenv("ENVIRONMENT", "development"),
    }
