# frontend/utils/chart_benchmarks.py
"""
Utilidades para añadir líneas de referencia/benchmarks a gráficos Plotly.

Issue #510: Benchmarks y Líneas de Referencia en Gráficos
Permite añadir líneas de referencia configurables (mínimo, objetivo, excelente)
para contextualizar rendimiento en gráficos del dashboard VentaLibre.
"""

from typing import Any, Dict, List, Literal, Optional

import plotly.graph_objects as go


# Mapeo de estilos de línea Plotly
LINE_STYLE_MAP = {
    "solid": "solid",
    "dash": "dash",
    "dot": "dot",
    "dashdot": "dashdot",
}

# Defaults para tipos de target comunes
DEFAULT_TARGET_CONFIGS = {
    "margin_min": {
        "value": 18.0,
        "color": "#dc3545",  # Rojo Bootstrap
        "line_style": "dot",
        "label": "Mínimo 18%",
    },
    "margin_target": {
        "value": 25.0,
        "color": "#28a745",  # Verde Bootstrap
        "line_style": "dash",
        "label": "Objetivo 25%",
    },
    "margin_excellent": {
        "value": 32.0,
        "color": "#17a2b8",  # Azul Bootstrap
        "line_style": "solid",
        "label": "Excelente 32%",
    },
    "hhi_max": {
        "value": 2500,
        "color": "#ffc107",  # Amarillo Bootstrap
        "line_style": "dash",
        "label": "HHI Máx",
    },
}


def add_benchmark_line(
    fig: go.Figure,
    value: float,
    axis: Literal["x", "y"] = "y",
    color: str = "#28a745",
    line_style: str = "dash",
    label: Optional[str] = None,
    annotation_position: str = "top right",
    line_width: float = 1.5,
    opacity: float = 0.8,
) -> go.Figure:
    """
    Añade una línea de referencia horizontal o vertical a una figura Plotly.

    Args:
        fig: Figura Plotly a modificar
        value: Valor donde dibujar la línea
        axis: "x" para vertical, "y" para horizontal
        color: Color de la línea en formato hex
        line_style: Estilo de línea (solid, dash, dot)
        label: Etiqueta para la anotación
        annotation_position: Posición de la anotación
        line_width: Grosor de la línea
        opacity: Opacidad de la línea (0-1)

    Returns:
        Figura modificada
    """
    dash_style = LINE_STYLE_MAP.get(line_style, "dash")

    if axis == "y":
        fig.add_hline(
            y=value,
            line_dash=dash_style,
            line_color=color,
            line_width=line_width,
            opacity=opacity,
            annotation_text=label or "",
            annotation_position=annotation_position,
            annotation_font_size=9,
            annotation_font_color=color,
        )
    else:
        fig.add_vline(
            x=value,
            line_dash=dash_style,
            line_color=color,
            line_width=line_width,
            opacity=opacity,
            annotation_text=label or "",
            annotation_position=annotation_position,
            annotation_font_size=9,
            annotation_font_color=color,
        )

    return fig


def add_benchmark_lines(
    fig: go.Figure,
    targets: List[Dict[str, Any]],
    axis: Literal["x", "y"] = "y",
    annotation_position: str = "top right",
) -> go.Figure:
    """
    Añade múltiples líneas de referencia a una figura Plotly.

    Args:
        fig: Figura Plotly a modificar
        targets: Lista de targets con formato:
            [{value, color, line_style, label}, ...]
        axis: "x" para verticales, "y" para horizontales
        annotation_position: Posición de las anotaciones

    Returns:
        Figura modificada
    """
    for target in targets:
        if target.get("value") is not None:
            add_benchmark_line(
                fig=fig,
                value=target["value"],
                axis=axis,
                color=target.get("color", "#28a745"),
                line_style=target.get("line_style", "dash"),
                label=target.get("label"),
                annotation_position=annotation_position,
            )

    return fig


def add_margin_benchmarks(
    fig: go.Figure,
    targets: Optional[List[Dict[str, Any]]] = None,
    show_min: bool = True,
    show_target: bool = True,
    show_excellent: bool = False,
) -> go.Figure:
    """
    Añade líneas de benchmark de margen a una figura.

    Shortcut conveniente para el caso común de mostrar líneas
    de margen mínimo, objetivo y excelente.

    Args:
        fig: Figura Plotly a modificar
        targets: Targets personalizados (si None, usa defaults)
        show_min: Mostrar línea de margen mínimo
        show_target: Mostrar línea de margen objetivo
        show_excellent: Mostrar línea de margen excelente

    Returns:
        Figura modificada
    """
    if targets:
        # Usar targets proporcionados
        return add_benchmark_lines(fig, targets, axis="y")

    # Usar defaults
    lines_to_add = []
    if show_min:
        lines_to_add.append(DEFAULT_TARGET_CONFIGS["margin_min"])
    if show_target:
        lines_to_add.append(DEFAULT_TARGET_CONFIGS["margin_target"])
    if show_excellent:
        lines_to_add.append(DEFAULT_TARGET_CONFIGS["margin_excellent"])

    return add_benchmark_lines(fig, lines_to_add, axis="y")


def add_quadrant_zones(
    fig: go.Figure,
    x_threshold: float,
    y_threshold: float,
    x_range: tuple = (0, None),
    y_range: tuple = (0, None),
    colors: Optional[Dict[str, str]] = None,
    opacity: float = 0.1,
) -> go.Figure:
    """
    Añade zonas de cuadrante coloreadas a una figura scatter.

    Crea 4 zonas basadas en thresholds de X e Y:
    - Top-Right: "star" (alto X, alto Y)
    - Top-Left: "opportunity" (bajo X, alto Y)
    - Bottom-Right: "cash_cow" (alto X, bajo Y)
    - Bottom-Left: "dog" (bajo X, bajo Y)

    Args:
        fig: Figura Plotly a modificar
        x_threshold: Valor de corte en eje X
        y_threshold: Valor de corte en eje Y
        x_range: Rango del eje X (min, max)
        y_range: Rango del eje Y (min, max)
        colors: Colores personalizados por zona
        opacity: Opacidad de las zonas

    Returns:
        Figura modificada
    """
    default_colors = {
        "star": "#28a745",  # Verde
        "opportunity": "#ffc107",  # Amarillo
        "cash_cow": "#17a2b8",  # Azul
        "dog": "#dc3545",  # Rojo
    }
    zone_colors = colors or default_colors

    # Calcular rangos
    x_min, x_max = x_range
    y_min, y_max = y_range

    # Si no hay max, usar un valor grande
    if x_max is None:
        x_max = x_threshold * 3
    if y_max is None:
        y_max = y_threshold * 2

    # Top-Right: Star
    fig.add_shape(
        type="rect",
        x0=x_threshold,
        y0=y_threshold,
        x1=x_max,
        y1=y_max,
        fillcolor=zone_colors["star"],
        opacity=opacity,
        layer="below",
        line_width=0,
    )

    # Top-Left: Opportunity
    fig.add_shape(
        type="rect",
        x0=x_min,
        y0=y_threshold,
        x1=x_threshold,
        y1=y_max,
        fillcolor=zone_colors["opportunity"],
        opacity=opacity,
        layer="below",
        line_width=0,
    )

    # Bottom-Right: Cash Cow
    fig.add_shape(
        type="rect",
        x0=x_threshold,
        y0=y_min,
        x1=x_max,
        y1=y_threshold,
        fillcolor=zone_colors["cash_cow"],
        opacity=opacity,
        layer="below",
        line_width=0,
    )

    # Bottom-Left: Dog
    fig.add_shape(
        type="rect",
        x0=x_min,
        y0=y_min,
        x1=x_threshold,
        y1=y_threshold,
        fillcolor=zone_colors["dog"],
        opacity=opacity,
        layer="below",
        line_width=0,
    )

    return fig


def create_benchmark_legend_text(targets: List[Dict[str, Any]]) -> str:
    """
    Crea texto de leyenda para los benchmarks.

    Args:
        targets: Lista de targets

    Returns:
        Texto formateado para leyenda
    """
    lines = []
    for target in targets:
        if target.get("label") and target.get("value") is not None:
            lines.append(f"• {target['label']}")

    return " | ".join(lines) if lines else ""
