"""
Callback Helpers - Utilidades reutilizables para callbacks Dash.

Contiene patrones comunes extraídos de callbacks para mejorar clarity y reducir duplicación.
"""

from typing import Any, Dict, Optional

import structlog
from dash import ctx, no_update

logger = structlog.get_logger(__name__)


def unified_toast_trigger(*store_inputs: Optional[Dict[str, Any]]) -> Optional[Dict[str, Any]]:
    """
    Patrón "Unified Toast Handler" - Un callback que centraliza múltiples toasts.

    Solución a DASH002 (REGLA #11): Cuando múltiples callbacks necesitan escribir
    a 'toast-trigger-store', usar stores intermedios + este handler unificado.

    Args:
        *store_inputs: Outputs de stores intermedios (toast data dictionaries)

    Returns:
        Toast data del store que fue actualizado, o no_update si no hay cambios.

    Ejemplo de uso:
        ```python
        # PASO 1: Crear stores intermedios en el layout
        dcc.Store(id='operation-toast-store'),
        dcc.Store(id='sync-toast-store'),
        dcc.Store(id='validation-toast-store')

        # PASO 2: Callbacks escriben a sus stores intermedios
        @app.callback(
            Output('operation-toast-store', 'data'),
            Input('save-button', 'n_clicks')
        )
        def handle_operation(n_clicks):
            return success_toast("Operación exitosa", "Éxito")

        # PASO 3: Callback unificado usando este helper
        @app.callback(
            Output('toast-trigger-store', 'data'),
            [Input('operation-toast-store', 'data'),
             Input('sync-toast-store', 'data'),
             Input('validation-toast-store', 'data')],
            prevent_initial_call=True
        )
        def unified_toast_handler_callback(operation_toast, sync_toast, validation_toast):
            return unified_toast_trigger(operation_toast, sync_toast, validation_toast)
        ```

    Ventajas:
        - ✅ Un solo callback escribe a toast-trigger-store (cumple REGLA #11)
        - ✅ Desacopla lógica de negocio de notificaciones
        - ✅ Escalable: Agregar nuevo toast = agregar nuevo Input al callback
        - ✅ Thread-safe: ctx.triggered detecta fuente exacta

    Implementación:
        Usado en:
        - frontend/callbacks/settings.py (3 stores intermedios)
        - frontend/callbacks/admin/sync_operations.py (3 stores intermedios)
    """
    if not ctx.triggered:
        return no_update

    # Detectar cuál store fue actualizado
    trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]
    triggered_value = ctx.triggered[0]["value"]

    # Logs estructurados para debugging
    logger.info("unified_toast_trigger", trigger_id=trigger_id, has_value=triggered_value is not None)

    # Retornar el toast del store que fue actualizado
    if triggered_value:
        return triggered_value

    return no_update


def create_intermediate_store_pattern(store_id: str) -> Dict[str, Any]:
    """
    Helper para crear configuración de stores intermedios con pattern consistente.

    Args:
        store_id: ID del dcc.Store a crear (ej: 'operation-toast-store')

    Returns:
        Dict con propiedades del dcc.Store

    Ejemplo:
        ```python
        # En lugar de repetir:
        dcc.Store(id='my-toast-store', storage_type='memory')

        # Usar:
        dcc.Store(**create_intermediate_store_pattern('my-toast-store'))
        ```
    """
    return {"id": store_id, "storage_type": "memory"}  # No persistencia necesaria para toasts
