"""
Helper functions para verificación de autenticación en callbacks.

Pivot 2026: Soporta dos modos de autenticación:
- Cloud (Hub): JWT tokens via OAuth
- Local (Desktop): PIN-based lock screen (siempre considera autenticado)
"""

import logging
import os
from typing import Any, Dict, Optional, Tuple

logger = logging.getLogger(__name__)

# Pivot 2026: Detect local mode once at module load
IS_LOCAL_MODE = os.getenv("KAIFARMA_LOCAL", "").lower() == "true"


class AuthStateInconsistentError(Exception):
    """
    Excepción lanzada cuando el estado de autenticación es inconsistente.

    Ejemplo: authenticated=True pero user=None o sin nombre.

    Los callbacks deben capturar esta excepción y redirigir a login.
    """
    pass


def is_user_authenticated(auth_state: Optional[Dict[str, Any]]) -> bool:
    """
    Determina si un usuario está autenticado para acceder al contenido de la app.

    Esta función centraliza la lógica de verificación de autenticación usada
    en múltiples callbacks (routing, sidebar, navigation).

    Pivot 2026: Soporta dos modos:
    - Local (KAIFARMA_LOCAL=true): Siempre retorna True. La seguridad real
      la maneja el lock screen modal con PIN (LocalSecurityManager).
    - Cloud/Hub: Requiere JWT válido via OAuth.

    Args:
        auth_state: Diccionario con estado de autenticación del usuario.
                   Debe contener clave 'authenticated' (bool).
                   Puede ser None si no hay sesión.

    Returns:
        bool: True si el usuario puede acceder al contenido.
              - Local mode: Siempre True
              - Cloud mode: True solo con JWT válido

    Examples:
        >>> # Modo local: siempre autenticado (PIN maneja seguridad)
        >>> # (asumiendo KAIFARMA_LOCAL=true)
        >>> is_user_authenticated(None)
        True

        >>> # Modo cloud: requiere auth_state con authenticated=True
        >>> is_user_authenticated({'authenticated': True, 'user': {...}})
        True

        >>> # Modo cloud: no autenticado
        >>> is_user_authenticated({'authenticated': False})
        False
    """
    # Pivot 2026: En modo local, siempre autenticado
    # La seguridad real la maneja el lock screen con PIN
    if IS_LOCAL_MODE:
        return True

    # Cloud mode: requiere autenticación JWT real
    # Validación defensiva con type checking explícito
    is_authenticated = isinstance(auth_state, dict) and auth_state.get("authenticated", False) is True

    return is_authenticated


def extract_user_info(auth_state: Optional[Dict[str, Any]]) -> Tuple[str, str]:
    """
    Extrae nombre de usuario y rol desde auth_state con valores por defecto.

    Args:
        auth_state: Diccionario con estado de autenticación del usuario.
                   Debe contener clave 'user' con 'full_name', 'username' y 'role'.
                   Puede ser None si no hay sesión.

    Returns:
        Tuple[str, str]: (user_name, user_role) con valores por defecto si no autenticado.
                        user_name: full_name o username
                        user_role: "admin" o "user"

    Raises:
        AuthStateInconsistentError: Si el usuario está autenticado pero faltan datos críticos.
                                   Los callbacks deben capturar esto y redirigir a login.

    Examples:
        >>> # Usuario autenticado con full_name
        >>> extract_user_info({
        ...     'authenticated': True,
        ...     'user': {'full_name': 'Diego Ruiz', 'role': 'admin'}
        ... })
        ('Diego Ruiz', 'admin')

        >>> # Usuario autenticado solo con username
        >>> extract_user_info({
        ...     'authenticated': True,
        ...     'user': {'username': 'dgruiz', 'role': 'user'}
        ... })
        ('dgruiz', 'user')

        >>> # Usuario no autenticado
        >>> extract_user_info(None)
        ('Usuario', 'user')

    Note:
        Fail-fast: Si authenticated=True pero faltan datos de usuario,
        lanza ValueError para exponer bugs de estado inconsistente.
    """
    # Valores por defecto para usuarios NO autenticados
    default_name = "Usuario"
    default_role = "user"

    # Caso 1: Usuario NO autenticado → Retornar defaults (normal)
    if not auth_state or not auth_state.get("authenticated"):
        return default_name, default_role

    # Caso 2: Usuario autenticado → DEBE tener datos válidos
    # (Si no, es un bug que debe ser visible)

    user = auth_state.get("user")

    # ❌ Estado inconsistente: authenticated=True pero user=None
    if user is None:
        error_msg = (
            "Estado inconsistente: Usuario autenticado sin datos de usuario. "
            f"auth_state: {auth_state}"
        )
        logger.error(error_msg)
        raise AuthStateInconsistentError(error_msg)

    # ❌ Estado inconsistente: authenticated=True pero sin nombre
    user_name = user.get("full_name") or user.get("username")
    if not user_name:
        error_msg = (
            "Estado inconsistente: Usuario autenticado sin nombre. "
            f"user: {user}"
        )
        logger.error(error_msg)
        raise AuthStateInconsistentError(error_msg)

    # ✅ Todo OK: Extraer rol con fallback
    user_role = user.get("role", default_role)

    return user_name, user_role


def get_auth_headers_from_tokens(auth_tokens: Optional[Dict[str, Any]]) -> Optional[Dict[str, str]]:
    """
    Extrae headers de autenticación desde auth-tokens-store.

    REGLA #7.6: Multi-Worker Token Restoration Pattern

    En Render multi-worker (Gunicorn), cada worker tiene su propio singleton
    auth_context. Sincronizar al singleton no es confiable porque el siguiente
    request puede ir a otro worker.

    Esta función extrae el token y retorna headers EXPLÍCITOS para pasar
    directamente a request_coordinator.make_request(auth_headers=...).

    Args:
        auth_tokens: Datos del store auth-tokens-store.
                    Estructura: {"tokens": {"access": "...", "refresh": "..."}, "timestamp": "..."}

    Returns:
        Dict con Authorization header si hay token válido, None si no hay token.
        Ejemplo: {"Authorization": "Bearer eyJhbG..."}

    Usage:
        @app.callback(
            ...,
            State("auth-tokens-store", "data")
        )
        def my_callback(..., auth_tokens):
            auth_headers = get_auth_headers_from_tokens(auth_tokens)

            response = request_coordinator.make_request(
                "/api/v1/endpoint",
                method="GET",
                auth_headers=auth_headers  # Pasa headers explícitamente
            )

    Note:
        - Requiere State("auth-tokens-store", "data") en el callback
        - Bypassa el singleton auth_context completamente
        - Thread-safe para multi-worker environments
    """
    if not auth_tokens or "tokens" not in auth_tokens:
        logger.debug("[AUTH_HELPERS] No auth_tokens available for header extraction")
        return None

    try:
        from utils.auth import auth_manager

        if auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"]):
            token = auth_manager.get_access_token()
            if token:
                logger.debug("[AUTH_HELPERS] Successfully extracted auth headers from tokens store")
                return {"Authorization": f"Bearer {token}"}
            else:
                logger.warning("[AUTH_HELPERS] restore succeeded but get_access_token returned None")
        else:
            logger.warning("[AUTH_HELPERS] Failed to restore tokens from encrypted tokens")

    except Exception as e:
        logger.error(f"[AUTH_HELPERS] Error extracting auth headers: {e}")

    return None


def should_show_session_expired_toast(auth_state: Optional[Dict], tokens_data: Optional[Dict]) -> bool:
    """
    Determina si debe mostrarse el toast de "Sesión expirada".

    Solo retorna True si:
    - Usuario NO está autenticado actualmente
    - Había tokens previos (sesión existente)

    Casos:
    - Usuario nuevo sin sesión → False (no mostrar toast)
    - Sesión expirada con tokens → True (mostrar toast)
    - Usuario autenticado → False (no aplica)

    Args:
        auth_state: Estado de autenticación
        tokens_data: Datos de tokens almacenados (puede contener tokens encriptados)

    Returns:
        bool: True si debe mostrarse toast de sesión expirada

    Examples:
        >>> # Usuario nuevo (sin sesión previa)
        >>> should_show_session_expired_toast(None, None)
        False

        >>> # Usuario con sesión expirada (tenía tokens)
        >>> should_show_session_expired_toast(
        ...     {'authenticated': False},
        ...     {'tokens': 'encrypted_blob', 'storage_type': 'local'}
        ... )
        True

        >>> # Usuario autenticado (no mostrar toast)
        >>> should_show_session_expired_toast(
        ...     {'authenticated': True, 'user': {...}},
        ...     {'tokens': 'valid_tokens'}
        ... )
        False
    """
    # Caso 1: Usuario autenticado → No mostrar toast
    if is_user_authenticated(auth_state):
        return False

    # Caso 2: Usuario NO autenticado
    # - Si NO hay tokens → Usuario nuevo (NO mostrar)
    # - Si SÍ hay tokens → Sesión expirada (SÍ mostrar)
    had_previous_session = tokens_data is not None and tokens_data.get("tokens") is not None

    return had_previous_session
