# frontend/utils/auth/base.py
"""
Abstract Base Interface for Authentication Managers (Pivot 2026).

Defines the contract that both Local (PIN) and Cloud (JWT) auth managers must fulfill.
This enables the Strategy Pattern for authentication across different deployment modes.

Author: Pivot 2026 - Auth Refactor
Date: 2026-01
"""

from abc import ABC, abstractmethod
from typing import Any, Dict, Optional


class AuthManagerBase(ABC):
    """
    Abstract base class for authentication managers.

    Both LocalAuthManager (PIN-based) and CloudAuthManager (JWT/OAuth)
    implement this interface, allowing callbacks to use authentication
    without knowing which system is active.
    """

    # =========================================================================
    # Core Authentication State
    # =========================================================================

    @abstractmethod
    def is_authenticated(self) -> bool:
        """
        Check if the user/terminal is currently authenticated.

        Returns:
            True if authenticated (unlocked for local, valid token for cloud)
        """
        pass

    @abstractmethod
    def get_access_token(self) -> Optional[str]:
        """
        Get the current access token (if applicable).

        For local mode, this may return a pseudo-token or None.
        For cloud mode, this returns the JWT access token.

        Returns:
            Access token string or None
        """
        pass

    # =========================================================================
    # Session Management
    # =========================================================================

    @abstractmethod
    def logout(self) -> None:
        """
        End the current session.

        For local mode: locks the terminal.
        For cloud mode: invalidates tokens and clears session.
        """
        pass

    @abstractmethod
    def get_current_user(self) -> Optional[Dict[str, Any]]:
        """
        Get information about the current authenticated user/session.

        Returns:
            Dictionary with user/session info, or None if not authenticated
        """
        pass

    # =========================================================================
    # Token Persistence (for dcc.Store integration)
    # =========================================================================

    @abstractmethod
    def get_encrypted_tokens(self) -> Optional[Dict[str, str]]:
        """
        Get tokens in encrypted/serializable form for dcc.Store persistence.

        Returns:
            Dictionary with encrypted token data, or None
        """
        pass

    @abstractmethod
    def restore_from_encrypted_tokens(
        self, encrypted_tokens: Dict[str, str], **kwargs
    ) -> bool:
        """
        Restore session from encrypted/serialized token data.

        Args:
            encrypted_tokens: Token data from dcc.Store
            **kwargs: Additional arguments (e.g., auth_headers for cloud mode)

        Returns:
            True if restoration was successful
        """
        pass

    # =========================================================================
    # Token Refresh (may be no-op for local)
    # =========================================================================

    @abstractmethod
    def refresh_access_token(self) -> bool:
        """
        Refresh the access token if applicable.

        For local mode: no-op, always returns True (no tokens to refresh).
        For cloud mode: uses refresh_token to get new access_token.

        Returns:
            True if refresh was successful or not needed
        """
        pass

    # =========================================================================
    # Auth Mode Detection
    # =========================================================================

    @property
    @abstractmethod
    def auth_mode(self) -> str:
        """
        Get the authentication mode identifier.

        Returns:
            'local' for PIN-based auth, 'cloud' for JWT/OAuth auth
        """
        pass

    @property
    def is_local_mode(self) -> bool:
        """Check if running in local (PIN) mode."""
        return self.auth_mode == "local"

    @property
    def is_cloud_mode(self) -> bool:
        """Check if running in cloud (JWT/OAuth) mode."""
        return self.auth_mode == "cloud"
