# frontend/utils/auth/auth_local.py
"""
Local Authentication Manager - PIN-based (Pivot 2026).

This implementation uses the LocalSecurityManager for PIN-based
authentication in local Windows installations.

NO external dependencies (no jwt, no cryptography required).

Author: Pivot 2026 - Auth Refactor
Date: 2026-01
"""

import logging
import os
from typing import Any, Dict, Optional

import requests

from .base import AuthManagerBase

logger = logging.getLogger(__name__)

# Backend URL (same machine in local mode)
BACKEND_URL = os.getenv("BACKEND_URL", "http://127.0.0.1:8000")

# Security: Validate backend URL is localhost (PIN sent in plaintext)
_ALLOWED_HOSTS = ("127.0.0.1", "localhost", "::1")
_parsed_host = BACKEND_URL.split("://")[-1].split(":")[0].split("/")[0]
if _parsed_host not in _ALLOWED_HOSTS:
    raise ValueError(
        f"SECURITY: Local auth requires localhost backend. "
        f"Got BACKEND_URL={BACKEND_URL} (host={_parsed_host}). "
        f"PIN transmission over network is not secure!"
    )


class AuthManagerLocal(AuthManagerBase):
    """
    PIN-based authentication manager for local installations.

    Uses the backend's LocalSecurityManager via HTTP endpoints.
    No JWT tokens - authentication state lives in backend memory.

    Security: Only allows localhost connections (PIN sent in plaintext over HTTP).
    """

    def __init__(self):
        """Initialize the local auth manager."""
        self._cached_session: Optional[Dict[str, Any]] = None
        logger.info("[AUTH_LOCAL] AuthManagerLocal initialized (localhost only)")

    # =========================================================================
    # Core Authentication State
    # =========================================================================

    def is_authenticated(self) -> bool:
        """
        Check if terminal is unlocked.

        Queries the backend's /auth/local/status endpoint.
        """
        try:
            response = requests.get(
                f"{BACKEND_URL}/api/v1/auth/local/status",
                timeout=2,
            )
            if response.status_code == 200:
                data = response.json()
                is_unlocked = data.get("is_unlocked", False)
                # Cache session data
                if is_unlocked:
                    self._cached_session = data
                return is_unlocked
            return False
        except requests.RequestException as e:
            logger.warning(f"[AUTH_LOCAL] Status check failed: {e}")
            # If we can't reach backend, assume not authenticated
            return False

    def get_access_token(self) -> Optional[str]:
        """
        Get access token - returns pseudo-token for local mode.

        Local mode doesn't use JWT tokens, but some code paths
        may check for token presence. We return a marker value
        to indicate local auth is active.
        """
        if self.is_authenticated():
            return "LOCAL_SESSION_ACTIVE"
        return None

    # =========================================================================
    # Session Management
    # =========================================================================

    def logout(self) -> None:
        """
        Lock the terminal.

        Calls the backend's /auth/local/lock endpoint.
        """
        try:
            response = requests.post(
                f"{BACKEND_URL}/api/v1/auth/local/lock",
                timeout=5,
            )
            if response.status_code == 200:
                logger.info("[AUTH_LOCAL] Terminal locked")
                self._cached_session = None
            else:
                logger.warning(f"[AUTH_LOCAL] Lock failed: {response.status_code}")
        except requests.RequestException as e:
            logger.error(f"[AUTH_LOCAL] Lock request failed: {e}")

    def get_current_user(self) -> Optional[Dict[str, Any]]:
        """
        Get current session info.

        Returns pharmacy and role information from the session.
        """
        if not self.is_authenticated():
            return None

        if self._cached_session:
            return {
                "pharmacy_name": self._cached_session.get("pharmacy_name", ""),
                "role": self._cached_session.get("role", "operativo"),
                "is_local": True,
                "auto_lock_minutes": self._cached_session.get("auto_lock_minutes", 5),
            }

        return None

    # =========================================================================
    # Token Persistence (simplified for local mode)
    # =========================================================================

    def get_encrypted_tokens(self) -> Optional[Dict[str, str]]:
        """
        Get tokens for persistence - simplified for local mode.

        Local mode doesn't persist tokens in dcc.Store.
        The session state lives in backend memory.
        """
        # Return minimal data to satisfy interface
        if self.is_authenticated():
            return {"mode": "local", "active": "true"}
        return None

    def restore_from_encrypted_tokens(
        self, encrypted_tokens: Dict[str, str], **kwargs
    ) -> bool:
        """
        Restore from tokens - no-op for local mode.

        Local mode session is managed by backend, not frontend tokens.
        The PIN must be re-entered after app restart.
        """
        # Nothing to restore - user must enter PIN
        logger.debug("[AUTH_LOCAL] restore_from_encrypted_tokens called (no-op)")
        return self.is_authenticated()

    # =========================================================================
    # Token Refresh (no-op for local)
    # =========================================================================

    def refresh_access_token(self) -> bool:
        """
        Refresh token - no-op for local mode.

        Local mode doesn't use JWT tokens with expiration.
        The session stays active until locked or auto-lock triggers.
        """
        # No tokens to refresh
        return True

    # =========================================================================
    # Auth Mode Detection
    # =========================================================================

    @property
    def auth_mode(self) -> str:
        """Return 'local' to identify PIN-based auth."""
        return "local"

    # =========================================================================
    # Local-Specific Methods
    # =========================================================================

    def unlock(self, pin: str) -> Dict[str, Any]:
        """
        Unlock the terminal with PIN.

        Args:
            pin: 4-6 digit PIN

        Returns:
            Dict with success status and message/error
        """
        try:
            response = requests.post(
                f"{BACKEND_URL}/api/v1/auth/local/unlock",
                json={"pin": pin},
                timeout=5,
            )

            if response.status_code == 200:
                logger.info("[AUTH_LOCAL] Terminal unlocked")
                return {"success": True, "message": "Terminal desbloqueado"}
            else:
                error = response.json().get("detail", "PIN incorrecto")
                logger.warning(f"[AUTH_LOCAL] Unlock failed: {error}")
                return {"success": False, "error": error}

        except requests.RequestException as e:
            logger.error(f"[AUTH_LOCAL] Unlock request failed: {e}")
            return {"success": False, "error": f"Error de conexión: {e}"}

    def set_pin(self, new_pin: str) -> Dict[str, Any]:
        """
        Change the terminal PIN (titular only).

        Args:
            new_pin: New 4-6 digit PIN

        Returns:
            Dict with success status and message/error
        """
        try:
            response = requests.post(
                f"{BACKEND_URL}/api/v1/auth/local/set-pin",
                json={"pin": new_pin},
                timeout=5,
            )

            if response.status_code == 200:
                logger.info("[AUTH_LOCAL] PIN changed")
                return {"success": True, "message": "PIN actualizado"}
            elif response.status_code == 403:
                return {"success": False, "error": "Solo el titular puede cambiar el PIN"}
            elif response.status_code == 401:
                return {"success": False, "error": "Terminal debe estar desbloqueado"}
            else:
                error = response.json().get("detail", "Error al cambiar PIN")
                return {"success": False, "error": error}

        except requests.RequestException as e:
            logger.error(f"[AUTH_LOCAL] Set PIN request failed: {e}")
            return {"success": False, "error": f"Error de conexión: {e}"}

    def get_status(self) -> Dict[str, Any]:
        """
        Get detailed terminal status.

        Returns:
            Dict with is_unlocked, pharmacy_name, role, state, auto_lock_minutes
        """
        try:
            response = requests.get(
                f"{BACKEND_URL}/api/v1/auth/local/status",
                timeout=2,
            )

            if response.status_code == 200:
                return response.json()
            else:
                return {
                    "is_unlocked": False,
                    "state": "error",
                    "pharmacy_name": "",
                    "role": "operativo",
                    "auto_lock_minutes": 5,
                }

        except requests.RequestException:
            return {
                "is_unlocked": False,
                "state": "offline",
                "pharmacy_name": "",
                "role": "operativo",
                "auto_lock_minutes": 5,
            }

    def is_titular(self) -> bool:
        """Check if current session has titular (owner) role."""
        if self._cached_session:
            return self._cached_session.get("role") == "titular"
        # Fetch fresh status
        status = self.get_status()
        return status.get("role") == "titular"
