# frontend/utils/auth/__init__.py
"""
Authentication Module Factory (Pivot 2026).

This module provides a unified authentication interface that automatically
selects the appropriate implementation based on the deployment mode:

- LOCAL MODE (KAIFARMA_LOCAL=true): PIN-based authentication
- CLOUD MODE (default): JWT/OAuth authentication

The detection logic is designed to be FAIL-SAFE:
1. Multiple detection methods with fallbacks
2. Clear logging of selected mode
3. Sensible defaults if detection fails
4. NEVER raises exceptions during import

Usage:
    from utils.auth import auth_manager

    if auth_manager.is_authenticated():
        user = auth_manager.get_current_user()

Author: Pivot 2026 - Auth Refactor
Date: 2026-01
"""

import logging
import os
import sys
from typing import Union

logger = logging.getLogger(__name__)


# =============================================================================
# FAIL-SAFE MODE DETECTION
# =============================================================================

def _detect_auth_mode() -> str:
    """
    Detect the authentication mode with multiple fallback strategies.

    Detection priority (first match wins):
    1. KAIFARMA_LOCAL environment variable (explicit)
    2. KAIFARMA_MODE environment variable (explicit mode name)
    3. Presence of local-mode markers (version.json in AppData)
    4. Check if running from packaged app location
    5. Default to 'cloud' (safest fallback for production)

    Returns:
        'local' or 'cloud'
    """
    detected_mode = None
    detection_source = "default"

    try:
        # Strategy 1: Check KAIFARMA_LOCAL env var (primary method)
        kaifarma_local = os.getenv("KAIFARMA_LOCAL", "").strip().lower()
        if kaifarma_local in ("true", "1", "yes", "on"):
            detected_mode = "local"
            detection_source = "KAIFARMA_LOCAL=true"
        elif kaifarma_local in ("false", "0", "no", "off"):
            detected_mode = "cloud"
            detection_source = "KAIFARMA_LOCAL=false"

        # Strategy 2: Check KAIFARMA_MODE env var (explicit mode name)
        if detected_mode is None:
            kaifarma_mode = os.getenv("KAIFARMA_MODE", "").strip().lower()
            if kaifarma_mode == "local":
                detected_mode = "local"
                detection_source = "KAIFARMA_MODE=local"
            elif kaifarma_mode == "cloud":
                detected_mode = "cloud"
                detection_source = "KAIFARMA_MODE=cloud"

        # Strategy 3: Check for local installation markers
        if detected_mode is None:
            try:
                local_appdata = os.getenv("LOCALAPPDATA", "")
                if local_appdata:
                    version_file = os.path.join(local_appdata, "kaiFarma", "version.json")
                    if os.path.exists(version_file):
                        detected_mode = "local"
                        detection_source = f"version.json exists: {version_file}"
            except Exception:
                pass  # Ignore errors in marker detection

        # Strategy 4: Check if running from packaged app location
        if detected_mode is None:
            try:
                exe_path = sys.executable.lower() if hasattr(sys, "executable") else ""
                if "kaifarma" in exe_path and "appdata" in exe_path:
                    detected_mode = "local"
                    detection_source = f"exe path contains kaifarma+appdata: {exe_path}"
            except Exception:
                pass  # Ignore errors in path detection

        # Strategy 5: Default fallback
        if detected_mode is None:
            # Default to cloud mode (safer for production - requires proper auth)
            detected_mode = "cloud"
            detection_source = "default fallback (no markers found)"

    except Exception as e:
        # Ultimate fallback if ANY exception occurs
        logger.error(f"[AUTH] Mode detection failed with exception: {e}")
        detected_mode = "cloud"
        detection_source = f"exception fallback: {e}"

    # Log the detection result (always, for debugging)
    logger.info(f"[AUTH] Mode detected: {detected_mode.upper()} (source: {detection_source})")

    return detected_mode


# =============================================================================
# SINGLETON INITIALIZATION
# =============================================================================

# Detect mode once at import time
_AUTH_MODE = _detect_auth_mode()

# Initialize the appropriate auth manager based on detected mode
_auth_manager_instance = None


def _initialize_auth_manager():
    """
    Initialize the auth manager singleton.

    This is separated from module-level code to allow for better error handling
    and to avoid circular imports.
    """
    global _auth_manager_instance

    if _auth_manager_instance is not None:
        return _auth_manager_instance

    try:
        if _AUTH_MODE == "local":
            # Local mode: PIN-based authentication (no jwt dependency)
            from .auth_local import AuthManagerLocal
            _auth_manager_instance = AuthManagerLocal()
            logger.info("[AUTH] Initialized AuthManagerLocal (PIN-based)")
        else:
            # Cloud mode: JWT/OAuth authentication
            from .auth_cloud import AuthManagerCloud
            _auth_manager_instance = AuthManagerCloud()
            logger.info("[AUTH] Initialized AuthManagerCloud (JWT/OAuth)")

    except ImportError as e:
        # Handle missing dependencies gracefully
        logger.error(f"[AUTH] Failed to import auth manager: {e}")

        if _AUTH_MODE == "local":
            # If local mode fails, something is wrong with our code
            raise ImportError(
                f"Failed to initialize local auth manager: {e}. "
                "This should not happen - auth_local.py has no external dependencies."
            ) from e
        else:
            # If cloud mode fails (e.g., missing jwt), try local as fallback
            logger.warning("[AUTH] Cloud auth failed, attempting local fallback...")
            try:
                from .auth_local import AuthManagerLocal
                _auth_manager_instance = AuthManagerLocal()
                logger.warning("[AUTH] Using AuthManagerLocal as fallback (cloud dependencies missing)")
            except Exception as fallback_error:
                raise ImportError(
                    f"Failed to initialize any auth manager. "
                    f"Cloud error: {e}, Local error: {fallback_error}"
                ) from e

    except Exception as e:
        # Catch-all for any other initialization errors
        logger.error(f"[AUTH] Unexpected error initializing auth manager: {e}")
        raise

    return _auth_manager_instance


# =============================================================================
# PUBLIC API
# =============================================================================

# Initialize on first access (lazy initialization)
def get_auth_manager():
    """
    Get the singleton auth manager instance.

    Returns:
        AuthManagerLocal or AuthManagerCloud depending on mode
    """
    return _initialize_auth_manager()


# For backwards compatibility: create auth_manager as a property-like access
# This allows `from utils.auth import auth_manager` to work
class _AuthManagerProxy:
    """
    Proxy class that provides lazy access to the auth manager singleton.

    This allows the module to be imported without immediately initializing
    the auth manager, which is useful for avoiding circular imports.
    """

    def __getattr__(self, name):
        return getattr(get_auth_manager(), name)

    def __repr__(self):
        manager = get_auth_manager()
        return f"<AuthManagerProxy mode={manager.auth_mode}>"


# Create the proxy instance that will be imported as 'auth_manager'
auth_manager = _AuthManagerProxy()


# =============================================================================
# CONVENIENCE EXPORTS
# =============================================================================

# Export the base class for type hints
from .base import AuthManagerBase

# Export utility functions from cloud module (backwards compatibility)
# These are safe to import even in local mode as they don't use jwt
def create_authenticated_client(manager=None):
    """Create an authenticated HTTP session."""
    import requests
    if manager is None:
        manager = get_auth_manager()
    session = requests.Session()
    token = manager.get_access_token()
    if token and token != "LOCAL_SESSION_ACTIVE":
        session.headers.update({"Authorization": f"Bearer {token}"})
    return session


def require_auth(func):
    """Decorator for callbacks requiring authentication (legacy, no-op)."""
    def wrapper(*args, **kwargs):
        return func(*args, **kwargs)
    return wrapper


# =============================================================================
# MODULE-LEVEL CONSTANTS
# =============================================================================

# Expose the detected mode for external checks
AUTH_MODE = _AUTH_MODE
IS_LOCAL_MODE = _AUTH_MODE == "local"
IS_CLOUD_MODE = _AUTH_MODE == "cloud"


# =============================================================================
# TYPE HINTS
# =============================================================================

# For type checking, expose the union type
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from .auth_local import AuthManagerLocal
    from .auth_cloud import AuthManagerCloud

    AuthManager = Union[AuthManagerLocal, AuthManagerCloud]
