"""
Audit Script - Callback IDs

Analiza todos los callbacks registrados y extrae los IDs que realmente se usan.
Compara con CRITICAL_IDS en layout_validator.py para detectar false positives.

Uso:
    python frontend/utils/audit_callback_ids.py
"""

import ast
import logging
from pathlib import Path
from typing import Set, Dict, List

logging.basicConfig(level=logging.INFO, format='%(message)s')
logger = logging.getLogger(__name__)


def extract_callback_ids_from_file(file_path: Path) -> Dict[str, List[str]]:
    """
    Extrae IDs de callbacks de un archivo Python usando AST parsing.

    Returns:
        Dict con estructura: {"file": "path", "callbacks": [{...}]}
    """
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            tree = ast.parse(f.read(), filename=str(file_path))
    except Exception as e:
        logger.warning(f"Error parsing {file_path}: {e}")
        return {}

    callback_ids = []

    # Buscar decoradores @app.callback o @callback
    for node in ast.walk(tree):
        if isinstance(node, ast.FunctionDef):
            for decorator in node.decorator_list:
                # @app.callback(...) o @callback(...)
                if isinstance(decorator, ast.Call):
                    decorator_name = None
                    if isinstance(decorator.func, ast.Name):
                        decorator_name = decorator.func.id
                    elif isinstance(decorator.func, ast.Attribute):
                        decorator_name = decorator.func.attr

                    if decorator_name in ['callback', 'clientside_callback']:
                        ids = extract_ids_from_decorator(decorator, node.name)
                        if ids:
                            callback_ids.append({
                                'function': node.name,
                                'line': node.lineno,
                                'ids': ids
                            })

    return callback_ids


def extract_ids_from_decorator(decorator: ast.Call, func_name: str) -> Set[str]:
    """
    Extrae IDs de componentes de un decorador @callback.

    Busca strings en Output(...), Input(...), State(...).
    """
    ids = set()

    for arg in decorator.args:
        ids.update(extract_ids_from_node(arg))

    return ids


def extract_ids_from_node(node: ast.AST) -> Set[str]:
    """
    Extrae IDs recursivamente de un nodo AST.

    Busca:
    - Output("my-id", ...)
    - Input("my-id", ...)
    - State("my-id", ...)
    - [Output(...), Input(...)]
    """
    ids = set()

    # Listas de callbacks
    if isinstance(node, ast.List):
        for elem in node.elts:
            ids.update(extract_ids_from_node(elem))

    # Llamadas a Output/Input/State
    elif isinstance(node, ast.Call):
        func_name = None
        if isinstance(node.func, ast.Name):
            func_name = node.func.id
        elif isinstance(node.func, ast.Attribute):
            func_name = node.func.attr

        if func_name in ['Output', 'Input', 'State']:
            # Primer argumento es el ID
            if node.args and isinstance(node.args[0], ast.Constant):
                component_id = node.args[0].value
                if isinstance(component_id, str):
                    ids.add(component_id)

    return ids


def audit_all_callbacks() -> Dict[str, Set[str]]:
    """
    Audita todos los archivos Python del frontend para extraer IDs de callbacks.

    Returns:
        Dict: {"file_path": Set[ids]}
    """
    frontend_path = Path(__file__).parent.parent

    # Directorios a escanear
    dirs_to_scan = [
        frontend_path / "callbacks",
        frontend_path / "components",
        frontend_path / "layouts",
        frontend_path / "pages",
    ]

    # Incluir app.py
    files_to_scan = [frontend_path / "app.py"]

    # Agregar todos los archivos .py de directorios
    for directory in dirs_to_scan:
        if directory.exists():
            files_to_scan.extend(directory.rglob("*.py"))

    all_ids = set()
    file_results = {}

    logger.info("=" * 70)
    logger.info("CALLBACK IDS AUDIT")
    logger.info("=" * 70)
    logger.info("")

    for file_path in files_to_scan:
        callback_data = extract_callback_ids_from_file(file_path)
        if callback_data:
            file_ids = set()
            for callback in callback_data:
                file_ids.update(callback['ids'])

            if file_ids:
                file_results[str(file_path.relative_to(frontend_path))] = file_ids
                all_ids.update(file_ids)

                logger.info(f"📄 {file_path.relative_to(frontend_path)}")
                logger.info(f"   {len(callback_data)} callbacks, {len(file_ids)} unique IDs")

    logger.info("")
    logger.info(f"TOTAL: {len(all_ids)} unique component IDs across {len(file_results)} files")
    logger.info("")

    return all_ids, file_results


def compare_with_critical_ids(actual_ids: Set[str]) -> None:
    """
    Compara IDs reales de callbacks con CRITICAL_IDS en layout_validator.py.
    """
    try:
        import sys
        from pathlib import Path
        # Añadir root del proyecto al path si no está
        project_root = Path(__file__).parent.parent.parent
        if str(project_root) not in sys.path:
            sys.path.insert(0, str(project_root))

        from frontend.utils.layout_validator import CRITICAL_IDS
    except ImportError as e:
        logger.error(f"❌ No se pudo importar CRITICAL_IDS de layout_validator.py: {e}")
        return

    critical_set = set(CRITICAL_IDS)

    logger.info("=" * 70)
    logger.info("COMPARISON: CRITICAL_IDS vs ACTUAL CALLBACK IDS")
    logger.info("=" * 70)
    logger.info("")

    # IDs en CRITICAL_IDS pero NO en callbacks (false positives)
    false_positives = critical_set - actual_ids

    if false_positives:
        logger.warning(f"⚠️  FALSE POSITIVES: {len(false_positives)} IDs en CRITICAL_IDS que NO se usan en callbacks:")
        logger.warning("")
        for component_id in sorted(false_positives):
            logger.warning(f"   - {component_id}")
        logger.warning("")
        logger.warning("   Acción: REMOVER estos IDs de:")
        logger.warning("   1. frontend/utils/layout_validator.py CRITICAL_IDS")
        logger.warning("   2. frontend/app.py skeleton (si existen como placeholders)")
        logger.warning("")
    else:
        logger.info("✅ No hay false positives - todos los IDs en CRITICAL_IDS se usan en callbacks")
        logger.info("")

    # IDs en callbacks pero NO en CRITICAL_IDS (missing)
    missing = actual_ids - critical_set

    if missing:
        logger.warning(f"⚠️  MISSING: {len(missing)} IDs usados en callbacks pero NO en CRITICAL_IDS:")
        logger.warning("")
        for component_id in sorted(missing):
            logger.warning(f"   - {component_id}")
        logger.warning("")
        logger.warning("   Acción: AGREGAR estos IDs a:")
        logger.warning("   1. frontend/utils/layout_validator.py CRITICAL_IDS")
        logger.warning("   2. frontend/app.py skeleton (si no existen)")
        logger.warning("")
    else:
        logger.info("✅ No hay IDs faltantes - CRITICAL_IDS cubre todos los callbacks")
        logger.info("")

    # Resumen
    logger.info("=" * 70)
    logger.info("SUMMARY")
    logger.info("=" * 70)
    logger.info(f"IDs en CRITICAL_IDS:     {len(critical_set)}")
    logger.info(f"IDs en callbacks reales: {len(actual_ids)}")
    logger.info(f"False positives:         {len(false_positives)}")
    logger.info(f"Missing:                 {len(missing)}")
    logger.info("")

    if false_positives or missing:
        logger.warning("⚠️  ACTION REQUIRED: Actualizar CRITICAL_IDS para reflejar callbacks reales")
        return False
    else:
        logger.info("✅ CRITICAL_IDS está perfectamente sincronizado con callbacks")
        return True


if __name__ == "__main__":
    actual_ids, file_results = audit_all_callbacks()
    is_synchronized = compare_with_critical_ids(actual_ids)

    # Exit code para CI/CD
    import sys
    sys.exit(0 if is_synchronized else 1)
