# frontend/utils/api_client.py
"""
Cliente para comunicación con el backend de xfarma.
Maneja autenticación, errores y timeouts.
"""

import logging
import os
import time
from dataclasses import dataclass
from typing import Any, Dict, Optional

import requests

logger = logging.getLogger(__name__)


@dataclass
class ApiResponse:
    """Respuesta estándar de la API"""

    success: bool
    data: Any = None
    message: str = ""
    status_code: int = 200


class BackendClient:
    """Cliente para comunicación con el backend xfarma

    IMPORTANTE: Thread-safe para multi-worker environments.
    La sesión se crea en cada request en lugar de mantener una sesión global.
    """

    def __init__(self):
        # Obtener backend URL con fallback robusto para producción
        self.backend_url = os.getenv("BACKEND_URL", "http://localhost:8000")

        # Log para debug en producción
        logger.info(f"BackendClient initialized with URL: {self.backend_url}")

        # Detect if running on Render and adjust timeouts accordingly
        is_render = os.getenv("RENDER") is not None

        # Configuración de retry con variables de entorno para fácil ajuste
        self.max_retries = int(os.getenv("API_MAX_RETRIES", "3"))
        self.retry_delay = float(os.getenv("API_BASE_RETRY_DELAY", "2" if is_render else "1"))
        self.retry_multiplier = float(os.getenv("API_RETRY_MULTIPLIER", "2"))
        self.timeout = int(os.getenv("API_TIMEOUT", "45" if is_render else "30"))

        if is_render:
            logger.info(
                f"Running on Render - Timeout: {self.timeout}s, Max retries: {self.max_retries}, Base delay: {self.retry_delay}s"
            )
        else:
            logger.info(
                f"Running locally - Timeout: {self.timeout}s, Max retries: {self.max_retries}, Base delay: {self.retry_delay}s"
            )

        # CRITICAL FIX: NO mantener sesión como atributo de instancia
        # Esto causaba HTTP 500 errors en Render multi-worker
        # self.session = requests.Session()  # ELIMINADO

        # Headers por defecto (se aplicarán en cada request)
        self.default_headers = {"Content-Type": "application/json", "User-Agent": "xfarma-frontend/1.0"}

    def _make_request(self, method: str, endpoint: str, **kwargs) -> ApiResponse:
        """Realizar petición HTTP con manejo de errores y reintentos

        Thread-safe: Crea nueva sesión/conexión para cada request.
        """
        # Añadir versionado automático si el endpoint no lo tiene
        if endpoint.startswith("/api/") and not endpoint.startswith("/api/v"):
            endpoint = endpoint.replace("/api/", "/api/v1/", 1)
        url = f"{self.backend_url}{endpoint}"
        last_error = None

        # Aplicar headers por defecto a kwargs
        if "headers" not in kwargs:
            kwargs["headers"] = {}
        kwargs["headers"].update(self.default_headers)

        # ✅ FIX (Issue #xxx): Usar auth_manager (thread-safe para multi-worker)
        # Errores 401 en producción causados por uso de auth_context deprecado
        # auth_manager es la fuente de verdad oficial desde Issue #348 (RBAC)
        from utils.auth import auth_manager

        access_token = auth_manager.get_access_token()
        if access_token:
            # Inyectar JWT token en headers de autenticación
            kwargs["headers"]["Authorization"] = f"Bearer {access_token}"
            logger.debug(
                f"[API_CLIENT] Injected JWT token (length: {len(access_token)}) for {endpoint}"
            )
        else:
            logger.warning(f"[API_CLIENT] No access token available for {endpoint}")

        for attempt in range(self.max_retries):
            try:
                # ✅ FIX ENCODING: Asegurar que los parámetros se envíen en UTF-8
                # Issue: "params" puede contener caracteres españoles (Ñ) que requests debe encodear correctamente
                # Solución: Si params es una lista de tuplas (de _prepare_list_params), convertir valores a UTF-8 explícitamente
                if 'params' in kwargs:
                    params = kwargs['params']
                    if isinstance(params, list) and params:
                        # _prepare_list_params retorna lista de tuplas: [('names', 'TECNIMEDE ESPAÑA'), ...]
                        # Asegurar que cada valor string esté en UTF-8 (Python 3 maneja esto naturalmente)
                        # requests lo encodeará correctamente si es un string de Python
                        encoded_params = []
                        for key, value in params:
                            if isinstance(value, str):
                                # Asegurar que el string está en UTF-8 (Python 3 maneja esto naturalmente)
                                # requests lo encodeará correctamente si es un string de Python
                                encoded_params.append((key, value))
                            else:
                                encoded_params.append((key, value))
                        kwargs['params'] = encoded_params

                # CRITICAL FIX: Usar requests directamente sin sesión persistente
                # Esto garantiza thread-safety in multi-worker environments
                # Extraer timeout de kwargs si existe, sino usar default
                request_timeout = kwargs.pop('timeout', self.timeout)
                response = requests.request(method=method, url=url, timeout=request_timeout, **kwargs)

                # 🔍 DEBUG: Log de URL generada para detectar problemas de encoding
                if 'params' in kwargs and any('TECNIMEDE' in str(v) or 'ESPAÑA' in str(v) for k, v in (kwargs['params'] if isinstance(kwargs['params'], list) else [])):
                    logger.debug(f"[API_CLIENT_ENCODING] Request URL: {response.url}")
                    logger.debug(f"[API_CLIENT_ENCODING] Params enviados: {kwargs.get('params')}")

                # ✅ FIX: Forzar UTF-8 encoding para caracteres españoles (Ñ, tildes)
                # Issue: requests infiere encoding incorrectamente de headers HTTP
                # Solución: Forzar UTF-8 ANTES de parsear JSON
                response.encoding = 'utf-8'

                # Verificar status code
                if response.status_code >= 400:
                    return ApiResponse(
                        success=False, message=f"Error HTTP {response.status_code}", status_code=response.status_code
                    )

                # Intentar parsear JSON
                try:
                    data = response.json()
                except ValueError:
                    data = response.text

                return ApiResponse(success=True, data=data, status_code=response.status_code)

            except requests.exceptions.Timeout as e:
                last_error = e
                if attempt < self.max_retries - 1:
                    wait_time = self.retry_delay * (self.retry_multiplier**attempt)  # Exponential backoff
                    logger.warning(
                        f"Timeout en intento {attempt + 1}/{self.max_retries} para {endpoint}. "
                        f"Reintentando en {wait_time}s..."
                    )
                    time.sleep(wait_time)
                    continue

            except requests.exceptions.ConnectionError as e:
                last_error = e
                if attempt < self.max_retries - 1:
                    wait_time = self.retry_delay * (self.retry_multiplier**attempt)  # Exponential backoff
                    logger.warning(
                        f"Error de conexión en intento {attempt + 1}/{self.max_retries} para {endpoint}. "
                        f"Reintentando en {wait_time}s..."
                    )
                    time.sleep(wait_time)
                    continue

            except Exception as e:
                logger.error(f"Error inesperado en API: {str(e)}")
                return ApiResponse(success=False, message=f"Error inesperado: {str(e)}", status_code=500)

        # Si llegamos aquí, todos los reintentos fallaron
        if isinstance(last_error, requests.exceptions.Timeout):
            return ApiResponse(
                success=False,
                message="El sistema está procesando muchas solicitudes. Por favor, intente nuevamente en unos momentos.",
                status_code=408,
            )
        elif isinstance(last_error, requests.exceptions.ConnectionError):
            return ApiResponse(
                success=False,
                message="No se puede conectar con el servidor. Por favor, verifique su conexión e intente nuevamente.",
                status_code=503,
            )
        else:
            return ApiResponse(
                success=False,
                message="El servidor está temporalmente fuera de servicio. Por favor, intente más tarde o contacte con soporte.",
                status_code=500,
            )

    # Métodos de salud del sistema
    def check_health(self) -> ApiResponse:
        """Verificar estado del backend"""
        return self._make_request("GET", "/health")

    def get_system_status(self, include_details: bool = False) -> ApiResponse:
        """Obtener estado completo del sistema

        Args:
            include_details: Incluir detalles del sistema (catálogo, fuentes externas)
        """
        params = {}
        if include_details:
            params["include_details"] = "true"
        return self._make_request("GET", "/api/v1/system/status", params=params)

    def get_pharmacy_dashboard_summary(self, pharmacy_id: str) -> ApiResponse:
        """Obtener resumen del dashboard para una farmacia específica

        Args:
            pharmacy_id: ID de la farmacia
        """
        return self._make_request("GET", f"/api/v1/pharmacy/{pharmacy_id}/dashboard-summary")

    def get_initial_dashboard_data(self, pharmacy_id: str) -> ApiResponse:
        """Obtener todos los datos iniciales del dashboard en una sola llamada.

        Endpoint unificado que combina:
        - system_status (catálogo CIMA/nomenclator)
        - pharmacy_dashboard_summary (ventas, enriquecimiento)
        - sales_trend (últimos 30 días para el gráfico)

        Optimización: Reduce 5-6 API calls a 1 sola llamada HTTP.

        Args:
            pharmacy_id: ID de la farmacia

        Returns:
            ApiResponse con datos combinados para renderizar el dashboard completo
        """
        return self._make_request("GET", f"/api/v1/pharmacy/{pharmacy_id}/initial-data")

    # Métodos de datos de ventas
    def get_sales_summary(self, pharmacy_id: Optional[str] = None, days: int = 30) -> ApiResponse:
        """Obtener resumen de ventas"""
        params = {"days": days}
        if pharmacy_id:
            params["pharmacy_id"] = pharmacy_id

        return self._make_request("GET", "/api/sales/summary", params=params)

    def get_sales_trends(self, pharmacy_id: Optional[str] = None, period: str = "daily", days: int = 30) -> ApiResponse:
        """Obtener tendencias de ventas"""
        params = {"period": period, "days": days}
        if pharmacy_id:
            params["pharmacy_id"] = pharmacy_id

        return self._make_request("GET", "/api/sales/trends", params=params)

    def get_top_products(self, pharmacy_id: Optional[str] = None, limit: int = 10, days: int = 30) -> ApiResponse:
        """Obtener productos más vendidos"""
        params = {"limit": limit, "days": days}
        if pharmacy_id:
            params["pharmacy_id"] = pharmacy_id

        return self._make_request("GET", "/api/sales/top-products", params=params)

    # Métodos de análisis enriquecido con datos CIMA/nomenclator
    def get_enriched_summary(
        self, pharmacy_id: str, start_date: Optional[str] = None, end_date: Optional[str] = None
    ) -> ApiResponse:
        """
        Obtener resumen de ventas enriquecidas con datos CIMA/nomenclator

        Args:
            pharmacy_id: ID de la farmacia (UUID)
            start_date: Fecha inicio (YYYY-MM-DD) opcional
            end_date: Fecha fin (YYYY-MM-DD) opcional

        Returns:
            ApiResponse con métricas de enriquecimiento, genéricos y categorías terapéuticas
        """
        params = {}
        if start_date:
            params["start_date"] = start_date
        if end_date:
            params["end_date"] = end_date

        return self._make_request("GET", f"/api/sales/enriched/summary/{pharmacy_id}", params=params)

    def get_generic_opportunities(
        self,
        pharmacy_id: str,
        start_date: Optional[str] = None,
        end_date: Optional[str] = None,
        min_savings: float = 0.0,
    ) -> ApiResponse:
        """
        Obtener análisis detallado de oportunidades de genéricos

        Args:
            pharmacy_id: ID de la farmacia (UUID)
            start_date: Fecha inicio opcional
            end_date: Fecha fin opcional
            min_savings: Mínimo ahorro potencial en €

        Returns:
            ApiResponse con oportunidades de ahorro marca vs genérico
        """
        params = {"min_savings": min_savings}
        if start_date:
            params["start_date"] = start_date
        if end_date:
            params["end_date"] = end_date

        return self._make_request("GET", f"/api/sales/enriched/generic-opportunities/{pharmacy_id}", params=params)

    def get_therapeutic_analysis(
        self, pharmacy_id: str, start_date: Optional[str] = None, end_date: Optional[str] = None
    ) -> ApiResponse:
        """
        Obtener análisis por categorías terapéuticas ATC

        Args:
            pharmacy_id: ID de la farmacia (UUID)
            start_date: Fecha inicio opcional
            end_date: Fecha fin opcional

        Returns:
            ApiResponse con distribución por códigos ATC y análisis prescripción/OTC
        """
        params = {}
        if start_date:
            params["start_date"] = start_date
        if end_date:
            params["end_date"] = end_date

        return self._make_request("GET", f"/api/sales/enriched/therapeutic-analysis/{pharmacy_id}", params=params)

    # === MÉTODOS DE ENRIQUECIMIENTO Y PROGRESO ===

    def get_enrichment_progress(self, pharmacy_id: str) -> ApiResponse:
        """
        Obtener progreso del enriquecimiento de datos

        Args:
            pharmacy_id: ID de la farmacia (UUID)

        Returns:
            ApiResponse con estadísticas de enriquecimiento
        """
        return self._make_request("GET", f"/api/v1/upload/enrichment/progress?pharmacy_id={pharmacy_id}")

    def trigger_enrichment(self, pharmacy_id: str, upload_id: Optional[str] = None) -> ApiResponse:
        """
        Disparar manualmente el proceso de enriquecimiento

        Args:
            pharmacy_id: ID de la farmacia (UUID)
            upload_id: ID del upload específico (opcional)

        Returns:
            ApiResponse con confirmación del proceso
        """
        params = {"pharmacy_id": pharmacy_id}
        if upload_id:
            params["upload_id"] = upload_id

        return self._make_request("POST", "/api/v1/upload/enrichment/trigger", json=params)

    def get_enrichment_status(self, pharmacy_id: str) -> ApiResponse:
        """
        Obtener estado y calidad del proceso de enriquecimiento

        Args:
            pharmacy_id: ID de la farmacia (UUID)

        Returns:
            ApiResponse con métricas de cobertura, métodos de matching y fuentes de datos
        """
        return self._make_request("GET", f"/api/sales/enrichment/status/{pharmacy_id}")

    # Métodos de upload
    def upload_file(
        self, file_data: bytes, filename: str, file_type: str, pharmacy_id: Optional[str] = None
    ) -> ApiResponse:
        """Subir archivo de datos de farmacia"""
        files = {"file": (filename, file_data, "application/octet-stream")}
        data = {"file_type": file_type}
        if pharmacy_id:
            data["pharmacy_id"] = pharmacy_id

        return self._make_request("POST", "/upload", files=files, data=data)

    def get_upload_history(self, pharmacy_id: Optional[str] = None, limit: int = 50) -> ApiResponse:
        """Obtener historial de uploads"""
        params = {"limit": limit}
        if pharmacy_id:
            params["pharmacy_id"] = pharmacy_id

        return self._make_request("GET", "/api/v1/upload/history", params=params)

    def get_upload_status(self, upload_id: str) -> ApiResponse:
        """Obtener estado de un upload específico"""
        return self._make_request("GET", f"/api/v1/upload/status/{upload_id}")

    def get_upload_delete_preview(self, upload_id: str) -> ApiResponse:
        """
        Obtener preview de lo que se eliminará si se borra un upload.

        Retorna estadísticas sin realizar ninguna modificación:
        - Nombre del archivo
        - Número de registros de ventas
        - Rango de fechas de los datos
        - Importe total de ventas

        Args:
            upload_id: UUID del upload

        Returns:
            ApiResponse con estadísticas de lo que se eliminará
        """
        return self._make_request("GET", f"/api/v1/upload/{upload_id}/delete-preview")

    def delete_upload(self, upload_id: str) -> ApiResponse:
        """
        Eliminar un upload específico y todos los datos de ventas asociados.

        ADVERTENCIA: Esta operación es IRREVERSIBLE y elimina:
        - El registro del upload
        - Todos los datos de ventas de este upload
        - Todos los datos de enriquecimiento de las ventas

        Args:
            upload_id: UUID del upload a eliminar

        Returns:
            ApiResponse con confirmación de eliminación

        Note:
            Timeout extendido a 300s para uploads grandes (37k+ registros).
            404 se trata como éxito (upload ya eliminado por request anterior).
        """
        response = self._make_request("DELETE", f"/api/v1/upload/{upload_id}", timeout=300)
        # Treat 404 as success - upload already deleted (possibly by timed-out previous request)
        if not response.success and response.status_code == 404:
            return ApiResponse(
                success=True,
                data={"message": "Upload ya eliminado"},
                status_code=200
            )
        return response

    # === MÉTODOS DE ANÁLISIS OPTIMIZADO DE GENÉRICOS (Power BI-style) ===
    # Los métodos anteriores han sido reemplazados por el nuevo sistema de medidas
    # Ver: /api/generic-analysis/* y /api/measures/* para la nueva arquitectura

    # === MÉTODOS DE GESTIÓN DE PHARMACY PARTNERS ===

    def initialize_pharmacy_partners(self, pharmacy_id: str, force_refresh: bool = False) -> ApiResponse:
        """
        Inicializa automáticamente los laboratorios partners para una farmacia

        Args:
            pharmacy_id: ID de la farmacia (UUID)
            force_refresh: Forzar re-inicialización

        Returns:
            ApiResponse con partners inicializados
        """
        params = {"force_refresh": force_refresh}
        return self._make_request("POST", f"/api/pharmacy-partners/initialize/{pharmacy_id}", params=params)

    def get_pharmacy_partners(self, pharmacy_id: str) -> ApiResponse:
        """
        Obtiene laboratorios partners de una farmacia

        Args:
            pharmacy_id: ID de farmacia

        Returns:
            ApiResponse con lista de partners y estado de selección
        """
        return self._make_request("GET", f"/api/pharmacy-partners/{pharmacy_id}")

    def get_selected_partners(self, pharmacy_id: str) -> ApiResponse:
        """
        Obtiene solo los laboratorios partners seleccionados para una farmacia

        Args:
            pharmacy_id: ID de farmacia

        Returns:
            ApiResponse con partners seleccionados para uso en análisis
        """
        return self._make_request("GET", f"/api/pharmacy-partners/{pharmacy_id}/selected")

    def update_partner_selection(self, pharmacy_id: str, selections: list) -> ApiResponse:
        """
        Actualiza selección de múltiples partners

        Args:
            pharmacy_id: ID de farmacia
            selections: Lista de {laboratory_name, is_selected}

        Returns:
            ApiResponse con estado actualizado de partners
        """
        data = {"selections": selections}
        return self._make_request("PUT", f"/api/pharmacy-partners/{pharmacy_id}/selection", json=data)

    # === MÉTODOS DE CONFIGURACIÓN DE FARMACIA ===

    def get_pharmacy_data(self, pharmacy_id: str) -> ApiResponse:
        """
        Obtiene datos completos de una farmacia

        Args:
            pharmacy_id: ID de la farmacia

        Returns:
            ApiResponse con datos de la farmacia
        """
        return self._make_request("GET", f"/api/v1/pharmacy/{pharmacy_id}")

    def update_pharmacy_data(self, pharmacy_id: str, data: Dict[str, Any]) -> ApiResponse:
        """
        Actualiza datos de una farmacia

        Args:
            pharmacy_id: ID de la farmacia
            data: Datos a actualizar (name, code, address, phone, erp_type, etc.)

        Returns:
            ApiResponse con datos actualizados
        """
        return self._make_request("PUT", f"/api/v1/pharmacy/{pharmacy_id}", json=data)

    def get_sales_date_range(self, pharmacy_id: str) -> ApiResponse:
        """
        Obtener rango de fechas de ventas disponibles

        Args:
            pharmacy_id: ID de la farmacia

        Returns:
            ApiResponse con min_date y max_date
        """
        return self._make_request("GET", f"/api/sales/date-range/{pharmacy_id}")

    # === MÉTODOS DE RE-ENRIQUECIMIENTO ===

    def check_reenrichment_status(self, pharmacy_id: str) -> ApiResponse:
        """
        Verificar qué datos necesitan re-enriquecimiento para una farmacia

        Args:
            pharmacy_id: ID de la farmacia

        Returns:
            ApiResponse con estado de re-enriquecimiento y recomendaciones
        """
        return self._make_request("GET", f"/api/v1/reenrichment/status/{pharmacy_id}")

    def execute_reenrichment(self, pharmacy_id: str) -> ApiResponse:
        """
        Ejecutar re-enriquecimiento para una farmacia específica

        Args:
            pharmacy_id: ID de la farmacia

        Returns:
            ApiResponse con confirmación del proceso de re-enriquecimiento
        """
        return self._make_request("POST", f"/api/v1/reenrichment/execute/{pharmacy_id}")

    def get_recent_catalog_changes(self, since_hours: int = 24) -> ApiResponse:
        """
        Mostrar productos del catálogo actualizados recientemente

        Args:
            since_hours: Horas hacia atrás para mostrar cambios (por defecto 24)

        Returns:
            ApiResponse con productos actualizados recientemente
        """
        return self._make_request("GET", f"/api/v1/reenrichment/catalog-changes?since_hours={since_hours}")

    def reenrich_failed_records(self, pharmacy_id: str, limit: int = 100) -> ApiResponse:
        """
        Re-enriquecer registros que fallaron previamente

        Args:
            pharmacy_id: ID de la farmacia
            limit: Número máximo de registros a procesar

        Returns:
            ApiResponse con confirmación del proceso
        """
        return self._make_request("POST", f"/api/v1/reenrichment/failed-records/{pharmacy_id}?limit={limit}")

    # === MÉTODOS DE ADMINISTRACIÓN PELIGROSOS ===

    def delete_all_pharmacy_data(self, pharmacy_id: Optional[str] = None) -> ApiResponse:
        """
        Borrar TODOS los datos de ventas de una farmacia de forma permanente.

        ADVERTENCIA: Esta operación es IRREVERSIBLE y elimina:
        - Todos los archivos de ventas (file_uploads)
        - Todos los datos de ventas procesados (sales_data)
        - Todos los datos de enriquecimiento (sales_enrichment)

        Se mantienen:
        - Configuración de la farmacia
        - Usuarios y configuraciones de la aplicación
        - Configuraciones de partners

        Args:
            pharmacy_id: ID de la farmacia (opcional, por defecto farmacia actual)

        Returns:
            ApiResponse con confirmación de borrado o error
        """
        params = {}
        if pharmacy_id:
            params["pharmacy_id"] = pharmacy_id

        return self._make_request("DELETE", "/api/v1/admin/delete-all-data", params=params)


# === CLASES DE CONVENIENCIA ===


class APIClient:
    """Wrapper simplificado del cliente de backend para uso en componentes Dash"""

    def __init__(self):
        self.backend_client = BackendClient()
        self._default_pharmacy_id = None  # Se carga dinámicamente

    def get_default_pharmacy_id(self) -> str:
        """
        Obtener pharmacy_id por defecto dinámicamente.

        ⚠️ DEPRECADO: Usar get_current_pharmacy_id() from utils.pharmacy_context

        Raises:
            ValueError: Si no se puede obtener pharmacy_id del backend
        """
        if self._default_pharmacy_id is None:
            try:
                response = self.backend_client._make_request("GET", "/api/sales/pharmacies")
                if response.success and response.data:
                    self._default_pharmacy_id = response.data.get("default_pharmacy_id")
                if not self._default_pharmacy_id:
                    # NO hay fallback - falla explícitamente
                    raise ValueError(
                        "Unable to get default pharmacy ID from backend. "
                        "Backend returned empty response. "
                        "Ensure backend is running: .\\scripts\\dev.ps1"
                    )
            except Exception as e:
                if isinstance(e, ValueError):
                    raise  # Re-raise ValueError as-is
                raise ValueError(
                    f"Error getting default pharmacy ID from backend: {str(e)}. "
                    f"Ensure backend is running: .\\scripts\\dev.ps1"
                )
        return self._default_pharmacy_id

    def get_available_pharmacies(self) -> Dict[str, Any]:
        """Obtener farmacias disponibles con datos"""
        return self.get("/api/sales/pharmacies")

    def _prepare_list_params(self, params: Optional[Dict] = None) -> Dict[str, Any]:
        """
        Preparar parámetros manejando listas de forma correcta para FastAPI.

        FastAPI espera múltiples parámetros con el mismo nombre para List[str]:
        ?partner_codes=111&partner_codes=426&partner_codes=644

        Args:
            params: Parámetros originales que pueden contener listas

        Returns:
            Parámetros preparados para requests
        """
        if not params:
            return {}

        prepared_params = []
        for key, value in params.items():
            if isinstance(value, list):
                # Para listas, crear múltiples parámetros con el mismo nombre
                for item in value:
                    prepared_params.append((key, str(item)))
            else:
                prepared_params.append((key, value))

        return prepared_params

    def get(self, endpoint: str, params: Optional[Dict] = None) -> Dict[str, Any]:
        """
        GET request simplificado que retorna directamente los datos

        Args:
            endpoint: Endpoint de la API (ej: '/api/sales/summary')
            params: Parámetros opcionales

        Returns:
            Dict con los datos de respuesta o diccionario vacío si hay error
        """
        try:
            # Preparar parámetros manejando listas correctamente
            prepared_params = self._prepare_list_params(params)
            response = self.backend_client._make_request("GET", endpoint, params=prepared_params)
            if response.success and response.data is not None:
                return response.data
            else:
                logger.warning(
                    f"API {endpoint} retornó datos vacíos o error: {response.message if response else 'No response'}"
                )
                return {}
        except Exception as e:
            logger.error(f"Error en GET {endpoint}: {str(e)}")
            return {}

    def post(self, endpoint: str, data: Optional[Dict] = None) -> Dict[str, Any]:
        """POST request simplificado"""
        try:
            response = self.backend_client._make_request("POST", endpoint, json=data or {})
            if response.success and response.data is not None:
                return response.data
            else:
                logger.warning(
                    f"API POST {endpoint} retornó datos vacíos o error: {response.message if response else 'No response'}"
                )
                return {}
        except Exception as e:
            logger.error(f"Error en POST {endpoint}: {str(e)}")
            return {}

    def put(self, endpoint: str, json: Optional[Dict] = None) -> Dict[str, Any]:
        """PUT request simplificado"""
        try:
            response = self.backend_client._make_request("PUT", endpoint, json=json or {})
            if response.success and response.data is not None:
                return response.data
            else:
                logger.warning(
                    f"API PUT {endpoint} retornó datos vacíos o error: {response.message if response else 'No response'}"
                )
                return {}
        except Exception as e:
            logger.error(f"Error en PUT {endpoint}: {str(e)}")
            return {}


# Instancia global del cliente
backend_client = BackendClient()
api_client = APIClient()  # Cliente simplificado para componentes Dash
# Force reload - 2025-09-24 20:09
