"""
Script de prueba para validar el sistema de diseño xFarma.
Este script verifica que todos los tokens y componentes funcionen correctamente.
"""

from components.base import BaseButton, BaseCard, Title
from dash import html
from styles.design_tokens import BORDER_RADIUS, COLORS, SHADOWS, SPACING, TYPOGRAPHY
from utils.style_helpers import color, shadow, spacing, text_style


def test_design_tokens():
    """Verificar que los design tokens se importen correctamente."""
    print("\n=== Test 1: Design Tokens ===")
    print(f"[OK] Colores: {len(COLORS)} tokens disponibles")
    print(f"  - Primary: {COLORS['primary']}")
    print(f"  - Success: {COLORS['success']}")
    print(f"[OK] Spacing: {len(SPACING)} tokens disponibles")
    print(f"  - Small: {SPACING['s']}")
    print(f"  - Medium: {SPACING['m']}")
    print(f"[OK] Tipografia: {len(TYPOGRAPHY)} variantes disponibles")
    print(f"  - h1 size: {TYPOGRAPHY['h1']['size']}")
    print(f"  - body size: {TYPOGRAPHY['body']['size']}")
    print(f"[OK] Sombras: {len(SHADOWS)} tamanos disponibles")
    print(f"[OK] Border Radius: {len(BORDER_RADIUS)} tamanos disponibles")


def test_components():
    """Verificar que los componentes base se creen correctamente."""
    print("\n=== Test 2: Componentes Base ===")

    # Test BaseCard
    card = BaseCard(children=[html.P("Test card content")], variant="default", shadow="sm")
    print("[OK] BaseCard - default variant creada correctamente")

    card_highlighted = BaseCard(children=[html.P("Highlighted card")], variant="highlighted", shadow="lg")
    print("[OK] BaseCard - highlighted variant creada correctamente")

    # Test BaseCard con props ARIA (fix CRITICAL)
    card_aria = BaseCard(children=[html.P("Card con accesibilidad")], role="region", aria_label="Resumen de ventas")
    # Verificar que retorna html.Div wrapper cuando hay props ARIA
    assert card_aria.__class__.__name__ == "Div", "BaseCard con ARIA debe retornar html.Div wrapper"
    print("[OK] BaseCard - props ARIA con wrapper html.Div (CRITICAL fix verificado)")

    # Test BaseCard sin props ARIA
    card_no_aria = BaseCard(children=[html.P("Card sin ARIA")])
    # Verificar que retorna dbc.Card directamente cuando no hay props ARIA
    assert card_no_aria.__class__.__name__ == "Card", "BaseCard sin ARIA debe retornar dbc.Card directo"
    print("[OK] BaseCard - sin ARIA retorna dbc.Card directo")

    # Test BaseButton
    btn_primary = BaseButton("Guardar", variant="primary", size="md")
    print("[OK] BaseButton - primary variant creada correctamente")

    btn_danger = BaseButton("Eliminar", variant="danger", size="sm", outline=True)
    print("[OK] BaseButton - danger outline creada correctamente")

    # Test Title
    title_h1 = Title("Titulo Principal", level=1)
    print("[OK] Title - level 1 (h1) creada correctamente")

    title_h3 = Title("Subtitulo", level=3, color="primary")
    print("[OK] Title - level 3 (h3) con color creada correctamente")


def test_style_helpers():
    """Verificar que los helpers de estilo funcionen correctamente."""
    print("\n=== Test 3: Style Helpers ===")

    # Test spacing
    uniform_spacing = spacing(m=True)
    print(f"[OK] spacing(m=True) = {uniform_spacing}")

    custom_spacing = spacing(top="l", bottom="l")
    print(f"[OK] spacing(top='l', bottom='l') = {custom_spacing}")

    # Test spacing con valores CSS válidos (fix HIGH)
    custom_px = spacing(top="20px", left="1.5rem")
    print(f"[OK] spacing(top='20px', left='1.5rem') = {custom_px} (valores CSS custom)")

    # Test spacing con valor inválido (debe generar warning y usar fallback)
    import warnings

    with warnings.catch_warnings(record=True) as w:
        warnings.simplefilter("always")
        invalid_spacing = spacing(top="invalid_value")
        assert len(w) == 1, "Debe generar warning para valor inválido"
        assert "Invalid spacing value" in str(w[0].message)
        print(
            f"[OK] spacing(top='invalid_value') = {invalid_spacing} - Warning generado correctamente (HIGH fix verificado)"
        )

    # Test color
    primary_color = color("primary")
    print(f"[OK] color('primary') = {primary_color}")

    # Test shadow
    medium_shadow = shadow("md")
    print(f"[OK] shadow('md') = {medium_shadow[:30]}...")

    # Test text_style
    h3_style = text_style("h3")
    print(f"[OK] text_style('h3') = fontSize: {h3_style['fontSize']}, weight: {h3_style['fontWeight']}")


def test_integration():
    """Verificar que los componentes funcionen juntos."""
    print("\n=== Test 4: Integracion ===")

    # Crear una card completa con título y botón
    integrated_card = BaseCard(
        children=[
            Title("Ventas Totales", level=4, margin_bottom=SPACING["s"]),
            html.H2("EUR 25,450", style={"color": COLORS["success"], "marginBottom": SPACING["s"]}),
            html.P("^ 12.5% vs mes anterior", style={"fontSize": "0.875rem", "marginBottom": SPACING["m"]}),
            BaseButton("Ver Detalles", variant="primary", size="sm"),
        ],
        variant="highlighted",
        shadow="md",
    )
    print("[OK] Card completa con Title + contenido + Button creada correctamente")

    # Usar helpers con componentes
    styled_element = html.Div(
        "Elemento con helpers",
        style={
            "padding": spacing(m=True),
            "color": color("text_primary"),
            "backgroundColor": color("bg_secondary"),
            "boxShadow": shadow("sm"),
            **text_style("body"),
        },
    )
    print("[OK] Elemento con style helpers aplicados correctamente")


def main():
    """Ejecutar todas las pruebas."""
    print("\n" + "=" * 50)
    print("SISTEMA DE DISENO xFarma - TEST DE VALIDACION")
    print("=" * 50)

    try:
        test_design_tokens()
        test_components()
        test_style_helpers()
        test_integration()

        print("\n" + "=" * 50)
        print("RESULTADO: TODOS LOS TESTS PASARON CORRECTAMENTE")
        print("=" * 50 + "\n")
        return True

    except Exception as e:
        print(f"\n[ERROR] ERROR EN TESTS: {e}")
        import traceback

        traceback.print_exc()
        return False


if __name__ == "__main__":
    success = main()
    exit(0 if success else 1)
