"""
kaiFarma Design System - Design Tokens

Tokens centralizados para mantener consistencia visual en toda la aplicación.
Estos valores deben ser la única fuente de verdad para propiedades de diseño.

Uso:
    from styles.design_tokens import COLORS, SPACING, TYPOGRAPHY

    style = {
        "color": COLORS["primary"],
        "padding": SPACING["m"],
        "fontSize": TYPOGRAPHY["h3"]["size"]
    }
"""

# ============================================================================
# COLORES (Alineado con context/style-guide.md)
# ============================================================================
#
# RATIOS DE CONTRASTE WCAG AA (mínimo 4.5:1 para texto normal, 3:1 para texto grande)
# Combinaciones verificadas:
# - text_primary (#343A40) sobre bg_primary (#FFFFFF): ~12.6:1 ✅ AAA
# - text_secondary (#6C757D) sobre bg_primary (#FFFFFF): ~5.9:1 ✅ AA
# - primary (#0066CC) sobre white: ~7.0:1 ✅ AAA
# - success (#28A745) sobre white: ~4.5:1 ✅ AA
# - info (#17A2B8) sobre white: ~4.6:1 ✅ AA
# - warning (#FFC107) sobre white: ~1.5:1 ❌ Solo para fondos, no texto
# - danger (#DC3545) sobre white: ~4.9:1 ✅ AA
#
# RECOMENDACIONES:
# - Para warning: usar sobre fondos oscuros o como fondo con texto oscuro
# - Para botones: texto blanco sobre colores semánticos (excepto warning)

COLORS = {
    # Colores primarios (kaiFarma Style Guide)
    "primary": "#0066CC",  # Primary Blue - CTAs y acciones principales
    "primary_light": "#3399FF",  # Variante más clara
    "primary_dark": "#004C99",  # Variante más oscura
    "secondary": "#6C757D",  # Professional Gray
    "secondary_light": "#ADB5BD",  # Gris claro
    "secondary_dark": "#495057",  # Gris oscuro
    # Colores semánticos
    "success": "#28A745",  # Pharmacy Green - uploads exitosos, sync OK, genéricos
    "success_light": "#71DD8A",  # Verde claro
    "success_dark": "#1E7E34",  # Verde oscuro
    "info": "#17A2B8",  # Teal - mensajes informativos, tips
    "info_light": "#63D1EB",  # Teal claro
    "info_dark": "#117A8B",  # Teal oscuro
    "warning": "#FFC107",  # Amber - estados pendientes, atención
    "warning_light": "#FFD54F",  # Amber claro
    "warning_dark": "#D39E00",  # Amber oscuro
    "danger": "#DC3545",  # Red - errores, sync fallido, crítico
    "danger_light": "#F1707A",  # Rojo claro
    "danger_dark": "#BD2130",  # Rojo oscuro
    # Colores neutrales (grises - Bootstrap standard)
    "white": "#FFFFFF",
    "gray_100": "#F8F9FA",  # Fondo página
    "gray_200": "#E9ECEF",  # Fondo secundario
    "gray_300": "#DEE2E6",  # Bordes
    "gray_400": "#CED4DA",  # Bordes hover
    "gray_500": "#ADB5BD",  # Texto muted
    "gray_600": "#6C757D",  # Texto secundario
    "gray_700": "#495057",  # Texto medio
    "gray_800": "#343A40",  # Texto principal
    "gray_900": "#212529",  # Texto muy oscuro
    "black": "#000000",
    # Colores de fondo
    "bg_primary": "#FFFFFF",  # Fondo principal (cards, inputs)
    "bg_secondary": "#F8F9FA",  # Fondo página
    "bg_tertiary": "#E9ECEF",  # Fondo terciario
    "bg_dark": "#343A40",  # Fondo oscuro (sidebar)
    # Colores de texto
    "text_primary": "#343A40",  # Texto principal (Dark Gray)
    "text_secondary": "#6C757D",  # Texto secundario (Professional Gray)
    "text_muted": "#ADB5BD",  # Texto deshabilitado
    "text_inverse": "#FFFFFF",  # Texto sobre fondos oscuros
    # Colores de borde
    "border_light": "#E9ECEF",
    "border_medium": "#DEE2E6",  # Border Gray del style-guide
    "border_dark": "#CED4DA",
    # =========================================================================
    # COLORES ESPECÍFICOS kaiFarma (Pharmacy-Specific Colors del style-guide)
    # =========================================================================
    "generic_medication": "#28A745",  # Verde - Medicamentos genéricos
    "brand_medication": "#0066CC",  # Azul - Medicamentos de marca
    "laboratory_partner": "#6F42C1",  # Púrpura - Datos de laboratorio
    "erp_source": "#FD7E14",  # Naranja - Indicadores de archivo ERP
}


# ============================================================================
# SPACING (Sistema base 4px)
# ============================================================================

SPACING = {
    "xs": "4px",  # Extra pequeño
    "s": "8px",  # Pequeño
    "m": "16px",  # Medio (base)
    "l": "24px",  # Grande
    "xl": "32px",  # Extra grande
    "2xl": "48px",  # 2x extra grande
    "3xl": "64px",  # 3x extra grande
}


# ============================================================================
# TIPOGRAFÍA (Alineado con context/style-guide.md)
# ============================================================================
#
# Font Family: Inter, system-ui, -apple-system, sans-serif
# Monospace: 'Courier New', Courier, monospace (para códigos nacionales)

TYPOGRAPHY = {
    # Encabezados (style-guide.md Font Scale)
    "h1": {
        "size": "2rem",  # 32px - Page titles
        "weight": "700",  # Bold
        "line_height": "1.2",
    },
    "h2": {
        "size": "1.5rem",  # 24px - Section headers
        "weight": "600",  # SemiBold
        "line_height": "1.2",
    },
    "h3": {
        "size": "1.25rem",  # 20px - Subsection headers
        "weight": "600",  # SemiBold
        "line_height": "1.2",
    },
    "h4": {
        "size": "1.125rem",  # 18px - Body Large / Important text
        "weight": "500",  # Medium
        "line_height": "1.3",
    },
    "h5": {
        "size": "1rem",  # 16px - Body emphasis
        "weight": "500",  # Medium
        "line_height": "1.4",
    },
    # Texto de cuerpo
    "body": {
        "size": "1rem",  # 16px - Default text
        "weight": "400",  # Regular
        "line_height": "1.5",
    },
    "body_large": {
        "size": "1.125rem",  # 18px - Important text, lead paragraphs
        "weight": "400",  # Regular
        "line_height": "1.5",
    },
    "body_small": {
        "size": "0.875rem",  # 14px - Small / Captions / Metadata
        "weight": "400",  # Regular
        "line_height": "1.5",
    },
    "caption": {
        "size": "0.75rem",  # 12px - Tiny / Labels / Tags
        "weight": "400",  # Regular
        "line_height": "1.4",
    },
}

# Font weights reference (para uso directo)
FONT_WEIGHTS = {
    "regular": "400",
    "medium": "500",
    "semibold": "600",
    "bold": "700",
}


# ============================================================================
# SOMBRAS
# ============================================================================

SHADOWS = {
    "none": "none",
    "sm": "0 0.125rem 0.25rem rgba(0, 0, 0, 0.075)",  # Sombra pequeña
    "md": "0 0.5rem 1rem rgba(0, 0, 0, 0.15)",  # Sombra media (bootstrap shadow)
    "lg": "0 1rem 3rem rgba(0, 0, 0, 0.175)",  # Sombra grande
    "xl": "0 2rem 4rem rgba(0, 0, 0, 0.2)",  # Sombra extra grande
}


# ============================================================================
# BORDER RADIUS
# ============================================================================

BORDER_RADIUS = {
    "none": "0",
    "sm": "0.25rem",  # 4px - bordes sutiles
    "md": "0.375rem",  # 6px - bootstrap default
    "lg": "0.5rem",  # 8px - bordes pronunciados
    "xl": "1rem",  # 16px - bordes muy redondeados
    "full": "9999px",  # Completamente redondeado (círculos, pills)
}


# ============================================================================
# Z-INDEX LAYERS (para gestión de profundidad)
# ============================================================================

Z_INDEX = {
    "base": 0,
    "dropdown": 1000,
    "sticky": 1020,
    "fixed": 1030,
    "modal_backdrop": 1040,
    "modal": 1050,
    "popover": 1060,
    "tooltip": 1070,
}


# ============================================================================
# TRANSITIONS (Alineado con context/style-guide.md)
# ============================================================================
#
# Easing: ease-in-out (standard), ease-out (entry), ease-in (exit)

TRANSITIONS = {
    "quick": "150ms ease-in-out",  # Hover states, small changes
    "standard": "250ms ease-in-out",  # Transitions, modal appearances
    "slow": "400ms ease-in-out",  # Page transitions, large movements
}

# Aliases para compatibilidad
TRANSITIONS["fast"] = TRANSITIONS["quick"]
TRANSITIONS["normal"] = TRANSITIONS["standard"]


# ============================================================================
# BREAKPOINTS (para diseño responsive consistente)
# ============================================================================

BREAKPOINTS = {
    "mobile": "375px",  # Móvil (viewport mínimo estándar)
    "tablet": "768px",  # Tablet (viewport medio)
    "desktop": "1440px",  # Desktop (viewport estándar)
}
