# Sistema de Diseño xFarma - Design Tokens

## 🎨 ¿Qué son los Design Tokens?

Los **design tokens** son la capa más básica de un sistema de diseño. Son valores nombrados que representan decisiones de diseño (colores, espaciado, tipografía, etc.) de forma centralizada y reutilizable.

**Beneficios:**
- ✅ **Consistencia visual** en toda la aplicación
- ✅ **Mantenibilidad** - cambiar un valor actualiza toda la app
- ✅ **Escalabilidad** - fácil agregar nuevos componentes
- ✅ **Colaboración** - lenguaje común entre diseño y desarrollo

---

## 📦 Fuente de Verdad

> ⚠️ **IMPORTANTE**: La fuente única de verdad para todos los valores es:
>
> **`frontend/styles/design_tokens.py`**
>
> Los valores mostrados en esta documentación son solo ejemplos de uso.
> Siempre consultar el archivo Python para valores actuales.

**Documentos relacionados:**
- `context/style-guide.md` - Especificación visual de diseño
- `context/design-principles.md` - Principios UX/UI

---

## 📦 Tokens Disponibles

### 1. Colores (`COLORS`)

Los colores están organizados en categorías semánticas:

```python
from styles.design_tokens import COLORS

# Colores principales de la marca
COLORS["primary"]        # Primary Blue - CTAs y acciones
COLORS["primary_light"]  # Variante más clara
COLORS["primary_dark"]   # Variante más oscura

COLORS["secondary"]       # Professional Gray
COLORS["secondary_light"] # Gris claro
COLORS["secondary_dark"]  # Gris oscuro

# Colores semánticos
COLORS["success"]  # Operaciones exitosas, genéricos
COLORS["info"]     # Información
COLORS["warning"]  # Advertencias
COLORS["danger"]   # Errores/peligro

# Colores neutrales (escala de grises)
COLORS["white"]     # Fondo cards
COLORS["gray_100"]  # Fondo muy claro
COLORS["gray_300"]  # Bordes
COLORS["gray_600"]  # Texto secundario
COLORS["gray_800"]  # Texto principal
COLORS["black"]

# Colores de fondo
COLORS["bg_primary"]   # Fondo principal (blanco)
COLORS["bg_secondary"] # Fondo secundario (gris claro)
COLORS["bg_dark"]      # Fondo oscuro (sidebar)

# Colores de texto
COLORS["text_primary"]   # Texto principal
COLORS["text_secondary"] # Texto secundario
COLORS["text_muted"]     # Texto deshabilitado

# Colores de borde
COLORS["border_light"]  # Borde claro
COLORS["border_medium"] # Borde medio
COLORS["border_dark"]   # Borde oscuro

# Colores específicos xFarma
COLORS["generic_medication"]  # Verde - Medicamentos genéricos
COLORS["brand_medication"]    # Azul - Medicamentos de marca
COLORS["laboratory_partner"]  # Púrpura - Datos de laboratorio
COLORS["erp_source"]          # Naranja - Indicadores ERP
```

**Uso:**
```python
from styles.design_tokens import COLORS

style = {
    "color": COLORS["primary"],
    "backgroundColor": COLORS["bg_secondary"],
    "borderColor": COLORS["border_light"]
}
```

---

### 2. Espaciado (`SPACING`)

Sistema de espaciado basado en múltiplos de **4px** para mantener consistencia visual.

> **Nota**: Las claves usan nombres cortos: `'s'`, `'m'`, `'l'` (no 'sm', 'md', 'lg')

```python
from styles.design_tokens import SPACING

SPACING["xs"]   # Extra pequeño (4px)
SPACING["s"]    # Pequeño (8px)
SPACING["m"]    # Medio - base (16px)
SPACING["l"]    # Grande (24px) - padding cards
SPACING["xl"]   # Extra grande (32px) - gaps secciones
SPACING["2xl"]  # 2x extra grande (48px)
SPACING["3xl"]  # 3x extra grande (64px)
```

**Cuándo usar cada tamaño:**
- `xs` (4px): Espaciado mínimo entre elementos muy relacionados
- `s` (8px): Espaciado entre elementos relacionados (ej: icono + texto)
- `m` (16px): Espaciado estándar (padding de cards, márgenes)
- `l` (24px): Padding interno de cards (OBLIGATORIO)
- `xl` (32px): Espaciado entre secciones (OBLIGATORIO)
- `2xl`/`3xl`: Espaciado de secciones completas

**Uso:**
```python
from styles.design_tokens import SPACING

# Padding uniforme
style = {"padding": SPACING["m"]}  # 16px en todos los lados

# Padding vertical y horizontal
style = {
    "paddingTop": SPACING["l"],
    "paddingBottom": SPACING["l"],
    "paddingLeft": SPACING["m"],
    "paddingRight": SPACING["m"]
}

# Margen inferior
style = {"marginBottom": SPACING["xl"]}
```

---

### 3. Tipografía (`TYPOGRAPHY`)

Escala tipográfica consistente con tamaños, pesos y alturas de línea definidos.

```python
from styles.design_tokens import TYPOGRAPHY

# Encabezados
TYPOGRAPHY["h1"]  # Títulos principales (2rem/32px)
TYPOGRAPHY["h2"]  # Títulos de sección (1.5rem/24px)
TYPOGRAPHY["h3"]  # Subtítulos importantes (1.25rem/20px)
TYPOGRAPHY["h4"]  # Subtítulos secundarios (1.125rem/18px)
TYPOGRAPHY["h5"]  # Títulos terciarios (1rem/16px)

# Texto de cuerpo
TYPOGRAPHY["body"]       # Texto estándar (1rem/16px)
TYPOGRAPHY["body_large"] # Texto destacado (1.125rem/18px)
TYPOGRAPHY["body_small"] # Texto pequeño (0.875rem/14px)
TYPOGRAPHY["caption"]    # Texto muy pequeño (0.75rem/12px)
```

Cada variante incluye:
- `size`: Tamaño de fuente en rem
- `weight`: Peso de la fuente (400-700)
- `line_height`: Altura de línea (1.2-1.5)

**Uso:**
```python
from styles.design_tokens import TYPOGRAPHY

# Aplicar estilo h3
style = {
    "fontSize": TYPOGRAPHY["h3"]["size"],
    "fontWeight": TYPOGRAPHY["h3"]["weight"],
    "lineHeight": TYPOGRAPHY["h3"]["line_height"]
}
```

---

### 4. Sombras (`SHADOWS`)

Sombras predefinidas para crear profundidad visual consistente.

```python
from styles.design_tokens import SHADOWS

SHADOWS["none"] # Sin sombra
SHADOWS["sm"]   # Sombra pequeña (cards, botones)
SHADOWS["md"]   # Sombra media (modales, dropdowns)
SHADOWS["lg"]   # Sombra grande (elementos flotantes)
SHADOWS["xl"]   # Sombra extra grande (énfasis máximo)
```

**Uso:**
```python
from styles.design_tokens import SHADOWS

# Card con sombra pequeña
style = {"boxShadow": SHADOWS["sm"]}

# Modal con sombra grande
style = {"boxShadow": SHADOWS["lg"]}
```

---

### 5. Border Radius (`BORDER_RADIUS`)

Radios de borde consistentes para esquinas redondeadas.

```python
from styles.design_tokens import BORDER_RADIUS

BORDER_RADIUS["none"] # Sin redondeo
BORDER_RADIUS["sm"]   # Bordes sutiles (4px)
BORDER_RADIUS["md"]   # Bootstrap default (6px)
BORDER_RADIUS["lg"]   # Bordes pronunciados (8px)
BORDER_RADIUS["xl"]   # Muy redondeado (16px)
BORDER_RADIUS["full"] # Círculos/pills (9999px)
```

**Uso:**
```python
from styles.design_tokens import BORDER_RADIUS

# Card con bordes redondeados
style = {"borderRadius": BORDER_RADIUS["md"]}

# Botón pill (completamente redondeado)
style = {"borderRadius": BORDER_RADIUS["full"]}
```

---

### 6. Transiciones (`TRANSITIONS`)

Duraciones y easing para animaciones consistentes.

```python
from styles.design_tokens import TRANSITIONS

TRANSITIONS["quick"]    # 150ms - Hover states
TRANSITIONS["standard"] # 250ms - Transiciones generales
TRANSITIONS["slow"]     # 400ms - Movimientos grandes
```

---

### 7. Breakpoints (`BREAKPOINTS`)

Puntos de quiebre para diseño responsive.

```python
from styles.design_tokens import BREAKPOINTS

BREAKPOINTS["mobile"]  # 375px - Viewport mínimo
BREAKPOINTS["tablet"]  # 768px - Viewport medio (mínimo soportado)
BREAKPOINTS["desktop"] # 1440px - Viewport estándar
```

---

## 📋 Principios del Sistema de Diseño

### 1. **Consistencia sobre Personalización**
Siempre usa los tokens antes de valores custom.

```python
# ✅ CORRECTO
style = {"padding": SPACING["m"]}

# ❌ INCORRECTO
style = {"padding": "15px"}  # Valor arbitrario
```

### 2. **Semántica sobre Valores Literales**
Usa nombres semánticos de colores.

```python
# ✅ CORRECTO
style = {"color": COLORS["success"]}

# ❌ INCORRECTO
style = {"color": "#28A745"}  # Valor hardcoded
```

### 3. **Composición de Estilos**
Combina tokens para crear estilos complejos.

```python
from styles.design_tokens import COLORS, SPACING, SHADOWS, BORDER_RADIUS

style = {
    "padding": SPACING["m"],
    "backgroundColor": COLORS["bg_primary"],
    "borderRadius": BORDER_RADIUS["md"],
    "boxShadow": SHADOWS["sm"],
    "color": COLORS["text_primary"]
}
```

---

## 🎯 Ejemplos de Uso Común

### Ejemplo 1: Card con Header y Content
```python
from dash import html
import dash_bootstrap_components as dbc
from styles.design_tokens import COLORS, SPACING, SHADOWS, BORDER_RADIUS

card = dbc.Card(
    [
        dbc.CardHeader(
            "Título de la Card",
            style={
                "backgroundColor": COLORS["primary"],
                "color": COLORS["text_inverse"],
                "padding": SPACING["m"],
                "fontWeight": "500"
            }
        ),
        dbc.CardBody(
            [
                html.P("Contenido de la card"),
                html.Button("Acción", style={"marginTop": SPACING["s"]})
            ],
            style={"padding": SPACING["m"]}
        )
    ],
    style={
        "borderRadius": BORDER_RADIUS["md"],
        "boxShadow": SHADOWS["sm"],
        "marginBottom": SPACING["l"]
    }
)
```

### Ejemplo 2: Lista con Items Separados
```python
from dash import html
from styles.design_tokens import COLORS, SPACING, BORDER_RADIUS

items = html.Div([
    html.Div(
        f"Item {i}",
        style={
            "padding": SPACING["m"],
            "marginBottom": SPACING["s"],
            "backgroundColor": COLORS["bg_tertiary"],
            "borderRadius": BORDER_RADIUS["md"],
            "borderLeft": f"4px solid {COLORS['primary']}"
        }
    )
    for i in range(1, 4)
])
```

### Ejemplo 3: Botones con Diferentes Variantes
```python
from dash import html
import dash_bootstrap_components as dbc
from styles.design_tokens import SPACING

buttons = html.Div([
    dbc.Button("Primario", color="primary", style={"marginRight": SPACING["s"]}),
    dbc.Button("Secundario", color="secondary", style={"marginRight": SPACING["s"]}),
    dbc.Button("Peligro", color="danger", outline=True)
], style={"display": "flex", "gap": SPACING["s"]})
```

---

## 🚀 Migración de Código Existente

### Checklist de Migración por Componente

Al migrar un componente, verificar:

- [ ] **Colores**: Reemplazar hex values con `COLORS["*"]`
- [ ] **Spacing**: Reemplazar px values con `SPACING["*"]`
- [ ] **Tipografía**: Usar `TYPOGRAPHY["*"]` para tamaños de fuente
- [ ] **Sombras**: Reemplazar box-shadow con `SHADOWS["*"]`
- [ ] **Botones**: Reemplazar `dbc.Button` con `BaseButton`
- [ ] **Cards**: Reemplazar `dbc.Card` con `BaseCard`
- [ ] **Títulos**: Reemplazar `html.H1-H5` con `Title`
- [ ] **Tests**: Verificar que tests siguen pasando

### Herramientas de Migración

```powershell
# Buscar colores hardcoded (hex values)
Get-ChildItem -Recurse -Filter *.py | Select-String -Pattern "#[0-9A-Fa-f]{6}"

# Buscar spacing hardcoded (px values)
Get-ChildItem -Recurse -Filter *.py | Select-String -Pattern '"\d+px"'

# Buscar componentes candidatos para BaseCard
Get-ChildItem -Recurse -Filter *.py | Select-String -Pattern "dbc\.Card"
```

---

## 📚 Recursos Adicionales

- **Componentes Base**: Ver `components/base/README.md`
- **Design Tokens Completos**: Ver `styles/design_tokens.py`
- **Style Guide**: Ver `context/style-guide.md`
- **Design Principles**: Ver `context/design-principles.md`

---

## 🤝 Contribuir

Al agregar nuevos tokens:
1. **Verifica** que no exista un token similar
2. **Usa nombres semánticos** (no literales)
3. **Documenta** el propósito y casos de uso
4. **Mantén consistencia** con el sistema existente
5. **Actualiza** `context/style-guide.md` si es necesario

---

**xFarma Design System** - Construyendo interfaces consistentes y escalables
