# frontend/stores/alerts_store.py
"""
Global Alerts Store - Agregador centralizado de alertas (Pivot 2026)

Arquitectura "Store tonto, Componente tonto":
- Cada modulo (encargos, recetas, inventario) publica su estado aqui
- El componente ProactiveAlerts solo lee y pinta, sin logica de negocio
- Permite que nuevos modulos se integren sin tocar el componente de alertas

Estructura del Store:
{
    "summary": {
        "critical_count": 3,
        "warning_count": 5,
        "info_count": 1,
        "last_update": "2026-01-10T15:30:00Z"
    },
    "modules": {
        "encargos": {
            "status": "critical",  # critical | warning | info | ok
            "count": 1129,
            "value": 19453.22,
            "msg": "1,129 encargos > 15 dias",
            "action_label": "Gestionar",
            "action_link": "/encargos",
            "icon": "mdi:clipboard-text-clock"
        },
        ...
    },
    "system": {
        "erp": {
            "status": "ok",
            "provider": "farmanager",
            "last_sync": "2026-01-10T15:18:00Z",
            "minutes_since_sync": 12,
            "msg": "Conectado - Sync hace 12 min"
        },
        "cima": {...},
        "license": {...}
    }
}

Usage:
    # En app.py (skeleton)
    dcc.Store(id='global-alerts-store', data=get_empty_alerts_state())

    # En callback de agregacion
    @app.callback(
        Output('global-alerts-store', 'data'),
        [Input('encargos-summary-store', 'data'),
         Input('recetas-summary-store', 'data'),
         ...]
    )
    def aggregate_alerts(encargos, recetas, ...):
        return build_alerts_state(encargos, recetas, ...)

See: docs/PROPUESTA_DASHBOARD_PIVOT_2026.md
"""

from datetime import datetime, timezone
from typing import Any, Dict, Literal, Optional, TypedDict


# =============================================================================
# Type Definitions (usando TypedDict para compatibilidad con frontend)
# =============================================================================

AlertStatus = Literal["critical", "warning", "info", "ok"]


# ModuleAlert: Split into required and optional fields for type safety
class _ModuleAlertRequired(TypedDict):
    """Required fields for module alerts."""

    status: AlertStatus
    msg: str
    action_link: str


class ModuleAlert(_ModuleAlertRequired, total=False):
    """Estado de alerta de un modulo individual.

    Required: status, msg, action_link
    Optional: count, value, action_label, icon
    """

    count: int
    value: Optional[float]  # Valor economico si aplica (EUR)
    action_label: str
    icon: str


# SystemStatus: Split into required and optional fields
class _SystemStatusRequired(TypedDict):
    """Required fields for system status."""

    status: AlertStatus
    msg: str


class SystemStatus(_SystemStatusRequired, total=False):
    """Estado de un componente del sistema.

    Required: status, msg
    Optional: provider, last_sync, minutes_since_sync
    """

    provider: Optional[str]  # ej: "farmanager", "farmatic"
    last_sync: Optional[str]  # ISO timestamp
    minutes_since_sync: Optional[int]


# AlertsSummary: All fields required
class AlertsSummary(TypedDict):
    """Resumen agregado de todas las alertas."""

    critical_count: int
    warning_count: int
    info_count: int
    last_update: str


# AlertsState: All fields required
class AlertsState(TypedDict):
    """Estado completo del store de alertas."""

    summary: AlertsSummary
    modules: Dict[str, ModuleAlert]
    system: Dict[str, SystemStatus]


# =============================================================================
# Factory Functions
# =============================================================================


def get_empty_alerts_state() -> Dict[str, Any]:
    """
    Retorna estado inicial vacio para el store.

    Usar en app.py:
        dcc.Store(id='global-alerts-store', data=get_empty_alerts_state())
    """
    return {
        "summary": {
            "critical_count": 0,
            "warning_count": 0,
            "info_count": 0,
            "last_update": datetime.now(timezone.utc).isoformat(),
        },
        "modules": {},
        "system": {},
    }


def build_module_alert(
    status: AlertStatus,
    count: int,
    msg: str,
    action_link: str,
    icon: str = "mdi:alert-circle",
    action_label: str = "Ver",
    value: Optional[float] = None,
) -> Dict[str, Any]:
    """
    Construye un objeto de alerta para un modulo.

    Args:
        status: Nivel de severidad (critical, warning, info, ok)
        count: Numero de items que requieren atencion
        msg: Mensaje descriptivo para el usuario
        action_link: URL de navegacion al gestionar
        icon: Icono dash-iconify
        action_label: Texto del boton de accion
        value: Valor economico en EUR (opcional)

    Returns:
        Dict compatible con ModuleAlert

    Example:
        encargos_alert = build_module_alert(
            status="critical",
            count=1129,
            msg="1,129 encargos > 15 dias",
            action_link="/encargos",
            icon="mdi:clipboard-text-clock",
            action_label="Gestionar",
            value=19453.22
        )
    """
    return {
        "status": status,
        "count": count,
        "value": value,
        "msg": msg,
        "action_label": action_label,
        "action_link": action_link,
        "icon": icon,
    }


def build_system_status(
    status: AlertStatus,
    msg: str,
    provider: Optional[str] = None,
    last_sync: Optional[str] = None,
) -> Dict[str, Any]:
    """
    Construye un objeto de estado para un componente del sistema.

    Args:
        status: Estado actual (ok, warning, critical)
        msg: Mensaje descriptivo
        provider: Nombre del proveedor (ej: farmanager)
        last_sync: Timestamp ISO de ultima sincronizacion

    Returns:
        Dict compatible con SystemStatus
    """
    minutes_since = None
    if last_sync:
        try:
            sync_dt = datetime.fromisoformat(last_sync.replace("Z", "+00:00"))
            now = datetime.now(timezone.utc)
            minutes_since = int((now - sync_dt).total_seconds() / 60)
        except (ValueError, TypeError):
            pass

    return {
        "status": status,
        "provider": provider,
        "last_sync": last_sync,
        "minutes_since_sync": minutes_since,
        "msg": msg,
    }


# =============================================================================
# Aggregation Logic
# =============================================================================


def aggregate_alerts(
    modules: Dict[str, Dict[str, Any]],
    system: Dict[str, Dict[str, Any]],
) -> Dict[str, Any]:
    """
    Agrega alertas de todos los modulos en un estado unificado.

    Esta funcion se llama desde el callback de agregacion para
    construir el estado completo del store.

    Args:
        modules: Dict de alertas por modulo (encargos, recetas, etc.)
        system: Dict de estados del sistema (erp, cima, license)

    Returns:
        Dict compatible con AlertsState

    Example:
        @app.callback(
            Output('global-alerts-store', 'data'),
            [Input('encargos-widget-summary', 'data'),
             Input('erp-status-store', 'data')]
        )
        def update_alerts_store(encargos_data, erp_data):
            modules = {}
            system = {}

            if encargos_data:
                modules['encargos'] = build_encargos_alert(encargos_data)

            if erp_data:
                system['erp'] = build_erp_status(erp_data)

            return aggregate_alerts(modules, system)
    """
    # Contar por severidad
    critical = 0
    warning = 0
    info = 0

    for module_data in modules.values():
        status = module_data.get("status", "ok")
        if status == "critical":
            critical += 1
        elif status == "warning":
            warning += 1
        elif status == "info":
            info += 1

    # Incluir sistema en el conteo
    for sys_data in system.values():
        status = sys_data.get("status", "ok")
        if status == "critical":
            critical += 1
        elif status == "warning":
            warning += 1

    summary = {
        "critical_count": critical,
        "warning_count": warning,
        "info_count": info,
        "last_update": datetime.now(timezone.utc).isoformat(),
    }

    return {
        "summary": summary,
        "modules": modules,
        "system": system,
    }


# =============================================================================
# Module-Specific Builders
# =============================================================================


def build_encargos_alert(summary_data: Dict[str, Any]) -> Dict[str, Any]:
    """
    Construye alerta de encargos desde datos del API /customer-orders/summary.

    Args:
        summary_data: Respuesta del endpoint de summary

    Returns:
        Dict compatible con ModuleAlert
    """
    if not summary_data:
        return build_module_alert(
            status="ok",
            count=0,
            msg="Sin encargos pendientes",
            action_link="/encargos",
            icon="mdi:clipboard-text-clock",
        )

    total = summary_data.get("total_pending", 0)
    total_value = summary_data.get("total_value", 0)
    by_severity = summary_data.get("by_severity", {})

    critical_count = by_severity.get("critical", {}).get("count", 0)
    critical_value = by_severity.get("critical", {}).get("value", 0)

    # Determinar severidad
    if critical_count > 0:
        status: AlertStatus = "critical"
        msg = f"{critical_count:,} encargos > 15 dias ({critical_value:,.0f} EUR)".replace(",", ".")
    elif total > 0:
        status = "warning"
        msg = f"{total:,} encargos pendientes".replace(",", ".")
    else:
        status = "ok"
        msg = "Sin encargos pendientes"

    return build_module_alert(
        status=status,
        count=critical_count if critical_count > 0 else total,
        value=critical_value if critical_count > 0 else total_value,
        msg=msg,
        action_link="/encargos",
        icon="mdi:clipboard-text-clock",
        action_label="Gestionar",
    )


def build_erp_status(erp_data: Dict[str, Any]) -> Dict[str, Any]:
    """
    Construye estado del ERP desde datos de sync.

    Args:
        erp_data: Datos del estado de sincronizacion ERP

    Returns:
        Dict compatible con SystemStatus
    """
    if not erp_data:
        return build_system_status(
            status="warning",
            msg="Sin conexion ERP",
        )

    provider = erp_data.get("provider", "desconocido")
    last_sync = erp_data.get("last_sync")
    connected = erp_data.get("connected", False)

    if not connected:
        return build_system_status(
            status="critical",
            msg=f"ERP {provider} desconectado",
            provider=provider,
        )

    # Calcular frescura
    status: AlertStatus = "ok"
    msg = "Conectado"

    if last_sync:
        try:
            sync_dt = datetime.fromisoformat(last_sync.replace("Z", "+00:00"))
            now = datetime.now(timezone.utc)
            minutes = int((now - sync_dt).total_seconds() / 60)

            if minutes > ERP_SYNC_CRITICAL_MINUTES:  # >24h
                status = "critical"
                msg = f"Ultima sync hace {minutes // 60}h - Datos obsoletos"
            elif minutes > ERP_SYNC_WARNING_MINUTES:  # >1h
                status = "warning"
                msg = f"Ultima sync hace {minutes}min - Revisar"
            else:
                status = "ok"
                msg = f"Conectado - Sync hace {minutes}min"
        except (ValueError, TypeError):
            msg = "Conectado"
    else:
        msg = "Conectado - Sin sync reciente"

    return build_system_status(
        status=status,
        msg=msg,
        provider=provider,
        last_sync=last_sync,
    )


# =============================================================================
# Threshold Constants (Configurable)
# =============================================================================

# Encargos: Umbrales de severidad
ENCARGOS_CRITICAL_DAYS = 15  # Dias para considerar critico
ENCARGOS_WARNING_DAYS = 7  # Dias para considerar warning

# ERP Sync: Umbrales de frescura (minutos)
ERP_SYNC_OK_MINUTES = 30  # Verde: Datos frescos
ERP_SYNC_WARNING_MINUTES = 60  # Ambar: Revisar
ERP_SYNC_CRITICAL_MINUTES = 1440  # Rojo: 24h - Datos obsoletos

# CIMA Sync: Umbrales (dias)
CIMA_SYNC_OK_DAYS = 7
CIMA_SYNC_WARNING_DAYS = 14
CIMA_SYNC_CRITICAL_DAYS = 30
