#!/usr/bin/env python3
"""
Script para corregir formato de números en todo el frontend
Convierte de formato americano (:,) a formato español usando helpers
"""

import re
from pathlib import Path


def fix_number_formatting(file_path: str) -> int:
    """
    Corregir formato de números en un archivo
    Returns: número de cambios realizados
    """
    changes = 0

    with open(file_path, "r", encoding="utf-8") as f:
        content = f.read()

    original_content = content

    # Patrones de reemplazo
    patterns = [
        # :,.0f}€ -> format_currency(value, decimals=0)
        (r'f"[^"]*\{([^}:]+):,\.0f\}€[^"]*"', lambda m: replace_currency_format(m, 0)),
        (r"f'[^']*\{([^}:]+):,\.0f\}€[^']*'", lambda m: replace_currency_format(m, 0)),
        # :,.2f}€ -> format_currency(value, decimals=2)
        (r'f"[^"]*\{([^}:]+):,\.2f\}€[^"]*"', lambda m: replace_currency_format(m, 2)),
        (r"f'[^']*\{([^}:]+):,\.2f\}€[^']*'", lambda m: replace_currency_format(m, 2)),
        # :,} -> format_number(value)
        (r'f"[^"]*\{([^}:]+):,\}[^"]*"', lambda m: replace_number_format(m)),
        (r"f'[^']*\{([^}:]+):,\}[^']*'", lambda m: replace_number_format(m)),
        # :.1f}% -> replace . with ,
        (
            r'(f"[^"]*\{[^}:]+:\.1f\}%[^"]*")',
            lambda m: (
                m.group(1).replace("f}%", 'f}%".replace(".", ",")') if not ".replace(" in m.group(1) else m.group(1)
            ),
        ),
        (
            r"(f'[^']*\{[^}:]+:\.1f\}%[^']*')",
            lambda m: (
                m.group(1).replace("f}%", "f}%'.replace('.', ',')") if not ".replace(" in m.group(1) else m.group(1)
            ),
        ),
    ]

    for pattern, replacement in patterns:
        new_content, count = re.subn(pattern, replacement, content)
        content = new_content
        changes += count

    # Solo escribir si hay cambios
    if content != original_content:
        # Verificar que tiene los imports necesarios
        if not has_helper_imports(content):
            content = add_helper_imports(content)

        with open(file_path, "w", encoding="utf-8") as f:
            f.write(content)

    return changes


def replace_currency_format(match, decimals):
    """Reemplazar formato de moneda"""
    var_name = match.group(1).strip()
    return f"format_currency({var_name}, decimals={decimals})"


def replace_number_format(match):
    """Reemplazar formato de número"""
    var_name = match.group(1).strip()
    return f"format_number({var_name})"


def has_helper_imports(content: str) -> bool:
    """Verificar si el archivo ya tiene los imports de helpers"""
    return "from utils.helpers import" in content and "format_currency" in content


def add_helper_imports(content: str) -> str:
    """Agregar imports de helpers si no existen"""
    lines = content.split("\n")

    # Buscar donde insertar los imports (después de otros imports)
    insert_pos = 0
    for i, line in enumerate(lines):
        if line.strip().startswith("import ") or line.strip().startswith("from "):
            insert_pos = i + 1
        elif line.strip() == "":
            continue
        else:
            break

    # Agregar import
    import_line = "from utils.helpers import format_currency, format_number, format_percentage"

    if insert_pos < len(lines):
        lines.insert(insert_pos, import_line)
    else:
        lines.append(import_line)

    return "\n".join(lines)


def main():
    """Ejecutar corrección en todos los archivos relevantes"""

    frontend_dir = Path(__file__).parent.parent
    total_changes = 0
    files_changed = 0

    # Archivos a procesar (excluyendo helpers.py y este script)
    patterns_to_include = ["components/*.py", "callbacks/*.py", "layouts/*.py"]

    files_to_process = []
    for pattern in patterns_to_include:
        files_to_process.extend(frontend_dir.glob(pattern))

    # Excluir archivos problemáticos
    exclude_files = {"helpers.py", "fix_number_formatting.py"}

    for file_path in files_to_process:
        if file_path.name in exclude_files:
            continue

        print(f"Procesando: {file_path.relative_to(frontend_dir)}")

        try:
            changes = fix_number_formatting(str(file_path))
            if changes > 0:
                print(f"  OK {changes} cambios realizados")
                total_changes += changes
                files_changed += 1
            else:
                print("  - Sin cambios")
        except Exception as e:
            print(f"  ERROR: {e}")

    print("\nResumen:")
    print(f"  Archivos procesados: {len(files_to_process)}")
    print(f"  Archivos modificados: {files_changed}")
    print(f"  Total de cambios: {total_changes}")


if __name__ == "__main__":
    main()
