# frontend/pages/settings_page.py
"""
Página unificada de Ajustes con tabs (Issue #155).
Consolida configuración de perfil, farmacia y preferencias.
"""

import dash_bootstrap_components as dbc

# ✅ REGLA #5: OBLIGATORIO usar Design System
from components.base import BaseButton, BaseCard, Title
from dash import dcc, html
from styles.design_tokens import SPACING


def create_settings_layout(active_tab="perfil"):
    """
    Crea layout unificado de ajustes con tabs.

    Args:
        active_tab: Tab activo ("perfil", "farmacia", "preferencias", "suscripcion")

    Returns:
        Layout completo de la página de ajustes
    """

    # Tabs principales
    # NOTA: dbc.Tab.label solo acepta strings simples, no componentes
    # Para iconos, usamos unicode o simplemente texto
    tabs = dbc.Tabs(
        [
            dbc.Tab(
                label="👤 Mi Perfil",
                tab_id="perfil",
                label_style={"padding": "12px 20px"},
            ),
            dbc.Tab(
                label="🏥 Datos de Farmacia",
                tab_id="farmacia",
                label_style={"padding": "12px 20px"},
            ),
            dbc.Tab(
                label="⚙️ Preferencias",
                tab_id="preferencias",
                label_style={"padding": "12px 20px"},
            ),
            # Issue #420: FREE tier experience - Subscription tab
            dbc.Tab(
                label="💳 Suscripción",
                tab_id="suscripcion",
                label_style={"padding": "12px 20px"},
            ),
        ],
        id="settings-tabs",
        active_tab=active_tab,
        className="mb-4",
    )

    return html.Div(
        [
            # Store intermedio para tab activo (REGLA #11)
            dcc.Store(id="settings-active-tab-store", data={"active_tab": active_tab}),
            # Header de página
            html.Div(
                [
                    Title("Ajustes", level=1),
                    html.P(
                        "Configura tu perfil, datos de farmacia y preferencias del sistema", className="text-muted mb-4"
                    ),
                ],
                className="mb-4",
            ),
            # Tabs
            tabs,
            # Contenido dinámico del tab activo (reemplazado por callback)
            html.Div(id="settings-tab-content"),
        ],
        className="container-fluid px-4 py-4",
    )


def create_profile_tab():
    """
    Tab de perfil personal (Mi Perfil).
    Usa Design System completo.
    """
    return BaseCard(
        children=[
            Title("Perfil Personal", level=3, margin_bottom=SPACING["l"]),
            # Formulario de perfil
            dbc.Form(
                [
                    # Nombre completo
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Nombre Completo", html_for="profile-full-name"),
                                    dbc.Input(type="text", id="profile-full-name", placeholder="Ej: Juan Pérez García"),
                                ],
                                width=12,
                                md=6,
                            ),
                            # Email (read-only)
                            dbc.Col(
                                [
                                    dbc.Label("Email", html_for="profile-email"),
                                    dbc.Input(
                                        type="email",
                                        id="profile-email",
                                        placeholder="usuario@farmacia.com",
                                        disabled=True,
                                    ),
                                    html.Small("El email no se puede modificar", className="text-muted"),
                                ],
                                width=12,
                                md=6,
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Teléfono
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Teléfono", html_for="profile-phone"),
                                    dbc.Input(type="tel", id="profile-phone", placeholder="+34 600 111 222"),
                                ],
                                width=12,
                                md=6,
                            ),
                            # Username (read-only)
                            dbc.Col(
                                [
                                    dbc.Label("Usuario", html_for="profile-username"),
                                    dbc.Input(
                                        type="text", id="profile-username", placeholder="juanperez", disabled=True
                                    ),
                                    html.Small("El usuario no se puede modificar", className="text-muted"),
                                ],
                                width=12,
                                md=6,
                            ),
                        ],
                        className="mb-4",
                    ),
                    # Sección de contraseña
                    html.Hr(className="my-4"),
                    Title("Cambiar Contraseña", level=4, margin_bottom=SPACING["m"]),
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Contraseña Actual", html_for="profile-current-password"),
                                    dbc.Input(type="password", id="profile-current-password", placeholder="••••••••"),
                                ],
                                width=12,
                                md=4,
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Nueva Contraseña", html_for="profile-new-password"),
                                    dbc.Input(type="password", id="profile-new-password", placeholder="••••••••"),
                                ],
                                width=12,
                                md=4,
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Confirmar Contraseña", html_for="profile-confirm-password"),
                                    dbc.Input(type="password", id="profile-confirm-password", placeholder="••••••••"),
                                ],
                                width=12,
                                md=4,
                            ),
                        ],
                        className="mb-4",
                    ),
                    # Notificaciones
                    html.Hr(className="my-4"),
                    Title("Notificaciones", level=4, margin_bottom=SPACING["m"]),
                    dbc.Checklist(
                        id="profile-notifications",
                        options=[
                            {"label": "Notificaciones de subida de archivos", "value": "uploads"},
                            {"label": "Notificaciones de errores del sistema", "value": "errors"},
                            {"label": "Notificaciones de análisis completados", "value": "analysis"},
                        ],
                        value=["uploads", "errors"],
                        className="mb-4",
                    ),
                    # Botones de acción
                    html.Div(
                        [
                            BaseButton("Guardar Cambios", id="save-profile-btn", variant="primary", size="md"),
                            BaseButton(
                                "Cancelar",
                                id="cancel-profile-btn",
                                variant="secondary",
                                outline=True,
                                size="md",
                                className="ms-2",
                            ),
                        ],
                        className="d-flex justify-content-end mt-4",
                    ),
                ]
            ),
            # Feedback de guardado
            html.Div(id="profile-save-feedback", className="mt-3"),
        ],
        variant="default",
        shadow="sm",
    )


def create_pharmacy_tab():
    """
    Tab de datos de farmacia (Datos de Farmacia).
    Usa Design System completo.
    """
    return BaseCard(
        children=[
            Title("Datos de Farmacia", level=3, margin_bottom=SPACING["l"]),
            # Formulario de farmacia
            dbc.Form(
                [
                    # Nombre y código
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Nombre de Farmacia", html_for="pharmacy-name"),
                                    dbc.Input(type="text", id="pharmacy-name", placeholder="Ej: Farmacia Central"),
                                ],
                                width=12,
                                md=8,
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Código de Farmacia", html_for="pharmacy-code"),
                                    dbc.Input(type="text", id="pharmacy-code", placeholder="12345"),
                                ],
                                width=12,
                                md=4,
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Dirección
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Dirección", html_for="pharmacy-address"),
                                    dbc.Input(type="text", id="pharmacy-address", placeholder="Calle Mayor 1"),
                                ],
                                width=12,
                            )
                        ],
                        className="mb-3",
                    ),
                    # Ciudad y teléfono
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Ciudad", html_for="pharmacy-city"),
                                    dbc.Input(type="text", id="pharmacy-city", placeholder="Madrid"),
                                ],
                                width=12,
                                md=6,
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Teléfono", html_for="pharmacy-phone"),
                                    dbc.Input(type="tel", id="pharmacy-phone", placeholder="+34 91 123 45 67"),
                                ],
                                width=12,
                                md=6,
                            ),
                        ],
                        className="mb-4",
                    ),
                    # Configuración ERP
                    html.Hr(className="my-4"),
                    Title("Configuración ERP", level=4, margin_bottom=SPACING["m"]),
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Sistema ERP Principal", html_for="primary-erp-type"),
                                    dbc.Select(
                                        id="primary-erp-type",
                                        options=[
                                            {"label": "Farmatic", "value": "farmatic"},
                                            {"label": "Farmanager", "value": "farmanager"},
                                            {"label": "Otro", "value": "other"},
                                        ],
                                        value="farmatic",
                                    ),
                                ],
                                width=12,
                                md=6,
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Versión de ERP", html_for="erp-version"),
                                    dbc.Input(type="text", id="erp-version", placeholder="Ej: v2024.1"),
                                ],
                                width=12,
                                md=6,
                            ),
                        ],
                        className="mb-4",
                    ),
                    # Formatos de datos
                    html.Hr(className="my-4"),
                    Title("Formatos de Datos", level=4, margin_bottom=SPACING["m"]),
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Formato de Fecha", html_for="date-format"),
                                    dbc.Select(
                                        id="date-format",
                                        options=[
                                            {"label": "DD/MM/YYYY", "value": "dd/mm/yyyy"},
                                            {"label": "YYYY-MM-DD", "value": "yyyy-mm-dd"},
                                        ],
                                        value="dd/mm/yyyy",
                                    ),
                                ],
                                width=12,
                                md=6,
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Separador Decimal", html_for="decimal-separator"),
                                    dbc.Select(
                                        id="decimal-separator",
                                        options=[
                                            {"label": "Coma (,)", "value": "comma"},
                                            {"label": "Punto (.)", "value": "dot"},
                                        ],
                                        value="comma",
                                    ),
                                ],
                                width=12,
                                md=6,
                            ),
                        ],
                        className="mb-4",
                    ),
                    # Botones de acción
                    html.Div(
                        [
                            BaseButton("Guardar Cambios", id="save-pharmacy-btn", variant="primary", size="md"),
                            BaseButton(
                                "Cancelar",
                                id="cancel-pharmacy-btn",
                                variant="secondary",
                                outline=True,
                                size="md",
                                className="ms-2",
                            ),
                        ],
                        className="d-flex justify-content-end mt-4",
                    ),
                ]
            ),
            # Feedback de guardado
            html.Div(id="pharmacy-save-feedback", className="mt-3"),
        ],
        variant="default",
        shadow="sm",
    )


def create_preferences_tab():
    """
    Tab de preferencias del sistema (Preferencias).
    Usa Design System completo.
    """
    return BaseCard(
        children=[
            Title("Preferencias del Sistema", level=3, margin_bottom=SPACING["l"]),
            # Formulario de preferencias
            dbc.Form(
                [
                    # Interfaz
                    Title("Interfaz", level=4, margin_bottom=SPACING["m"]),
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Tema Visual", html_for="theme-selector"),
                                    dbc.Select(
                                        id="theme-selector",
                                        options=[
                                            {"label": "Claro (por defecto)", "value": "light"},
                                            {"label": "Oscuro (próximamente)", "value": "dark", "disabled": True},
                                        ],
                                        value="light",
                                    ),
                                    html.Small("El tema oscuro estará disponible próximamente", className="text-muted"),
                                ],
                                width=12,
                                md=6,
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Idioma", html_for="language-selector"),
                                    dbc.Select(
                                        id="language-selector",
                                        options=[{"label": "Español", "value": "es"}],
                                        value="es",
                                        disabled=True,
                                    ),
                                    html.Small("Actualmente solo disponible en español", className="text-muted"),
                                ],
                                width=12,
                                md=6,
                            ),
                        ],
                        className="mb-4",
                    ),
                    # Actualización automática
                    html.Hr(className="my-4"),
                    Title("Actualización de Datos", level=4, margin_bottom=SPACING["m"]),
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Intervalo de Actualización", html_for="refresh-interval"),
                                    dbc.Select(
                                        id="refresh-interval",
                                        options=[
                                            {"label": "15 segundos", "value": "15"},
                                            {"label": "30 segundos (recomendado)", "value": "30"},
                                            {"label": "60 segundos", "value": "60"},
                                        ],
                                        value="30",
                                    ),
                                    html.Small(
                                        "Frecuencia de actualización automática del dashboard", className="text-muted"
                                    ),
                                ],
                                width=12,
                                md=6,
                            )
                        ],
                        className="mb-4",
                    ),
                    # Notificaciones del sistema
                    html.Hr(className="my-4"),
                    Title("Notificaciones del Sistema", level=4, margin_bottom=SPACING["m"]),
                    dbc.Checklist(
                        id="system-notifications",
                        options=[
                            {"label": "Alertas del sistema", "value": "system"},
                            {"label": "Alertas de stock bajo", "value": "stock"},
                            {"label": "Resumen diario por email", "value": "daily"},
                        ],
                        value=["system"],
                        className="mb-3",
                    ),
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Email para Notificaciones", html_for="notification-email"),
                                    dbc.Input(
                                        type="email", id="notification-email", placeholder="notificaciones@farmacia.com"
                                    ),
                                ],
                                width=12,
                                md=6,
                            )
                        ],
                        className="mb-4",
                    ),
                    # Botones de acción
                    html.Div(
                        [
                            BaseButton("Guardar Preferencias", id="save-preferences-btn", variant="primary", size="md"),
                            BaseButton(
                                "Restaurar Valores por Defecto",
                                id="reset-preferences-btn",
                                variant="secondary",
                                outline=True,
                                size="md",
                                className="ms-2",
                            ),
                        ],
                        className="d-flex justify-content-end mt-4",
                    ),
                ]
            ),
            # Feedback de guardado
            html.Div(id="preferences-save-feedback", className="mt-3"),
        ],
        variant="default",
        shadow="sm",
    )


# =============================================================================
# Issue #420: FREE tier experience - Subscription Tab
# =============================================================================

# Plan configuration (mirrors backend PLAN_FEATURES in subscription_limits.py)
PLAN_CONFIG = {
    "free": {
        "name": "FREE",
        "badge_color": "secondary",
        "storage_mb": 100,
        "data_months": 3,
        "max_sales": "10.000",
        "max_uploads": 5,
        "backup_days": 7,
        "support": "Comunidad",
    },
    "pro": {
        "name": "PRO",
        "badge_color": "info",
        "storage_mb": 1024,
        "data_months": None,  # Unlimited
        "max_sales": "100.000",
        "max_uploads": 50,
        "backup_days": 30,
        "support": "Email",
    },
    "max": {
        "name": "MAX",
        "badge_color": "success",
        "storage_mb": 10240,
        "data_months": None,  # Unlimited
        "max_sales": "1.000.000",
        "max_uploads": 500,
        "backup_days": 90,
        "support": "Prioritario",
    },
}


def _create_plan_comparison_table():
    """
    Create the 3-column plan comparison table.

    Returns:
        dbc.Table with plan comparison
    """
    features = [
        ("Almacenamiento", "100 MB", "1 GB", "10 GB"),
        ("Historial de datos", "3 meses", "Ilimitado", "Ilimitado"),
        ("Registros de ventas", "10.000", "100.000", "1.000.000"),
        ("Subidas de archivos", "5", "50", "500"),
        ("Retención de backups", "7 días", "30 días", "90 días"),
        ("Soporte", "Comunidad", "Email", "Prioritario"),
    ]

    header = html.Thead(
        html.Tr(
            [
                html.Th("Característica", style={"width": "40%"}),
                html.Th(
                    dbc.Badge("FREE", color="secondary"),
                    className="text-center",
                ),
                html.Th(
                    dbc.Badge("PRO", color="info"),
                    className="text-center",
                ),
                html.Th(
                    dbc.Badge("MAX", color="success"),
                    className="text-center",
                ),
            ]
        )
    )

    rows = []
    for feature in features:
        name, free_val, pro_val, max_val = feature
        rows.append(
            html.Tr(
                [
                    html.Td(name),
                    html.Td(free_val, className="text-center"),
                    html.Td(pro_val, className="text-center text-info fw-bold"),
                    html.Td(max_val, className="text-center text-success fw-bold"),
                ]
            )
        )

    return dbc.Table(
        [header, html.Tbody(rows)],
        bordered=True,
        hover=True,
        responsive=True,
        striped=True,
        className="mt-3",
    )


def create_subscription_tab():
    """
    Tab de suscripción (Suscripción).
    Muestra plan actual, uso de almacenamiento y comparativa de planes.

    Issue #420: FREE tier experience - Subscription page.

    Uses Design System completo (REGLA #5).
    """
    return html.Div(
        [
            # Section 1: Current Plan
            BaseCard(
                children=[
                    Title("Tu Plan Actual", level=3, margin_bottom=SPACING["l"]),
                    html.Div(
                        id="subscription-current-plan-content",
                        children=[
                            # Placeholder - will be filled by callback
                            dbc.Spinner(
                                html.Div("Cargando información del plan..."),
                                size="sm",
                                color="primary",
                            )
                        ],
                    ),
                ],
                variant="highlighted",
                shadow="sm",
            ),
            # Section 2: Storage Usage
            BaseCard(
                children=[
                    Title("Uso de Almacenamiento", level=4, margin_bottom=SPACING["m"]),
                    html.Div(
                        id="subscription-storage-content",
                        children=[
                            dbc.Spinner(
                                html.Div("Calculando almacenamiento..."),
                                size="sm",
                                color="primary",
                            )
                        ],
                    ),
                ],
                variant="default",
                shadow="sm",
                className="mt-4",
            ),
            # Section 3: Plan Limits Info (for FREE users)
            BaseCard(
                children=[
                    Title("Límites de tu Plan", level=4, margin_bottom=SPACING["m"]),
                    html.Div(
                        id="subscription-limits-content",
                        children=[
                            html.P(
                                [
                                    html.I(className="fas fa-clock me-2 text-warning"),
                                    "Con el plan FREE solo puedes acceder a los datos de los ",
                                    html.Strong("últimos 3 meses"),
                                    " desde tu última venta registrada.",
                                ],
                                className="mb-0",
                            )
                        ],
                    ),
                ],
                variant="default",
                shadow="sm",
                className="mt-4",
            ),
            # Section 4: Plan Comparison Table
            BaseCard(
                children=[
                    Title("Comparativa de Planes", level=4, margin_bottom=SPACING["m"]),
                    _create_plan_comparison_table(),
                ],
                variant="default",
                shadow="sm",
                className="mt-4",
            ),
            # Section 5: Upgrade CTA
            BaseCard(
                children=[
                    html.Div(
                        [
                            Title("¿Listo para más?", level=4, margin_bottom=SPACING["s"]),
                            html.P(
                                "Actualiza tu plan para desbloquear más almacenamiento, "
                                "historial ilimitado y soporte prioritario.",
                                className="text-muted mb-3",
                            ),
                            html.A(
                                BaseButton(
                                    html.Span(
                                        [
                                            html.I(className="fas fa-envelope me-2"),
                                            "Contactar para Actualizar",
                                        ]
                                    ),
                                    id="subscription-upgrade-btn",
                                    variant="success",
                                    size="lg",
                                ),
                                href="mailto:ventas@xfarma.es?subject=Solicitud%20de%20upgrade%20de%20plan",
                                style={"textDecoration": "none"},
                            ),
                        ],
                        className="text-center",
                    )
                ],
                variant="highlighted",
                shadow="md",
                className="mt-4",
            ),
            # Feedback div
            html.Div(id="subscription-feedback", className="mt-3"),
        ]
    )
