"""
Layout para Dashboard Venta Libre (Issue #461, #471, #491, #493, #506)

Análisis de ventas de productos OTC por categoría NECESIDAD.
Tab 1: Visión General (#491) - Treemap + Evolución Temporal + YoY + Top Contributors
Tab 2: Categorías y Marcas (#493) - HHI, Cuadrante Valor, Brand Duel, Evolución, Boxplot
Tab 3: Inventario (#471) - KPIs, ABC, Rotación de parafarmacia
Tab 4: Producto y Surtido (#494) - Ficha, KPIs, Alternativas, Complementarios
Tab 5: Insights (#506) - Motor de detección automática de anomalías

Nota: Tab Gestión eliminado (Issue #488) - funcionalidades movidas a /admin.
"""

import dash_bootstrap_components as dbc
from dash import dcc, html

from components.inventory import create_inventory_tab
from components.ventalibre.brand_analysis import (
    create_brand_duel_section,
    create_brands_chart_card,
    create_brands_table_card,
    create_category_selector,
    create_hhi_education_modal,
    create_hhi_indicator,
    create_hhi_matrix_card,
    create_market_evolution_card,
    create_price_boxplot_card,
    create_value_quadrant_card,
)
from components.ventalibre.context_menu import create_context_menu_modal
from components.ventalibre.l2_drilldown import create_l2_drilldown_modal
from components.ventalibre.roi_tracker_card import create_roi_actions_modal
from components.insights import InsightsPanel
from styles.design_tokens import COLORS, SPACING


def _create_global_date_filter():
    """
    Filtro de fechas global visible en todos los tabs (Issue #537).

    Proporciona consistencia al usuario mostrando siempre el período
    de análisis seleccionado, independientemente del tab activo.

    Incluye banner de restricción FREE tier (Issue #420).
    """
    return dbc.Card(
        dbc.CardBody(
            [
                dbc.Row(
                    [
                        # Icono y label
                        dbc.Col(
                            html.Div(
                                [
                                    html.I(
                                        className="fas fa-calendar-alt me-2",
                                        style={"color": COLORS["primary"], "fontSize": "1.2rem"},
                                    ),
                                    html.Span(
                                        "Período de Análisis",
                                        style={"fontWeight": "600", "fontSize": "0.95rem"},
                                    ),
                                ],
                                className="d-flex align-items-center",
                            ),
                            width="auto",
                            className="d-flex align-items-center",
                        ),
                        # DatePickerRange
                        dbc.Col(
                            dcc.DatePickerRange(
                                id="ventalibre-date-range",
                                display_format="DD/MM/YYYY",
                                start_date_placeholder_text="Cargando...",
                                end_date_placeholder_text="Cargando...",
                                calendar_orientation="horizontal",
                                first_day_of_week=1,
                                month_format="MMMM YYYY",
                                className="w-100",
                                clearable=False,
                                persistence=True,
                                persistence_type="session",
                            ),
                            width=12,
                            md=5,
                            lg=4,
                            className="mt-2 mt-md-0",
                        ),
                        # Slider visual de fechas
                        dbc.Col(
                            dcc.RangeSlider(
                                id="ventalibre-date-slider",
                                min=0,
                                max=100,  # Se configura dinámicamente
                                step=1,
                                marks={},  # Se configura dinámicamente con meses
                                value=[0, 100],
                                tooltip={"placement": "bottom", "always_visible": False},
                                persistence=True,
                                persistence_type="session",
                            ),
                            width=12,
                            md=True,  # Flex grow
                            className="mt-2 mt-md-0 d-none d-lg-block",  # Oculto en móvil/tablet
                        ),
                    ],
                    className="align-items-center g-3",
                ),
                # Issue #420: FREE tier date restriction banner (full width below)
                html.Div(
                    id="ventalibre-date-restriction-banner",
                    className="mt-2",
                    style={"display": "none"},
                ),
            ],
            className="py-2 px-3",
        ),
        className="mb-3 shadow-sm",
        style={
            "backgroundColor": COLORS["bg_tertiary"],
            "border": f"1px solid {COLORS['border_light']}",
            "position": "relative",  # Issue #548: Contexto de apilamiento para calendario
            "zIndex": 1100,  # Mayor que Tabs content para que calendario aparezca encima
        },
    )


def get_ventalibre_layout():
    """Layout principal con tabs."""
    return html.Div([
        # NOTA: Stores definidos en app.py skeleton (REGLA #0.5)

        # ===================================================================
        # HEADER
        # ===================================================================
        html.Div([
            dbc.Row([
                dbc.Col([
                    html.H1([
                        html.I(className="fas fa-shopping-bag me-3", style={"color": COLORS["info"]}),
                        "Análisis Venta Libre",
                    ], className="mb-2"),
                    html.P(
                        "Análisis de ventas de productos OTC por categoría NECESIDAD",
                        className="text-muted mb-0",
                    ),
                ], width=12, lg=8),
                # Issue #511: Botones de descarga (Dirección PDF + Suelo Excel)
                dbc.Col([
                    dbc.ButtonGroup([
                        dbc.Button(
                            html.Span([html.I(className="fas fa-file-pdf me-2"), "Informe PDF"]),
                            id="ventalibre-download-pdf-btn",
                            color="primary",
                            outline=True,
                            size="sm",
                            title="Descargar resumen ejecutivo para Dirección",
                        ),
                        dbc.Button(
                            html.Span([html.I(className="fas fa-file-excel me-2"), "Pick-Lists"]),
                            id="ventalibre-download-excel-btn",
                            color="success",
                            outline=True,
                            size="sm",
                            title="Descargar listas operativas para Suelo",
                        ),
                        # Issue #513: Plan de Acción
                        dbc.Button(
                            html.Span([html.I(className="fas fa-tasks me-2"), "Plan Acción"]),
                            id="ventalibre-download-action-plan-btn",
                            color="warning",
                            outline=True,
                            size="sm",
                            title="Descargar plan de acción con insights priorizados",
                        ),
                    ], className="float-end"),
                ], width=12, lg=4, className="d-flex align-items-center justify-content-lg-end mt-3 mt-lg-0"),
            ], className="align-items-center"),
        ]),

        # ===================================================================
        # FILTRO GLOBAL DE FECHAS (Issue #537)
        # Visible en todos los tabs para consistencia
        # ===================================================================
        _create_global_date_filter(),

        # ===================================================================
        # TABS
        # ===================================================================
        dbc.Tabs([
            # TAB 1: VISIÓN GENERAL (Issue #491 - antes "Análisis")
            dbc.Tab(
                label="Visión General",
                tab_id="tab-analisis",  # Mantener ID para no romper callbacks
                children=_create_analisis_tab(),
            ),
            # TAB 2: CATEGORÍAS Y MARCAS (Issue #493)
            dbc.Tab(
                label="Categorías y Marcas",
                tab_id="tab-brands",
                children=_create_brands_tab(),
            ),
            # TAB 3: INVENTARIO (Issue #471)
            dbc.Tab(
                label="Inventario",
                tab_id="tab-inventario",
                children=create_inventory_tab(
                    id_prefix="ventalibre",
                    product_type="venta_libre",
                    title="Inventario Parafarmacia",
                    description="Análisis de rotación y rentabilidad del stock de productos de venta libre",
                ),
            ),
            # TAB 4: PRODUCTO Y SURTIDO (Issue #494)
            dbc.Tab(
                label="Producto y Surtido",
                tab_id="tab-producto",
                children=_create_producto_tab(),
            ),
            # TAB 5: INSIGHTS (Issue #506)
            dbc.Tab(
                label="Insights",
                tab_id="tab-insights",
                children=_create_insights_tab(),
            ),
        ], id="ventalibre-tabs", active_tab="tab-analisis"),

        # Modal feedback (fuera de tabs)
        _create_feedback_modal(),

        # Modal L2 drill-down (Issue #505)
        create_l2_drilldown_modal(),

        # Modal educativo HHI (diagnóstico cruzado HHI × Margen)
        create_hhi_education_modal(),

        # Modal ROI Tracker - Acciones pendientes (Issue #514)
        create_roi_actions_modal(),

        # Modal Context Menu - Navegación contextual (Issue #512)
        create_context_menu_modal(),

        # Alerts container
        html.Div(id="ventalibre-alerts-container"),

        # Toast para feedback de narrativa (Issue #509)
        dbc.Toast(
            id="narrative-feedback-toast",
            header="Feedback",
            icon="success",
            is_open=False,
            dismissable=True,
            duration=3000,
            style={"position": "fixed", "bottom": 20, "right": 20, "zIndex": 1050},
        ),

    ], className="container-fluid px-4 py-3")


def _create_filters_panel():
    """
    Panel de filtros específicos para Tab 1 (Análisis).

    Nota: El filtro de fechas se movió al header global (Issue #537).
    Este panel solo contiene filtros específicos del Tab 1.

    Filtros incluidos:
    - EmployeeFilter (PRO only)
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-filter me-2", style={"color": COLORS["info"]}),
                    html.Span("Filtros Adicionales", style={"fontWeight": "600"}),
                ],
                style={
                    "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
                    "border": "none",
                    "padding": f"{SPACING.get('s', '0.5rem')} {SPACING.get('m', '1rem')}"
                },
            ),
            dbc.CardBody(
                [
                    # Filtro de Empleados (PRO only)
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.H6(
                                        "👥 Filtro de Empleados",
                                        className="mb-2",
                                        style={"fontWeight": "600"},
                                    ),
                                    dcc.Dropdown(
                                        id="ventalibre-employee-filter",
                                        options=[],  # Se carga dinámicamente
                                        placeholder="Todos los empleados",
                                        multi=False,
                                        clearable=True,
                                        persistence=True,
                                        persistence_type="session",
                                    ),
                                    html.Small(
                                        [
                                            html.I(className="fas fa-info-circle me-1"),
                                            "Vacío = Todos los empleados",
                                        ],
                                        className="text-muted mt-1 d-block",
                                    ),
                                ],
                                width=12,
                            ),
                        ],
                    ),
                ],
                style={"padding": SPACING.get("m", "1rem")},
            ),
        ],
        className="h-100 shadow-sm",
    )


def _create_context_panel():
    """
    Panel de contexto visual con treemap (similar a /prescription).

    Muestra distribución jerárquica de ventas de venta libre
    mediante un treemap interactivo.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-chart-pie me-2", style={"color": COLORS["info"]}),
                    html.Span("Contexto de Ventas", style={"fontWeight": "600"}),
                ],
                style={
                    "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
                    "border": "none",
                    "padding": f"{SPACING.get('s', '0.5rem')} {SPACING.get('m', '1rem')}"
                },
            ),
            dbc.CardBody(
                [
                    # Treemap de contexto
                    dcc.Loading(
                        id="ventalibre-context-treemap-loading",
                        type="default",
                        children=dcc.Graph(
                            id="ventalibre-context-treemap",
                            config={"displayModeBar": False},
                            style={"height": "280px"},
                            figure={
                                "data": [],
                                "layout": {
                                    "margin": {"t": 10, "b": 10, "l": 10, "r": 10},
                                    "annotations": [{
                                        "text": "Cargando contexto...",
                                        "showarrow": False,
                                        "font": {"size": 14, "color": COLORS.get("text_secondary", "#6c757d")}
                                    }]
                                }
                            }
                        ),
                    ),
                ],
                style={"padding": SPACING.get("s", "0.5rem")},
            ),
        ],
        className="h-100 shadow-sm",
        style={"border": f"2px solid {COLORS.get('border_light', '#dee2e6')}"},
    )


def _create_analisis_tab():
    """Contenido de Tab 1: Análisis."""
    return html.Div([
        # ===================================================================
        # ROW 50/50: FILTROS + CONTEXTO (similar a /prescription)
        # ===================================================================
        dbc.Row(
            [
                dbc.Col([_create_filters_panel()], width=12, lg=6, className="mb-4"),
                dbc.Col([_create_context_panel()], width=12, lg=6, className="mb-4"),
            ]
        ),

        # ===================================================================
        # EXECUTIVE SUMMARY (Issue #509) - NLG con Insight Engine
        # ===================================================================
        html.Div(id="ventalibre-executive-summary-container", className="mb-4"),

        # ===================================================================
        # ROI TRACKER WIDGET (Issue #514) - Feedback Loop de Acciones
        # ===================================================================
        html.Div(id="ventalibre-roi-tracker-container", className="mb-4"),

        # ===================================================================
        # Treemap + Tabla
        # ===================================================================
        dbc.Row([
            # Treemap (izquierda)
            dbc.Col([
                dbc.Card([
                    dbc.CardHeader([
                        html.Div([
                            html.I(className="fas fa-chart-pie me-2"),
                            "Ventas por NECESIDAD",
                            # Badge contador de categorías minoritarias (W14a)
                            dbc.Badge(
                                id="ventalibre-other-count",
                                children="0",
                                color="secondary",
                                className="ms-2",
                                style={"display": "none"},  # Se muestra solo si hay otras
                            ),
                        ], className="d-flex align-items-center"),
                        # Toggle Treemap/Barras (Issue #492)
                        dbc.ButtonGroup([
                            dbc.Button(
                                html.Span([
                                    html.I(className="fas fa-th-large me-1"),
                                    "Treemap",
                                ]),
                                id="ventalibre-chart-treemap-btn",
                                color="primary",
                                outline=False,
                                size="sm",
                                className="px-2",
                            ),
                            dbc.Button(
                                html.Span([
                                    html.I(className="fas fa-bars me-1"),
                                    "Barras",
                                ]),
                                id="ventalibre-chart-bars-btn",
                                color="primary",
                                outline=True,
                                size="sm",
                                className="px-2",
                            ),
                        ], size="sm"),
                    ], className="d-flex justify-content-between align-items-center"),
                    dbc.CardBody([
                        # Treemap principal (solo categorías ≥2%)
                        dcc.Loading([
                            html.Div(id="ventalibre-treemap-container"),
                        ], type="circle"),

                        # ============================================================
                        # W14a: Sección colapsible "Otras Categorías" (minoritarias)
                        # ============================================================
                        html.Div([
                            # Botón toggle
                            dbc.Button(
                                html.Span([
                                    html.I(className="fas fa-chevron-down me-2", id="ventalibre-toggle-icon"),
                                    "Ver categorías minoritarias",
                                ]),
                                id="ventalibre-toggle-other",
                                size="sm",
                                color="link",
                                className="p-0 mt-2",
                            ),

                            # Collapse con tabla
                            dbc.Collapse([
                                html.Hr(className="my-2"),
                                html.H6([
                                    html.I(className="fas fa-list-ul me-2"),
                                    "Otras Categorías",
                                    html.Small(
                                        " (< 2% de ventas)",
                                        className="text-muted",
                                    ),
                                ], className="mb-2"),
                                # Tabla scrollable de categorías minoritarias
                                html.Div(
                                    id="ventalibre-other-categories-table",
                                    style={"maxHeight": "200px", "overflowY": "auto"},
                                ),
                            ], id="ventalibre-other-collapse", is_open=False),
                        ], id="ventalibre-other-section", style={"display": "none"}),
                        # ============================================================
                    ]),
                ], className="h-100 shadow-sm"),
            ], width=12, lg=6, className="mb-4"),

            # Tabla productos (derecha)
            dbc.Col([
                dbc.Card([
                    dbc.CardHeader([
                        html.I(className="fas fa-list me-2"),
                        "Productos",
                        # Badge de categoría seleccionada con botón para limpiar
                        html.Span([
                            dbc.Badge(
                                id="ventalibre-category-badge",
                                children="Todas",
                                color="info",
                                className="ms-2",
                            ),
                            # Botón X para limpiar filtro (oculto por defecto)
                            dbc.Button(
                                html.I(className="fas fa-times"),
                                id="ventalibre-clear-category",
                                color="link",
                                size="sm",
                                className="p-0 ms-1 text-danger",
                                style={"display": "none"},  # Oculto inicialmente
                                title="Mostrar todas las categorías",
                            ),
                        ]),
                    ]),
                    dbc.CardBody([
                        # Búsqueda
                        dbc.InputGroup([
                            dbc.InputGroupText(html.I(className="fas fa-search")),
                            dbc.Input(
                                id="ventalibre-search",
                                placeholder="Buscar producto...",
                                type="text",
                                debounce=True,
                            ),
                        ], className="mb-3"),

                        # Tabla de productos
                        dcc.Loading([
                            html.Div(id="ventalibre-products-container"),
                        ], type="circle"),

                        # Paginación
                        html.Div([
                            dbc.Button(
                                html.I(className="fas fa-chevron-left"),
                                id="ventalibre-prev-page",
                                color="secondary",
                                size="sm",
                                className="me-2",
                            ),
                            html.Span(id="ventalibre-page-info", className="mx-2"),
                            dbc.Button(
                                html.I(className="fas fa-chevron-right"),
                                id="ventalibre-next-page",
                                color="secondary",
                                size="sm",
                            ),
                        ], className="d-flex justify-content-center align-items-center mt-3"),
                    ]),
                ], className="h-100 shadow-sm"),
            ], width=12, lg=6, className="mb-4"),
        ]),

        # ===================================================================
        # Issue #491: EVOLUCIÓN TEMPORAL DE VENTAS
        # Gráfico de evolución + Tabla YoY + Top Contributors
        # ===================================================================

        # Row 1: Evolution Chart (full width)
        dbc.Row([
            dbc.Col([
                dbc.Card([
                    dbc.CardHeader([
                        html.Div([
                            html.I(
                                className="fas fa-chart-line me-2",
                                style={"color": COLORS["info"]},
                            ),
                            "Evolución Temporal de Ventas",
                        ], className="d-flex align-items-center"),
                        # Period selector (6m, 12m, 24m)
                        dbc.RadioItems(
                            id="ventalibre-evolution-period",
                            options=[
                                {"label": "6m", "value": 6},
                                {"label": "12m", "value": 12},
                                {"label": "24m", "value": 24},
                            ],
                            value=12,
                            inline=True,
                            className="ms-auto",
                            inputClassName="btn-check",
                            labelClassName="btn btn-outline-primary btn-sm",
                            labelCheckedClassName="active",
                        ),
                    ], className="d-flex justify-content-between align-items-center"),
                    dbc.CardBody([
                        dcc.Loading(
                            id="ventalibre-evolution-chart-loading",
                            type="circle",
                            children=html.Div(id="ventalibre-evolution-container"),
                        ),
                    ]),
                ], className="shadow-sm"),
            ], width=12, className="mb-4"),
        ]),

        # Row 2: YoY Table (6 cols) + Top Contributors (6 cols)
        dbc.Row([
            # YoY Comparison Table
            dbc.Col([
                dbc.Card([
                    dbc.CardHeader([
                        html.I(
                            className="fas fa-calendar-alt me-2",
                            style={"color": COLORS["success"]},
                        ),
                        "Comparación Año vs Año",
                    ], style={"backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa")}),
                    dbc.CardBody([
                        dcc.Loading(
                            id="ventalibre-yoy-loading",
                            type="circle",
                            children=html.Div(id="ventalibre-yoy-container"),
                        ),
                    ]),
                ], className="h-100 shadow-sm"),
            ], width=12, lg=6, className="mb-4"),

            # Top Contributors
            dbc.Col([
                dbc.Card([
                    dbc.CardHeader([
                        html.Div([
                            html.I(
                                className="fas fa-trophy me-2",
                                style={"color": COLORS["warning"]},
                            ),
                            "Top Contribuidores al Cambio",
                        ], className="d-flex align-items-center"),
                        # Direction filter
                        dbc.RadioItems(
                            id="ventalibre-contributors-direction",
                            options=[
                                {"label": "Todos", "value": "all"},
                                {"label": html.I(className="fas fa-arrow-up text-success"), "value": "up"},
                                {"label": html.I(className="fas fa-arrow-down text-danger"), "value": "down"},
                            ],
                            value="all",
                            inline=True,
                            className="ms-auto",
                            inputClassName="btn-check",
                            labelClassName="btn btn-outline-secondary btn-sm",
                            labelCheckedClassName="active",
                        ),
                    ], className="d-flex justify-content-between align-items-center"),
                    dbc.CardBody([
                        dcc.Loading(
                            id="ventalibre-contributors-loading",
                            type="circle",
                            children=html.Div(id="ventalibre-contributors-container"),
                        ),
                    ]),
                ], className="h-100 shadow-sm"),
            ], width=12, lg=6, className="mb-4"),
        ]),
    ])


def _create_brands_tab():
    """
    Contenido de Tab 2: Categorías y Marcas (Issue #493).

    Análisis de marcas por categoría NECESIDAD con métricas consultoría:
    - HHI (concentración de mercado)
    - Cuadrante de Valor (Margen vs Volumen)
    - Brand Duel Mode (comparación lado a lado)
    - Evolución de cuota de mercado
    - Boxplot de precios (canibalización)
    """
    return html.Div([
        # ===================================================================
        # ROW 1: SELECTOR CATEGORÍA + HHI (4/8 cols)
        # ===================================================================
        dbc.Row([
            # Selector de categoría NECESIDAD
            dbc.Col([
                create_category_selector(),
            ], width=12, lg=4, className="mb-4"),

            # Indicador HHI
            dbc.Col([
                create_hhi_indicator(),
            ], width=12, lg=8, className="mb-4"),
        ]),

        # ===================================================================
        # ROW 2: TREEMAP/BARRAS + TABLA MARCAS (6/6 cols)
        # ===================================================================
        dbc.Row([
            # Treemap/Barras de marcas
            dbc.Col([
                create_brands_chart_card(),
            ], width=12, lg=6, className="mb-4"),

            # Tabla detalle por marca
            dbc.Col([
                create_brands_table_card(),
            ], width=12, lg=6, className="mb-4"),
        ]),

        # ===================================================================
        # ROW 3: CUADRANTE VALOR + EVOLUCIÓN CUOTA (6/6 cols)
        # ===================================================================
        dbc.Row([
            # Cuadrante de Valor (Scatter Margen vs Volumen)
            dbc.Col([
                create_value_quadrant_card(),
            ], width=12, lg=6, className="mb-4"),

            # Evolución cuota de mercado (áreas apiladas 100%)
            dbc.Col([
                create_market_evolution_card(),
            ], width=12, lg=6, className="mb-4"),
        ]),

        # ===================================================================
        # ROW 4: BOXPLOT PRECIOS (12 cols)
        # ===================================================================
        dbc.Row([
            dbc.Col([
                create_price_boxplot_card(),
            ], width=12, className="mb-4"),
        ]),

        # ===================================================================
        # ROW 5: MATRIZ HHI × MARGEN - Vista Estratégica (12 cols)
        # Issue #539: Scatter plot de TODAS las categorías para diagnóstico
        # ===================================================================
        dbc.Row([
            dbc.Col([
                create_hhi_matrix_card(),
            ], width=12, className="mb-4"),
        ]),

        # ===================================================================
        # ROW 6: BRAND DUEL MODE (colapsible)
        # ===================================================================
        dbc.Row([
            dbc.Col([
                create_brand_duel_section(),
            ], width=12),
        ]),
    ])


def _create_producto_tab():
    """
    Tab 4: Producto y Surtido (Issue #494).

    Centro de Mando de Rentabilidad por producto:
    - Buscador con autocompletado
    - Ficha producto + KPIs
    - Alternativas (switching)
    - Complementarios (MBA)
    - Scatter competitivo
    - Decision Box (semáforo)
    """
    from components.ventalibre import (
        create_product_search,
        create_product_ficha_placeholder,
        create_kpis_placeholder,
        create_alternatives_placeholder,
        create_complementary_placeholder,
        create_scatter_placeholder,
        create_decision_placeholder,
    )

    return html.Div([
        # ===================================================================
        # ROW 1: BUSCADOR DE PRODUCTOS
        # ===================================================================
        dbc.Row([
            dbc.Col([
                create_product_search(),
            ], width=12),
        ], className="mb-4"),

        # ===================================================================
        # ROW 2: FICHA + KPIS
        # ===================================================================
        dbc.Row([
            dbc.Col([
                html.Div(
                    id="ventalibre-product-ficha-container",
                    children=create_product_ficha_placeholder(),
                ),
            ], width=12, lg=4, className="mb-3"),
            dbc.Col([
                html.Div(
                    id="ventalibre-product-kpis-container",
                    children=create_kpis_placeholder(),
                ),
            ], width=12, lg=8, className="mb-3"),
        ]),

        # ===================================================================
        # ROW 3: ALTERNATIVAS + COMPLEMENTARIOS
        # ===================================================================
        dbc.Row([
            dbc.Col([
                html.Div(
                    id="ventalibre-alternatives-container",
                    children=create_alternatives_placeholder(),
                ),
            ], width=12, lg=6, className="mb-3"),
            dbc.Col([
                html.Div(
                    id="ventalibre-complementary-container",
                    children=create_complementary_placeholder(),
                ),
            ], width=12, lg=6, className="mb-3"),
        ]),

        # ===================================================================
        # ROW 4: SCATTER COMPETITIVO + DECISION BOX
        # ===================================================================
        dbc.Row([
            dbc.Col([
                html.Div(
                    id="ventalibre-competitive-container",
                    children=create_scatter_placeholder(),
                ),
            ], width=12, lg=8, className="mb-3"),
            dbc.Col([
                html.Div(
                    id="ventalibre-decision-box-container",
                    children=create_decision_placeholder(),
                ),
            ], width=12, lg=4, className="mb-3"),
        ]),
    ])


def _create_insights_tab():
    """
    Tab 5: Insights (Issue #506).

    Motor de detección automática de anomalías con 10 reglas,
    acciones concretas e impacto económico estimado.
    """
    return html.Div([
        InsightsPanel(id_prefix="ventalibre"),
    ])


def _create_feedback_modal():
    """Modal para reportar errores de clasificación."""
    return dbc.Modal([
        dbc.ModalHeader(dbc.ModalTitle("Reportar Error de Clasificación")),
        dbc.ModalBody([
            html.Div(id="ventalibre-feedback-content"),

            html.Hr(),

            dbc.Label("¿Cuál debería ser la categoría correcta?"),
            dcc.Dropdown(
                id="ventalibre-feedback-category",
                options=[],  # Se llena dinámicamente
                placeholder="Seleccionar categoría...",
                className="mb-3",
            ),

            dbc.Label("Comentario (opcional)"),
            dbc.Textarea(
                id="ventalibre-feedback-comment",
                placeholder="Añade contexto si lo deseas...",
                className="mb-3",
            ),
        ]),
        dbc.ModalFooter([
            dbc.Button("Cancelar", id="ventalibre-feedback-cancel", color="secondary"),
            dbc.Button("Enviar Reporte", id="ventalibre-feedback-submit", color="primary"),
        ]),
    ], id="ventalibre-feedback-modal", is_open=False, centered=True)
