from utils.helpers import format_number

# frontend/layouts/upload.py
"""
Layout de la página de carga de archivos de xfarma.
Permite subir archivos ERP y ver el historial de cargas.

Issue #429: Layout compacto 2 columnas
Issue #476: Carga de ficheros de inventario desde ERPs
"""

import dash_bootstrap_components as dbc
from components.common import create_loading_spinner
from dash import dcc, html
from styles.design_tokens import BORDER_RADIUS
from styles.design_tokens import COLORS as DS_COLORS
from styles.design_tokens import SPACING, TRANSITIONS, TYPOGRAPHY


def create_upload_type_selector() -> html.Div:
    """
    Información sobre detección automática de tipo de archivo.

    Issue #476: El sistema detecta automáticamente si es ventas o inventario.
    El selector manual ya no es necesario.
    """
    return html.Div(
        [
            # Mensaje de detección automática
            html.Div(
                [
                    html.I(className="fas fa-magic me-2 text-info"),
                    html.Span([
                        html.Strong("Detección automática: "),
                        "Sube tu archivo de ",
                        html.Strong("ventas"),
                        " o ",
                        html.Strong("inventario"),
                        " - el sistema lo detectará automáticamente."
                    ]),
                ],
                className="small text-muted",
            ),
            # Selector oculto para mantener compatibilidad con callbacks existentes
            dbc.RadioItems(
                id="upload-type-selector",
                options=[
                    {"label": "Ventas", "value": "sales"},
                    {"label": "Inventario", "value": "inventory"},
                ],
                value="sales",
                style={"display": "none"},  # Oculto - detección automática
            ),
            # Contenedor de opciones de inventario (oculto)
            html.Div(
                id="inventory-options-container",
                style={"display": "none"},
                children=[
                    dcc.DatePickerSingle(
                        id="inventory-snapshot-date",
                        display_format="DD/MM/YYYY",
                        first_day_of_week=1,
                        style={"display": "none"},
                    ),
                ],
            ),
        ],
        className="mb-2",
    )


def get_upload_layout() -> html.Div:
    """
    Obtener layout compacto de la página de carga de archivos.
    Layout 2 columnas: Upload + Ayuda ERP | KPIs + Historial

    Returns:
        Componente html.Div con el layout de upload
    """

    return html.Div(
        [
            # Header compacto
            html.Div(
                [
                    html.H1([html.I(className="fas fa-upload me-2"), "Cargar Datos"], className="mb-1"),
                    html.P("Sube archivos de tu sistema ERP para analizar las ventas", className="text-muted mb-2 small"),
                ],
            ),
            # === ROW 1: Layout 2 columnas - Upload + Info ===
            dbc.Row(
                [
                    # COLUMNA IZQUIERDA: Upload zone
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [html.I(className="fas fa-upload me-2"), "Subir Archivo"],
                                        className="py-2"
                                    ),
                                    dbc.CardBody(
                                        [
                                            # Selector de tipo de archivo (Issue #476)
                                            create_upload_type_selector(),
                                            # Upload con validación y accesibilidad
                                            html.Div(
                                                [
                                                    dcc.Loading(
                                                        id="upload-loading",
                                                        type="circle",
                                                        color=DS_COLORS["info"],
                                                        children=[
                                                            dcc.Upload(
                                                                id="upload-data",
                                                                children=html.Div(
                                                                    [
                                                                        html.Div(
                                                                            [
                                                                                html.I(
                                                                                    className="fas fa-cloud-upload-alt",
                                                                                    style={
                                                                                        "fontSize": "48px",
                                                                                        "color": DS_COLORS["info"],
                                                                                        "marginBottom": "12px",
                                                                                    },
                                                                                )
                                                                            ],
                                                                            **{"aria-hidden": "true"},
                                                                        ),
                                                                        html.H6(
                                                                            "Arrastra tu archivo aquí o haz clic",
                                                                            className="mb-1",
                                                                            style={"color": DS_COLORS["text_primary"]},
                                                                        ),
                                                                        html.P(
                                                                            ".csv, .xlsx, .xls | Máx: 50MB",
                                                                            className="mb-0 small text-muted",
                                                                            id="upload-format-info",
                                                                        ),
                                                                    ],
                                                                    style={
                                                                        "display": "flex",
                                                                        "flexDirection": "column",
                                                                        "alignItems": "center",
                                                                        "justifyContent": "center",
                                                                        "height": "100%",
                                                                        "padding": "16px",
                                                                    },
                                                                ),
                                                                style={
                                                                    "height": "160px",
                                                                    "borderWidth": "2px",
                                                                    "borderStyle": "dashed",
                                                                    "borderRadius": BORDER_RADIUS["lg"],
                                                                    "borderColor": DS_COLORS["info"],
                                                                    "textAlign": "center",
                                                                    "backgroundColor": DS_COLORS["bg_tertiary"],
                                                                    "cursor": "pointer",
                                                                    "transition": TRANSITIONS["normal"],
                                                                },
                                                                multiple=False,
                                                                accept=".csv,.xlsx,.xls",
                                                                max_size=52428800,
                                                            )
                                                        ],
                                                    )
                                                ],
                                                **{
                                                    "role": "region",
                                                    "aria-label": "Zona de carga de archivos ERP",
                                                },
                                            ),
                                            # Feedback containers (compactos)
                                            html.Div(id="upload-immediate-feedback", className="mt-2", style={"display": "none"}, **{"aria-live": "assertive"}),
                                            html.Div(id="file-preview-container", className="mt-2", style={"display": "none"}, **{"aria-live": "polite"}),
                                            html.Div(id="upload-alerts", className="mt-2"),
                                            html.Div(id="upload-progress-container", className="mt-2", style={"display": "none"}, **{"aria-live": "polite"}),
                                        ],
                                        className="py-2"
                                    ),
                                ],
                                className="shadow-sm h-100",
                            ),
                        ],
                        width=12,
                        lg=7,
                        className="mb-3 mb-lg-0",
                    ),
                    # COLUMNA DERECHA: Instrucciones ERP + KPIs
                    dbc.Col(
                        [
                            # Card Instrucciones ERP (compacta)
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [html.I(className="fas fa-info-circle me-2"), "Instrucciones"],
                                        className="py-2"
                                    ),
                                    dbc.CardBody(
                                        html.Div(id="erp-help-accordion-container"),
                                        className="py-2",
                                        style={"fontSize": "0.85rem"}
                                    )
                                ],
                                className="shadow-sm mb-3"
                            ),
                            # Card KPIs (compacta)
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [html.I(className="fas fa-chart-bar me-2"), "Resumen"],
                                        className="py-2"
                                    ),
                                    dbc.CardBody(
                                        html.Div(
                                            id="upload-system-status",
                                            children=[create_loading_spinner(size="sm", text="Cargando...")],
                                        ),
                                        className="py-2"
                                    )
                                ],
                                className="shadow-sm"
                            ),
                            # Resumen último upload
                            html.Div(id="last-upload-summary", className="mt-3"),
                        ],
                        width=12,
                        lg=5,
                    ),
                ],
                className="mb-3",
            ),
            # === ROW 2: Historial de Cargas ===
            dbc.Card(
                [
                    dbc.CardHeader(
                        [html.I(className="fas fa-history me-2"), "Historial de Cargas"],
                        className="py-2"
                    ),
                    dbc.CardBody(
                        html.Div(
                            id="upload-history-container",
                            children=[create_loading_spinner(size="sm", text="Cargando historial...")]
                        ),
                        className="py-2"
                    )
                ],
                className="shadow-sm"
            ),
            # === MODAL: Confirmación de eliminación ===
            dbc.Modal(
                [
                    dbc.ModalHeader(
                        html.Div(
                            [
                                html.I(className="fas fa-exclamation-triangle me-2 text-danger"),
                                html.Span("Confirmar Eliminación"),
                            ],
                            className="d-flex align-items-center",
                        ),
                        close_button=True,
                        className="bg-danger bg-opacity-10",
                    ),
                    dbc.ModalBody(
                        [
                            dbc.Alert(
                                html.Span([
                                    html.I(className="fas fa-exclamation-triangle me-2"),
                                    html.Strong("¡Atención! "),
                                    "Esta acción eliminará permanentemente el archivo y todos sus datos de ventas asociados.",
                                ]),
                                color="danger",
                                className="mb-3",
                            ),
                            # Preview de datos a eliminar (cargado dinámicamente)
                            html.Div(id="delete-upload-preview", className="mb-3"),
                        ]
                    ),
                    dbc.ModalFooter(
                        [
                            dbc.Button(
                                "Cancelar",
                                id="delete-upload-cancel",
                                color="secondary",
                                outline=True,
                                className="me-2",
                            ),
                            dbc.Button(
                                id="delete-upload-confirm",
                                color="danger",
                                children=html.Span([html.I(className="fas fa-trash-alt me-2"), "Eliminar"]),
                            ),
                        ]
                    ),
                ],
                id="delete-upload-modal",
                is_open=False,
                centered=True,
                backdrop="static",
                keyboard=False,
                size="md",
            ),
            # Stores para manejo de estado
            dcc.Store(id="upload-history-data"),
            dcc.Store(id="delete-upload-target", storage_type="memory"),
            dcc.Store(id="delete-in-progress", storage_type="memory"),  # Para patrón 2-fases
            dcc.Interval(id="app-load", interval=60000, n_intervals=0),  # 60s para reducir carga
        ]
    )


def create_upload_success_alert(filename: str, stats: dict) -> dbc.Alert:
    """Crear alerta de éxito después de upload completado."""
    total_rows = stats.get("total_rows", 0)
    processed_rows = stats.get("rows_processed", 0)

    return dbc.Alert(
        [
            html.I(className="fas fa-check-circle me-2"),
            html.Strong("¡Archivo procesado!"),
            html.Span(f" {format_number(processed_rows).replace(',', '.')} de {format_number(total_rows).replace(',', '.')} filas", className="ms-1"),
        ],
        color="success",
        dismissable=True,
        duration=10000,
        className="py-2 mb-0"
    )


def create_upload_error_alert(filename: str, error_message: str) -> dbc.Alert:
    """Crear alerta de error en upload."""
    return dbc.Alert(
        [
            html.I(className="fas fa-exclamation-triangle me-2"),
            html.Strong("Error: "),
            html.Span(error_message[:100]),
        ],
        color="danger",
        dismissable=True,
        className="py-2 mb-0"
    )


def create_validation_warnings(warnings: list) -> html.Div:
    """Crear lista de advertencias de validación."""
    if not warnings:
        return html.Div()

    return dbc.Alert(
        [
            html.I(className="fas fa-exclamation-circle me-2"),
            html.Strong(f"{len(warnings)} advertencias"),
        ],
        color="warning",
        dismissable=True,
        className="py-2 mb-0"
    )


def get_supported_formats_info() -> html.Div:
    """Obtener información detallada de formatos soportados."""
    return html.Div(
        [
            html.Small("Formatos: CSV (;/,), Excel (.xlsx, .xls)", className="text-muted"),
            html.Small(" | Columnas: fecha, código, producto, cantidad, importe", className="text-muted d-block"),
        ]
    )
