# frontend/layouts/prescription.py
"""
Layout del Dashboard de Análisis de Ventas de Prescripción.

Este dashboard permite a las farmacias analizar sus ventas de prescripción
con clasificación terapéutica ATC (Anatomical Therapeutic Chemical).

REDISEÑO Issue #444: Sistema de 4 tabs:
1. Tab "Visión General y Evolución":
   - Filtros (fechas, categorías, grupos ATC, empleados, labs)
   - Panel contexto visual (treemap 15 categorías)
   - Evolución temporal con drill-down
   - Análisis comparativo YoY (waterfall + top contributors)
2. Tab "Clientes y patologías":
   - Próximamente: Análisis por pacientes y patologías
   - Filtro "Nivel de detalle ATC" aquí
3. Tab "Tendencias y estacionalidad":
   - Próximamente: Análisis de estacionalidad
4. Tab "Inventario" (Issue #471):
   - KPIs de inventario, ABC, tabla de rotación
"""

import dash_bootstrap_components as dbc
from dash import dcc, html
from dash_iconify import DashIconify

from components.base import BaseCard, Title
from components.inventory import create_inventory_tab
from components.prescription.heatmap_hourly import create_heatmap_container
from components.prescription.monthly_index_chart import create_monthly_index_container
from components.prescription.trend_decomposition_chart import (
    create_decomposition_container,
    create_forecast_container,
)
# Issue #532: category_patterns_chart removed - replaced by ATC filter
from components.prescription.stockout_risk_table import (
    create_stockout_risk_container,
)
from components.prescription.anomaly_alerts import (
    create_anomaly_alerts_container,
)
from components.skeleton_loader import create_skeleton_loader, create_table_skeleton
from styles.design_tokens import COLORS, SPACING
from utils.constants import PRESCRIPTION_CATEGORIES, ATC_LEVEL_1_CODES, format_category_name


def _create_global_date_filter():
    """
    Filtro de fechas global visible en todos los tabs (Issue #537).

    Proporciona consistencia al usuario mostrando siempre el período
    de análisis seleccionado, independientemente del tab activo.

    Incluye banner de restricción FREE tier (Issue #420).
    """
    return dbc.Card(
        dbc.CardBody(
            [
                dbc.Row(
                    [
                        # Icono y label
                        dbc.Col(
                            html.Div(
                                [
                                    html.I(
                                        className="fas fa-calendar-alt me-2",
                                        style={"color": COLORS["primary"], "fontSize": "1.2rem"},
                                    ),
                                    html.Span(
                                        "Período de Análisis",
                                        style={"fontWeight": "600", "fontSize": "0.95rem"},
                                    ),
                                ],
                                className="d-flex align-items-center",
                            ),
                            width="auto",
                            className="d-flex align-items-center",
                        ),
                        # DatePickerRange
                        dbc.Col(
                            dcc.DatePickerRange(
                                id="prescription-date-range",
                                display_format="DD/MM/YYYY",
                                start_date_placeholder_text="Cargando...",
                                end_date_placeholder_text="Cargando...",
                                calendar_orientation="horizontal",
                                first_day_of_week=1,
                                month_format="MMMM YYYY",
                                className="w-100",
                                clearable=False,
                                persistence=True,
                                persistence_type="session",
                            ),
                            width=12,
                            md=5,
                            lg=4,
                            className="mt-2 mt-md-0",
                        ),
                        # Slider visual de fechas
                        dbc.Col(
                            dcc.RangeSlider(
                                id="prescription-date-slider",
                                min=0,
                                max=100,  # Se configura dinámicamente
                                step=1,
                                marks={},  # Se configura dinámicamente con meses
                                value=[0, 100],
                                tooltip={"placement": "bottom", "always_visible": False},
                                updatemode="mouseup",  # Issue #540: Solo actualizar al soltar (reduce callbacks)
                                persistence=True,
                                persistence_type="session",
                            ),
                            width=12,
                            md=True,  # Flex grow
                            className="mt-2 mt-md-0 d-none d-lg-block",  # Oculto en móvil/tablet
                        ),
                    ],
                    className="align-items-center g-3",
                ),
                # Issue #420: FREE tier date restriction banner (full width below)
                html.Div(
                    id="prescription-date-restriction-banner",
                    className="mt-2",
                    style={"display": "none"},
                ),
            ],
            className="py-2 px-3",
        ),
        className="mb-3 shadow-sm",
        style={
            "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
            "border": f"1px solid {COLORS.get('border_light', '#dee2e6')}",
            "position": "relative",  # Issue #548: Contexto de apilamiento para calendario
            "zIndex": 1100,  # Mayor que Tabs content para que calendario aparezca encima
        },
    )


def get_prescription_layout():
    """
    Layout completo del dashboard de prescripción con sistema de 3 tabs.

    Issue #444: Reorganización en tabs:
    - Tab 1: Visión General y Evolución (contenido actual)
    - Tab 2: Clientes y patologías (próximamente)
    - Tab 3: Tendencias y estacionalidad (próximamente)

    Returns:
        html.Div: Layout estructurado con tabs y componentes
    """

    return html.Div(
        [
            # Stores definidos en app.py skeleton (REGLA #0.5):
            # - prescription-date-range-store
            # - prescription-products-store
            # - prescription-temporal-drill-store
            # Header
            create_header(),
            # ===================================================================
            # FILTRO GLOBAL DE FECHAS (Issue #537)
            # Visible en todos los tabs para consistencia
            # ===================================================================
            _create_global_date_filter(),
            # ===================================================================
            # SISTEMA DE TABS (Issue #444)
            # ===================================================================
            dbc.Tabs(
                id="prescription-tabs",
                active_tab="tab-overview",
                children=[
                    # TAB 1: Visión General y Evolución
                    dbc.Tab(
                        label="Visión General y Evolución",
                        tab_id="tab-overview",
                        children=[
                            html.Div(
                                [
                                    # Layout superior 50/50: Filtros + Contexto
                                    dbc.Row(
                                        [
                                            dbc.Col([create_filters_tab1()], width=12, lg=6, className="mb-4"),
                                            dbc.Col([create_context_panel()], width=12, lg=6, className="mb-4"),
                                        ]
                                    ),
                                    # Sección principal 50/50: Evolución + Tabla
                                    dbc.Row(
                                        [
                                            dbc.Col([create_evolution_section()], width=12, lg=6, className="mb-4"),
                                            dbc.Col([create_products_table_section()], width=12, lg=6, className="mb-4"),
                                        ],
                                    ),
                                    # Sección Análisis Comparativo YoY
                                    html.Section(
                                        [
                                            html.H4(
                                                [
                                                    DashIconify(icon="mdi:chart-timeline-variant", width=24, color=COLORS["primary"], className="me-2"),
                                                    "Análisis Comparativo de Crecimiento (YoY)",
                                                ],
                                                className="mb-3",
                                            ),
                                            html.P(
                                                "Comparación año sobre año de las ventas de prescripción",
                                                className="text-muted mb-4",
                                            ),
                                            # Issue #458: Waterfalls lado a lado (categoría + producto)
                                            dbc.Row(
                                                [
                                                    dbc.Col([create_waterfall_section()], width=12, lg=6, className="mb-4"),
                                                    dbc.Col([create_product_waterfall_section()], width=12, lg=6, className="mb-4"),
                                                ],
                                            ),
                                            # Issue #458: Top Contributors full-width debajo
                                            dbc.Row(
                                                [
                                                    dbc.Col([create_top_contributors_section()], width=12, className="mb-4"),
                                                ],
                                            ),
                                        ],
                                        id="comparative-analysis-section",
                                        className="mt-4",
                                    ),
                                ],
                                className="pt-3",
                            ),
                        ],
                    ),
                    # TAB 2: Clientes y patologías
                    dbc.Tab(
                        label="Clientes y patologías",
                        tab_id="tab-clients",
                        children=[
                            html.Div(
                                [
                                    # Filtro ATC movido aquí
                                    create_atc_filter_section(),
                                    # Placeholder próximamente
                                    create_coming_soon_placeholder(
                                        "Análisis por clientes y patologías",
                                        "Esta sección permitirá analizar ventas por paciente, "
                                        "patologías y grupos terapéuticos ATC con detalle.",
                                    ),
                                ],
                                className="pt-3",
                            ),
                        ],
                    ),
                    # TAB 3: Tendencias y estacionalidad (Issue #489, #502)
                    # NOTE: dbc.Tab label solo acepta strings en dbc 2.0.4
                    # Los badges dinámicos se muestran dentro del contenido del tab
                    dbc.Tab(
                        label="Tendencias y estacionalidad",
                        tab_id="tab-tendencias",
                        children=[
                            html.Div(
                                [
                                    create_seasonality_tab_content(),
                                ],
                                className="pt-3",
                            ),
                        ],
                    ),
                    # TAB 4: Inventario (Issue #471)
                    dbc.Tab(
                        label="Inventario",
                        tab_id="tab-inventario",
                        children=[
                            create_inventory_tab(
                                id_prefix="prescription",
                                product_type="medicamento",
                                title="Inventario Medicamentos",
                                description="Análisis de rotación y rentabilidad del stock de medicamentos",
                            ),
                        ],
                    ),
                ],
                className="mb-3",
            ),
        ],
        style={"padding": "20px"},
    )


def create_header():
    """Header con título y descripción del dashboard"""
    return html.Div(
        [
            html.H1(
                [
                    html.I(className="fas fa-pills me-3", style={"color": COLORS["primary"]}),
                    "Análisis de Ventas de Prescripción",
                ],
                style={"marginBottom": "10px"},
            ),
            html.P(
                "Análisis terapéutico de ventas con clasificación ATC (Anatomical Therapeutic Chemical)",
                className="text-muted",
                style={"fontSize": "16px"},
            ),
        ],
        className="mb-4",
    )


def create_coming_soon_placeholder(title: str, description: str = "") -> dbc.Card:
    """
    Crea un placeholder visual para secciones próximamente disponibles.

    Args:
        title: Título de la sección
        description: Descripción opcional

    Returns:
        dbc.Card: Tarjeta con mensaje "Próximamente"
    """
    return dbc.Card(
        [
            dbc.CardBody(
                [
                    html.Div(
                        [
                            DashIconify(
                                icon="mdi:clock-outline",
                                width=64,
                                color=COLORS["text_secondary"],
                                className="mb-3",
                            ),
                            html.H4(
                                title,
                                className="text-muted mb-2",
                            ),
                            html.P(
                                description or "Esta funcionalidad estará disponible próximamente.",
                                className="text-muted mb-3",
                                style={"maxWidth": "500px"},
                            ),
                            dbc.Badge(
                                "Próximamente",
                                color="info",
                                pill=True,
                                className="px-3 py-2",
                            ),
                        ],
                        className="text-center py-5",
                    ),
                ],
            ),
        ],
        className="border-dashed",
        style={
            "borderStyle": "dashed",
            "borderColor": COLORS["border_light"],
            "backgroundColor": COLORS["bg_secondary"],
        },
    )


def create_seasonality_tab_content() -> html.Div:
    """
    Crea el contenido del Tab 3: Tendencias y Estacionalidad (Issue #489).

    Layout:
    - Row 0: ATC Filter (Issue #532)
    - Row 1: KPIs (Tendencia YoY, Próximo Pico, Anomalías)
    - Row 2: Heatmap Día×Hora (50%) + Placeholder STL (50%)
    - Row 3: Índice Mensual (full width)

    Returns:
        html.Div: Contenido completo del tab
    """
    # Build ATC filter options from constants
    atc_options = [{"label": "Todos los grupos ATC", "value": ""}]
    for atc in ATC_LEVEL_1_CODES:
        atc_options.append({
            "label": f"{atc['code']} - {atc['name']}",
            "value": atc["code"]
        })

    return html.Div(
        [
            # =========================================================
            # ROW 0: Period Display + ATC Filter (Issue #526, #532)
            # =========================================================
            dbc.Row(
                [
                    # Issue #526: Indicador de período analizado
                    dbc.Col(
                        dbc.Card(
                            dbc.CardBody(
                                [
                                    html.Div(
                                        [
                                            DashIconify(
                                                icon="mdi:calendar-range",
                                                width=24,
                                                color=COLORS["info"],
                                                className="me-2",
                                            ),
                                            html.Span(
                                                "Período de Análisis",
                                                style={"fontWeight": "600"},
                                            ),
                                        ],
                                        className="d-flex align-items-center mb-2",
                                    ),
                                    html.Div(
                                        id="seasonality-period-display",
                                        children=[
                                            dbc.Badge(
                                                "Cargando...",
                                                color="secondary",
                                                className="me-2",
                                            ),
                                        ],
                                        className="d-flex align-items-center flex-wrap",
                                    ),
                                    html.Small(
                                        "Los gráficos de heatmap e índice mensual usan este período. "
                                        "STL y Forecast usan todo el historial disponible.",
                                        className="text-muted mt-2 d-block",
                                    ),
                                ],
                            ),
                            className="shadow-sm",
                            style={
                                "backgroundColor": COLORS.get("bg_secondary", "#f8f9fa"),
                                "border": f"1px solid {COLORS.get('info', '#17a2b8')}40",
                            },
                        ),
                        width=12,
                        lg=6,
                        className="mb-3 mb-lg-0",
                    ),
                    # ATC Filter (Issue #532)
                    dbc.Col(
                        dbc.Card(
                            dbc.CardBody(
                                [
                                    html.Div(
                                        [
                                            DashIconify(
                                                icon="mdi:medical-bag",
                                                width=24,
                                                color=COLORS["primary"],
                                                className="me-2",
                                            ),
                                            html.Span(
                                                "Filtrar por Grupo Terapéutico ATC",
                                                style={"fontWeight": "600"},
                                            ),
                                        ],
                                        className="d-flex align-items-center mb-3",
                                    ),
                                    dcc.Dropdown(
                                        id="seasonality-atc-filter",
                                        options=atc_options,
                                        value="",
                                        placeholder="Seleccionar grupo ATC...",
                                        clearable=True,
                                        style={"minWidth": "300px"},
                                        className="w-100 w-md-auto",
                                    ),
                                    html.Small(
                                        "Filtra todos los gráficos de estacionalidad por código ATC nivel 1 (A-V)",
                                        className="text-muted mt-2 d-block",
                                    ),
                                ],
                            ),
                            className="shadow-sm",
                        ),
                        width=12,
                        lg=6,
                    ),
                ],
                className="mb-4",
            ),
            # =========================================================
            # ROW 1: KPIs de Estacionalidad
            # =========================================================
            dbc.Row(
                [
                    # KPI 1: Tendencia YoY
                    dbc.Col(
                        dbc.Card(
                            dbc.CardBody(
                                [
                                    html.Div(
                                        [
                                            DashIconify(
                                                icon="mdi:trending-up",
                                                width=24,
                                                color=COLORS["primary"],
                                                className="me-2",
                                            ),
                                            html.Span("Tendencia YoY", className="text-muted"),
                                        ],
                                        className="d-flex align-items-center mb-2",
                                    ),
                                    html.Div(
                                        [
                                            html.H3(
                                                id="seasonality-trend-yoy-value",
                                                children="--",
                                                className="mb-0 d-inline",
                                            ),
                                            html.I(
                                                id="seasonality-trend-direction-icon",
                                                className="mdi mdi-minus ms-2",
                                            ),
                                        ],
                                        className="d-flex align-items-center",
                                    ),
                                ],
                                className="text-center",
                            ),
                            className="h-100 shadow-sm",
                        ),
                        width=12,
                        lg=4,
                        className="mb-3",
                    ),
                    # KPI 2: Próximo Pico Estacional (Issue #527: tooltip explicativo)
                    dbc.Col(
                        dbc.Card(
                            dbc.CardBody(
                                [
                                    html.Div(
                                        [
                                            DashIconify(
                                                icon="mdi:calendar-star",
                                                width=24,
                                                color=COLORS["warning"],
                                                className="me-2",
                                            ),
                                            html.Span("Próximo Pico", className="text-muted"),
                                            DashIconify(
                                                id="next-peak-info-icon",
                                                icon="mdi:information-outline",
                                                width=16,
                                                color=COLORS["secondary"],
                                                className="ms-1",
                                                style={"cursor": "pointer", "opacity": "0.7"},
                                            ),
                                        ],
                                        className="d-flex align-items-center mb-2",
                                    ),
                                    html.H5(
                                        id="seasonality-next-peak-value",
                                        children="--",
                                        className="mb-0",
                                    ),
                                    # Issue #527: Tooltip explicativo del algoritmo
                                    dbc.Tooltip(
                                        html.Div([
                                            html.Strong("¿Cómo se calcula?"),
                                            html.Br(),
                                            "Basado en patrones estacionales de farmacias españolas:",
                                            html.Br(),
                                            html.Br(),
                                            html.Strong("Antigripales"), " (Nov-Feb): ",
                                            "Temporada de gripe y resfriados.",
                                            html.Br(),
                                            html.Strong("Antihistamínicos"), " (Mar-Jun): ",
                                            "Temporada de alergias primaverales.",
                                            html.Br(),
                                            html.Br(),
                                            html.Em("Los días se calculan hasta el inicio del próximo pico."),
                                        ]),
                                        target="next-peak-info-icon",
                                        placement="bottom",
                                        style={"maxWidth": "300px"},
                                    ),
                                ],
                                className="text-center",
                            ),
                            className="h-100 shadow-sm",
                        ),
                        width=12,
                        lg=4,
                        className="mb-3",
                    ),
                    # KPI 3: Anomalías Detectadas
                    dbc.Col(
                        dbc.Card(
                            dbc.CardBody(
                                [
                                    html.Div(
                                        [
                                            DashIconify(
                                                icon="mdi:alert-circle-outline",
                                                width=24,
                                                color=COLORS["danger"],
                                                className="me-2",
                                            ),
                                            html.Span("Anomalías", className="text-muted"),
                                        ],
                                        className="d-flex align-items-center mb-2",
                                    ),
                                    html.H3(
                                        id="seasonality-anomalies-value",
                                        children="0",
                                        className="mb-0",
                                    ),
                                ],
                                className="text-center",
                            ),
                            className="h-100 shadow-sm",
                        ),
                        width=12,
                        lg=4,
                        className="mb-3",
                    ),
                ],
                className="mb-4",
            ),
            # =========================================================
            # Issue #502: Export Button
            # =========================================================
            dbc.Row(
                [
                    dbc.Col(
                        dbc.Button(
                            html.Span([
                                DashIconify(
                                    icon="mdi:download",
                                    width=20,
                                    className="me-2",
                                ),
                                "Exportar Previsión Mensual",
                            ]),
                            id="seasonality-export-btn",
                            color="success",
                            outline=True,
                            size="sm",
                            className="float-end",
                        ),
                        width=12,
                        className="text-end mb-3",
                    ),
                ],
            ),
            # =========================================================
            # ROW 2: Heatmap + Índice Mensual
            # =========================================================
            html.Div(
                id="monthly-index-data-quality-alert",
                className="mb-3",
            ),
            dbc.Row(
                [
                    # Heatmap Día × Hora (50%)
                    dbc.Col(
                        create_heatmap_container(),
                        width=12,
                        lg=6,
                        className="mb-4",
                    ),
                    # Índice Mensual (50%)
                    dbc.Col(
                        create_monthly_index_container(),
                        width=12,
                        lg=6,
                        className="mb-4",
                    ),
                ],
            ),
            # =========================================================
            # ROW 3: STL Decomposition (Issue #498)
            # =========================================================
            html.Div(
                id="stl-data-quality-alert",
                className="mb-3",
            ),
            html.Div(
                id="stl-summary-container",
                className="mb-3",
            ),
            dbc.Row(
                [
                    dbc.Col(
                        create_decomposition_container(),
                        width=12,
                    ),
                ],
                className="mb-4",
            ),
            # =========================================================
            # ROW 4: Forecast Holt-Winters (Issue #498)
            # =========================================================
            html.Div(
                id="forecast-summary-container",
                className="mb-3",
            ),
            dbc.Row(
                [
                    dbc.Col(
                        create_forecast_container(),
                        width=12,
                    ),
                ],
                className="mb-4",
            ),
            # Issue #532: Category Patterns section removed, replaced by ATC filter
            # =========================================================
            # ROW 5: Stock-out Risk Matrix (Issue #500)
            # =========================================================
            dbc.Row(
                [
                    dbc.Col(
                        create_stockout_risk_container(),
                        width=12,
                    ),
                ],
                className="mb-4",
            ),
            # =========================================================
            # ROW 7: Anomaly Detection Alerts (Issue #501)
            # =========================================================
            dbc.Row(
                [
                    dbc.Col(
                        create_anomaly_alerts_container(),
                        width=12,
                    ),
                ],
                className="mb-4",
            ),
            # =========================================================
            # ROW 8: Cuadro Explicativo
            # =========================================================
            dbc.Card(
                dbc.CardBody(
                    [
                        html.Div(
                            [
                                DashIconify(
                                    icon="mdi:information-outline",
                                    width=24,
                                    color=COLORS["info"],
                                    className="me-2",
                                ),
                                html.Span(
                                    "Guía de Interpretación",
                                    style={"fontWeight": "600", "fontSize": "1.1rem"},
                                ),
                            ],
                            className="d-flex align-items-center mb-3",
                        ),
                        dbc.Row(
                            [
                                dbc.Col(
                                    [
                                        html.H6(
                                            "📊 Índice Mensual",
                                            className="mb-2",
                                            style={"fontWeight": "600"},
                                        ),
                                        html.P(
                                            "Muestra cómo varían las ventas según el mes del año. "
                                            "Un índice > 100 indica meses con ventas superiores a la media anual "
                                            "(ej: enero/febrero por gripes), mientras que < 100 indica meses más flojos "
                                            "(ej: agosto por vacaciones).",
                                            className="text-muted small mb-0",
                                        ),
                                    ],
                                    width=12,
                                    lg=4,
                                    className="mb-3 mb-lg-0",
                                ),
                                dbc.Col(
                                    [
                                        html.H6(
                                            "📈 Descomposición STL",
                                            className="mb-2",
                                            style={"fontWeight": "600"},
                                        ),
                                        html.P(
                                            "Separa las ventas en tres componentes: "
                                            "Tendencia (dirección general del negocio), "
                                            "Estacionalidad (patrones que se repiten cada año) y "
                                            "Residuo (variaciones atípicas o ruido). "
                                            "Requiere mínimo 12 meses de datos.",
                                            className="text-muted small mb-0",
                                        ),
                                    ],
                                    width=12,
                                    lg=4,
                                    className="mb-3 mb-lg-0",
                                ),
                                dbc.Col(
                                    [
                                        html.H6(
                                            "🔮 Forecast",
                                            className="mb-2",
                                            style={"fontWeight": "600"},
                                        ),
                                        html.P(
                                            "Proyección de ventas para los próximos 6 meses usando Holt-Winters. "
                                            "Tiene en cuenta la tendencia actual y los patrones estacionales. "
                                            "La banda sombreada indica el intervalo de confianza (80%).",
                                            className="text-muted small mb-0",
                                        ),
                                    ],
                                    width=12,
                                    lg=4,
                                ),
                            ],
                        ),
                    ],
                ),
                className="shadow-sm",
                style={
                    "backgroundColor": COLORS["bg_secondary"],
                    "border": f"1px solid {COLORS['border_medium']}",
                },
            ),
        ],
    )


def create_atc_filter_section() -> dbc.Card:
    """
    Sección de filtro de Nivel de detalle ATC para Tab 2.

    Movido desde Tab 1 según Issue #444.

    Returns:
        dbc.Card: Panel con selector de nivel ATC
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-filter me-2", style={"color": COLORS["info"]}),
                    html.Span("Filtros de Análisis ATC", style={"fontWeight": "600"}),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.H6(
                                        "🔬 Nivel de Detalle ATC",
                                        className="mb-2",
                                        style={"fontWeight": "600"},
                                    ),
                                    dcc.Dropdown(
                                        id="atc-level-filter",
                                        options=[
                                            {"label": "Nivel 1 - Grupo Anatómico (A, C, N...)", "value": 1},
                                            {"label": "Nivel 2 - Grupo Terapéutico (A02, C09...)", "value": 2},
                                            {"label": "Nivel 3 - Subgrupo Farmacológico (A02B...)", "value": 3},
                                            {"label": "Nivel 4 - Subgrupo Químico (A02BC...)", "value": 4},
                                            {"label": "Nivel 5 - Principio Activo (A02BC01...)", "value": 5},
                                        ],
                                        value=1,
                                        clearable=False,
                                        className="w-100",
                                        persistence=True,
                                        persistence_type="session",
                                    ),
                                    html.Small(
                                        "Selecciona el nivel de granularidad para el análisis por clasificación ATC",
                                        className="text-muted mt-2 d-block",
                                    ),
                                ],
                                width=12,
                                lg=6,
                            ),
                        ],
                    ),
                ],
                style={"padding": SPACING["m"]},
            ),
        ],
        className="mb-4",
    )


def create_filters_tab1():
    """
    Panel de filtros específicos para Tab 1 (Issue #444).

    Nota: El filtro de fechas se movió al header global (Issue #537).
    Este panel solo contiene filtros específicos del Tab 1.

    Filtros incluidos:
    - Grupos ATC
    - Laboratorios
    - Empleados (PRO only)

    NOTA: "Nivel de detalle ATC" movido a Tab 2 según Issue #444.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-filter me-2", style={"color": COLORS["info"]}),
                    html.Span("Filtros Adicionales", style={"fontWeight": "600"}),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # Fila 1: Grupos ATC (ancho completo - Categorías movidas al gráfico de evolución)
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.H6(
                                        "🧬 Grupos ATC",
                                        className="mb-2",
                                        style={"fontWeight": "600"},
                                    ),
                                    dcc.Dropdown(
                                        id="prescription-atc-codes-filter",
                                        options=[
                                            {"label": f"{atc['code']} - {atc['name']}", "value": atc["code"]}
                                            for atc in ATC_LEVEL_1_CODES
                                        ],
                                        multi=True,
                                        placeholder="Todos los grupos ATC",
                                        className="w-100",
                                        clearable=True,
                                        searchable=True,
                                        persistence=True,
                                        persistence_type="session",
                                    ),
                                ],
                                width=12,
                                lg=6,
                                className="mb-3",
                            ),
                            # Placeholder oculto para mantener ID requerido por callbacks existentes
                            # FIX: Añadir multi=True para evitar error "Invalid prop value of type array"
                            dcc.Dropdown(
                                id="prescription-categories-filter",
                                options=[],
                                value=[],
                                multi=True,
                                style={"display": "none"},
                            ),
                        ],
                        className="g-3",
                    ),
                    # Fila 3: Laboratorios + Empleados
                    dbc.Row(
                        [
                            # Filtro de Laboratorios (50%)
                            dbc.Col(
                                [
                                    html.H6(
                                        "🏭 Filtro de Laboratorios",
                                        className="mb-2",
                                        style={"fontWeight": "600"},
                                    ),
                                    dcc.Dropdown(
                                        id="prescription-laboratories-filter",
                                        multi=True,
                                        placeholder="Todos los laboratorios",
                                        options=[],
                                        value=[],
                                        className="w-100",
                                        persistence=True,
                                        persistence_type="session",
                                        clearable=True,
                                        searchable=True,
                                    ),
                                ],
                                width=12,
                                lg=6,
                                className="mb-3",
                            ),
                            # Filtro de empleados - PRO tier
                            dbc.Col(
                                [
                                    html.H6(
                                        "👥 Filtro de Empleados",
                                        className="mb-2",
                                        style={"fontWeight": "600"},
                                    ),
                                    dcc.Dropdown(
                                        id="prescription-employee-filter",
                                        multi=True,
                                        placeholder="Todos los empleados",
                                        options=[],
                                        value=[],
                                        className="w-100",
                                        persistence=True,
                                        persistence_type="session",
                                        clearable=True,
                                        searchable=True,
                                    ),
                                    html.Div(
                                        [
                                            dbc.Badge(
                                                html.Span([
                                                    html.I(className="fas fa-crown me-1"),
                                                    "Disponible en PRO",
                                                ]),
                                                id="prescription-employee-filter-badge",
                                                color="secondary",
                                                className="mt-2",
                                                pill=True,
                                                style={"fontSize": "0.7rem", "opacity": "0.8"},
                                            ),
                                            dbc.Tooltip(
                                                "El filtro de empleados permite analizar ventas por vendedor. "
                                                "Esta función está disponible en planes PRO y MAX.",
                                                target="prescription-employee-filter-badge",
                                                placement="top",
                                            ),
                                        ]
                                    ),
                                ],
                                width=12,
                                lg=6,
                                className="mb-3",
                            ),
                        ],
                        className="g-3",
                    ),
                    # Fila 4: Botón Aplicar Filtros
                    dbc.Row(
                        dbc.Col(
                            dbc.Button(
                                html.Span([
                                    html.I(className="fas fa-check me-2"),
                                    "Aplicar Filtros",
                                ]),
                                id="prescription-apply-filters-btn",
                                color="primary",
                                className="px-4",
                            ),
                            width="auto",
                            className="ms-auto",
                        ),
                        className="mt-3",
                    ),
                ],
                style={"padding": SPACING["m"]},
            ),
        ],
        className="mb-3",
    )


# Alias para compatibilidad con código antiguo
def create_filters():
    """Alias de create_filters_tab1 para compatibilidad."""
    return create_filters_tab1()


def create_context_panel():
    """
    Panel de contexto visual con treemap (Issue #436, #444).

    Muestra distribución jerárquica de ventas totales de la farmacia
    mediante un treemap interactivo. Igual que /generics.

    Issue #444: Ahora muestra jerarquía completa:
    - Total Ventas
      - Con Conjunto Homogéneo (sustituible)
        - En Vademécum (analizable)
      - Sin Conjunto Homogéneo

    Returns:
        dbc.Card: Panel con treemap y métricas compactas
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-chart-pie me-2", style={"color": COLORS["info"]}),
                    html.Span("Contexto de Ventas de Prescripción", style={"fontWeight": "600"}),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # =========================================================
                    # TREEMAP: Categorías de Prescripción
                    # =========================================================
                    dcc.Loading(
                        id="prescription-context-treemap-loading",
                        type="default",
                        children=dcc.Graph(
                            id="prescription-context-treemap",
                            config={"displayModeBar": False},
                            style={"height": "280px"},
                            figure={
                                "data": [],
                                "layout": {
                                    "margin": {"t": 10, "b": 10, "l": 10, "r": 10},
                                    "annotations": [{
                                        "text": "Cargando contexto...",
                                        "showarrow": False,
                                        "font": {"size": 14, "color": COLORS["text_secondary"]}
                                    }]
                                }
                            }
                        ),
                    ),
                    html.Hr(className="my-2"),
                    # =========================================================
                    # Issue #436: MÉTRICAS COMPACTAS
                    # =========================================================
                    dcc.Loading(
                        id="prescription-context-panel-loading",
                        type="default",
                        children=html.Div(
                            id="prescription-context-panel-content",
                            children=[
                                # Grid 2x2 para métricas compactas
                                dbc.Row(
                                    [
                                        dbc.Col(
                                            [
                                                html.Small("Ventas Totales:", className="text-muted"),
                                                html.H6(
                                                    id="prescription-context-total-sales",
                                                    children="--",
                                                    className="mb-0 text-primary",
                                                ),
                                                html.Small(
                                                    id="prescription-context-total-percentage",
                                                    children="--",
                                                    className="text-muted",
                                                    style={"fontSize": "0.75rem"},
                                                ),
                                            ],
                                            width=6,
                                            className="mb-2",
                                        ),
                                        dbc.Col(
                                            [
                                                html.Small("Unidades:", className="text-muted"),
                                                html.H6(
                                                    id="prescription-context-total-units",
                                                    children="--",
                                                    className="mb-0 text-info",
                                                ),
                                                html.Small(
                                                    id="prescription-context-units-percentage",
                                                    children="--",
                                                    className="text-muted",
                                                    style={"fontSize": "0.75rem"},
                                                ),
                                            ],
                                            width=6,
                                            className="mb-2",
                                        ),
                                    ]
                                ),
                            ],
                        ),
                    )
                ],
                style={"padding": SPACING["s"]},
            ),
        ],
        className="mb-3",
        style={"border": f"2px solid {COLORS['border_light']}"},
    )


def create_products_table_section():
    """
    Sección de acordeón de categorías con drill-down a productos (Issue #441).

    Acordeón dinámico que muestra las 15 categorías de prescripción:
    - Header: Nombre de categoría + Ventas (€) + % del total
    - Contenido expandible: Top 10 productos de esa categoría

    Se actualiza según los filtros seleccionados.

    Returns:
        dbc.Card: Panel con acordeón de categorías
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    DashIconify(icon="mdi:view-list", width=18, color=COLORS["primary"], className="me-2"),
                    html.Span("Categorías por Ventas", style={"fontWeight": "600"}),
                    html.Small(
                        " (click para ver productos)",
                        className="text-muted ms-2",
                        style={"fontSize": "0.75rem"},
                    ),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # Contenedor para el acordeón (poblado por callback)
                    # Store prescription-category-products-store definido en skeleton (app.py)
                    dcc.Loading(
                        id="prescription-products-table-loading",
                        type="default",
                        children=html.Div(
                            id="prescription-products-table-container",
                            children=create_table_skeleton(rows=10),
                            style={"minHeight": "400px", "maxHeight": "600px", "overflowY": "auto"},
                        ),
                    ),
                ],
                style={"padding": SPACING["s"]},
            ),
        ],
        className="h-100",
    )


def create_evolution_section():
    """
    Sección de gráfico de evolución temporal con drill-down.

    Gráfico de barras que muestra evolución de ventas agregadas con
    drill-down por períodos (Trimestre → Mes → Quincena).

    Returns:
        dbc.Card: Panel con controles drill-down y gráfico
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    DashIconify(icon="mdi:chart-bar", width=18, color=COLORS["success"], className="me-2"),
                    html.Span("Evolución Temporal de Ventas", style={"fontWeight": "600"}),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # =========================================================
                    # FILTRO DE CATEGORÍAS (movido desde Filtros de Análisis)
                    # Solo afecta al contenido de esta sección
                    # =========================================================
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.H6(
                                        "📋 Categorías de Prescripción",
                                        className="mb-2",
                                        style={"fontWeight": "600"},
                                    ),
                                    dcc.Dropdown(
                                        id="prescription-evolution-categories-filter",
                                        options=[
                                            {"label": format_category_name(cat), "value": cat}
                                            for cat in PRESCRIPTION_CATEGORIES
                                        ],
                                        multi=True,
                                        placeholder="Todas las categorías (mostrar todas)",
                                        className="w-100",
                                        clearable=True,
                                        searchable=True,
                                        persistence=True,
                                        persistence_type="session",
                                    ),
                                    html.Small(
                                        "Filtra las categorías a mostrar en el gráfico de evolución",
                                        className="text-muted mt-1 d-block",
                                    ),
                                ],
                                width=12,
                                className="mb-3",
                            ),
                        ],
                    ),
                    html.Hr(className="my-2"),
                    # Controles de drill-down (sin selector Top N - ahora se muestran todas o las filtradas)
                    dbc.Row(
                        [
                            # Hidden inputs para compatibilidad con callbacks existentes
                            dbc.Select(
                                id="ribbon-top-n",
                                options=[{"label": "Todas", "value": "100"}],
                                value="100",
                                style={"display": "none"},
                            ),
                            dbc.RadioItems(
                                id="evolution-chart-type",
                                value="ribbon",
                                options=[{"label": "", "value": "ribbon"}],
                                style={"display": "none"},
                            ),
                            # Indicador de nivel actual
                            dbc.Col(
                                [
                                    html.Small("Granularidad:", className="text-muted me-2"),
                                    dbc.Badge(
                                        id="prescription-drill-level-badge",
                                        children="Trimestre",
                                        color="info",
                                        pill=True,
                                    ),
                                ],
                                width="auto",
                            ),
                            # Breadcrumb de navegación
                            dbc.Col(
                                html.Div(
                                    id="prescription-drill-breadcrumb",
                                    className="text-muted small",
                                ),
                                width="auto",
                            ),
                            # Botón drill-up
                            dbc.Col(
                                dbc.Button(
                                    html.Span([
                                        DashIconify(icon="mdi:arrow-up", width=16, className="me-1"),
                                        "Subir Nivel",
                                    ]),
                                    id="prescription-drill-up-btn",
                                    color="secondary",
                                    size="sm",
                                    outline=True,
                                    disabled=True,  # Deshabilitado en nivel raíz
                                ),
                                width="auto",
                                className="ms-auto",
                            ),
                        ],
                        className="mb-3 align-items-center",
                    ),
                    html.Small(
                        "Haz clic en una barra para ver más detalle temporal",
                        className="text-muted d-block mb-2",
                    ),
                    # Gráfico de evolución
                    dcc.Loading(
                        id="prescription-evolution-chart-loading",
                        type="default",
                        children=dcc.Graph(
                            id="prescription-evolution-chart",
                            config={"displayModeBar": False},
                            style={"height": "400px"},
                            figure={
                                "data": [],
                                "layout": {
                                    "margin": {"t": 20, "b": 40, "l": 60, "r": 20},
                                    "annotations": [{
                                        "text": "Aplica filtros para ver la evolución temporal",
                                        "showarrow": False,
                                        "font": {"size": 14, "color": COLORS["text_secondary"]}
                                    }]
                                }
                            }
                        ),
                    ),
                ],
                style={"padding": SPACING["s"]},
            ),
        ],
        className="h-100",
    )


def create_waterfall_section():
    """
    Sección de análisis waterfall de crecimiento (Issue #444: Simplificado a YoY automático).

    Gráfico waterfall que muestra contribución de cada categoría
    al crecimiento/decrecimiento total. Siempre YoY (año sobre año).

    Issue #444: Simplificado - el período anterior se calcula automáticamente
    restando 1 año al rango de fechas seleccionado en los filtros principales.

    Returns:
        dbc.Card: Panel con display de períodos y gráfico waterfall
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    DashIconify(icon="mdi:chart-waterfall", width=18, color=COLORS["info"], className="me-2"),
                    html.Span("Variación por Categoría (YoY)", style={"fontWeight": "600"}),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # Issue #444: Stores ocultos para mantener compatibilidad con callbacks existentes
                    dcc.Store(id="waterfall-comparison-type", data="yoy"),
                    dcc.Store(id="waterfall-period-base", data=None),
                    # Fila: Display de períodos comparados (automático desde filtros)
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Small("Período actual:", className="text-muted d-block mb-1"),
                                    html.Div(
                                        id="waterfall-current-period-display",
                                        children="(según filtros aplicados)",
                                        className="border rounded px-3 py-2 bg-light",
                                        style={"fontSize": "0.9rem"},
                                    ),
                                ],
                                width="auto",
                                className="me-3",
                            ),
                            dbc.Col(
                                [
                                    html.Small("vs", className="text-muted d-block mb-1 text-center"),
                                    html.Div(style={"height": "38px", "display": "flex", "alignItems": "center"}),
                                ],
                                width="auto",
                                className="px-2",
                            ),
                            dbc.Col(
                                [
                                    html.Small("Período anterior (-1 año):", className="text-muted d-block mb-1"),
                                    html.Div(
                                        id="waterfall-period-comparison-display",
                                        children="(auto-calculado)",
                                        className="border rounded px-3 py-2 bg-light text-muted",
                                        style={"fontSize": "0.9rem"},
                                    ),
                                    # Store definido en app.py skeleton (REGLA #0.5)
                                ],
                                width="auto",
                            ),
                        ],
                        className="mb-3 align-items-end",
                    ),
                    html.Small(
                        "Top 10 categorías que más contribuyen al cambio en ventas",
                        className="text-muted d-block mb-3",
                    ),
                    # Contenedor para el gráfico waterfall
                    dcc.Loading(
                        id="waterfall-analysis-loading",
                        type="default",
                        children=html.Div(
                            id="waterfall-analysis-container",
                            children=[
                                # Placeholder para el chart (REGLA #0.5 - Skeleton Pattern)
                                dcc.Graph(
                                    id="waterfall-analysis-chart",
                                    figure={"data": [], "layout": {"height": 500}},
                                    config={"displayModeBar": False},
                                    style={"height": "500px"},
                                ),
                            ],
                            style={"minHeight": "500px"},
                        ),
                    ),
                ],
                style={"padding": SPACING["m"]},
            ),
        ],
        className="mb-4",
    )


def create_product_waterfall_section():
    """
    Sección de waterfall de crecimiento por producto (Issue #458).

    Gráfico waterfall que muestra contribución de cada producto
    al crecimiento/decrecimiento de la categoría seleccionada.

    Se activa cuando el usuario hace clic en una barra del waterfall
    de categorías o en un chip de categoría.

    Returns:
        dbc.Card: Panel con gráfico waterfall por producto
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    DashIconify(icon="mdi:chart-waterfall", width=18, color=COLORS["success"], className="me-2"),
                    html.Span("Variación por Producto", style={"fontWeight": "600"}),
                    html.Small(
                        id="product-waterfall-category-badge",
                        className="ms-2",
                    ),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # Placeholder inicial: Indica al usuario que seleccione categoría
                    html.Div(
                        id="product-waterfall-placeholder",
                        children=[
                            html.Div(
                                [
                                    DashIconify(
                                        icon="mdi:cursor-default-click-outline",
                                        width=48,
                                        color=COLORS["text_secondary"],
                                        className="mb-3",
                                    ),
                                    html.P(
                                        "Selecciona una categoría",
                                        className="text-muted mb-1",
                                        style={"fontSize": "1rem", "fontWeight": "500"},
                                    ),
                                    html.Small(
                                        "Haz clic en una barra del gráfico de categorías "
                                        "o en un chip para ver el detalle por producto",
                                        className="text-muted",
                                    ),
                                ],
                                className="text-center py-5",
                            ),
                        ],
                        style={"display": "block"},
                    ),
                    # Contenedor para el gráfico waterfall de productos
                    dcc.Loading(
                        id="product-waterfall-loading",
                        type="default",
                        children=html.Div(
                            id="product-waterfall-container",
                            children=[
                                dcc.Graph(
                                    id="product-waterfall-chart",
                                    figure={"data": [], "layout": {"height": 500}},
                                    config={"displayModeBar": False},
                                    style={"height": "500px", "display": "none"},
                                ),
                            ],
                            style={"minHeight": "500px"},
                        ),
                    ),
                ],
                style={"padding": SPACING["m"]},
            ),
        ],
        className="mb-4",
    )


def create_top_contributors_section():
    """
    Sección de tabla Top Contributors (Issue #436 FASE 2.2).

    Tabla de principios activos que más contribuyen al cambio en ventas,
    con filtros de límite y dirección.

    Feature: Click-to-filter desde waterfall chart.

    Returns:
        dbc.Card: Panel con filtros y tabla de top contributors
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    DashIconify(icon="mdi:format-list-numbered", width=18, color=COLORS["warning"], className="me-2"),
                    html.Span("Principales contribuyentes a la variación", style={"fontWeight": "600"}),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # Store para filtro definido en skeleton (app.py) - REGLA #0.5
                    # Indicador de filtro activo + botón limpiar (estáticos, controlados por callback)
                    html.Div(
                        id="contributors-filter-row",
                        children=[
                            dbc.Alert(
                                [
                                    DashIconify(icon="mdi:filter", width=16, className="me-2"),
                                    html.Span("Filtrando por: ", className="me-1"),
                                    html.Strong(id="contributors-filter-category-name", children=""),
                                    dbc.Button(
                                        [
                                            DashIconify(icon="mdi:close", width=14),
                                            html.Small(" Limpiar", className="ms-1"),
                                        ],
                                        id="contributors-clear-filter-btn",
                                        color="link",
                                        size="sm",
                                        className="ms-3 p-0 text-decoration-none",
                                        style={"color": COLORS["info"]},
                                    ),
                                ],
                                id="contributors-category-filter-indicator",
                                color="info",
                                className="mb-3 py-2 d-flex align-items-center",
                                style={"fontSize": "0.9rem"},
                            ),
                        ],
                        style={"display": "none"},  # Controlado por callback
                    ),
                    # Controles de filtrado
                    dbc.Row(
                        [
                            # Límite de resultados
                            dbc.Col(
                                [
                                    html.Small("Mostrar:", className="text-muted me-2"),
                                    dcc.Dropdown(
                                        id="top-contributors-limit",
                                        options=[
                                            {"label": "Top 10", "value": 10},
                                            {"label": "Top 20", "value": 20},
                                            {"label": "Top 50", "value": 50},
                                        ],
                                        value=10,
                                        clearable=False,
                                        style={"width": "120px"},
                                        persistence=True,
                                        persistence_type="session",
                                    ),
                                ],
                                width="auto",
                            ),
                            # Filtro de dirección
                            dbc.Col(
                                [
                                    html.Small("Dirección:", className="text-muted me-2"),
                                    dbc.RadioItems(
                                        id="top-contributors-direction",
                                        options=[
                                            {"label": "Todos", "value": "all"},
                                            {"label": "↗️ Subidas", "value": "up"},
                                            {"label": "↘️ Bajadas", "value": "down"},
                                        ],
                                        value="all",
                                        inline=True,
                                        className="d-inline-flex",
                                        persistence=True,
                                        persistence_type="session",
                                    ),
                                ],
                                width="auto",
                                className="ms-3",
                            ),
                        ],
                        className="mb-3 align-items-center",
                    ),
                    html.Small(
                        id="contributors-help-text",
                        children="Principios activos ordenados por impacto en el cambio de ventas. "
                                 "Haz clic en una barra del gráfico de variación para filtrar.",
                        className="text-muted d-block mb-3",
                    ),
                    # Contenedor para la tabla
                    dcc.Loading(
                        id="top-contributors-loading",
                        type="default",
                        children=html.Div(
                            id="top-contributors-container",
                            style={"minHeight": "400px", "maxHeight": "600px", "overflowY": "auto"},
                        ),
                    ),
                ],
                style={"padding": SPACING["m"]},
            ),
        ],
        className="mb-4",
    )


# format_category_name importada desde utils.constants (DRY - Issue #436)


