"""
Homepage - Panel de Dashboard Principal
Rediseño completo del Issue #152 - Dashboard profesional con KPIs mejorados

Mejoras implementadas:
- ✅ Saludo personalizado con nombre de usuario
- ✅ 6+ KPIs con iconografía dash-iconify
- ✅ 3 KPIs Year-over-Year (QTD, MTD, WTD)
- ✅ Feed de actividad reciente
- ✅ Uso completo del Design System (BaseCard, design tokens)
- ✅ Layout responsive y profesional

Pivot 2026:
- ✅ ERPStatusBanner contextual (Issue #601)
- ✅ Cazador de Encargos widget (Issue #604)
"""

from datetime import datetime

import dash
import dash_bootstrap_components as dbc

# Componentes existentes

# Nuevos componentes dashboard (Issue #152)
from components.dashboard import create_activity_feed, create_kpi_card, create_yoy_kpi_card
from components.encargos import create_encargos_widget  # Pivot 2026: Cazador de Encargos
from components.erp_status_banner import create_erp_status_banner  # Pivot 2026: Issue #601
from components.proactive_alerts import AlertsGrid  # Pivot 2026: Issue #602
from components.base import BaseCard
from stores.alerts_store import (
    get_empty_alerts_state,
    aggregate_alerts,
    build_encargos_alert,
    build_erp_status,
)
from components.feature_flags import FeatureFlags, is_feature_enabled
from components.system_status_unified import create_homepage_unified_status
from dash import dcc, html
from dash_iconify import DashIconify
from styles.design_tokens import COLORS, SPACING

# Utils
from utils.helpers import format_currency, format_number, format_percentage

# Logging
import logging
logger = logging.getLogger(__name__)


# ===================================================================
# HELPERS - User Name Extraction (Issue #316)
# ===================================================================

# Priority order for name field extraction
# Aligned with backend UserResponse schema and OAuth provider fields
NAME_FIELD_PRIORITY = [
    "full_name",      # Backend UserResponse primary field
    "name",           # OAuth common field
    "first_name",     # Alternative naming
    "display_name",   # OAuth display name
    "given_name",     # OAuth given name
    "nickname",       # OAuth nickname fallback
]


def capitalize_spanish_name(name: str) -> str:
    """
    Capitaliza nombres españoles respetando preposiciones.

    Ejemplos:
    - "maría garcía" → "María García"
    - "josé de la cruz" → "José de la Cruz"
    - "ana maría del carmen" → "Ana María del Carmen"
    """
    spanish_prepositions = {"de", "del", "la", "las", "los", "y"}
    words = name.split()
    return " ".join(
        word.lower() if word.lower() in spanish_prepositions else word.capitalize()
        for word in words
    )


def extract_user_name(user_info: dict) -> str:
    """
    Extrae el nombre de usuario con múltiples fallbacks.

    Issue #316: Extracción robusta que maneja diferentes estructuras de user_info
    desde OAuth providers (Google, Microsoft) y registro manual.

    Args:
        user_info: Diccionario con información del usuario

    Returns:
        Nombre extraído o cadena vacía si no se encuentra

    Examples:
        >>> extract_user_name({"full_name": "María García"})
        "María García"
        >>> extract_user_name({"email": "pedro@farmacia.es"})
        "Pedro"
        >>> extract_user_name({})
        ""
    """
    if not user_info or not isinstance(user_info, dict):
        return ""

    # Intentar campos en orden de prioridad
    for field in NAME_FIELD_PRIORITY:
        if candidate := user_info.get(field):
            # Manejar tipos no-string de OAuth providers
            candidate_str = str(candidate) if not isinstance(candidate, str) else candidate
            if cleaned := candidate_str.strip():
                return capitalize_spanish_name(cleaned)

    # Fallback final: prefix del email
    if email := user_info.get("email"):
        if isinstance(email, str) and "@" in email:
            username = email.split("@")[0]
            return capitalize_spanish_name(username)

    return ""


def create_homepage_layout() -> html.Div:
    """
    Crea el layout principal de la homepage con dashboard profesional.

    Nuevo diseño (Issue #152):
    - Saludo personalizado + fecha
    - Grid de 6 KPIs con iconografía
    - Gráfico de tendencia de ventas
    - Feed de actividad reciente
    - Cards legacy adaptadas (catalog control, fuentes externas)
    """
    return html.Div(
        [
            # ===================================================================
            # SECCIÓN 0: ERP STATUS BANNER (Pivot 2026 - Issue #601)
            # Banner contextual que muestra estado de conexion ERP
            # Solo visible cuando hay problemas de sync (>30 min)
            # ===================================================================
            html.Div(id="erp-status-banner-container"),
            # ===================================================================
            # SECCIÓN 1: SALUDO PERSONALIZADO Y CONTEXTO (Nuevo - Issue #152)
            # ===================================================================
            create_greeting_section(),
            # ===================================================================
            # SECCIÓN 1.5: ALERTAS PROACTIVAS (Pivot 2026 - Issue #602)
            # Grid de alertas ordenadas por severidad (critical > warning > info)
            # Se renderiza desde global-alerts-store
            # ===================================================================
            html.Div(
                id="proactive-alerts-container",
                className="mb-4",
            ),
            # ===================================================================
            # SECCIÓN 1.6: QUICK ACTIONS (Pivot 2026 - Issue #603)
            # Acciones rapidas: Sync, Nuevo Encargo, Ver Recetas
            # ===================================================================
            html.Div(
                id="quick-actions-container",
                children=[create_quick_actions_section()],
                className="mb-4",
            ),
            # ===================================================================
            # SECCIÓN 2: KPIs PRINCIPALES (Nuevo - Issue #152)
            # ===================================================================
            html.Div(
                id="kpis-row",
                children=[create_kpis_placeholder()],  # Se actualiza con callback
                className="mb-4",
            ),
            # ===================================================================
            # SECCIÓN 3: KPIs YoY Y ACTIVIDAD
            # ===================================================================
            dbc.Row(
                [
                    # KPIs YoY (columna izquierda - 2/3)
                    dbc.Col(
                        [
                            html.Div(
                                id="yoy-kpis-container",
                                children=[create_yoy_kpis_placeholder()],
                            ),
                        ],
                        width=12,
                        lg=8,
                        className="mb-4",
                    ),
                    # Feed de actividad (columna derecha - 1/3)
                    dbc.Col(
                        [create_activity_feed()],
                        width=12,
                        lg=4,
                        className="mb-4",
                    ),
                ],
            ),
            # Store para datos YoY
            dcc.Store(id="yoy-kpis-store", data=None),
            # ===================================================================
            # SECCIÓN 4: CAZADOR DE ENCARGOS (Pivot 2026)
            # ===================================================================
            dbc.Row(
                [
                    dbc.Col(
                        [
                            BaseCard(
                                children=[create_encargos_widget()],
                                shadow="sm",
                                padding=SPACING["l"],
                            ),
                        ],
                        width=12,
                        className="mb-4",
                    ),
                ],
            ),
            # ===================================================================
            # SECCIÓN 5: PANEL DE ESTADO UNIFICADO (Condicional - Feature Flag)
            # ===================================================================
            html.Div(id="unified-status-container", className="mb-4"),
            # ===================================================================
            # SECCIÓN 6: CARDS LEGACY ADAPTADAS (Mantenidas para compatibilidad)
            # ===================================================================
            html.Div(
                id="legacy-system-panels",
                children=[
                    dbc.Row(
                        [
                            # Catalog Control Panel (lado izquierdo)
                            dbc.Col(
                                [html.Div(id="catalog-control-panel-container")],
                                width=12,
                                lg=6,
                                className="mb-4",
                            ),
                            # Fuentes Externas Card (lado derecho)
                            dbc.Col(
                                [create_external_sources_card()],
                                width=12,
                                lg=6,
                                className="mb-4",
                            ),
                        ]
                    )
                ],
            ),
            # ===================================================================
            # STORES Y INTERVALS (Mantenidos para compatibilidad)
            # ===================================================================
            dcc.Store(id="enrichment-progress-store"),
            # NOTE: homepage-data-store, homepage-feature-flags-store, erp-sync-status-store
            # are defined in app.py skeleton (REGLA #0.5) - do not duplicate here
        ]
    )


def create_greeting_section() -> html.Div:
    """
    Crea la sección de saludo personalizado con nombre y fecha (Issue #152).

    Returns:
        html.Div: Sección de saludo con contexto temporal
    """
    from utils.helpers import MESES_ESPANOL, DIAS_SEMANA_ESPANOL

    now = datetime.now()

    # Usar helpers centralizados para localización española (Issue #XXX)
    weekday = DIAS_SEMANA_ESPANOL.get(now.weekday(), "").lower()
    month_name = MESES_ESPANOL.get(now.month, "").lower()

    # Formato: "Lunes, 1 de octubre de 2024"
    date_str = f"{weekday.capitalize()}, {now.day} de {month_name} de {now.year}"

    # Determinar saludo según hora del día
    hour = now.hour
    if hour < 12:
        greeting = "Buenos días"
        icon = "mdi:weather-sunny"
    elif hour < 20:
        greeting = "Buenas tardes"
        icon = "mdi:weather-sunset"
    else:
        greeting = "Buenas noches"
        icon = "mdi:weather-night"

    return html.Div(
        [
            # Fila con saludo e icono
            html.Div(
                [
                    DashIconify(
                        icon=icon,
                        width=32,
                        height=32,
                        style={
                            "color": COLORS["primary"],
                            "marginRight": SPACING["s"],
                        },
                    ),
                    html.Div(
                        [
                            html.H2(
                                [
                                    greeting,
                                    html.Span(
                                        id="user-greeting-name",
                                        children="",  # Se actualiza con callback
                                        style={"marginLeft": "8px"},
                                    ),
                                ],
                                className="mb-1",
                                style={
                                    "fontWeight": "600",
                                    "color": COLORS["text_primary"],
                                },
                            ),
                            html.P(
                                [
                                    html.I(
                                        className="far fa-calendar-alt me-2",
                                        style={"color": COLORS["text_secondary"]},
                                    ),
                                    date_str,
                                ],
                                className="text-muted mb-0",
                                style={"fontSize": "0.95rem"},
                            ),
                        ]
                    ),
                ],
                className="d-flex align-items-center",
            ),
            html.Hr(className="my-4"),
        ],
        className="mb-4",
    )


def create_quick_actions_section() -> html.Div:
    """
    Crea la seccion de acciones rapidas (Pivot 2026 - Issue #603).

    Acciones disponibles:
    - Sincronizar ERP: Fuerza sync manual
    - Nuevo Encargo: Acceso rapido a crear encargo
    - Ver Recetas: Acceso al modulo de recetas

    Returns:
        html.Div: Fila con botones de accion rapida
    """
    actions = [
        {
            "id": "btn-quick-sync",
            "icon": "mdi:sync",
            "label": "Sincronizar",
            "color": "primary",
            "outline": True,
        },
        {
            "id": "btn-quick-encargo",
            "icon": "mdi:clipboard-plus",
            "label": "Nuevo Encargo",
            "color": "success",
            "outline": True,
            "href": "/encargos",
        },
        {
            "id": "btn-quick-recetas",
            "icon": "mdi:file-document-outline",
            "label": "Ver Recetas",
            "color": "info",
            "outline": True,
            "href": "/prescription",
        },
    ]

    buttons = []
    for action in actions:
        btn_props = {
            "id": action["id"],
            "color": action["color"],
            "outline": action.get("outline", False),
            "size": "sm",
            "className": "me-2 mb-2",
        }
        if "href" in action:
            btn_props["href"] = action["href"]

        buttons.append(
            dbc.Button(
                [
                    DashIconify(
                        icon=action["icon"],
                        width=18,
                        className="me-1",
                    ),
                    action["label"],
                ],
                **btn_props,
            )
        )

    return html.Div(
        [
            html.Small(
                "Acciones rapidas",
                className="text-muted d-block mb-2",
                style={"fontSize": "0.8rem", "textTransform": "uppercase", "letterSpacing": "0.5px"},
            ),
            html.Div(buttons, className="d-flex flex-wrap"),
        ],
    )


def create_kpis_placeholder() -> html.Div:
    """
    Placeholder inicial para KPIs mientras se cargan datos.

    Returns:
        html.Div: Grid de skeleton loaders para KPIs
    """
    from components.skeleton_loader import create_card_skeleton

    skeleton_kpis = [create_card_skeleton() for _ in range(6)]

    return dbc.Row(
        [
            dbc.Col(
                [skeleton],
                width=12,
                sm=6,
                md=4,
                lg=2,
                className="mb-3",
            )
            for skeleton in skeleton_kpis
        ]
    )


def create_yoy_kpis_placeholder() -> html.Div:
    """
    Placeholder inicial para KPIs YoY mientras se cargan datos.

    Returns:
        html.Div: Grid de skeleton loaders para 3 KPIs YoY
    """
    from components.skeleton_loader import create_card_skeleton

    skeleton_kpis = [create_card_skeleton() for _ in range(3)]

    return dbc.Row(
        [
            dbc.Col(
                [skeleton],
                width=12,
                md=4,
                className="mb-3",
            )
            for skeleton in skeleton_kpis
        ]
    )


def create_yoy_kpis_row(yoy_data: dict) -> dbc.Row:
    """
    Crea la fila con 3 KPIs de variación YoY.

    Args:
        yoy_data: Dict con keys qtd_yoy, mtd_yoy, wtd_yoy desde Measures API

    Returns:
        dbc.Row: Fila con los 3 KPIs YoY (trimestre, mes, semana)
    """
    # Extraer datos de cada KPI
    qtd = yoy_data.get("qtd_yoy", {})
    mtd = yoy_data.get("mtd_yoy", {})
    wtd = yoy_data.get("wtd_yoy", {})

    # Crear KPIs
    qtd_kpi = create_yoy_kpi_card(
        title=qtd.get("label", "Trimestre YoY"),
        change_percent=qtd.get("change_percent", 0),
        subtitle="vs mismo período año anterior",
        component_id="kpi-qtd-yoy",
    )

    mtd_kpi = create_yoy_kpi_card(
        title=mtd.get("label", "Mes YoY"),
        change_percent=mtd.get("change_percent", 0),
        subtitle="vs mismo período año anterior",
        component_id="kpi-mtd-yoy",
    )

    wtd_kpi = create_yoy_kpi_card(
        title=wtd.get("label", "Semana YoY"),
        change_percent=wtd.get("change_percent", 0),
        subtitle="vs mismo período año anterior",
        component_id="kpi-wtd-yoy",
    )

    return dbc.Row(
        [
            dbc.Col([qtd_kpi], width=12, md=4, className="mb-3"),
            dbc.Col([mtd_kpi], width=12, md=4, className="mb-3"),
            dbc.Col([wtd_kpi], width=12, md=4, className="mb-3"),
        ]
    )


def create_kpis_row(dashboard_data: dict) -> dbc.Row:
    """
    Crea la fila de KPIs con datos reales (Issue #152).

    Args:
        dashboard_data: Datos del dashboard desde API

    Returns:
        dbc.Row: Fila con 6 KPIs configurados

    Note:
        Esta función se usa en el callback para actualizar KPIs con datos reales.
    """
    sales = dashboard_data.get("sales", {})
    catalog = dashboard_data.get("catalog", {})
    system_status = dashboard_data.get("system", {})
    cima = dashboard_data.get("cima", {})
    nomenclator = dashboard_data.get("nomenclator", {})

    # KPI 1: Período de Ventas (rango de fechas analizables)
    date_range = sales.get("date_range", {})
    start_date = date_range.get("start")
    end_date = date_range.get("end")

    if start_date and end_date:
        # Formatear fechas: "Ene 2025 - Ago 2025"
        from datetime import datetime
        try:
            start_dt = datetime.fromisoformat(start_date.replace("Z", "+00:00"))
            end_dt = datetime.fromisoformat(end_date.replace("Z", "+00:00"))
            # Meses en español abreviados
            meses = ["Ene", "Feb", "Mar", "Abr", "May", "Jun",
                     "Jul", "Ago", "Sep", "Oct", "Nov", "Dic"]
            start_str = f"{meses[start_dt.month - 1]} {start_dt.year}"
            end_str = f"{meses[end_dt.month - 1]} {end_dt.year}"
            if start_str == end_str:
                period_value = start_str
            else:
                period_value = f"{start_str} - {end_str}"
        except (ValueError, AttributeError):
            period_value = "--"
    else:
        period_value = "--"

    kpi1 = create_kpi_card(
        title="Período Ventas",
        value=period_value,
        icon="mdi:calendar-range",
        color="success",
        subtitle="Datos analizables",
        component_id="kpi-sales-period",
    )

    # KPI 2: Productos Únicos
    unique_products = sales.get("unique_products", 0)
    kpi2 = create_kpi_card(
        title="Productos Únicos",
        value=format_number(unique_products) if unique_products > 0 else "--",
        icon="mdi:package-variant",
        color="info",
        subtitle="En ventas",
        component_id="kpi-unique-products",
    )

    # KPI 3: CIMA (última actualización)
    cima_last_sync = cima.get("last_sync")
    cima_products = cima.get("total_products", 0)

    # Formatear fecha de última sync CIMA
    if cima_last_sync:
        from datetime import datetime, timezone
        try:
            sync_dt = datetime.fromisoformat(cima_last_sync.replace("Z", "+00:00"))
            now = datetime.now(timezone.utc)
            days_ago = (now - sync_dt).days
            if days_ago == 0:
                cima_value = "Hoy"
            elif days_ago == 1:
                cima_value = "Ayer"
            else:
                cima_value = f"Hace {days_ago} días"
        except (ValueError, AttributeError):
            cima_value = "--"
    else:
        cima_value = "--"

    kpi3 = create_kpi_card(
        title="CIMA",
        value=cima_value,
        icon="mdi:book-medical",
        color="info",
        subtitle=f"{format_number(cima_products)} productos" if cima_products > 0 else "Sin datos",
        component_id="kpi-cima",
    )

    # KPI 4: Nomenclátor (última actualización)
    nomenclator_last_sync = nomenclator.get("last_sync")
    homogeneous_groups = nomenclator.get("homogeneous_groups", 0)
    new_groups = nomenclator.get("new_groups", 0)

    # Formatear fecha de última sync Nomenclátor
    if nomenclator_last_sync:
        from datetime import datetime, timezone
        try:
            sync_dt = datetime.fromisoformat(nomenclator_last_sync.replace("Z", "+00:00"))
            now = datetime.now(timezone.utc)
            days_ago = (now - sync_dt).days
            if days_ago == 0:
                nomen_value = "Hoy"
            elif days_ago == 1:
                nomen_value = "Ayer"
            else:
                nomen_value = f"Hace {days_ago} días"
        except (ValueError, AttributeError):
            nomen_value = "--"
    else:
        nomen_value = "--"

    kpi4 = create_kpi_card(
        title="Nomenclátor",
        value=nomen_value,
        icon="mdi:book-open-page-variant",
        color="warning",
        subtitle=f"{format_number(homogeneous_groups)} grupos" if homogeneous_groups > 0 else "Sin grupos",
        trend=f"+{new_groups}" if new_groups > 0 else None,
        trend_positive=new_groups > 0,
        component_id="kpi-nomenclator",
    )

    # Grid responsive: 4 columnas en desktop, 2 en tablet, 1 en móvil
    return dbc.Row(
        [
            dbc.Col([kpi1], width=12, sm=6, md=6, lg=3, className="mb-3"),
            dbc.Col([kpi2], width=12, sm=6, md=6, lg=3, className="mb-3"),
            dbc.Col([kpi3], width=12, sm=6, md=6, lg=3, className="mb-3"),
            dbc.Col([kpi4], width=12, sm=6, md=6, lg=3, className="mb-3"),
        ]
    )


# ===================================================================
# FUNCIONES LEGACY (Mantenidas para compatibilidad)
# ===================================================================


def create_external_sources_card() -> dbc.Card:
    """
    Card con el estado de las fuentes de datos externas (LEGACY).
    Mantenida para compatibilidad con callbacks existentes.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.H4(
                        [
                            html.I(className="bi bi-cloud-download me-2"),
                            "Fuentes de Datos Externas",
                        ],
                        className="mb-0",
                    )
                ]
            ),
            dbc.CardBody(
                [
                    # CIMA
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.H6(
                                        [
                                            html.I(
                                                className="bi bi-circle-fill text-success me-2",
                                                id="cima-status-indicator",
                                            ),
                                            "CIMA (AEMPS)",
                                        ],
                                        className="mb-2",
                                    ),
                                    html.Small(
                                        [
                                            "Última sincronización: ",
                                            html.Span(
                                                id="cima-last-sync", children="--"
                                            ),
                                        ],
                                        className="text-muted",
                                    ),
                                ],
                                width=12,
                                className="mb-3",
                            )
                        ]
                    ),
                    # Nomenclátor
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.H6(
                                        [
                                            html.I(
                                                className="bi bi-circle-fill text-success me-2",
                                                id="nomen-status-indicator",
                                            ),
                                            "Nomenclátor",
                                        ],
                                        className="mb-2",
                                    ),
                                    html.Small(
                                        [
                                            "Última descarga: ",
                                            html.Span(
                                                id="nomen-last-sync", children="--"
                                            ),
                                        ],
                                        className="text-muted",
                                    ),
                                ],
                                width=12,
                                className="mb-3",
                            )
                        ]
                    ),
                    # Grupos Homogéneos
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.H6(
                                        [
                                            html.I(
                                                className="bi bi-circle-fill text-warning me-2",
                                                id="groups-status-indicator",
                                            ),
                                            "Grupos Homogéneos",
                                        ],
                                        className="mb-2",
                                    ),
                                    html.Small(
                                        [
                                            "Estado: ",
                                            html.Span(
                                                id="groups-status", children="--"
                                            ),
                                        ],
                                        className="text-muted",
                                    ),
                                ],
                                width=12,
                            )
                        ]
                    ),
                    html.Hr(),
                    dbc.Alert(
                        id="external-sources-alert",
                        children="Todas las fuentes están actualizadas",
                        color="success",
                        className="mb-0",
                    ),
                ]
            ),
        ]
    )


# ===================================================================
# CALLBACKS REGISTRATION (Se maneja en archivo separado)
# ===================================================================


def register_homepage_callbacks(app):
    """
    Registra los callbacks para la homepage rediseñada (Issue #152).

    Callbacks implementados:
    1. update_greeting_and_kpis - Actualiza saludo y KPIs
    2. update_trend_chart - Actualiza gráfico de tendencia
    3. update_activity_feed - Actualiza feed de actividad
    4. render_unified_or_legacy_status - Feature flag para panel unificado
    5. update_homepage_store - Store intermedio
    6. refresh_homepage_feature_flags - Refresco de feature flags
    7. fetch_dashboard_data - Obtención de datos del dashboard
    8. handle_homepage_data_updates - Callback unificado (REGLA #11)
    9. update_task_progress - Progreso de tareas

    Pivot 2026:
    10. update_erp_sync_status - Actualiza estado ERP desde dashboard data
    11. render_erp_status_banner - Renderiza banner ERP
    12. handle_erp_retry_sync - Maneja retry de sync ERP
    13. aggregate_global_alerts - Agrega alertas al global-alerts-store (#602)
    14. render_proactive_alerts - Renderiza grid de alertas proactivas (#602)
    15. handle_quick_sync - Maneja sync rapido desde QuickActions (#603)
    """
    from datetime import datetime, timedelta

    import dash_bootstrap_components as dbc
    from components.dashboard import create_activity_list, extract_activities_from_data
    from components.erp_status_banner import create_erp_status_banner
    from stores.alerts_store import build_erp_status
    from dash import Input, Output, State, ctx, no_update
    from dash.exceptions import PreventUpdate
    from utils.api_client import backend_client  # Para helper methods específicos
    from utils.request_coordinator import request_coordinator  # Para make_request estándar
    from utils.helpers import format_number

    logger = logging.getLogger(__name__)

    # ===================================================================
    # CALLBACK 0A: Actualizar ERP sync status store (Pivot 2026 - Issue #601)
    # ===================================================================
    @app.callback(
        Output("erp-sync-status-store", "data"),
        Input("homepage-data-store", "data"),
        State("erp-sync-status-store", "data"),  # Fix #4: Read current state for race condition check
        prevent_initial_call=True,
    )
    def update_erp_sync_status(homepage_data, current_erp_status):
        """
        Extrae datos de ERP desde dashboard data y los transforma al formato del store.

        El store erp-sync-status-store contiene el estado formateado del ERP
        que luego se usa para renderizar el banner.

        Fix #4 (Race Condition): If a manual sync is in progress (_syncing_in_progress=True),
        don't overwrite the optimistic state until the sync completes and we get fresh data.
        """
        # Fix #4: Don't overwrite if manual sync is in progress (unless expired)
        if current_erp_status and current_erp_status.get("_syncing_in_progress"):
            expires_str = current_erp_status.get("_syncing_expires")
            if expires_str:
                try:
                    expires_dt = datetime.fromisoformat(expires_str)
                    if datetime.now() < expires_dt:
                        logger.debug("[HOMEPAGE] ERP sync in progress - skipping store update")
                        raise PreventUpdate
                    else:
                        logger.debug("[HOMEPAGE] ERP sync flag expired - allowing update")
                except (ValueError, TypeError):
                    pass  # Invalid date, allow update

        if not homepage_data or not homepage_data.get("data"):
            return {}

        dashboard_data = homepage_data.get("data", {}).get("dashboard", {})
        system_data = dashboard_data.get("system", {})

        # Extraer componentes del sistema
        components = system_data.get("components", {})
        erp_component = components.get("erp", {})

        if not erp_component:
            # Si no hay datos de ERP, intentar construir desde otros datos
            # En modo local, el ERP puede no estar en components
            return {}

        # Construir datos de ERP para el store
        erp_data = {
            "provider": erp_component.get("provider", "farmanager"),
            "connected": erp_component.get("status", "unknown") in ["ready", "READY", "connected", "ok"],
            "last_sync": erp_component.get("last_success_at") or erp_component.get("last_sync"),
        }

        # Usar build_erp_status para obtener el formato correcto
        erp_status = build_erp_status(erp_data)

        logger.debug(f"[HOMEPAGE] ERP status updated: {erp_status.get('status')} - {erp_status.get('msg')}")

        return erp_status

    # ===================================================================
    # CALLBACK 0B: Renderizar ERP Status Banner (Pivot 2026 - Issue #601)
    # ===================================================================
    @app.callback(
        Output("erp-status-banner-container", "children"),
        Input("erp-sync-status-store", "data"),
        prevent_initial_call=True,
    )
    def render_erp_status_banner(erp_status):
        """
        Renderiza el banner de estado ERP si es necesario mostrarlo.

        El banner se oculta cuando:
        - No hay datos de ERP
        - El sync es menor a 30 minutos (todo OK)

        El banner se muestra cuando:
        - Sync >= 30 min (verde informativo)
        - Sync >= 60 min (ambar warning)
        - Sync >= 1440 min o desconectado (rojo critico)
        """
        if not erp_status:
            return None

        banner = create_erp_status_banner(erp_status)
        if banner:
            logger.info(f"[HOMEPAGE] ERP banner rendered: {erp_status.get('status')}")
        return banner

    # ===================================================================
    # CALLBACK 0C: Manejar retry de sync ERP (Pivot 2026 - Issue #601)
    # ===================================================================
    @app.callback(
        Output("erp-sync-status-store", "data", allow_duplicate=True),
        Input("btn-erp-retry-sync", "n_clicks"),
        State("auth-state", "data"),
        prevent_initial_call=True,
    )
    def handle_erp_retry_sync(n_clicks, auth_state):
        """
        Maneja el click en el boton de retry sync del banner ERP.

        Llama al endpoint de sync ERP y actualiza el store con el nuevo estado.
        """
        if not n_clicks:
            raise PreventUpdate

        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.warning("[HOMEPAGE] ERP retry sync attempted without authentication")
            raise PreventUpdate

        try:
            # Llamar al endpoint de sync ERP
            response = request_coordinator.make_request(
                "POST",
                "/api/v1/erp-sync/run",
                timeout=30,
            )

            if response and response.get("success"):
                logger.info("[HOMEPAGE] ERP sync triggered successfully")
                # Fix #4: Set _syncing_in_progress flag to prevent race condition
                # The flag expires after 30 seconds to prevent permanent blocking
                sync_expires = (datetime.now() + timedelta(seconds=30)).isoformat()
                return {
                    "status": "ok",
                    "msg": "Sincronizando datos...",
                    "provider": response.get("provider", "farmanager"),
                    "last_sync": datetime.now().isoformat(),
                    "minutes_since_sync": 0,
                    "_syncing_in_progress": True,
                    "_syncing_expires": sync_expires,
                }
            elif response and response.get("skipped"):
                # Sync already in progress
                logger.warning(f"[HOMEPAGE] ERP sync skipped: {response.get('reason')}")
                return {
                    "status": "warning",
                    "msg": response.get("reason", "Sincronización en progreso..."),
                    "provider": "farmanager",
                    "last_sync": None,
                    "minutes_since_sync": None,
                    "_syncing_in_progress": True,
                    "_syncing_expires": (datetime.now() + timedelta(seconds=30)).isoformat(),
                }
            else:
                logger.error(f"[HOMEPAGE] ERP sync failed: {response}")
                return {
                    "status": "critical",
                    "msg": "Error al sincronizar - Reintentar",
                    "provider": "unknown",
                    "last_sync": None,
                    "minutes_since_sync": None,
                }

        except Exception as e:
            logger.error(f"[HOMEPAGE] Error triggering ERP sync: {str(e)}")
            return {
                "status": "critical",
                "msg": f"Error: {str(e)[:50]}",
                "provider": "unknown",
                "last_sync": None,
                "minutes_since_sync": None,
            }

    # ===================================================================
    # CALLBACK 0D: Agregar alertas al global-alerts-store (Pivot 2026 - Issue #602)
    # ===================================================================
    @app.callback(
        Output("global-alerts-store", "data"),
        [
            Input("erp-sync-status-store", "data"),
            Input("homepage-data-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def aggregate_global_alerts(erp_status, homepage_data):
        """
        Agrega alertas de todos los modulos en el global-alerts-store.

        Lee de:
        - erp-sync-status-store: Estado de conexion ERP
        - homepage-data-store: Datos del dashboard (encargos, etc.)

        Escribe a global-alerts-store con formato AlertsState.
        """
        from stores.alerts_store import aggregate_alerts, build_encargos_alert, build_erp_status

        modules = {}
        system = {}

        # Agregar alerta de ERP al sistema
        if erp_status:
            system["erp"] = erp_status

        # Agregar alertas de modulos desde homepage data
        if homepage_data and homepage_data.get("data"):
            dashboard_data = homepage_data.get("data", {}).get("dashboard", {})

            # Encargos (si hay datos)
            encargos_summary = dashboard_data.get("encargos_summary")
            if encargos_summary:
                modules["encargos"] = build_encargos_alert(encargos_summary)

        # Agregar estado si no hay datos de ERP pero hay homepage data
        if not system.get("erp") and homepage_data:
            dashboard_data = homepage_data.get("data", {}).get("dashboard", {})
            system_data = dashboard_data.get("system", {})
            components = system_data.get("components", {})
            erp_component = components.get("erp", {})

            if erp_component:
                erp_data = {
                    "provider": erp_component.get("provider", "farmanager"),
                    "connected": erp_component.get("status", "unknown") in ["ready", "READY", "connected", "ok"],
                    "last_sync": erp_component.get("last_success_at") or erp_component.get("last_sync"),
                }
                system["erp"] = build_erp_status(erp_data)

        result = aggregate_alerts(modules, system)
        logger.debug(f"[HOMEPAGE] Global alerts aggregated: {result.get('summary', {})}")
        return result

    # ===================================================================
    # CALLBACK 0E: Renderizar ProactiveAlerts (Pivot 2026 - Issue #602)
    # ===================================================================
    @app.callback(
        Output("proactive-alerts-container", "children"),
        Input("global-alerts-store", "data"),
        prevent_initial_call=True,
    )
    def render_proactive_alerts(alerts_data):
        """
        Renderiza el grid de alertas proactivas desde global-alerts-store.
        """
        from components.proactive_alerts import AlertsGrid

        if not alerts_data:
            return AlertsGrid(None)

        return AlertsGrid(alerts_data)

    # ===================================================================
    # CALLBACK 0F: Quick Sync Action (Pivot 2026 - Issue #603)
    # ===================================================================
    @app.callback(
        Output("erp-sync-status-store", "data", allow_duplicate=True),
        Input("btn-quick-sync", "n_clicks"),
        State("auth-state", "data"),
        prevent_initial_call=True,
    )
    def handle_quick_sync(n_clicks, auth_state):
        """
        Maneja el click en el boton de sync rapido.
        Reutiliza la logica de handle_erp_retry_sync.
        """
        if not n_clicks:
            raise PreventUpdate

        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.warning("[HOMEPAGE] Quick sync attempted without authentication")
            raise PreventUpdate

        try:
            response = request_coordinator.make_request(
                "POST",
                "/api/v1/erp-sync/run",
                timeout=30,
            )

            if response and response.get("success"):
                logger.info("[HOMEPAGE] Quick sync triggered successfully")
                sync_expires = (datetime.now() + timedelta(seconds=30)).isoformat()
                return {
                    "status": "ok",
                    "msg": "Sincronizando datos...",
                    "provider": response.get("provider", "farmanager"),
                    "last_sync": datetime.now().isoformat(),
                    "minutes_since_sync": 0,
                    "_syncing_in_progress": True,
                    "_syncing_expires": sync_expires,
                }
            else:
                logger.error(f"[HOMEPAGE] Quick sync failed: {response}")
                return {
                    "status": "warning",
                    "msg": "Sync en progreso o no disponible",
                    "provider": "unknown",
                }

        except Exception as e:
            logger.error(f"[HOMEPAGE] Error in quick sync: {str(e)}")
            return {
                "status": "critical",
                "msg": f"Error: {str(e)[:50]}",
                "provider": "unknown",
            }

    # ===================================================================
    # CALLBACK 1: Actualizar saludo y KPIs (Nuevo - Issue #152)
    # ===================================================================
    @app.callback(
        [
            Output("user-greeting-name", "children"),
            Output("kpis-row", "children"),
        ],
        Input("homepage-data-store", "data"),
        State("auth-state", "data"),
        prevent_initial_call=True,
    )
    def update_greeting_and_kpis(homepage_data, auth_state):
        """
        Actualiza el saludo personalizado y los KPIs del dashboard.

        Issue #152: Saludo personalizado + 6 KPIs con iconografía.
        Issue #316: Extracción robusta de nombre de usuario con fallbacks múltiples.
        """
        # Extracción de nombre de usuario (Issue #316)
        user_name = ""
        if auth_state and auth_state.get("authenticated"):
            user_info = auth_state.get("user", {}) or {}
            extracted_name = extract_user_name(user_info)

            if extracted_name:
                user_name = f", {extracted_name}"
                logger.debug(f"[HOMEPAGE] Greeting personalized for user: {extracted_name}")
            else:
                logger.warning(f"[HOMEPAGE] Could not extract user name from user_info: {user_info}")
        else:
            logger.debug(f"[HOMEPAGE] User not authenticated or auth_state invalid")

        # KPIs desde datos del homepage
        kpis_row = create_kpis_placeholder()  # Default: skeleton

        if homepage_data and homepage_data.get("data"):
            dashboard_data = homepage_data.get("data", {}).get("dashboard", {})
            if dashboard_data:
                kpis_row = create_kpis_row(dashboard_data)
                logger.info("[HOMEPAGE] KPIs actualizados con datos reales")

        return user_name, kpis_row

    # ===================================================================
    # CALLBACK 2: Actualizar KPIs Year-over-Year (Nuevo - Issue #152)
    # ===================================================================
    # OPTIMIZADO: YoY incluido en /initial-data (1 query SQL consolidada)
    # Eliminada llamada API separada a Measures (12 queries → 1 query)
    @app.callback(
        Output("yoy-kpis-container", "children"),
        Input("homepage-data-store", "data"),  # REGLA #11: Usar store intermedio
        State("auth-state", "data"),
        prevent_initial_call=True,
    )
    def update_yoy_kpis(homepage_data, auth_state):
        """
        Muestra los KPIs Year-over-Year (YoY) desde los datos del store.

        OPTIMIZADO: Los datos YoY ahora vienen incluidos en /initial-data
        en lugar de hacer una llamada separada a Measures API.
        Esto reduce 12+ queries SQL a 1 sola query consolidada.

        Issue #152: Reemplaza gráfico de tendencia por 3 KPIs YoY.
        REGLA #11: Usa homepage-data-store como trigger.
        """
        from dash import no_update
        from utils.auth_helpers import is_user_authenticated

        try:
            # Verificar autenticación (REGLA #7.6)
            if not is_user_authenticated(auth_state):
                logger.debug("[HOMEPAGE] User not authenticated for YoY KPIs")
                return no_update

            if not homepage_data:
                return create_yoy_kpis_placeholder()

            # Obtener datos YoY del store (ya vienen de /initial-data)
            dashboard_data = homepage_data.get("data", {}).get("dashboard", {})
            yoy_kpis_raw = dashboard_data.get("yoy_kpis", {})

            if not yoy_kpis_raw:
                logger.debug("[HOMEPAGE] No YoY data in dashboard store")
                return create_yoy_kpis_placeholder()

            # Transformar al formato esperado por create_yoy_kpis_row
            yoy_data = {}
            for measure_name, value in yoy_kpis_raw.items():
                if not value:
                    continue

                # Datos vienen de /initial-data con formato:
                # {tipo, periodo_actual: {trimestre/mes/semana}, variacion: {ventas_porcentual}}
                tipo = value.get("tipo", "")
                periodo_actual = value.get("periodo_actual", {})
                periodo_anterior = value.get("periodo_anterior", {})
                variacion = value.get("variacion", {})

                # Generar label basado en tipo
                if tipo == "QTD":
                    label = f"{periodo_actual.get('trimestre', '')} vs {periodo_anterior.get('trimestre', '')}"
                elif tipo == "MTD":
                    label = f"{periodo_actual.get('mes', '')} vs {periodo_anterior.get('mes', '')}"
                elif tipo == "WTD":
                    label = f"{periodo_actual.get('semana', '')} vs {periodo_anterior.get('semana', '')}"
                else:
                    label = "YoY"

                yoy_data[measure_name] = {
                    "label": label,
                    "change_percent": variacion.get("ventas_porcentual", 0),
                }

            if not yoy_data:
                logger.debug("[HOMEPAGE] No valid YoY data after transformation")
                return create_yoy_kpis_placeholder()

            logger.info("[HOMEPAGE] YoY KPIs actualizados desde initial-data (optimizado)")
            return create_yoy_kpis_row(yoy_data)

        except Exception as e:
            logger.error(f"[HOMEPAGE] Error actualizando YoY KPIs: {str(e)}")
            return create_yoy_kpis_placeholder()

    # ===================================================================
    # CALLBACK 3: Actualizar feed de actividad (Nuevo - Issue #152)
    # ===================================================================
    @app.callback(
        Output("activity-feed-content", "children"),
        Input("homepage-data-store", "data"),
        prevent_initial_call=True,
    )
    def update_activity_feed(homepage_data):
        """
        Actualiza el feed de actividad reciente con últimas acciones del sistema.

        Issue #152: Feed con últimas 5 actividades (uploads, syncs, enrichments).
        """
        if not homepage_data or not homepage_data.get("data"):
            from components.dashboard.activity_feed import _create_activity_placeholder

            return _create_activity_placeholder()

        dashboard_data = homepage_data.get("data", {}).get("dashboard", {})

        # Extraer actividades desde datos del dashboard
        activities = extract_activities_from_data(dashboard_data)

        logger.info(
            f"[HOMEPAGE] Activity feed actualizado con {len(activities)} actividades"
        )

        return create_activity_list(activities)

    # ===================================================================
    # CALLBACKS LEGACY (Mantenidos para compatibilidad)
    # ===================================================================

    # FASE 4: Callback para renderizar componente unificado o legacy
    @app.callback(
        [
            Output("unified-status-container", "children"),
            Output("legacy-system-panels", "style"),
        ],
        Input("homepage-feature-flags-store", "data"),
        prevent_initial_call=False,
    )
    def render_unified_or_legacy_status(feature_flags):
        """
        Renderiza componente unificado o legacy según feature flags.
        Permite rollback inmediato cambiando flags.
        """
        try:
            # Verificar si el componente unificado está habilitado
            unified_enabled = is_feature_enabled("unified_status_panel", feature_flags)
            basic_mode_enabled = is_feature_enabled("basic_mode_enabled", feature_flags)

            if unified_enabled and basic_mode_enabled:
                # Modo NUEVO: Componente unificado en modo básico para farmacéuticos
                logger.info("[HOMEPAGE] Renderizando componente unificado en modo básico")

                unified_component = create_homepage_unified_status()

                # Botón para ver detalles técnicos
                admin_link = dbc.Card(
                    [
                        dbc.CardBody(
                            [
                                html.Div(
                                    [
                                        html.I(className="fas fa-cog me-2 text-primary"),
                                        html.Span("Para detalles técnicos completos, "),
                                        dcc.Link(
                                            "acceda al panel de administración",
                                            href="/admin",
                                            className="text-primary fw-bold",
                                        ),
                                    ],
                                    className="d-flex align-items-center justify-content-center",
                                )
                            ],
                            className="py-2",
                        )
                    ],
                    className="border-0 bg-light mb-3",
                )

                unified_content = [unified_component, admin_link]

                # Ocultar paneles legacy
                legacy_style = {"display": "none"}

                return unified_content, legacy_style

            else:
                # Modo LEGACY: Sistema actual
                logger.info("[HOMEPAGE] Usando sistema legacy (feature flags deshabilitados)")

                # Banner informativo sobre el nuevo sistema
                if feature_flags and not unified_enabled:
                    info_banner = dbc.Alert(
                        html.Div(
                            [
                                html.I(className="fas fa-info-circle me-2"),
                                "El nuevo panel de estado unificado está disponible. ",
                                "Contacte al administrador para habilitarlo.",
                            ]
                        ),
                        color="info",
                        dismissable=True,
                        className="mb-3",
                    )

                    unified_content = [info_banner]
                else:
                    unified_content = []

                # Mostrar paneles legacy
                legacy_style = {"display": "block"}

                return unified_content, legacy_style

        except Exception as e:
            logger.error(f"[HOMEPAGE] Error en render_unified_or_legacy_status: {str(e)}")

            # Fallback seguro: mostrar sistema legacy
            error_alert = dbc.Alert(
                [
                    html.I(className="fas fa-exclamation-triangle me-2"),
                    f"Error cargando componente de estado: {str(e)}. Usando sistema legacy.",
                ],
                color="warning",
                dismissable=True,
            )

            return [error_alert], {"display": "block"}

    # Store Intermedio para datos específicos de homepage
    @app.callback(
        Output("homepage-data-store", "data"),
        [Input("dashboard-data-store", "data"), Input("url", "pathname")],
        prevent_initial_call=True,
    )
    def update_homepage_store(dashboard_data, pathname):
        """Actualiza el store intermedio solo cuando estamos en homepage"""
        import datetime

        from dash.exceptions import PreventUpdate

        # Solo actualizar si estamos en la homepage
        # Fix Issue #152: pathname puede ser "/", "/homepage" o "/home"
        if pathname not in ["/", "/homepage", "/home"]:
            raise PreventUpdate

        return {
            "source": "homepage",
            "data": {"dashboard": dashboard_data or {}},
            "last_update": datetime.datetime.now().isoformat(),
        }

    # FASE 4: Callback para actualizar feature flags cuando se necesite
    @app.callback(
        Output("homepage-feature-flags-store", "data", allow_duplicate=True),
        Input("url", "pathname"),
        prevent_initial_call=True,
    )
    def refresh_homepage_feature_flags(pathname):
        """
        Refresca feature flags cuando se navega a homepage.
        Permite activación/desactivación en tiempo real.
        """
        # Fix Issue #152: pathname puede ser "/", "/homepage" o "/home"
        if pathname not in ["/", "/homepage", "/home"]:
            raise PreventUpdate

        # Recargar flags actuales
        current_flags = FeatureFlags.get_all_flags()
        logger.info(
            f"[HOMEPAGE] Feature flags refrescados: unified_status_panel={current_flags.get('unified_status_panel', {}).get('enabled', False)}"
        )

        return current_flags

    @app.callback(
        [
            Output("dashboard-update-interval", "disabled"),
            Output("catalog-initial-load-interval", "disabled"),
        ],
        Input("url", "pathname"),
        [
            State("dashboard-update-interval", "disabled"),
            State("catalog-initial-load-interval", "disabled"),
        ],
        prevent_initial_call=False,  # Ejecutar al cargar
    )
    def enable_dashboard_intervals(
        pathname, current_dash_disabled, current_catalog_disabled
    ):
        """
        Habilita intervals solo cuando usuario está en homepage.
        Optimizado para evitar actualizaciones innecesarias.

        Issue #315 FIX: Los intervals estaban permanentemente deshabilitados en app.py,
        impidiendo que fetch_dashboard_data() se disparara. Este callback habilita
        los intervals dinámicamente cuando el usuario navega a la homepage, permitiendo
        la carga de datos del dashboard.

        OPTIMIZACIÓN (Code Review): Usa State para verificar estado actual y evitar
        resets innecesarios cuando se navega entre variaciones de homepage (/, /homepage, /home).

        Args:
            pathname: Ruta URL actual
            current_dash_disabled: Estado actual del dashboard interval
            current_catalog_disabled: Estado actual del catalog interval

        Returns:
            tuple: (dashboard_interval_disabled, catalog_interval_disabled)
                - False, False: Habilitar ambos intervals en homepage
                - True, True: Deshabilitar en otras páginas
                - no_update: Si no hay cambio necesario
        """
        is_homepage = pathname in ["/", "/homepage", "/home"]

        if is_homepage:
            # Solo actualizar si están deshabilitados
            if current_dash_disabled or current_catalog_disabled:
                logger.info(
                    f"[HOMEPAGE] Habilitando intervals para carga de datos del dashboard (desde: {pathname})"
                )
                return False, False
        else:
            # Solo actualizar si están habilitados
            if not current_dash_disabled or not current_catalog_disabled:
                logger.info(
                    f"[HOMEPAGE] Deshabilitando intervals al salir de homepage (hacia: {pathname})"
                )
                return True, True

        # Sin cambios necesarios
        return dash.no_update, dash.no_update

    @app.callback(
        Output("dashboard-data-store", "data"),
        [
            Input("url", "pathname"),  # Trigger inmediato al navegar a /home
            Input("dashboard-update-interval", "n_intervals"),
            Input("catalog-initial-load-interval", "n_intervals"),
        ],
        State("auth-state", "data"),
        prevent_initial_call=True,
    )
    def fetch_dashboard_data(pathname, n_intervals, initial_load, auth_state):
        """
        Obtiene los datos del dashboard desde el endpoint unificado.

        Optimización: Reduce 5-6 API calls a 1 sola llamada HTTP.
        Issue #302: Verificación proactiva de autenticación antes de llamadas API.
        Fix: Trigger inmediato con pathname para evitar espera de intervals.
        """
        # Guard: Solo ejecutar en homepage
        if pathname not in ["/", "/homepage", "/home"]:
            raise PreventUpdate

        # Guard proactivo: NO hacer peticiones API si usuario no autenticado
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.debug("[HOMEPAGE] User not authenticated - skipping dashboard data fetch")
            raise PreventUpdate

        try:
            from utils.pharmacy_context import get_current_pharmacy_id

            pharmacy_id = get_current_pharmacy_id()

            # ✅ OPTIMIZACIÓN: Una sola llamada al endpoint unificado
            # Reemplaza: get_system_status + get_pharmacy_dashboard_summary + sales_trend
            response = backend_client.get_initial_dashboard_data(pharmacy_id)

            if response.success and response.data:
                data = response.data

                # Log de rendimiento
                execution_time = data.get("execution_time_ms", "?")
                sales_data = data.get("sales", {})
                catalog_data = data.get("catalog", {})

                logger.info(
                    f"[HOMEPAGE] Dashboard data loaded in {execution_time}ms: "
                    f"catalog={catalog_data.get('total_products', 0)} products, "
                    f"sales={sales_data.get('total_records', 0)}, "
                    f"enriched={sales_data.get('enriched_records', 0)}"
                )

                # Mapear la estructura del endpoint unificado a la estructura esperada por los callbacks
                # Esto mantiene compatibilidad con callbacks existentes
                combined_data = {
                    # Datos directos del endpoint
                    "pharmacy": data.get("pharmacy", {}),
                    "sales": sales_data,
                    "catalog": catalog_data,
                    "system": data.get("system", {}),
                    "sales_trend": data.get("sales_trend", {}),
                    "cima": data.get("cima", {}),  # KPI CIMA
                    "nomenclator": data.get("nomenclator", {}),  # KPI Nomenclátor
                    # Campos de compatibilidad con estructura anterior
                    "status": data.get("system", {}).get("status", "unknown"),
                    "message": data.get("system", {}).get("message", ""),
                    "components": data.get("system", {}).get("components", {}),
                    "quick_stats": {
                        "products_in_catalog": catalog_data.get("total_products", 0),
                        "has_sales_data": sales_data.get("total_records", 0) > 0,
                        "has_enrichment": sales_data.get("enriched_records", 0) > 0,
                    },
                    "timestamp": data.get("timestamp"),
                    # KPIs Year-over-Year (optimización Issue #XXX: incluidos en initial-data)
                    "yoy_kpis": data.get("yoy_kpis", {}),
                }

                return combined_data
            else:
                logger.error(f"Error fetching dashboard data: {response.message}")
                return {}

        except Exception as e:
            logger.error(f"Error obteniendo datos del dashboard: {str(e)}")

        return {}

    # REGLA #11: Callback unificado para TODOS los updates desde homepage-data-store
    @app.callback(
        [
            # Outputs de dashboard display (2)
            Output("external-sources-alert", "children"),
            Output("external-sources-alert", "color"),
            # Outputs de external sources dates (6)
            Output("cima-last-sync", "children"),
            Output("nomen-last-sync", "children"),
            Output("cima-status-indicator", "className"),
            Output("nomen-status-indicator", "className"),
            Output("groups-status", "children"),
            Output("groups-status-indicator", "className"),
        ],
        Input("homepage-data-store", "data"),
        prevent_initial_call=True,
    )
    def handle_homepage_legacy_updates(homepage_data):
        """
        Callback unificado para actualizar componentes legacy (fuentes externas).

        Mantiene compatibilidad con sistema legacy mientras se usa nuevo dashboard.
        """

        # Valores por defecto
        DEFAULT_LEGACY_OUTPUTS = (
            "Sin datos disponibles",
            "warning",
            "--",
            "--",
            "bi bi-circle-fill text-muted me-2",
            "bi bi-circle-fill text-muted me-2",
            "--",
            "bi bi-circle-fill text-muted me-2",
        )

        if not homepage_data or not homepage_data.get("data"):
            return DEFAULT_LEGACY_OUTPUTS

        homepage_inner_data = homepage_data.get("data")
        if not homepage_inner_data or not isinstance(homepage_inner_data, dict):
            return DEFAULT_LEGACY_OUTPUTS

        data = homepage_inner_data.get("dashboard", {})
        system_status = homepage_inner_data.get("system", {})

        # Determinar mensaje y color de fuentes externas
        if system_status and system_status.get("components"):
            components = system_status.get("components", {})
            cima_info = components.get("cima", {})
            nomen_info = components.get("nomenclator", {})
            groups_info = components.get("homogeneous_groups", {})

            cima_products = cima_info.get("products", 0)
            nomen_products = nomen_info.get("products", 0)

            status_parts = []

            if cima_products > 0:
                status_parts.append(f"✅ CIMA ({format_number(cima_products)})")
            else:
                status_parts.append("⚠️ CIMA (sin datos)")

            if nomen_products > 0:
                status_parts.append(f"✅ Nomenclátor ({format_number(nomen_products)})")
            else:
                status_parts.append("⚠️ Nomenclátor (sin datos)")

            groups_count = groups_info.get("count", 0)
            if groups_count > 0:
                status_parts.append(f"✅ Grupos ({format_number(groups_count)})")

            if cima_products == 0 or nomen_products == 0:
                sources_color = "warning"
                sources_message = "Fuentes parciales: " + " | ".join(status_parts)
            else:
                sources_color = "success"
                sources_message = "Fuentes disponibles: " + " | ".join(status_parts)

            # Formatear fechas
            def format_sync_date(date_str, has_data=False, products_count=0):
                if date_str:
                    try:
                        from datetime import datetime

                        dt = datetime.fromisoformat(date_str.replace("Z", "+00:00"))
                        return dt.strftime("%d/%m/%Y %H:%M")
                    except:
                        return date_str
                if has_data or products_count > 0:
                    return "Datos disponibles"
                return "Sin datos"

            def get_indicator_class(status, products_count):
                if products_count > 0:
                    return "bi bi-circle-fill text-success me-2"
                elif status in ["READY", "ready"]:
                    return "bi bi-circle-fill text-success me-2"
                elif status in ["INITIALIZING", "initializing"]:
                    return "bi bi-circle-fill text-info me-2"
                elif status in ["ERROR", "error"]:
                    return "bi bi-circle-fill text-danger me-2"
                else:
                    return "bi bi-circle-fill text-warning me-2"

            cima_date_str = cima_info.get("last_success_at")
            nomen_date_str = nomen_info.get("last_success_at")

            cima_date = format_sync_date(cima_date_str, cima_products > 0, cima_products)
            nomen_date = format_sync_date(nomen_date_str, nomen_products > 0, nomen_products)

            cima_status_value = cima_info.get("status", "unknown")
            nomen_status_value = nomen_info.get("status", "unknown")

            cima_indicator = get_indicator_class(cima_status_value, cima_products)
            nomen_indicator = get_indicator_class(nomen_status_value, nomen_products)

            groups_status_value = groups_info.get("status", "unknown")

            if groups_count > 0:
                groups_text = f"{groups_count:,} grupos".replace(",", ".")
                groups_indicator = get_indicator_class("ready", groups_count)
            else:
                groups_text = "Sin datos"
                groups_indicator = get_indicator_class(groups_status_value, groups_count)

            return (
                sources_message,
                sources_color,
                cima_date,
                nomen_date,
                cima_indicator,
                nomen_indicator,
                groups_text,
                groups_indicator,
            )

        return DEFAULT_LEGACY_OUTPUTS
