# frontend/layouts/generics.py
"""
Layout para el Panel de Análisis de Partners - REDESIGN 2025.
Paradigma: Partners vs No-Partners dentro del universo sustituible.

Issue #415: Hybrid layout con cabecera de filtros reutilizable.
Issue #426: Layout 50/50 filtros-contexto + Design System compliance.
"""

from datetime import datetime, timedelta

import dash_bootstrap_components as dbc
from dash import dcc, html

# ✅ REGLA #5: Design System - Importar tokens y componentes base
from components.base import BaseCard
from styles.design_tokens import COLORS, SPACING, TYPOGRAPHY

# ✅ FASE 4: Importar componentes reutilizables de filtros (FASE 1)
from components.filters import (
    ApplyButton,
    CommonFilterPanel,
    DateRangeFilter,
    EmployeeFilter,
)

# ✅ Issue #428: Gauge de Penetración ahora está inline en el layout


def get_generics_layout():
    """
    Layout redesigneado para análisis de partners.

    Issue #415 Nueva arquitectura HÍBRIDA:
    - CABECERA FILTROS (sticky): Fechas + Empleados + Botón Aplicar
    - Panel de Contexto (universo sustituible) - FIJO
    - Panel Selección Partners - DINÁMICO
    - Gráficos drill-down temporales
    - Tabla conjuntos homogéneos con detalle inline
    - Slider descuentos

    Returns:
        html.Div: Layout completo del panel partners
    """

    return html.Div(
        [
            # Header page (no sticky)
            html.Div(
                [
                    html.H1(
                        [
                            html.I(className="fas fa-handshake me-3", style={"color": COLORS["info"]}),
                            "Panel de Análisis de Partners",
                        ],
                        className="mb-2",
                        style={"fontSize": TYPOGRAPHY["h1"]["size"], "fontWeight": "600"},
                    ),
                    html.P(
                        "Análisis de partners vs. no-partners en el universo sustituible",
                        className="text-muted mb-4",
                        style={"fontSize": TYPOGRAPHY["body"]["size"]},
                    ),
                ],
                className="mb-4",
            ),
            # ===================================================================
            # Issue #426: LAYOUT SUPERIOR 50/50
            # IZQUIERDA: Filtros + (Partners Óptimos | Gauge lado a lado)
            # DERECHA: Contexto de Ventas (altura completa)
            # ===================================================================
            dbc.Row(
                [
                    # IZQUIERDA (50%): Filtros arriba + Partners/Gauge lado a lado abajo
                    dbc.Col(
                        [
                            # Card 1: Filtros de Análisis
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [
                                            html.I(className="fas fa-filter me-2", style={"color": COLORS["info"]}),
                                            html.Span("Filtros de Análisis", style={"fontWeight": "600"}),
                                        ],
                                        style={
                                            "backgroundColor": COLORS["bg_tertiary"],
                                            "border": "none",
                                            "padding": f"{SPACING['s']} {SPACING['m']}"
                                        },
                                    ),
                                    dbc.CardBody(
                                        [
                                            # Fila 1: Período de Análisis (ancho completo)
                                            dbc.Row(
                                                [
                                                    dbc.Col(
                                                        [
                                                            html.H6(
                                                                "📅 Período de Análisis",
                                                                className="mb-2",
                                                                style={"fontWeight": "600"},
                                                            ),
                                                            dcc.DatePickerRange(
                                                                id="generics-date-range",
                                                                display_format="DD/MM/YYYY",
                                                                start_date_placeholder_text="Cargando...",
                                                                end_date_placeholder_text="Cargando...",
                                                                calendar_orientation="horizontal",
                                                                first_day_of_week=1,
                                                                month_format="MMMM YYYY",
                                                                className="w-100",
                                                                clearable=False,
                                                                persistence=True,
                                                                persistence_type="session",
                                                            ),
                                                            # Issue #420: FREE tier date restriction banner
                                                            html.Div(
                                                                id="generics-date-restriction-banner",
                                                                style={"display": "none"},
                                                            ),
                                                        ],
                                                        width=12,
                                                        className="mb-3",
                                                    ),
                                                ],
                                            ),
                                            # Fila 2: Selector Visual de Fechas (ancho completo)
                                            dbc.Row(
                                                [
                                                    dbc.Col(
                                                        [
                                                            html.H6(
                                                                "📊 Selector Visual de Fechas",
                                                                className="mb-2",
                                                                style={"fontWeight": "600"},
                                                            ),
                                                            dcc.RangeSlider(
                                                                id="generics-date-slider",
                                                                min=0,
                                                                max=100,
                                                                step=1,
                                                                marks={},
                                                                value=[0, 100],
                                                                tooltip={"placement": "bottom", "always_visible": False},
                                                                persistence=True,
                                                                persistence_type="session",
                                                            ),
                                                        ],
                                                        width=12,
                                                        className="mb-3",
                                                    ),
                                                ],
                                            ),
                                            # Fila 3: Filtro de Empleados (ancho completo)
                                            dbc.Row(
                                                [
                                                    dbc.Col(
                                                        [
                                                            html.H6(
                                                                "👥 Filtro de Empleados",
                                                                className="mb-2",
                                                                style={"fontWeight": "600"},
                                                            ),
                                                            dcc.Dropdown(
                                                                id="generics-employee-filter",
                                                                multi=True,
                                                                placeholder="Todos los empleados",
                                                                options=[],
                                                                value=[],
                                                                className="w-100",
                                                                persistence=True,
                                                                persistence_type="session",
                                                                clearable=True,
                                                                searchable=True,
                                                            ),
                                                            html.Div(
                                                                [
                                                                    dbc.Badge(
                                                                        html.Span([
                                                                            html.I(className="fas fa-crown me-1"),
                                                                            "Disponible en PRO",
                                                                        ]),
                                                                        id="generics-employee-filter-badge",
                                                                        color="secondary",
                                                                        className="mt-2",
                                                                        pill=True,
                                                                        style={"fontSize": "0.7rem", "opacity": "0.8"},
                                                                    ),
                                                                    # Issue #435: Tooltip explicativo para FREE tier
                                                                    dbc.Tooltip(
                                                                        "El filtro de empleados permite analizar ventas por vendedor. "
                                                                        "Esta función está disponible en planes PRO y MAX.",
                                                                        target="generics-employee-filter-badge",
                                                                        placement="top",
                                                                    ),
                                                                ]
                                                            ),
                                                        ],
                                                        width=12,
                                                        className="mb-3",
                                                    ),
                                                ],
                                            ),
                                            # Fila 4: Botón alineado a la derecha
                                            dbc.Row(
                                                dbc.Col(
                                                    dbc.Button(
                                                        html.Span([
                                                            html.I(className="fas fa-check me-2"),
                                                            "Aplicar Filtros",
                                                        ]),
                                                        id="generics-apply-btn",
                                                        color="primary",
                                                        className="px-4",
                                                    ),
                                                    width="auto",
                                                    className="ms-auto",
                                                ),
                                                className="mt-3",
                                            ),
                                        ],
                                        style={"padding": SPACING["m"]},
                                    ),
                                ],
                                className="mb-3",
                            ),
                            # Card 2: Partners (mismo ancho que Filtros)
                            create_partners_selection_panel(),
                        ],
                        width=12,
                        lg=6,
                        className="mb-4",
                    ),
                    # DERECHA (50%): Contexto + Penetración/Partners Óptimos
                    dbc.Col(
                        [
                            # Contexto de Ventas
                            create_context_panel(),
                            # Penetración + Partners Óptimos lado a lado
                            dbc.Row(
                                [
                                    dbc.Col(
                                        create_penetration_gauge_card(),
                                        width=6,
                                    ),
                                    dbc.Col(
                                        create_optimal_partners_card(),
                                        width=6,
                                    ),
                                ],
                                className="g-2",
                            ),
                        ],
                        width=12,
                        lg=6,
                        className="mb-4",
                    ),
                ]
            ),
            # ===================================================================
            # FIN LAYOUT SUPERIOR
            # ===================================================================
            # ===================================================================
            # SECCIÓN PRINCIPAL: Gráficos 50/50 + Matriz + Detalle
            # ===================================================================
            # Gráficos de Ventas y Unidades lado a lado (50/50)
            dbc.Row(
                [
                    dbc.Col(
                        create_sales_chart(),
                        width=12,
                        lg=6,
                    ),
                    dbc.Col(
                        create_units_chart(),
                        width=12,
                        lg=6,
                    ),
                ],
                className="mb-4",
            ),
            # Matriz (8 cols) + Evolución y Descuento (4 cols)
            dbc.Row(
                [
                    # Matriz de conjuntos homogéneos (66.7%)
                    dbc.Col(
                        create_homogeneous_matrix(),
                        width=12,
                        lg=8,
                    ),
                    # Evolución + Descuento (33.3%)
                    dbc.Col(
                        [
                            create_homogeneous_detail_chart(),
                            create_discount_slider(),
                        ],
                        width=12,
                        lg=4,
                    ),
                ],
                className="mb-4",
            ),
            # Tabla Referencias Partners debajo de todo - full width
            dbc.Row([
                dbc.Col([
                    html.Div(
                        create_partner_references_table(),
                        style={"marginTop": SPACING["m"]},
                    )
                ], width=12)
            ]),
            # ===================================================================
            # STORES Y COMPONENTES MOVIDOS A SKELETON GLOBAL (REGLA #0.5)
            # Fix dependency cycle: componentes ahora están en app.py
            # ===================================================================
            # Los siguientes stores/intervals YA NO están aquí, están en app.py:
            # - context-store
            # - analysis-store
            # - temporal-drill-store
            # - partners-selection-store
            # - matrix-sort-store
            # - homogeneous-expansion-state
            # - partners-load-result-store
            # - context-refresh-interval
            # - download-homogeneous-csv
            #
            # Solo mantenemos stores específicos de generics que NO causan ciclos:
            # ===================================================================
            dcc.Store(id="homogeneous-detail-store"),  # Detalles de conjuntos expandidos (específico)
            # Modal para mensajes
            dbc.Modal(
                [
                    dbc.ModalHeader(dbc.ModalTitle(id="partners-modal-title")),
                    dbc.ModalBody(id="partners-modal-body"),
                    dbc.ModalFooter(dbc.Button("Cerrar", id="close-partners-modal-btn", className="ms-auto")),
                ],
                id="partners-modal",
                is_open=False,
            ),
        ]
    )


def create_penetration_gauge_card():
    """
    Card con gauge de penetración para la columna derecha.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-tachometer-alt me-2", style={"color": COLORS["primary"]}),
                    html.Span("Penetración", style={"fontWeight": "600", "fontSize": TYPOGRAPHY["body"]["size"]}),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    dcc.Loading(
                        id="penetration-gauge-loading",
                        type="default",
                        children=[
                            dcc.Graph(
                                id="partner-penetration-gauge-chart",
                                config={"displayModeBar": False, "responsive": True},
                                style={"height": "180px"},
                            ),
                            html.Div(
                                id="penetration-gauge-legend",
                                className="mt-1",
                            ),
                        ]
                    ),
                ],
                style={"padding": SPACING["s"]},
            ),
        ],
        className="mb-3",
    )


def create_optimal_partners_card():
    """
    Card con lista compacta de partners óptimos para la columna derecha.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-bullseye me-2", style={"color": COLORS["primary"]}),
                    html.Span("Partners Óptimos", style={"fontWeight": "600", "fontSize": TYPOGRAPHY["body"]["size"]}),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # Selector de cantidad de partners (inline)
                    html.Div(
                        [
                            html.Small(
                                "Analizar:",
                                className="text-muted me-2",
                                style={"fontWeight": "500", "fontSize": "0.8rem"}
                            ),
                            dcc.Dropdown(
                                id="optimal-partners-count-selector",
                                options=[
                                    {"label": f"Top {i}", "value": i}
                                    for i in range(1, 11)
                                ],
                                value=5,
                                clearable=False,
                                searchable=False,
                                persistence=True,
                                persistence_type="session",
                                style={"width": "90px", "display": "inline-block", "fontSize": "0.85rem"}
                            ),
                        ],
                        className="d-flex align-items-center mb-3"
                    ),
                    # Lista compacta de partners con cobertura
                    dcc.Loading(
                        id="optimal-partners-loading",
                        type="default",
                        children=html.Div(
                            id="optimal-partners-compact-list",
                            children=[
                                html.Div(
                                    [
                                        html.I(className="fas fa-info-circle me-2", style={"color": COLORS["info"]}),
                                        html.Small(
                                            "Aplica filtros para ver partners óptimos",
                                            className="text-muted",
                                            style={"fontSize": "0.8rem"}
                                        ),
                                    ],
                                    className="text-center p-3",
                                    style={
                                        "backgroundColor": COLORS["bg_tertiary"],
                                        "borderRadius": "4px",
                                    }
                                )
                            ]
                        ),
                    ),
                ],
                style={"padding": SPACING["m"]},
            ),
        ],
        className="mb-3",
    )


def create_context_panel():
    """
    Panel de contexto con treemap visual + métricas numéricas.

    Issue #415: Treemap interactivo mostrando jerarquía de ventas completa.
    Issue #426: Terminología actualizada y treemap con flujo completo:
    Total → Venta Libre / Prescripción → Con Conjunto Homogéneo → En Vademécum → Oportunidad

    Versión híbrida: Treemap arriba + métricas compactas abajo.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-chart-pie me-2", style={"color": COLORS["info"]}),
                    html.Span("Contexto de Ventas", style={"fontWeight": "600", "fontSize": TYPOGRAPHY["body"]["size"]}),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # =========================================================
                    # Issue #426: TREEMAP VISUAL CON FLUJO COMPLETO
                    # =========================================================
                    dcc.Loading(
                        id="context-treemap-loading",
                        type="default",
                        children=dcc.Graph(
                            id="generics-context-treemap",
                            config={"displayModeBar": False},
                            style={"height": "220px"},  # Aumentado para más niveles
                            figure={
                                "data": [],
                                "layout": {
                                    "margin": {"t": 10, "b": 10, "l": 10, "r": 10},
                                    "annotations": [{
                                        "text": "Cargando contexto...",
                                        "showarrow": False,
                                        "font": {"size": 14, "color": COLORS["text_secondary"]}
                                    }]
                                }
                            }
                        ),
                    ),
                    html.Hr(className="my-2"),
                    # =========================================================
                    # Issue #426: MÉTRICAS CON TERMINOLOGÍA ACTUALIZADA
                    # =========================================================
                    dcc.Loading(
                        id="context-panel-loading",
                        type="default",
                        children=html.Div(
                            id="context-panel-content",
                            children=[
                                # Grid 2x2 para métricas compactas
                                dbc.Row(
                                    [
                                        dbc.Col(
                                            [
                                                html.Small("Con Conjunto Homogéneo:", className="text-muted"),
                                                html.H6(
                                                    id="context-total-substitutable",
                                                    children="--",
                                                    className="mb-0 text-primary",
                                                ),
                                                html.Small(
                                                    id="context-total-percentage",
                                                    children="--",
                                                    className="text-muted",
                                                    style={"fontSize": TYPOGRAPHY["caption"]["size"]},
                                                ),
                                            ],
                                            width=6,
                                            className="mb-2",
                                        ),
                                        dbc.Col(
                                            [
                                                html.Small("En Vademécum:", className="text-muted"),
                                                html.H6(
                                                    id="context-analyzable-amount",
                                                    children="--",
                                                    className="mb-0 text-info",
                                                ),
                                                html.Small(
                                                    id="context-analyzable-percentage",
                                                    children="--",
                                                    className="text-muted",
                                                    style={"fontSize": TYPOGRAPHY["caption"]["size"]},
                                                ),
                                            ],
                                            width=6,
                                            className="mb-2",
                                        ),
                                    ]
                                ),
                                dbc.Row(
                                    [
                                        dbc.Col(
                                            [
                                                html.Small("Ya Partners:", className="text-muted"),
                                                html.H6(
                                                    id="context-current-partners",
                                                    children="--",
                                                    className="mb-0 text-success",
                                                ),
                                                html.Small(
                                                    id="context-current-percentage",
                                                    children="--",
                                                    className="text-muted",
                                                    style={"fontSize": TYPOGRAPHY["caption"]["size"]},
                                                ),
                                            ],
                                            width=6,
                                        ),
                                        dbc.Col(
                                            [
                                                html.Small("Oportunidad:", className="text-muted"),
                                                html.H6(
                                                    id="context-opportunity",
                                                    children="--",
                                                    className="mb-0 text-warning",
                                                ),
                                                html.Small(
                                                    id="context-opportunity-percentage",
                                                    children="--",
                                                    className="text-muted",
                                                    style={"fontSize": TYPOGRAPHY["caption"]["size"]},
                                                ),
                                            ],
                                            width=6,
                                        ),
                                    ]
                                ),
                            ],
                        ),
                    )
                ],
                style={"padding": SPACING["s"]},
            ),
        ],
        className="mb-3",
        style={"border": f"2px solid {COLORS['border_light']}"},
    )


def create_partners_selection_panel():
    """
    Panel de selección de laboratorios partners.
    Versión compacta para columna derecha.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-flask me-2", style={"color": COLORS["success"]}),
                    html.Span("Partners", style={"fontWeight": "600", "fontSize": TYPOGRAPHY["body"]["size"]}),
                    dbc.Button(
                        html.I(className="fas fa-sync"),
                        id="refresh-partners-list-btn",
                        size="sm",
                        color="light",
                        className="float-end",
                        title="Actualizar lista de partners",
                        style={"padding": f"{SPACING['xs']} {SPACING['s']}"},
                    ),
                ],
                style={"padding": f"{SPACING['s']} {SPACING['m']}"},
            ),
            dbc.CardBody(
                [
                    # Información compacta de partners seleccionados
                    html.Div(
                        [
                            html.Small("Seleccionados:", className="text-muted"),
                            html.H6(id="selected-partners-count", children="0 de 0", className="mb-2 text-primary"),
                        ]
                    ),
                    # Dropdown con búsqueda para partners (altura limitada)
                    dcc.Loading(
                        id="partners-dropdown-loading",
                        type="default",
                        children=html.Div(
                            [
                                html.Label("Seleccionar Partners:", className="form-label text-muted mb-2"),
                                dcc.Dropdown(
                                    id="partners-dropdown",
                                    value=[],  # ✅ FIX: Inicializar con lista vacía para prevenir None
                                    placeholder="Buscar y seleccionar partners...",
                                    multi=True,
                                    searchable=False,  # ✅ FIX ORDEN: searchable=True ordena alfabéticamente automáticamente
                                    clearable=True,
                                    style={"marginBottom": SPACING["m"]},
                                    optionHeight=35,
                                    maxHeight=200,
                                ),
                            ]
                        ),
                    ),
                    html.Hr(className="my-2"),
                    # Indicador de análisis automático
                    html.Div(
                        [
                            html.Small(
                                [
                                    html.I(className="fas fa-bolt me-1", style={"color": COLORS["warning"]}),
                                    "El análisis se actualiza automáticamente",
                                ],
                                className="text-muted text-center d-block mt-2",
                            )
                        ]
                    ),
                ],
                style={"padding": SPACING["m"]},
            ),
        ]
    )


def create_sales_chart():
    """
    Gráfico de ventas por período con drill-down.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-chart-bar me-2", style={"color": COLORS["info"]}),
                    html.Span("Ventas por Período (€)", style={"fontWeight": "600"}),
                    html.Div(
                        [
                            html.Small(
                                id="drill-level-indicator",
                                children="Nivel: Trimestre",
                                className="text-muted me-3",
                            ),
                            dbc.Button(
                                html.Div([html.I(className="fas fa-arrow-up me-1"), "Volver"]),
                                id="drill-up-btn",
                                size="sm",
                                color="light",
                                disabled=True,
                                className="float-end",
                            ),
                        ],
                        className="float-end",
                    ),
                ]
            ),
            dbc.CardBody(
                [
                    dcc.Loading(
                        id="sales-chart-loading",
                        type="default",
                        children=dcc.Graph(
                            id="sales-drill-down-chart",
                            config={"displayModeBar": False},
                            style={"height": "300px"},
                        ),
                    )
                ]
            ),
        ],
        className="h-100",
    )


def create_units_chart():
    """
    Gráfico de unidades por período.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-cubes me-2", style={"color": COLORS["primary"]}),
                    html.Span("Unidades por Período", style={"fontWeight": "600"}),
                ]
            ),
            dbc.CardBody(
                [
                    dcc.Loading(
                        id="units-chart-loading",
                        type="default",
                        children=dcc.Graph(
                            id="units-drill-down-chart",
                            config={"displayModeBar": False},
                            style={"height": "300px"},
                        ),
                    )
                ]
            ),
        ],
        className="h-100",
    )


def create_drill_down_charts():
    """
    Gráficos de drill-down temporal con barras 100% apiladas.
    DEPRECATED: Usar create_sales_chart() y create_units_chart() por separado.
    """
    return html.Div(
        [
            create_sales_chart(),
            html.Div(style={"marginBottom": SPACING["m"]}),
            create_units_chart(),
        ]
    )


def create_discount_slider():
    """
    Slider de simulación de descuentos para partners.
    Versión compacta para columna derecha.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-percentage me-2", style={"color": COLORS["warning"]}),
                    html.Span("Descuento", style={"fontWeight": "600", "fontSize": TYPOGRAPHY["body"]["size"]}),
                ],
                style={
                    "backgroundColor": COLORS["warning_light"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # Slider compacto
                    html.Div(
                        [
                            dcc.Slider(
                                id="partner-discount-slider",
                                min=10,
                                max=50,
                                step=5,
                                value=25,
                                marks={10: "10%", 30: "30%", 50: "50%"},  # Menos marcas para espacio compacto
                                tooltip={"placement": "bottom", "always_visible": True},
                                updatemode='drag',  # ✅ FIX: Solo dispara callback al soltar (no durante drag)
                                className="mb-3",
                            )
                        ]
                    ),
                    # Resultado compacto
                    html.Div(
                        [
                            html.Small("Ahorro adicional:", className="text-muted"),
                            html.H6(id="discount-additional-savings", children="--", className="mb-0 text-success"),
                        ],
                        className="text-center",
                    ),
                ],
                style={"padding": SPACING["m"]},
            ),
        ],
        style={"backgroundColor": COLORS["warning_light"]},
    )


def create_homogeneous_detail_chart():
    """
    Gráfico temporal de ventas/unidades del conjunto homogéneo seleccionado.

    Issue #346: Muestra evolución temporal del conjunto seleccionado.
    Actualizado dinámicamente vía callback cuando se selecciona un conjunto.
    Incluye botón drill-up para volver al nivel anterior.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-chart-bar me-2", style={"color": COLORS["info"]}),
                    html.Span("Evolución del Conjunto Seleccionado", style={"fontWeight": "600", "fontSize": TYPOGRAPHY["body"]["size"]}),
                ],
                style={
                    "backgroundColor": COLORS["info_light"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                },
            ),
            dbc.CardBody(
                [
                    # Issue #346: Botón drill-up (consistente con gráficos principales)
                    html.Div(
                        [
                            dbc.Button(
                                html.Span([html.I(className="fas fa-arrow-up me-1"), "Volver"]),  # FIX DASH001: Envolver lista en html.Span
                                id="homogeneous-drill-up-btn",
                                size="sm",
                                color="secondary",
                                outline=True,
                                disabled=True,  # Inicialmente deshabilitado (nivel quarter)
                                className="mb-2",
                            )
                        ],
                        className="d-flex justify-content-end",  # Alineado a la derecha
                    ),
                    dcc.Loading(
                        id="homogeneous-detail-chart-loading",
                        type="default",
                        children=html.Div(
                            id="homogeneous-detail-chart-container",
                            children=[
                                html.Div(
                                    [
                                        html.I(className="fas fa-mouse-pointer me-2 text-info"),
                                        html.P(
                                            "Selecciona un conjunto de la matriz para ver su evolución temporal",
                                            className="text-muted text-center mb-0",
                                        ),
                                    ],
                                    className="text-center py-3",
                                )
                            ],
                        ),
                    )
                ],
                style={"padding": SPACING["m"]},
            ),
        ],
        style={"backgroundColor": COLORS["info_light"]},
        className="mt-3",
    )


def create_homogeneous_matrix():
    """
    Matriz PowerBI-style para conjuntos homogéneos con jerarquía expandible.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-table me-2", style={"color": COLORS["danger"]}),
                    html.Span("Conjuntos Homogéneos", style={"fontWeight": "600"}),
                    dbc.ButtonGroup(
                        [
                            dbc.Button(
                                html.Div([html.I(className="fas fa-expand-arrows-alt me-1"), "Expandir Todo"]),
                                id="expand-all-homogeneous-btn",
                                color="info",
                                size="sm",
                            ),
                            dbc.Button(
                                html.Div([html.I(className="fas fa-download me-1"), "Exportar"]),
                                id="export-homogeneous-btn",
                                color="success",
                                size="sm",
                            ),
                        ],
                        className="float-end",
                    ),
                ]
            ),
            dbc.CardBody(
                [
                    # Búsqueda con autocompletado dinámico
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Div(
                                        [
                                            html.I(
                                                className="fas fa-search",
                                                style={
                                                    "position": "absolute",
                                                    "left": "12px",
                                                    "top": "8px",
                                                    "color": COLORS["text_secondary"],
                                                    "zIndex": "10",
                                                },
                                            ),
                                            dcc.Dropdown(
                                                id="homogeneous-search-input",
                                                placeholder="Buscar conjunto (empieza a escribir)...",
                                                options=[],  # Se llenará dinámicamente
                                                value=None,
                                                searchable=True,
                                                clearable=True,
                                                style={"paddingLeft": "30px"},
                                                optionHeight=40,
                                            ),
                                        ],
                                        style={"position": "relative"},
                                    )
                                ],
                                lg=12,
                            )
                        ],
                        className="mb-3",
                    ),
                    # Contenedor matriz PowerBI con ordenación en headers y scroll (Issue #346)
                    dcc.Loading(
                        id="matrix-loading",
                        type="default",
                        children=html.Div(
                            id="homogeneous-matrix-container",
                            style={"maxHeight": "500px", "overflowX": "auto", "overflowY": "auto"},
                        ),
                    ),
                    # Resumen inferior
                    html.Hr(),
                    html.Div(id="homogeneous-matrix-summary", className="text-muted text-center"),
                ]
            ),
        ]
    )


def create_partner_references_table():
    """
    Tabla de referencias de productos partners (33% width).
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-pills me-2", style={"color": COLORS["success"]}),
                    html.Span("Referencias Partners", style={"fontWeight": "600"}),
                ]
            ),
            dbc.CardBody(
                [
                    # Resumen compacto
                    html.Div(
                        [
                            dbc.Row(
                                [
                                    dbc.Col(
                                        [
                                            html.Small("Referencias:", className="text-muted d-block"),
                                            html.H6(
                                                id="partner-refs-count", children="--", className="mb-0 text-primary"
                                            ),
                                        ],
                                        width=6,
                                    ),
                                    dbc.Col(
                                        [
                                            html.Small("Ventas:", className="text-muted d-block"),
                                            html.H6(
                                                id="partner-refs-sales", children="--", className="mb-0 text-success"
                                            ),
                                        ],
                                        width=6,
                                    ),
                                ],
                                className="mb-3",
                            )
                        ]
                    ),
                    # Filtro rápido
                    dbc.InputGroup(
                        [
                            dbc.InputGroupText(html.I(className="fas fa-search")),
                            dbc.Input(
                                id="partner-refs-search",
                                placeholder="Buscar código o descripción...",
                                debounce=False,
                                size="sm",
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Tabla de referencias
                    dcc.Loading(
                        id="references-loading",
                        type="default",
                        children=html.Div(
                            id="partner-references-container", style={"maxHeight": "500px", "overflowY": "auto"}
                        ),
                    ),
                ]
            ),
        ]
    )
