# frontend/layouts/dashboard.py
"""
Layout del dashboard farmacéutico principal.
Navegación lateral fija con 6 secciones especializadas.
"""

import dash_bootstrap_components as dbc
from components.common import create_loading_spinner
from dash import dcc, html
from utils.constants import APP_CONFIG, COLORS


def get_dashboard_layout() -> html.Div:
    """
    Dashboard farmacéutico principal.
    """

    return html.Div(
        [
            # Header
            html.Div(
                [
                    html.H1(
                        [html.I(className="fas fa-chart-line me-2"), "Panel de Ventas Farmacéutico"], className="mb-2"
                    ),
                    html.P(
                        "Bienvenido al sistema de análisis farmacéutico especializado de xFarma",
                        className="text-muted mb-4",
                    ),
                ],
                className="mb-4",
            ),
            # Alert informativo
            dbc.Alert(
                html.Div(
                    [
                        html.I(className="fas fa-info-circle me-2"),
                        "Selecciona una sección en el menú lateral para comenzar el análisis.",
                    ]
                ),
                color="info",
                className="mb-4",
            ),
            # KPIs principales
            create_dashboard_kpis(),
            # Stores para datos
            dcc.Store(id="dashboard-data"),
            # Intervalos
            dcc.Interval(id="dashboard-interval", interval=APP_CONFIG["update_intervals"]["dashboard"], n_intervals=0),
        ]
    )


def create_dashboard_kpis() -> dbc.Row:
    """
    KPIs principales del dashboard.

    Returns:
        Row con KPIs básicos
    """

    return dbc.Row(
        [
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.Div(
                                        [
                                            html.I(
                                                className="fas fa-chart-bar",
                                                style={"fontSize": "2rem", "color": COLORS["primary"]},
                                            ),
                                        ],
                                        className="text-center mb-3",
                                    ),
                                    html.H5("Análisis Listo", className="text-center mb-2"),
                                    html.P("6 secciones especializadas", className="text-muted text-center mb-0"),
                                ]
                            )
                        ]
                    )
                ],
                width=12,
                md=6,
                lg=3,
                className="mb-3",
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.Div(
                                        [
                                            html.I(
                                                className="fas fa-prescription",
                                                style={"fontSize": "2rem", "color": COLORS["info"]},
                                            ),
                                        ],
                                        className="text-center mb-3",
                                    ),
                                    html.H5("Prescripción", className="text-center mb-2"),
                                    html.P("7 categorías ATC", className="text-muted text-center mb-0"),
                                ]
                            )
                        ]
                    )
                ],
                width=12,
                md=6,
                lg=3,
                className="mb-3",
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.Div(
                                        [
                                            html.I(
                                                className="fas fa-pills",
                                                style={"fontSize": "2rem", "color": COLORS["success"]},
                                            ),
                                        ],
                                        className="text-center mb-3",
                                    ),
                                    html.H5("Genéricos", className="text-center mb-2"),
                                    html.P("Motor inteligente", className="text-muted text-center mb-0"),
                                ]
                            )
                        ]
                    )
                ],
                width=12,
                md=6,
                lg=3,
                className="mb-3",
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.Div(
                                        [
                                            html.I(
                                                className="fas fa-shopping-cart",
                                                style={"fontSize": "2rem", "color": COLORS["warning"]},
                                            ),
                                        ],
                                        className="text-center mb-3",
                                    ),
                                    html.H5("Venta Libre", className="text-center mb-2"),
                                    html.P("ML + Cross-selling", className="text-muted text-center mb-0"),
                                ]
                            )
                        ]
                    )
                ],
                width=12,
                md=6,
                lg=3,
                className="mb-3",
            ),
        ]
    )


# Eliminamos create_nav_item ya que no es necesario


def create_nav_item_old(section_id: str, icon: str, title: str, description: str, active: bool = False) -> html.Div:
    """
    Crear item de navegación lateral.

    Args:
        section_id: ID único de la sección
        icon: Emoji o icono
        title: Título de la sección
        description: Descripción breve
        active: Si está activa

    Returns:
        Navigation item component
    """

    classes = "sidebar-nav-item"
    if active:
        classes += " active"

    return html.Div(
        [
            html.Div(
                [
                    html.Div(
                        [
                            html.Span(icon, className="nav-icon"),
                            html.Div(
                                [
                                    html.Strong(title, className="nav-title"),
                                    html.Div(description, className="nav-description"),
                                ],
                                className="nav-text",
                            ),
                        ],
                        className="nav-content",
                    )
                ],
                className="nav-inner",
            )
        ],
        className=classes,
        id=f"nav-{section_id}",
        **{"data-section": section_id},
    )


# Eliminamos create_welcome_section ya que está integrado en el layout principal


# Eliminamos create_welcome_kpis ya que ahora es create_dashboard_kpis


def create_welcome_kpis_old() -> dbc.Row:
    """
    KPIs de bienvenida para la sección por defecto.

    Returns:
        Row con KPIs básicos
    """

    return dbc.Row(
        [
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4("📊", className="text-center mb-3"),
                                    html.H5("Análisis Listo", className="text-center mb-2"),
                                    html.P("6 secciones especializadas", className="text-muted text-center mb-0"),
                                ]
                            )
                        ],
                        className="h-100 text-center border-primary",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4("🏥", className="text-center mb-3"),
                                    html.H5("Prescripción", className="text-center mb-2"),
                                    html.P("7 categorías ATC", className="text-muted text-center mb-0"),
                                ]
                            )
                        ],
                        className="h-100 text-center border-info",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4("💊", className="text-center mb-3"),
                                    html.H5("Genéricos", className="text-center mb-2"),
                                    html.P("Motor inteligente", className="text-muted text-center mb-0"),
                                ]
                            )
                        ],
                        className="h-100 text-center border-success",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4("🛍️", className="text-center mb-3"),
                                    html.H5("Venta Libre", className="text-center mb-2"),
                                    html.P("ML + Cross-selling", className="text-muted text-center mb-0"),
                                ]
                            )
                        ],
                        className="h-100 text-center border-warning",
                    )
                ],
                width=3,
            ),
        ],
        className="mb-4",
    )


# Funciones para crear contenido específico de cada sección
def create_ventas_section() -> html.Div:
    """Sección VENTAS - Análisis estratégico 13 meses"""
    return html.Div(
        [
            html.H2("📈 Análisis de Ventas Estratégico"),
            html.P("Análisis de 13 meses con drivers de crecimiento..."),
            create_loading_spinner(text="Cargando análisis de ventas..."),
        ]
    )


def create_prescripcion_section() -> html.Div:
    """Sección PRESCRIPCIÓN - 7 categorías + ATC"""
    return html.Div(
        [
            html.H2("🏥 Análisis de Prescripción"),
            html.P("7 categorías farmacéuticas con clasificación ATC..."),
            create_loading_spinner(text="Cargando análisis de prescripción..."),
        ]
    )


def create_genericos_section() -> html.Div:
    """Sección GENÉRICOS - Análisis de oportunidades con nuevo sistema"""

    # Importar el nuevo layout de genéricos
    from layouts.generics import get_generics_layout

    return get_generics_layout()


def create_venta_libre_section() -> html.Div:
    """Sección VENTA LIBRE - ML + cross-selling"""
    return html.Div(
        [
            html.H2("🛍️ Análisis Venta Libre"),
            html.P("Machine Learning aplicado a OTC y cross-selling..."),
            create_loading_spinner(text="Cargando análisis venta libre..."),
        ]
    )


def create_cargar_datos_section() -> html.Div:
    """Sección CARGAR DATOS - Upload auto-detección ERP con progreso de enriquecimiento"""

    # Importar componente de progreso de enriquecimiento
    from components.enrichment_progress import create_enrichment_progress_card

    return html.Div(
        [
            # Header de sección
            html.Div(
                [
                    html.H2(
                        [html.I(className="fas fa-upload me-3 text-primary"), "Sistema de Carga Inteligente"],
                        className="section-title mb-1",
                    ),
                    html.P(
                        [
                            "Carga automatizada con detección de duplicados, ",
                            html.Strong("enriquecimiento automático"),
                            " y actualización del dashboard en tiempo real",
                        ],
                        className="text-muted mb-4",
                    ),
                ]
            ),
            # Progreso de enriquecimiento
            dbc.Row(
                [
                    dbc.Col([create_enrichment_progress_card()], width=6),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [
                                            html.H5(
                                                [html.I(className="fas fa-cogs me-2 text-info"), "Proceso Automático"],
                                                className="mb-0 text-info",
                                            )
                                        ]
                                    ),
                                    dbc.CardBody(
                                        [
                                            html.Ol(
                                                [
                                                    html.Li(
                                                        [html.Strong("Carga de archivo "), "→ Validación formato ERP"]
                                                    ),
                                                    html.Li(
                                                        [html.Strong("Detección duplicados "), "→ Descarte automático"]
                                                    ),
                                                    html.Li(
                                                        [
                                                            html.Strong("Enriquecimiento CIMA/Nomenclator "),
                                                            "→ Matching inteligente",
                                                        ]
                                                    ),
                                                    html.Li(
                                                        [
                                                            html.Strong("Actualización dashboard "),
                                                            "→ Métricas en tiempo real",
                                                        ]
                                                    ),
                                                ],
                                                className="mb-0 small",
                                            ),
                                            html.Hr(),
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-info-circle me-2 text-primary"),
                                                    "El enriquecimiento con datos oficiales es automático tras cada carga",
                                                ],
                                                className="text-primary small",
                                            ),
                                        ]
                                    ),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=6,
                    ),
                ],
                className="mb-4",
            ),
            # Área de upload (placeholder)
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [
                                            html.H5(
                                                [
                                                    html.I(className="fas fa-file-upload me-2 text-success"),
                                                    "Área de Carga",
                                                ],
                                                className="mb-0 text-success",
                                            )
                                        ]
                                    ),
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-cloud-upload-alt fa-3x text-muted mb-3"),
                                                    html.H6(
                                                        "Arrastra tu archivo CSV/Excel aquí", className="text-muted"
                                                    ),
                                                    html.P(
                                                        "Soporta: Farmatic, Farmanager, Nixfarma, Unycop, IOFWin",
                                                        className="small text-muted mb-3",
                                                    ),
                                                    dbc.Button(
                                                        html.Div(
                                                            [
                                                                html.I(className="fas fa-folder-open me-2"),
                                                                "Seleccionar Archivo",
                                                            ]
                                                        ),
                                                        color="success",
                                                        outline=True,
                                                    ),
                                                ],
                                                className="text-center py-4",
                                            )
                                        ]
                                    ),
                                ],
                                className="shadow-sm border-success border-2 border-dashed",
                            )
                        ],
                        width=12,
                    )
                ]
            ),
        ]
    )


def create_configuracion_section() -> html.Div:
    """Sección CONFIGURACIÓN - Ajustes sistema"""
    return html.Div(
        [
            html.H2("⚙️ Configuración del Sistema"),
            html.P("Ajustes y preferencias de análisis..."),
            create_loading_spinner(text="Cargando configuración..."),
        ]
    )
