# frontend/layouts/config.py
"""
Layout de la página de configuración de xfarma.
Permite configurar ajustes de la aplicación y farmacia.
"""

import dash_bootstrap_components as dbc
from components.common import create_info_card, create_loading_spinner
from dash import dcc, html
from utils.constants import COLORS, ERP_TYPES


def get_config_layout() -> html.Div:
    """
    Obtener layout completo de la página de configuración.

    Returns:
        Componente html.Div con el layout de configuración
    """

    return html.Div(
        [
            # Header
            html.Div(
                [
                    html.H1([html.I(className="fas fa-cog me-2"), "Configuración"], className="mb-2"),
                    html.P("Ajustes de la aplicación y configuración de la farmacia", className="text-muted mb-4"),
                ],
                className="mb-4",
            ),
            # Tabs de configuración
            dbc.Tabs(
                [
                    # Tab de configuración general
                    dbc.Tab(
                        label="General",
                        tab_id="general-tab",
                        children=[html.Div([get_general_config_section()], className="p-4")],
                    ),
                    # Tab de configuración de farmacia
                    dbc.Tab(
                        label="Farmacia",
                        tab_id="pharmacy-tab",
                        children=[html.Div([get_pharmacy_config_section()], className="p-4")],
                    ),
                    # Tab de sistema
                    dbc.Tab(
                        label="Sistema",
                        tab_id="system-tab",
                        children=[html.Div([get_system_config_section()], className="p-4")],
                    ),
                    # Tab de ayuda
                    dbc.Tab(
                        label="Ayuda", tab_id="help-tab", children=[html.Div([get_help_section()], className="p-4")]
                    ),
                ],
                id="config-tabs",
                active_tab="general-tab",
            ),
            # Stores para configuración
            dcc.Store(id="config-data"),
            dcc.Store(id="pharmacy-data"),
            # Modal antiguo eliminado - usamos el modal v2 que está más abajo
        ]
    )  # Sin clase contenedora para consistencia


def get_general_config_section() -> html.Div:
    """
    Sección de configuración general.

    Returns:
        Componente html.Div con configuración general
    """

    return html.Div(
        [
            dbc.Row(
                [
                    # Configuraciones de aplicación
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [html.I(className="fas fa-cog me-2"), "Configuración de Aplicación"]
                                    ),
                                    dbc.CardBody(
                                        [
                                            # Tema de la aplicación
                                            html.Div(
                                                [
                                                    html.Label("Tema de la Interfaz", className="form-label fw-bold"),
                                                    dbc.RadioItems(
                                                        id="theme-selector",
                                                        options=[
                                                            {"label": "Claro", "value": "light"},
                                                            {"label": "Oscuro", "value": "dark"},
                                                            {"label": "Automático", "value": "auto"},
                                                        ],
                                                        value="light",
                                                        inline=True,
                                                    ),
                                                ],
                                                className="mb-3",
                                            ),
                                            # Idioma
                                            html.Div(
                                                [
                                                    html.Label("Idioma", className="form-label fw-bold"),
                                                    dbc.Select(
                                                        id="language-selector",
                                                        options=[
                                                            {"label": "Español", "value": "es"},
                                                            {"label": "Catalán", "value": "ca"},
                                                            {"label": "Euskera", "value": "eu"},
                                                        ],
                                                        value="es",
                                                    ),
                                                ],
                                                className="mb-3",
                                            ),
                                            # Frecuencia de actualización
                                            html.Div(
                                                [
                                                    html.Label(
                                                        "Actualización de Datos", className="form-label fw-bold"
                                                    ),
                                                    dbc.Select(
                                                        id="refresh-interval",
                                                        options=[
                                                            {"label": "30 segundos", "value": "30"},
                                                            {"label": "1 minuto", "value": "60"},
                                                            {"label": "5 minutos", "value": "300"},
                                                            {"label": "Manual", "value": "0"},
                                                        ],
                                                        value="30",
                                                    ),
                                                ],
                                                className="mb-3",
                                            ),
                                        ]
                                    ),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=12,
                        lg=6,
                    ),
                    # Configuración de notificaciones
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader([html.I(className="fas fa-bell me-2"), "Notificaciones"]),
                                    dbc.CardBody(
                                        [
                                            # Notificaciones de sistema
                                            html.Div(
                                                [
                                                    dbc.Switch(
                                                        id="notifications-system",
                                                        label="Notificaciones del sistema",
                                                        value=True,
                                                    )
                                                ],
                                                className="mb-3",
                                            ),
                                            # Alertas de stock
                                            html.Div(
                                                [
                                                    dbc.Switch(
                                                        id="notifications-stock",
                                                        label="Alertas de stock bajo",
                                                        value=True,
                                                    )
                                                ],
                                                className="mb-3",
                                            ),
                                            # Resumen diario
                                            html.Div(
                                                [
                                                    dbc.Switch(
                                                        id="notifications-daily",
                                                        label="Resumen diario por email",
                                                        value=False,
                                                    )
                                                ],
                                                className="mb-3",
                                            ),
                                            # Email para notificaciones
                                            html.Div(
                                                [
                                                    html.Label(
                                                        "Email para notificaciones", className="form-label fw-bold"
                                                    ),
                                                    dbc.Input(
                                                        id="notification-email",
                                                        type="email",
                                                        placeholder="correo@farmacia.com",
                                                    ),
                                                ]
                                            ),
                                        ]
                                    ),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=12,
                        lg=6,
                    ),
                ],
                className="mb-4",
            ),
            # Botones de acción
            html.Div(
                [
                    dbc.Button("Guardar Configuración", color="primary", id="save-general-config"),
                    dbc.Button("Restablecer", color="outline-secondary", className="ms-2", id="reset-general-config"),
                ],
                className="text-end",
            ),
        ]
    )


def get_pharmacy_config_section() -> html.Div:
    """
    Sección de configuración de farmacia.

    Returns:
        Componente html.Div con configuración de farmacia
    """

    return html.Div(
        [
            dbc.Row(
                [
                    # Datos de la farmacia
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader([html.I(className="fas fa-store me-2"), "Datos de la Farmacia"]),
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.Label("Nombre de la Farmacia", className="form-label fw-bold"),
                                                    dbc.Input(
                                                        id="pharmacy-name", type="text", placeholder="Farmacia San Juan"
                                                    ),
                                                ],
                                                className="mb-3",
                                            ),
                                            html.Div(
                                                [
                                                    html.Label("Código de Farmacia", className="form-label fw-bold"),
                                                    dbc.Input(id="pharmacy-code", type="text", placeholder="F001"),
                                                ],
                                                className="mb-3",
                                            ),
                                            html.Div(
                                                [
                                                    html.Label("Dirección", className="form-label fw-bold"),
                                                    dbc.Textarea(
                                                        id="pharmacy-address",
                                                        placeholder="Calle Principal, 123\n28001 Madrid",
                                                    ),
                                                ],
                                                className="mb-3",
                                            ),
                                            html.Div(
                                                [
                                                    html.Label("Teléfono", className="form-label fw-bold"),
                                                    dbc.Input(
                                                        id="pharmacy-phone", type="tel", placeholder="+34 91 123 45 67"
                                                    ),
                                                ],
                                                className="mb-3",
                                            ),
                                        ]
                                    ),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=12,
                        lg=6,
                    ),
                    # Configuración ERP
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader([html.I(className="fas fa-desktop me-2"), "Sistema ERP"]),
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.Label("Tipo de ERP Principal", className="form-label fw-bold"),
                                                    dbc.Select(
                                                        id="primary-erp-type",
                                                        options=[
                                                            {"label": name, "value": key}
                                                            for key, name in ERP_TYPES.items()
                                                        ],
                                                        value="farmatic",
                                                    ),
                                                ],
                                                className="mb-3",
                                            ),
                                            html.Div(
                                                [
                                                    html.Label("Versión", className="form-label fw-bold"),
                                                    dbc.Input(id="erp-version", type="text", placeholder="v2.1.0"),
                                                ],
                                                className="mb-3",
                                            ),
                                            html.Div(
                                                [
                                                    html.Label("Formato de Fechas", className="form-label fw-bold"),
                                                    dbc.Select(
                                                        id="date-format",
                                                        options=[
                                                            {"label": "dd/mm/yyyy", "value": "%d/%m/%Y"},
                                                            {"label": "yyyy-mm-dd", "value": "%Y-%m-%d"},
                                                            {"label": "mm/dd/yyyy", "value": "%m/%d/%Y"},
                                                        ],
                                                        value="%d/%m/%Y",
                                                    ),
                                                ],
                                                className="mb-3",
                                            ),
                                            html.Div(
                                                [
                                                    html.Label(
                                                        "Separador de Decimales", className="form-label fw-bold"
                                                    ),
                                                    dbc.RadioItems(
                                                        id="decimal-separator",
                                                        options=[
                                                            {"label": "Coma (,)", "value": ","},
                                                            {"label": "Punto (.)", "value": "."},
                                                        ],
                                                        value=",",
                                                        inline=True,
                                                    ),
                                                ]
                                            ),
                                        ]
                                    ),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=12,
                        lg=6,
                    ),
                ],
                className="mb-4",
            ),
            # Botones de acción
            html.Div(
                [
                    dbc.Button("Guardar Configuración", color="primary", id="save-pharmacy-config"),
                    dbc.Button("Restablecer", color="outline-secondary", className="ms-2", id="reset-pharmacy-config"),
                ],
                className="text-end",
            ),
        ]
    )


def get_system_config_section() -> html.Div:
    """
    Sección de configuración del sistema simplificada.

    Returns:
        Componente html.Div con información básica del sistema
    """

    return html.Div(
        [
            dbc.Row(
                [
                    # Información de la aplicación
                    dbc.Col(
                        [
                            create_info_card(
                                title="Información de la Aplicación",
                                icon="fas fa-info-circle",
                                color=COLORS["info"],
                                content=[
                                    html.Div(
                                        [
                                            html.P([html.Strong("Versión: "), "1.0.0"]),
                                            html.P([html.Strong("Última actualización: "), "25/12/2024"]),
                                            html.P([html.Strong("Entorno: "), "Desarrollo"]),
                                            html.P([html.Strong("Base de datos: "), "PostgreSQL 15"], className="mb-0"),
                                        ]
                                    )
                                ],
                            )
                        ],
                        width=12,
                        lg=6,
                    ),
                    # Estado básico del sistema
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader([html.I(className="fas fa-server me-2"), "Estado del Sistema"]),
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                id="system-status-basic",
                                                children=[create_loading_spinner(text="Verificando estado...")],
                                            ),
                                            html.Hr(),
                                            html.Div(
                                                [
                                                    dbc.Button(
                                                        html.Div(
                                                            [
                                                                html.I(className="fas fa-cogs me-2"),
                                                                "Administración Avanzada",
                                                            ]
                                                        ),
                                                        color="outline-primary",
                                                        href="/admin",
                                                        external_link=True,
                                                        className="w-100",
                                                    )
                                                ]
                                            ),
                                        ]
                                    ),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=12,
                        lg=6,
                    ),
                ],
                className="mb-4",
            ),
            # Herramientas básicas de mantenimiento
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader([html.I(className="fas fa-tools me-2"), "Herramientas Básicas"]),
                                    dbc.CardBody(
                                        [
                                            html.P(
                                                "Herramientas básicas para el usuario estándar.", className="text-muted"
                                            ),
                                            html.Div(
                                                [
                                                    dbc.Button(
                                                        html.Div(
                                                            [html.I(className="fas fa-broom me-2"), "Limpiar Cache"]
                                                        ),
                                                        color="outline-warning",
                                                        id="clear-cache-btn",
                                                        className="me-2 mb-2",
                                                    ),
                                                    dbc.Button(
                                                        html.Div(
                                                            [html.I(className="fas fa-sync me-2"), "Actualizar Datos"]
                                                        ),
                                                        color="outline-info",
                                                        id="update-system-btn",
                                                        className="me-2 mb-2",
                                                    ),
                                                ]
                                            ),
                                            html.Hr(),
                                            html.Div(
                                                [
                                                    html.P(
                                                        [
                                                            html.I(className="fas fa-info-circle text-info me-2"),
                                                            "Para operaciones avanzadas del sistema, utilice el ",
                                                            html.A(
                                                                "Panel de Administración",
                                                                href="/admin",
                                                                className="fw-bold",
                                                            ),
                                                            ".",
                                                        ],
                                                        className="text-muted small mb-0",
                                                    )
                                                ]
                                            ),
                                            html.Div(id="maintenance-status", className="mt-3"),
                                        ]
                                    ),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=12,
                    )
                ]
            ),
            # Alertas del sistema
            html.Div(id="system-alerts", className="mt-3"),
        ]
    )


def get_help_section() -> html.Div:
    """
    Sección de ayuda y documentación.

    Returns:
        Componente html.Div con ayuda
    """

    return html.Div(
        [
            dbc.Row(
                [
                    # Guías rápidas
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader([html.I(className="fas fa-question-circle me-2"), "Guías Rápidas"]),
                                    dbc.CardBody(
                                        [
                                            dbc.Accordion(
                                                [
                                                    dbc.AccordionItem(
                                                        [
                                                            html.P("1. Ve a la sección 'Cargar Datos'"),
                                                            html.P("2. Selecciona tu sistema ERP"),
                                                            html.P("3. Arrastra o selecciona el archivo de ventas"),
                                                            html.P("4. Espera a que se procese"),
                                                            html.P("5. Revisa los datos en el Dashboard"),
                                                        ],
                                                        title="¿Cómo subir datos de ventas?",
                                                    ),
                                                    dbc.AccordionItem(
                                                        [
                                                            html.P("Los archivos deben contener como mínimo:"),
                                                            html.Ul(
                                                                [
                                                                    html.Li("Fecha de venta"),
                                                                    html.Li("Código de producto"),
                                                                    html.Li("Nombre del producto"),
                                                                    html.Li("Cantidad vendida"),
                                                                    html.Li("Importe total"),
                                                                ]
                                                            ),
                                                        ],
                                                        title="¿Qué formato deben tener los archivos?",
                                                    ),
                                                    dbc.AccordionItem(
                                                        [
                                                            html.P("Si encuentras problemas:"),
                                                            html.Ul(
                                                                [
                                                                    html.Li("Verifica que el archivo no esté corrupto"),
                                                                    html.Li("Comprueba el formato de fechas"),
                                                                    html.Li(
                                                                        "Asegúrate de que los campos estén completos"
                                                                    ),
                                                                    html.Li(
                                                                        "Consulta la sección Sistema para ver el estado"
                                                                    ),
                                                                ]
                                                            ),
                                                        ],
                                                        title="¿Qué hacer si hay errores?",
                                                    ),
                                                ]
                                            )
                                        ]
                                    ),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=12,
                        lg=6,
                    ),
                    # Contacto y soporte
                    dbc.Col(
                        [
                            create_info_card(
                                title="Contacto y Soporte",
                                icon="fas fa-life-ring",
                                color=COLORS["primary"],
                                content=[
                                    html.P("¿Necesitas ayuda adicional?"),
                                    html.Div(
                                        [
                                            html.P(
                                                [
                                                    html.I(className="fas fa-envelope me-2"),
                                                    html.A("soporte@xfarma.com", href="mailto:soporte@xfarma.com"),
                                                ]
                                            ),
                                            html.P([html.I(className="fas fa-phone me-2"), "+34 900 123 456"]),
                                            html.P(
                                                [html.I(className="fas fa-clock me-2"), "L-V: 9:00-18:00"],
                                                className="mb-0",
                                            ),
                                        ]
                                    ),
                                ],
                            )
                        ],
                        width=12,
                        lg=6,
                    ),
                ],
                className="mb-4",
            ),
            # Documentación adicional
            dbc.Row(
                [
                    dbc.Col(
                        [
                            create_info_card(
                                title="Enlaces Útiles",
                                icon="fas fa-external-link-alt",
                                color=COLORS["secondary"],
                                content=[
                                    html.Div(
                                        [
                                            dbc.Button(
                                                html.Div([html.I(className="fas fa-book me-2"), "Manual de Usuario"]),
                                                color="outline-primary",
                                                size="sm",
                                                className="me-2 mb-2",
                                            ),
                                            dbc.Button(
                                                html.Div([html.I(className="fas fa-video me-2"), "Video Tutoriales"]),
                                                color="outline-success",
                                                size="sm",
                                                className="me-2 mb-2",
                                            ),
                                            dbc.Button(
                                                html.Div([html.I(className="fas fa-comments me-2"), "FAQ"]),
                                                color="outline-info",
                                                size="sm",
                                                className="mb-2",
                                            ),
                                        ]
                                    )
                                ],
                            )
                        ],
                        width=12,
                    )
                ]
            ),
        ]
    )
