"""
Layout para el Dashboard de Clustering/Taxonomía (Issue #458)

.. deprecated:: ADR-004
    Este módulo está deprecado desde ADR-004 (Dic 2024).
    El clustering dinámico ha sido reemplazado por grupos curados
    manualmente en Admin → Grupos Curados.

Visualización UMAP 2D interactiva para clasificación de productos venta libre.

Features:
- UMAP 2D scatter con colores por NECESIDAD
- Ghost points para productos movidos en sesión
- Panel de detalle de producto seleccionado
- Filtros por categoría y estado
- Métricas de cobertura y calidad
"""

import dash_bootstrap_components as dbc
from dash import dcc, html

# Design System imports (REGLA #5)
from components.base import BaseCard
from styles.design_tokens import COLORS, SPACING, TYPOGRAPHY

# Clustering components
from components.clustering import create_umap_scatter, create_umap_controls, create_umap_legend


def get_clustering_layout():
    """
    Layout para el dashboard de clustering/taxonomía.

    Estructura:
    - Header con título y KPIs
    - Panel izquierdo: UMAP scatter + controles
    - Panel derecho: Detalle producto + acciones

    Returns:
        html.Div: Layout completo
    """
    return html.Div([
        # ===================================================================
        # NOTA: Stores definidos en app.py skeleton (REGLA #0.5)
        # - umap-data-store (session)
        # - ghost-points-store (session)
        # - selected-product-store (memory)
        # Los stores NO se duplican aquí para evitar conflictos
        # ===================================================================

        # ===================================================================
        # DEPRECATION BANNER (ADR-004)
        # ===================================================================
        dbc.Alert([
            html.Div([
                html.I(className="fas fa-exclamation-triangle me-2"),
                html.Strong("Funcionalidad Deprecada"),
            ], className="d-flex align-items-center mb-2"),
            html.P([
                "El clustering dinámico ha sido reemplazado por ",
                html.Strong("Grupos Curados"),
                " (ADR-004). Esta página se eliminará en una futura versión.",
            ], className="mb-2"),
            dcc.Link(
                dbc.Button([
                    html.I(className="fas fa-arrow-right me-2"),
                    "Ir a Admin → Grupos Curados",
                ], color="warning", size="sm"),
                href="/admin?tab=curated-groups",
            ),
        ], color="warning", dismissable=True, className="mb-4"),

        # ===================================================================
        # HEADER
        # ===================================================================
        html.Div([
            html.H1([
                html.I(className="fas fa-project-diagram me-3", style={"color": COLORS["info"]}),
                "Taxonomía Venta Libre",
                html.Span(
                    " (Deprecado)",
                    style={"fontSize": "0.5em", "color": COLORS["warning"], "fontWeight": "normal"},
                ),
            ], className="mb-2", style={"fontSize": TYPOGRAPHY["h1"]["size"], "fontWeight": "600"}),
            html.P(
                "Visualización y validación de clasificación de productos parafarmacia",
                className="text-muted mb-4",
                style={"fontSize": TYPOGRAPHY["body"]["size"]},
            ),
        ], className="mb-4"),

        # ===================================================================
        # KPI CARDS
        # ===================================================================
        dbc.Row([
            dbc.Col([
                dbc.Card([
                    dbc.CardBody([
                        html.Div([
                            html.I(className="fas fa-layer-group fa-2x", style={"color": COLORS["info"]}),
                        ], className="text-center mb-2"),
                        html.H3(id="kpi-total-products", children="--", className="text-center mb-1"),
                        html.P("Productos", className="text-muted text-center small mb-0"),
                    ], className="py-3"),
                ], className="h-100 shadow-sm"),
            ], width=6, lg=3, className="mb-3"),

            dbc.Col([
                dbc.Card([
                    dbc.CardBody([
                        html.Div([
                            html.I(className="fas fa-map-marked-alt fa-2x", style={"color": COLORS["success"]}),
                        ], className="text-center mb-2"),
                        html.H3(id="kpi-with-coords", children="--", className="text-center mb-1"),
                        html.P("Con coordenadas", className="text-muted text-center small mb-0"),
                    ], className="py-3"),
                ], className="h-100 shadow-sm"),
            ], width=6, lg=3, className="mb-3"),

            dbc.Col([
                dbc.Card([
                    dbc.CardBody([
                        html.Div([
                            html.I(className="fas fa-percentage fa-2x", style={"color": COLORS["warning"]}),
                        ], className="text-center mb-2"),
                        html.H3(id="kpi-coverage", children="--", className="text-center mb-1"),
                        html.P("Cobertura", className="text-muted text-center small mb-0"),
                    ], className="py-3"),
                ], className="h-100 shadow-sm"),
            ], width=6, lg=3, className="mb-3"),

            dbc.Col([
                dbc.Card([
                    dbc.CardBody([
                        html.Div([
                            html.I(className="fas fa-check-circle fa-2x", style={"color": COLORS["success"]}),
                        ], className="text-center mb-2"),
                        html.H3(id="kpi-verified", children="--", className="text-center mb-1"),
                        html.P("Verificados", className="text-muted text-center small mb-0"),
                    ], className="py-3"),
                ], className="h-100 shadow-sm"),
            ], width=6, lg=3, className="mb-3"),
        ], className="mb-4"),

        # ===================================================================
        # MAIN CONTENT: UMAP + DETAIL PANEL
        # ===================================================================
        dbc.Row([
            # LEFT: UMAP Scatter
            dbc.Col([
                dbc.Card([
                    dbc.CardHeader([
                        html.I(className="fas fa-chart-scatter me-2", style={"color": COLORS["info"]}),
                        html.Span("Mapa UMAP 2D", style={"fontWeight": "600"}),
                        # Version badge
                        dbc.Badge(
                            id="umap-version-badge",
                            children="--",
                            color="secondary",
                            className="ms-2",
                        ),
                    ], style={
                        "backgroundColor": COLORS["bg_tertiary"],
                        "border": "none",
                        "padding": f"{SPACING['s']} {SPACING['m']}",
                    }),
                    dbc.CardBody([
                        # Controls
                        create_umap_controls(),

                        # Loading wrapper for UMAP scatter
                        dcc.Loading(
                            id="umap-loading",
                            type="circle",
                            children=[
                                html.Div(id="umap-scatter-container", children=[
                                    # Placeholder - will be populated by callback
                                    html.Div(
                                        "Cargando visualización UMAP...",
                                        className="text-center text-muted py-5",
                                    ),
                                ]),
                            ],
                        ),

                        # Legend
                        create_umap_legend(),

                        # Filter by NECESIDAD
                        html.Div([
                            dbc.Label("Filtrar por categoría:", className="small mt-3"),
                            dcc.Dropdown(
                                id="umap-necesidad-filter",
                                options=[],  # Populated by callback
                                value=None,
                                placeholder="Todas las categorías",
                                clearable=True,
                                className="small",
                            ),
                        ]),
                    ]),
                ], className="h-100 shadow-sm"),
            ], width=12, lg=7, className="mb-4"),

            # RIGHT: Product Detail Panel
            dbc.Col([
                dbc.Card([
                    dbc.CardHeader([
                        html.I(className="fas fa-info-circle me-2", style={"color": COLORS["info"]}),
                        html.Span("Detalle del Producto", style={"fontWeight": "600"}),
                    ], style={
                        "backgroundColor": COLORS["bg_tertiary"],
                        "border": "none",
                        "padding": f"{SPACING['s']} {SPACING['m']}",
                    }),
                    dbc.CardBody([
                        html.Div(
                            id="product-detail-panel",
                            children=[
                                html.Div([
                                    html.I(className="fas fa-mouse-pointer fa-3x text-muted mb-3"),
                                    html.P(
                                        "Haz clic en un punto del mapa para ver los detalles del producto",
                                        className="text-muted text-center",
                                    ),
                                ], className="text-center py-5"),
                            ],
                        ),
                    ]),
                ], className="h-100 shadow-sm"),
            ], width=12, lg=5, className="mb-4"),
        ]),

        # ===================================================================
        # STATUS/ALERTS SECTION
        # ===================================================================
        html.Div(id="clustering-alerts-container"),

    ], className="container-fluid px-4 py-3")


def create_product_detail_card(product: dict) -> html.Div:
    """
    Create product detail card for the right panel.

    Args:
        product: Product data dict

    Returns:
        html.Div with product details
    """
    risk_colors = {
        "high": "danger",
        "medium": "warning",
        "low": "success",
        "unknown": "secondary",
    }

    risk_level = product.get("risk_level", "unknown")
    risk_color = risk_colors.get(risk_level, "secondary")

    return html.Div([
        # Product name
        html.H5(product.get("product_name", "Unknown"), className="mb-3"),

        # Badges
        html.Div([
            dbc.Badge(
                product.get("necesidad", "Sin clasificar"),
                color="info",
                className="me-2",
            ),
            dbc.Badge(
                product.get("detected_brand", "Sin marca") or "Sin marca",
                color="secondary",
                className="me-2",
            ),
            dbc.Badge(
                "Verificado" if product.get("human_verified") else "Pendiente",
                color="success" if product.get("human_verified") else "warning",
            ),
        ], className="mb-3"),

        html.Hr(),

        # Metrics
        dbc.Row([
            dbc.Col([
                html.P("Confianza ML", className="small text-muted mb-1"),
                html.H6(
                    f"{product.get('ml_confidence', 0) * 100:.0f}%"
                    if product.get("ml_confidence") else "N/A"
                ),
            ], width=6),
            dbc.Col([
                html.P("Z-Score Centroide", className="small text-muted mb-1"),
                html.H6([
                    f"{product.get('centroid_distance_zscore', 0):.2f}"
                    if product.get("centroid_distance_zscore") else "N/A",
                    dbc.Badge(
                        risk_level.upper(),
                        color=risk_color,
                        className="ms-2",
                        style={"fontSize": "0.7rem"},
                    ),
                ]),
            ], width=6),
        ], className="mb-3"),

        dbc.Row([
            dbc.Col([
                html.P("Coordenadas UMAP", className="small text-muted mb-1"),
                html.H6(
                    f"({product.get('umap_x', 0):.2f}, {product.get('umap_y', 0):.2f})"
                ),
            ], width=12),
        ], className="mb-3"),

        html.Hr(),

        # Actions (placeholder for W15)
        html.Div([
            html.P("Acciones", className="small text-muted mb-2"),
            dbc.ButtonGroup([
                dbc.Button(
                    html.Span([html.I(className="fas fa-check me-1"), "Validar"]),
                    color="success", size="sm", disabled=True, className="me-1"
                ),
                dbc.Button(
                    html.Span([html.I(className="fas fa-exchange-alt me-1"), "Mover"]),
                    color="warning", size="sm", disabled=True, className="me-1"
                ),
                dbc.Button(
                    html.Span([html.I(className="fas fa-flag me-1"), "Reportar"]),
                    color="danger", size="sm", outline=True, disabled=True
                ),
            ], size="sm"),
            html.P(
                "Acciones disponibles en próxima versión (W15)",
                className="small text-muted mt-2 mb-0",
                style={"fontSize": "0.75rem"},
            ),
        ]),
    ])
