# frontend/layouts/auth.py
"""
Layout para páginas de autenticación (login/registro) con OAuth2
Soporta login tradicional y social (Google/Microsoft)
"""


import dash_bootstrap_components as dbc
from dash import html

# Configuración de backend
from utils.config import BACKEND_URL


def create_login_layout():
    """
    Crea el layout de la página de login con OAuth2

    Features:
    - Formulario de login tradicional
    - Botones de OAuth (Google/Microsoft)
    - Link a registro y recuperación de contraseña
    """
    return dbc.Container(
        [
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    # Logo y título
                                    html.Div(
                                        [
                                            html.I(className="fas fa-pills fa-3x text-primary mb-3"),
                                            html.H2("xFarma", className="text-primary mb-1"),
                                            html.P("Sistema de Gestión Farmacéutica", className="text-muted"),
                                        ],
                                        className="text-center mb-4",
                                    ),
                                    # Card de login
                                    dbc.Card(
                                        [
                                            dbc.CardBody(
                                                [
                                                    html.H4("Iniciar Sesión", className="card-title text-center mb-4"),
                                                    # Mensajes de error/éxito
                                                    html.Div(id="login-alert", className="mb-3"),
                                                    # Formulario tradicional
                                                    dbc.Form(
                                                        [
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label("Email", html_for="login-email"),
                                                                            dbc.Input(
                                                                                type="email",
                                                                                id="login-email",
                                                                                placeholder="tu@email.com",
                                                                                required=True,
                                                                            ),
                                                                            dbc.FormFeedback(
                                                                                "Por favor ingresa un email válido",
                                                                                type="invalid",
                                                                            ),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Contraseña", html_for="login-password"
                                                                            ),
                                                                            dbc.Input(
                                                                                type="password",
                                                                                id="login-password",
                                                                                placeholder="Tu contraseña",
                                                                                required=True,
                                                                            ),
                                                                            dbc.FormFeedback(
                                                                                "La contraseña es requerida",
                                                                                type="invalid",
                                                                            ),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Checkbox(
                                                                                id="remember-me",
                                                                                label="Recordarme",
                                                                                value=False,
                                                                                className="mb-3",
                                                                            ),
                                                                        ]
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Button(
                                                                [
                                                                    html.I(
                                                                        className="fas fa-sign-in-alt me-2",
                                                                        id="login-button-icon",
                                                                    ),
                                                                    html.Span("Iniciar Sesión", id="login-button-text"),
                                                                ],
                                                                id="login-button",
                                                                color="primary",
                                                                size="lg",
                                                                className="w-100 mb-3",
                                                                n_clicks=0,
                                                            ),
                                                        ]
                                                    ),
                                                    # Divider
                                                    html.Hr(className="my-4"),
                                                    # Social Login Status Notice
                                                    dbc.Alert(
                                                        html.Div(
                                                            [
                                                                html.I(className="fas fa-info-circle me-2"),
                                                                html.Strong("Nota: "),
                                                                "El inicio de sesión social (Google/Microsoft) está preparado pero requiere configuración. ",
                                                                "Por favor, usa email y contraseña para acceder.",
                                                            ]
                                                        ),
                                                        color="info",
                                                        className="small",
                                                        dismissable=True,
                                                    ),
                                                    html.P(
                                                        "O inicia sesión con:", className="text-center text-muted mb-3"
                                                    ),
                                                    # OAuth buttons
                                                    html.Div(
                                                        [
                                                            dbc.Button(
                                                                html.Div(
                                                                    [html.I(className="fab fa-google me-2"), "Google"]
                                                                ),
                                                                id="google-login-button",
                                                                color="danger",
                                                                outline=True,
                                                                size="lg",
                                                                className="w-100 mb-2",
                                                                href=f"{BACKEND_URL}/api/auth/oauth/google",
                                                                external_link=True,
                                                            ),
                                                            dbc.Button(
                                                                html.Div(
                                                                    [
                                                                        html.I(className="fab fa-microsoft me-2"),
                                                                        "Microsoft",
                                                                    ]
                                                                ),
                                                                id="microsoft-login-button",
                                                                color="info",
                                                                outline=True,
                                                                size="lg",
                                                                className="w-100 mb-3",
                                                                href=f"{BACKEND_URL}/api/auth/oauth/microsoft",
                                                                external_link=True,
                                                            ),
                                                        ],
                                                        id="oauth-buttons-container",
                                                    ),
                                                    # Links adicionales
                                                    html.Div(
                                                        [
                                                            html.A(
                                                                "¿Olvidaste tu contraseña?",
                                                                href="/auth/forgot-password",
                                                                className="d-block text-center mb-2",
                                                            ),
                                                            html.Div(
                                                                [
                                                                    html.Span("¿No tienes cuenta? "),
                                                                    html.A(
                                                                        "Regístrate aquí",
                                                                        href="/auth/register",
                                                                        className="fw-bold",
                                                                    ),
                                                                ],
                                                                className="text-center",
                                                            ),
                                                        ],
                                                        className="mt-3",
                                                    ),
                                                ]
                                            )
                                        ],
                                        className="shadow",
                                    ),
                                ],
                                style={"max-width": "450px", "margin": "0 auto"},
                            )
                        ],
                        width=12,
                        lg=6,
                        className="mx-auto",
                    )
                ],
                className="align-items-center min-vh-100",
            )
        ],
        fluid=True,
        className="bg-light",
    )


def create_register_layout():
    """
    Crea el layout de la página de registro

    Features:
    - Formulario de registro completo
    - Validación de campos
    - OAuth opcional
    """
    return dbc.Container(
        [
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    # Logo y título
                                    html.Div(
                                        [
                                            html.I(className="fas fa-pills fa-3x text-primary mb-3"),
                                            html.H2("xFarma", className="text-primary mb-1"),
                                            html.P("Crear Nueva Cuenta", className="text-muted"),
                                        ],
                                        className="text-center mb-4",
                                    ),
                                    # Card de registro
                                    dbc.Card(
                                        [
                                            dbc.CardBody(
                                                [
                                                    html.H4("Registro", className="card-title text-center mb-4"),
                                                    # Mensajes
                                                    html.Div(id="register-alert", className="mb-3"),
                                                    # Formulario
                                                    dbc.Form(
                                                        [
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Nombre Completo",
                                                                                html_for="register-fullname",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="text",
                                                                                id="register-fullname",
                                                                                placeholder="Juan Pérez",
                                                                            ),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Email *", html_for="register-email"
                                                                            ),
                                                                            dbc.Input(
                                                                                type="email",
                                                                                id="register-email",
                                                                                placeholder="tu@email.com",
                                                                                required=True,
                                                                            ),
                                                                            dbc.FormFeedback(
                                                                                "Email válido", type="valid"
                                                                            ),
                                                                            dbc.FormFeedback(
                                                                                "Por favor ingresa un email válido",
                                                                                type="invalid",
                                                                            ),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Nombre de Usuario *",
                                                                                html_for="register-username",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="text",
                                                                                id="register-username",
                                                                                placeholder="usuario123",
                                                                                required=True,
                                                                                minLength=3,
                                                                                maxLength=50,
                                                                            ),
                                                                            dbc.FormText("Mínimo 3 caracteres"),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Contraseña *",
                                                                                html_for="register-password",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="password",
                                                                                id="register-password",
                                                                                placeholder="Mínimo 8 caracteres",
                                                                                required=True,
                                                                                minLength=8,
                                                                            ),
                                                                            dbc.Progress(
                                                                                id="password-strength",
                                                                                className="mb-1",
                                                                                style={"height": "5px"},
                                                                            ),
                                                                            dbc.FormText(id="password-requirements"),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Confirmar Contraseña *",
                                                                                html_for="register-password-confirm",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="password",
                                                                                id="register-password-confirm",
                                                                                placeholder="Repite tu contraseña",
                                                                                required=True,
                                                                            ),
                                                                            dbc.FormFeedback(
                                                                                "Las contraseñas coinciden",
                                                                                type="valid",
                                                                            ),
                                                                            dbc.FormFeedback(
                                                                                "Las contraseñas no coinciden",
                                                                                type="invalid",
                                                                            ),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Teléfono", html_for="register-phone"
                                                                            ),
                                                                            dbc.Input(
                                                                                type="tel",
                                                                                id="register-phone",
                                                                                placeholder="+34 600 000 000",
                                                                            ),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            # Sección de datos de farmacia
                                                            html.Hr(className="my-4"),
                                                            html.H5("Datos de la Farmacia", className="mb-3"),
                                                            html.P(
                                                                "Se creará automáticamente una farmacia asociada a tu cuenta",
                                                                className="text-muted small mb-3",
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Nombre de la Farmacia *",
                                                                                html_for="register-pharmacy-name",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="text",
                                                                                id="register-pharmacy-name",
                                                                                placeholder="Farmacia San José",
                                                                                required=True,
                                                                                minLength=3,
                                                                            ),
                                                                            dbc.FormText("Mínimo 3 caracteres"),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Código de Farmacia",
                                                                                html_for="register-pharmacy-code",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="text",
                                                                                id="register-pharmacy-code",
                                                                                placeholder="FARM001",
                                                                            ),
                                                                            dbc.FormText(
                                                                                "Si no se especifica, se usará tu nombre de usuario"
                                                                            ),
                                                                        ],
                                                                        width=12,
                                                                        md=6,
                                                                        className="mb-3",
                                                                    ),
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Email de Farmacia",
                                                                                html_for="register-pharmacy-email",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="email",
                                                                                id="register-pharmacy-email",
                                                                                placeholder="farmacia@ejemplo.com",
                                                                            ),
                                                                            dbc.FormText(
                                                                                "Si no se especifica, se usará tu email personal"
                                                                            ),
                                                                        ],
                                                                        width=12,
                                                                        md=6,
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Dirección",
                                                                                html_for="register-pharmacy-address",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="text",
                                                                                id="register-pharmacy-address",
                                                                                placeholder="Calle Mayor, 123",
                                                                            ),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Ciudad",
                                                                                html_for="register-pharmacy-city",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="text",
                                                                                id="register-pharmacy-city",
                                                                                placeholder="Madrid",
                                                                            ),
                                                                        ],
                                                                        width=12,
                                                                        md=6,
                                                                        className="mb-3",
                                                                    ),
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Código Postal",
                                                                                html_for="register-pharmacy-postal-code",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="text",
                                                                                id="register-pharmacy-postal-code",
                                                                                placeholder="28001",
                                                                            ),
                                                                        ],
                                                                        width=12,
                                                                        md=6,
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label(
                                                                                "Teléfono de Farmacia",
                                                                                html_for="register-pharmacy-phone",
                                                                            ),
                                                                            dbc.Input(
                                                                                type="tel",
                                                                                id="register-pharmacy-phone",
                                                                                placeholder="+34 910 000 000",
                                                                            ),
                                                                            dbc.FormText(
                                                                                "Si no se especifica, se usará tu teléfono personal"
                                                                            ),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            html.Hr(className="my-4"),
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Checkbox(
                                                                                id="accept-terms",
                                                                                label="Acepto los términos y condiciones",
                                                                                value=False,
                                                                                className="mb-3",
                                                                            ),
                                                                        ]
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Button(
                                                                html.Div(
                                                                    [
                                                                        html.I(className="fas fa-user-plus me-2"),
                                                                        "Crear Cuenta",
                                                                    ]
                                                                ),
                                                                id="register-button",
                                                                color="primary",
                                                                size="lg",
                                                                className="w-100 mb-3",
                                                                n_clicks=0,
                                                                disabled=True,  # Se habilita cuando se aceptan términos
                                                            ),
                                                        ]
                                                    ),
                                                    # Divider
                                                    html.Hr(className="my-4"),
                                                    html.P(
                                                        "O regístrate con:", className="text-center text-muted mb-3"
                                                    ),
                                                    # OAuth buttons
                                                    html.Div(
                                                        [
                                                            dbc.Button(
                                                                html.Div(
                                                                    [html.I(className="fab fa-google me-2"), "Google"]
                                                                ),
                                                                color="danger",
                                                                outline=True,
                                                                className="w-100 mb-2",
                                                                href=f"{BACKEND_URL}/api/auth/oauth/google",
                                                            ),
                                                            dbc.Button(
                                                                html.Div(
                                                                    [
                                                                        html.I(className="fab fa-microsoft me-2"),
                                                                        "Microsoft",
                                                                    ]
                                                                ),
                                                                color="info",
                                                                outline=True,
                                                                className="w-100 mb-3",
                                                                href=f"{BACKEND_URL}/api/auth/oauth/microsoft",
                                                            ),
                                                        ]
                                                    ),
                                                    # Link a login
                                                    html.Div(
                                                        [
                                                            html.Span("¿Ya tienes cuenta? "),
                                                            html.A(
                                                                "Inicia sesión", href="/auth/login", className="fw-bold"
                                                            ),
                                                        ],
                                                        className="text-center mt-3",
                                                    ),
                                                ]
                                            )
                                        ],
                                        className="shadow",
                                    ),
                                ],
                                style={"max-width": "500px", "margin": "0 auto"},
                            )
                        ],
                        width=12,
                        lg=6,
                        className="mx-auto",
                    )
                ],
                className="align-items-center min-vh-100",
            )
        ],
        fluid=True,
        className="bg-light",
    )


def create_forgot_password_layout():
    """
    Crea el layout para recuperación de contraseña
    """
    return dbc.Container(
        [
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    # Logo y título
                                    html.Div(
                                        [
                                            html.I(className="fas fa-pills fa-3x text-primary mb-3"),
                                            html.H2("xFarma", className="text-primary mb-1"),
                                            html.P("Recuperar Contraseña", className="text-muted"),
                                        ],
                                        className="text-center mb-4",
                                    ),
                                    # Card
                                    dbc.Card(
                                        [
                                            dbc.CardBody(
                                                [
                                                    html.H4(
                                                        "¿Olvidaste tu contraseña?",
                                                        className="card-title text-center mb-3",
                                                    ),
                                                    html.P(
                                                        "Ingresa tu email y te enviaremos instrucciones para resetear tu contraseña.",
                                                        className="text-center text-muted mb-4",
                                                    ),
                                                    # Alert
                                                    html.Div(id="forgot-alert", className="mb-3"),
                                                    # Form
                                                    dbc.Form(
                                                        [
                                                            dbc.Row(
                                                                [
                                                                    dbc.Col(
                                                                        [
                                                                            dbc.Label("Email", html_for="forgot-email"),
                                                                            dbc.Input(
                                                                                type="email",
                                                                                id="forgot-email",
                                                                                placeholder="tu@email.com",
                                                                                required=True,
                                                                            ),
                                                                        ],
                                                                        className="mb-3",
                                                                    ),
                                                                ]
                                                            ),
                                                            dbc.Button(
                                                                html.Div(
                                                                    [
                                                                        html.I(className="fas fa-paper-plane me-2"),
                                                                        "Enviar Instrucciones",
                                                                    ]
                                                                ),
                                                                id="forgot-button",
                                                                color="primary",
                                                                size="lg",
                                                                className="w-100 mb-3",
                                                                n_clicks=0,
                                                            ),
                                                        ]
                                                    ),
                                                    # Link back
                                                    html.Div(
                                                        [html.A("← Volver al login", href="/auth/login")],
                                                        className="text-center mt-3",
                                                    ),
                                                ]
                                            )
                                        ],
                                        className="shadow",
                                    ),
                                ],
                                style={"max-width": "450px", "margin": "0 auto"},
                            )
                        ],
                        width=12,
                        lg=6,
                        className="mx-auto",
                    )
                ],
                className="align-items-center min-vh-100",
            )
        ],
        fluid=True,
        className="bg-light",
    )


def create_oauth_callback_layout():
    """
    Layout para manejar callbacks de OAuth
    Procesa tokens y redirige al dashboard
    """
    return dbc.Container(
        [
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.Div(
                                        [
                                            dbc.Spinner(color="primary", size="lg"),
                                            html.H4("Procesando autenticación...", className="mt-3"),
                                            html.P("Por favor espera mientras te redirigimos.", className="text-muted"),
                                        ],
                                        className="text-center",
                                    )
                                ],
                                className="d-flex align-items-center justify-content-center",
                                style={"min-height": "100vh"},
                            )
                        ]
                    )
                ]
            )
        ],
        fluid=True,
    )


def create_auth_error_layout():
    """
    Layout para mostrar errores de autenticación
    """
    return dbc.Container(
        [
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.Div(
                                        [
                                            html.I(className="fas fa-exclamation-circle fa-4x text-danger mb-3"),
                                            html.H3("Error de Autenticación", className="text-danger"),
                                            html.P(
                                                "Ha ocurrido un error durante el proceso de autenticación.",
                                                className="text-muted",
                                            ),
                                            html.P(id="auth-error-message", className="mb-4"),
                                            dbc.Button(
                                                html.Div(
                                                    [html.I(className="fas fa-arrow-left me-2"), "Volver al Login"]
                                                ),
                                                href="/auth/login",
                                                color="primary",
                                                size="lg",
                                            ),
                                        ],
                                        className="text-center",
                                    )
                                ],
                                className="d-flex align-items-center justify-content-center",
                                style={"min-height": "100vh"},
                            )
                        ]
                    )
                ]
            )
        ],
        fluid=True,
        className="bg-light",
    )
