"""
Layout para tab de Prescripción en admin dashboard.

Issue #408: Tab completo para gestión de clasificación de productos de prescripción.

Componentes principales:
- Stats header con KPIs (total, clasificados, tasa)
- Gráfico de distribución por categorías (barras clickables)
- Panel de acciones (batch classify, upload Excel, export CSV)
- Tabla interactiva de productos con paginación

REGLA #5: Usar Design System (components.base, styles.design_tokens)
REGLA #0.5: Stores en app.py skeleton (NO duplicar aquí)
"""

import logging
import dash_bootstrap_components as dbc
from dash import dcc, html

from components.base import BaseCard, BaseButton, Title
from styles.design_tokens import COLORS, SPACING

logger = logging.getLogger(__name__)


def create_prescription_tab() -> html.Div:
    """
    Tab de clasificación de prescripción (Issue #408).

    Estructura:
    1. Stats Header (3 KPI cards)
    2. Gráfico categorías (barras clickables + "No Clasificados")
    3. Panel acciones (3 botones + upload)
    4. Tabla productos (DataTable con paginación 50/página)

    Returns:
        html.Div: Layout completo del tab
    """
    return dbc.Container(
        [
            # ===================================================================
            # SECTION 1: STATS HEADER
            # ===================================================================
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div([
                                html.H3([
                                    html.I(className="fas fa-prescription me-2 text-primary"),
                                    "Clasificación de Prescripción"
                                ], className="mb-2"),
                                html.P(
                                    "Gestión de categorías de productos de prescripción",
                                    className="text-muted"
                                )
                            ])
                        ],
                        width=12,
                        className="mb-4"
                    )
                ],
            ),

            # KPI Cards
            html.Div(
                id="prescription-stats-header",
                children=[
                    dbc.Alert(
                        html.Div([
                            html.I(className="fas fa-spinner fa-spin me-2"),
                            "Cargando estadísticas..."
                        ]),
                        color="info",
                        className="text-center"
                    )
                ]
            ),

            # ===================================================================
            # SECTION 2: GRÁFICO CATEGORÍAS
            # ===================================================================
            html.Hr(className="my-4"),
            dbc.Row(
                [
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardHeader(
                                        html.H5([
                                            html.I(className="fas fa-chart-bar me-2 text-primary"),
                                            "Distribución por Categorías"
                                        ]),
                                        className="bg-light"
                                    ),
                                    dbc.CardBody(
                                        [
                                            html.P(
                                                "Haz clic en una barra para filtrar la tabla de productos",
                                                className="text-muted small mb-3"
                                            ),
                                            dcc.Graph(
                                                id="prescription-category-chart",
                                                config={
                                                    'displayModeBar': False,
                                                    'responsive': True
                                                },
                                                style={'height': '350px'}
                                            )
                                        ]
                                    )
                                ],
                                shadow="sm"
                            )
                        ],
                        width=12,
                        className="mb-4"
                    )
                ]
            ),

            # ===================================================================
            # SECTION 3: PANEL DE ACCIONES
            # ===================================================================
            html.Hr(className="my-4"),
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.H5([
                                html.I(className="fas fa-cogs me-2 text-warning"),
                                "Acciones de Clasificación"
                            ]),
                        ],
                        width=12,
                        className="mb-3"
                    )
                ]
            ),

            dbc.Row(
                [
                    dbc.Col(
                        [
                            BaseButton(
                                children=html.Div([
                                    html.I(className="fas fa-magic me-2"),
                                    "Clasificar No Clasificados"
                                ]),
                                id="prescription-batch-classify-btn",
                                variant="primary",
                                className="w-100 d-flex align-items-center justify-content-center"
                            ),
                            html.Small(
                                "Clasificación automática de productos sin categoría",
                                className="text-muted d-block text-center mt-2"
                            )
                        ],
                        width=12,
                        md=4,
                        className="mb-3"
                    ),
                    dbc.Col(
                        [
                            dcc.Upload(
                                id="prescription-upload-excel",
                                children=BaseButton(
                                    children=html.Div([
                                        html.I(className="fas fa-file-excel me-2"),
                                        "Importar desde Excel"
                                    ]),
                                    id="prescription-upload-excel-btn",
                                    variant="success",
                                    outline=True,
                                    className="w-100 d-flex align-items-center justify-content-center"
                                ),
                                multiple=False,
                                accept=".xlsx,.xls"
                            ),
                            html.Small(
                                "Subir Excel con columnas: codigo_nacional, categoria",
                                className="text-muted d-block text-center mt-2"
                            )
                        ],
                        width=12,
                        md=4,
                        className="mb-3"
                    ),
                    dbc.Col(
                        [
                            BaseButton(
                                children=html.Div([
                                    html.I(className="fas fa-download me-2"),
                                    "Exportar a CSV"
                                ]),
                                id="prescription-export-csv-btn",
                                variant="secondary",
                                outline=True,
                                className="w-100 d-flex align-items-center justify-content-center"
                            ),
                            html.Small(
                                "Descargar todos los productos con categorías",
                                className="text-muted d-block text-center mt-2"
                            ),
                            # Componente para descargar CSV
                            dcc.Download(id="prescription-download-csv")
                        ],
                        width=12,
                        md=4,
                        className="mb-3"
                    )
                ]
            ),

            # ===================================================================
            # SECTION 4: TABLA DE PRODUCTOS
            # ===================================================================
            html.Hr(className="my-4"),
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.H5([
                                        html.I(className="fas fa-table me-2 text-info"),
                                        html.Span("Todos los Productos", id="prescription-table-title")
                                    ]),
                                    html.P(
                                        id="prescription-filter-info",
                                        className="text-muted small mb-3"
                                    )
                                ]
                            )
                        ],
                        width=12,
                        className="mb-3"
                    )
                ]
            ),

            # Tabla de productos (ID único para admin - evitar duplicados con /prescription)
            html.Div(
                id="admin-prescription-products-table-container",
                children=[
                    dbc.Alert(
                        html.Div([
                            html.I(className="fas fa-info-circle me-2"),
                            "Cargando productos..."
                        ]),
                        color="info",
                        className="text-center"
                    )
                ]
            ),

            # Paginación
            html.Div(
                id="prescription-pagination-container",
                className="d-flex justify-content-center mt-4"
            ),

            # ===================================================================
            # COMPONENTES OCULTOS (para estados)
            # ===================================================================
            # Stores y componentes NO se duplican aquí (REGLA #0.5)
            # Los stores están en app.py skeleton:
            # - prescription-all-products-store (session)
            # - prescription-selected-category-store (memory)
            # - prescription-filtered-products-store (memory)
            # - prescription-current-page-store (memory)

        ],
        fluid=True,
        className="py-4"
    )
