"""
Panel de gestión de clasificación de productos de prescripción (Issue #16 Fase 2).

Este módulo contiene las funciones de layout para el tab de Prescripción en el panel admin.
Separado de admin.py para mejor organización del código.
"""

import dash_bootstrap_components as dbc
from dash import dcc, html

from components.base import BaseButton, BaseCard


# Prescription categories (14 total) - sincronizado con callbacks/admin/prescription.py
PRESCRIPTION_CATEGORIES = [
    "MEDICAMENTOS",
    "FORMULAS_MAGISTRALES",
    "VACUNAS_INDIVIDUALIZADAS",
    "TIRAS_REACTIVAS_GLUCOSA",
    "TIRAS_REACTIVAS_CETONEMIA",
    "AGUJAS_LANCETAS",
    "ABSORBENTES_INCONTINENCIA",
    "SISTEMAS_AUTOADMINISTRACION",
    "DISPOSITIVOS_INHALACION",
    "SUEROS",
    "MEDICAMENTOS_EXTRANJEROS",
    "PRODUCTOS_SANITARIOS",
    "OXIGENOTERAPIA",
    "OTROS",
]

# Category labels for UI
CATEGORY_LABELS = {
    "MEDICAMENTOS": "Medicamentos",
    "FORMULAS_MAGISTRALES": "Fórmulas Magistrales",
    "VACUNAS_INDIVIDUALIZADAS": "Vacunas Individualizadas",
    "TIRAS_REACTIVAS_GLUCOSA": "Tiras Glucosa",
    "TIRAS_REACTIVAS_CETONEMIA": "Tiras Cetonemia",
    "AGUJAS_LANCETAS": "Agujas y Lancetas",
    "ABSORBENTES_INCONTINENCIA": "Absorbentes",
    "SISTEMAS_AUTOADMINISTRACION": "Autoadministración",
    "DISPOSITIVOS_INHALACION": "Inhalación",
    "SUEROS": "Sueros",
    "MEDICAMENTOS_EXTRANJEROS": "Medicamentos Extranjeros",
    "PRODUCTOS_SANITARIOS": "Productos Sanitarios",
    "OXIGENOTERAPIA": "Oxigenoterapia",
    "OTROS": "Otros",
    "NO_CLASIFICADOS": "Sin Clasificar",
}


def create_prescription_panel() -> html.Div:
    """
    Panel de gestión de clasificación de productos de prescripción (Issue #16 Fase 2).

    Componentes:
    - KPI cards (Total, Clasificados, No Clasificados, Categorías Activas)
    - Gráfico de distribución de categorías (14 categorías)
    - Panel de acciones (Batch Classification, Upload Excel, Export)
    - Tabla interactiva de productos por categoría
    """
    return dbc.Container(
        [
            # KPI Cards
            dbc.Row(
                [
                    # Total productos de prescripción
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(
                                                        className="fas fa-pills me-2 text-primary",
                                                        style={"fontSize": "2rem"},
                                                    ),
                                                    html.Div(
                                                        [
                                                            html.Small(
                                                                "Total Prescripción",
                                                                className="text-muted text-uppercase",
                                                            ),
                                                            html.H5(
                                                                "--",
                                                                id="prescription-total-count",
                                                                className="mb-0 text-primary",
                                                                style={"fontSize": "1.25rem"},
                                                            ),
                                                        ],
                                                        className="ms-3",
                                                    ),
                                                ],
                                                className="d-flex align-items-center",
                                            )
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=12,
                        md=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # Clasificados
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(
                                                        className="fas fa-check-circle me-2 text-success",
                                                        style={"fontSize": "2rem"},
                                                    ),
                                                    html.Div(
                                                        [
                                                            html.Small(
                                                                "Clasificados",
                                                                className="text-muted text-uppercase",
                                                            ),
                                                            html.H5(
                                                                "--",
                                                                id="prescription-classified-count",
                                                                className="mb-0 text-success",
                                                                style={"fontSize": "1.25rem"},
                                                            ),
                                                        ],
                                                        className="ms-3",
                                                    ),
                                                ],
                                                className="d-flex align-items-center",
                                            )
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=12,
                        md=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # Porcentaje clasificados
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(
                                                        className="fas fa-percent me-2 text-info",
                                                        style={"fontSize": "2rem"},
                                                    ),
                                                    html.Div(
                                                        [
                                                            html.Small(
                                                                "% Clasificado",
                                                                className="text-muted text-uppercase",
                                                            ),
                                                            html.H5(
                                                                "--",
                                                                id="prescription-classified-percentage",
                                                                className="mb-0 text-info",
                                                                style={"fontSize": "1.25rem"},
                                                            ),
                                                        ],
                                                        className="ms-3",
                                                    ),
                                                ],
                                                className="d-flex align-items-center",
                                            )
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=12,
                        md=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # No clasificados
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(
                                                        className="fas fa-exclamation-triangle me-2 text-warning",
                                                        style={"fontSize": "2rem"},
                                                    ),
                                                    html.Div(
                                                        [
                                                            html.Small(
                                                                "No Clasificados",
                                                                className="text-muted text-uppercase",
                                                            ),
                                                            html.H5(
                                                                "--",
                                                                id="prescription-unclassified-count",
                                                                className="mb-0 text-warning",
                                                                style={"fontSize": "1.25rem"},
                                                            ),
                                                        ],
                                                        className="ms-3",
                                                    ),
                                                ],
                                                className="d-flex align-items-center",
                                            )
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=12,
                        md=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # Porcentaje no clasificados
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(
                                                        className="fas fa-percent me-2 text-secondary",
                                                        style={"fontSize": "2rem"},
                                                    ),
                                                    html.Div(
                                                        [
                                                            html.Small(
                                                                "% Sin Clasificar",
                                                                className="text-muted text-uppercase",
                                                            ),
                                                            html.H5(
                                                                "--",
                                                                id="prescription-unclassified-percentage",
                                                                className="mb-0 text-secondary",
                                                                style={"fontSize": "1.25rem"},
                                                            ),
                                                        ],
                                                        className="ms-3",
                                                    ),
                                                ],
                                                className="d-flex align-items-center",
                                            )
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=12,
                        md=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # Categorías activas
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(
                                                        className="fas fa-tags me-2 text-primary",
                                                        style={"fontSize": "2rem"},
                                                    ),
                                                    html.Div(
                                                        [
                                                            html.Small(
                                                                "Categorías Activas",
                                                                className="text-muted text-uppercase",
                                                            ),
                                                            html.H5(
                                                                "--",
                                                                id="prescription-active-categories-count",
                                                                className="mb-0 text-primary",
                                                                style={"fontSize": "1.25rem"},
                                                            ),
                                                        ],
                                                        className="ms-3",
                                                    ),
                                                ],
                                                className="d-flex align-items-center",
                                            )
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=12,
                        md=6,
                        lg=3,
                        className="mb-3",
                    ),
                ],
                className="mb-4",
            ),
            # Gráfico y Panel de Acciones
            dbc.Row(
                [
                    # Gráfico de distribución
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardHeader(
                                        [
                                            html.I(className="fas fa-chart-pie me-2 text-primary"),
                                            html.Span(
                                                "Distribución por Categorías",
                                                style={"fontWeight": "600"},
                                            ),
                                        ],
                                        style={"backgroundColor": "#f8f9fa", "border": "none"},
                                    ),
                                    dbc.CardBody(
                                        [
                                            dcc.Loading(
                                                id="prescription-chart-loading",
                                                type="default",
                                                children=html.Div(id="prescription-category-distribution-chart"),
                                            )
                                        ]
                                    ),
                                ],
                                shadow="sm",
                                className="mb-3",
                            ),
                        ],
                        width=12,
                        lg=8,
                    ),
                    # Panel de acciones
                    dbc.Col(
                        html.Div(
                            [
                                # Batch Classification
                                BaseCard(
                                    children=[
                                        dbc.CardHeader(
                                            [
                                                html.I(className="fas fa-magic me-2 text-success"),
                                                html.Span(
                                                    "Clasificación Automática",
                                                    style={"fontWeight": "600"},
                                                ),
                                            ],
                                            style={"backgroundColor": "#f8f9fa", "border": "none"},
                                        ),
                                        dbc.CardBody(
                                            [
                                                html.P(
                                                    "Clasificación automática con ML",
                                                    className="small text-muted mb-2",
                                                ),
                                                BaseButton(
                                                    children=html.Div(
                                                        [
                                                            html.I(className="fas fa-bolt me-2"),
                                                            "Clasificar",
                                                        ],
                                                        className="d-flex align-items-center justify-content-center"
                                                    ),
                                                    id="prescription-batch-classify-btn",
                                                    variant="success",
                                                    size="sm",
                                                    className="w-100",
                                                ),
                                                html.Div(id="prescription-batch-status", className="mt-2"),
                                            ],
                                            style={"padding": "0.75rem"}
                                        ),
                                    ],
                                    shadow="sm",
                                    className="mb-2",
                                    style={"flex": "1"}
                                ),
                            # Import Excel
                            BaseCard(
                                children=[
                                    dbc.CardHeader(
                                        [
                                            html.I(className="fas fa-file-excel me-2 text-primary"),
                                            html.Span(
                                                "Importar Excel",
                                                style={"fontWeight": "600"},
                                            ),
                                        ],
                                        style={"backgroundColor": "#f8f9fa", "border": "none"},
                                    ),
                                    dbc.CardBody(
                                        [
                                            html.P(
                                                "Importar desde Excel",
                                                className="small text-muted mb-2",
                                            ),
                                            dcc.Upload(
                                                id="prescription-upload-excel",
                                                children=html.Div(
                                                    [
                                                        html.I(className="fas fa-cloud-upload-alt me-2"),
                                                        "Seleccionar archivo"
                                                    ],
                                                    className="d-flex align-items-center justify-content-center",
                                                ),
                                                style={
                                                    "width": "100%",
                                                    "height": "60px",
                                                    "lineHeight": "60px",
                                                    "borderWidth": "2px",
                                                    "borderStyle": "dashed",
                                                    "borderRadius": "8px",
                                                    "borderColor": "#ced4da",
                                                    "textAlign": "center",
                                                    "cursor": "pointer",
                                                    "backgroundColor": "#f8f9fa",
                                                },
                                                multiple=False,
                                            ),
                                            html.Div(id="prescription-upload-status", className="mt-2"),
                                            html.Small(
                                                html.Span(
                                                    "Ver categorías válidas",
                                                    id="prescription-show-categories-link",
                                                    className="text-primary",
                                                    style={"cursor": "pointer", "textDecoration": "underline"},
                                                ),
                                                className="text-muted d-block mt-2",
                                            ),
                                        ],
                                        style={"padding": "0.75rem"}
                                    ),
                                ],
                                shadow="sm",
                                className="mb-2",
                                style={"flex": "1"}
                            ),
                            # Export
                            BaseCard(
                                children=[
                                    dbc.CardHeader(
                                        [
                                            html.I(className="fas fa-download me-2 text-secondary"),
                                            html.Span(
                                                "Exportar",
                                                style={"fontWeight": "600"},
                                            ),
                                        ],
                                        style={"backgroundColor": "#f8f9fa", "border": "none"},
                                    ),
                                    dbc.CardBody(
                                        [
                                            BaseButton(
                                                children=html.Div(
                                                    [
                                                        html.I(className="fas fa-file-csv me-2"),
                                                        "Descargar CSV",
                                                    ],
                                                    className="d-flex align-items-center justify-content-center"
                                                ),
                                                id="prescription-export-csv-btn",
                                                variant="secondary",
                                                size="sm",
                                                className="w-100",
                                            ),
                                            html.Small(
                                                "Exportar clasificaciones",
                                                className="text-muted d-block mt-2",
                                            ),
                                        ],
                                        style={"padding": "0.75rem"}
                                    ),
                                ],
                                shadow="sm",
                                className="mb-2",
                                style={"flex": "1"}
                            ),
                            # Listados de Referencia Oficiales (Issue #445)
                            BaseCard(
                                children=[
                                    dbc.CardHeader(
                                        [
                                            html.I(className="fas fa-list-alt me-2 text-info"),
                                            html.Span(
                                                "Listados Oficiales",
                                                style={"fontWeight": "600"},
                                            ),
                                        ],
                                        style={"backgroundColor": "#f8f9fa", "border": "none"},
                                    ),
                                    dbc.CardBody(
                                        [
                                            # Stats mini
                                            html.Div(
                                                id="reference-list-stats-mini",
                                                children=[
                                                    html.Small(
                                                        "Cargados: --",
                                                        className="text-muted",
                                                    ),
                                                ],
                                                className="mb-2",
                                            ),
                                            # Upload area
                                            dcc.Upload(
                                                id="reference-list-upload-excel",
                                                children=html.Div(
                                                    [
                                                        html.I(className="fas fa-file-excel me-2 text-success"),
                                                        "Subir Excel PF",
                                                    ],
                                                    className="d-flex align-items-center justify-content-center",
                                                ),
                                                style={
                                                    "width": "100%",
                                                    "height": "50px",
                                                    "lineHeight": "50px",
                                                    "borderWidth": "2px",
                                                    "borderStyle": "dashed",
                                                    "borderRadius": "8px",
                                                    "borderColor": "#17a2b8",
                                                    "textAlign": "center",
                                                    "cursor": "pointer",
                                                    "backgroundColor": "#f0f9ff",
                                                },
                                                multiple=False,
                                                accept=".xlsx,.xls",
                                            ),
                                            # Checkbox truncate
                                            dbc.Checkbox(
                                                id="reference-list-truncate-checkbox",
                                                label="Reemplazar datos",
                                                value=True,
                                                className="mt-2 small",
                                            ),
                                            # Status
                                            html.Div(id="reference-list-upload-status", className="mt-2"),
                                            html.Small(
                                                "Dietas, tiras, efectos",
                                                className="text-muted d-block mt-1",
                                            ),
                                        ],
                                        style={"padding": "0.75rem"}
                                    ),
                                ],
                                shadow="sm",
                                className="mb-2",
                                style={"flex": "1"}
                            ),
                            ],
                            style={
                                "display": "flex",
                                "flexDirection": "column",
                                "height": "100%",
                                "gap": "0.5rem"
                            }
                        ),
                        width=12,
                        lg=4,
                    ),
                ]
            ),
            # Tabla interactiva de productos por categoría
            dbc.Row(
                [
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardHeader(
                                        [
                                            html.I(className="fas fa-table me-2 text-primary"),
                                            html.Span(
                                                id="prescription-category-table-title",
                                                children="Productos por Categoría",
                                                style={"fontWeight": "600"},
                                            ),
                                        ],
                                        style={"backgroundColor": "#f8f9fa", "border": "none"},
                                    ),
                                    dbc.CardBody(
                                        [
                                            dcc.Loading(
                                                id="prescription-category-table-loading",
                                                type="default",
                                                children=html.Div(
                                                    id="prescription-category-table-container",
                                                    children=[],
                                                ),
                                            )
                                        ]
                                    ),
                                ],
                                shadow="sm",
                            ),
                        ],
                        width=12,
                    )
                ]
            ),
        ],
        fluid=True,
    )


def create_prescription_categories_modal() -> dbc.Modal:
    """
    Modal con listado de las 14 categorías de prescripción disponibles.
    Issue #16 Fase 2: Ayuda para clasificación manual.
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                dbc.ModalTitle(
                    [
                        html.I(className="fas fa-tags me-2 text-primary"),
                        "Categorías de Prescripción",
                    ]
                )
            ),
            dbc.ModalBody(
                [
                    html.P(
                        "Listado de las 14 categorías disponibles para clasificación:",
                        className="mb-3",
                    ),
                    html.Ol(
                        [
                            html.Li("MEDICAMENTOS - Medicamentos de prescripción general"),
                            html.Li("FORMULAS_MAGISTRALES - Fórmulas magistrales preparadas"),
                            html.Li("VACUNAS_INDIVIDUALIZADAS - Vacunas con prescripción personalizada"),
                            html.Li("TIRAS_REACTIVAS_GLUCOSA - Tiras reactivas para medición de glucosa"),
                            html.Li("TIRAS_REACTIVAS_CETONEMIA - Tiras reactivas para medición de cetonemia"),
                            html.Li("AGUJAS_LANCETAS - Agujas y lancetas para diabéticos"),
                            html.Li("ABSORBENTES_INCONTINENCIA - Productos absorbentes"),
                            html.Li("SISTEMAS_AUTOADMINISTRACION - Sistemas de autoadministración"),
                            html.Li("DISPOSITIVOS_INHALACION - Dispositivos y cámaras de inhalación"),
                            html.Li("SUEROS - Sueros y soluciones parenterales"),
                            html.Li("MEDICAMENTOS_EXTRANJEROS - Medicamentos extranjeros autorizados"),
                            html.Li("PRODUCTOS_SANITARIOS - Productos sanitarios especiales"),
                            html.Li("OXIGENOTERAPIA - Productos para oxigenoterapia"),
                            html.Li("OTROS - Otros productos de prescripción no categorizados"),
                        ],
                        className="small",
                    ),
                    dbc.Alert(
                        html.Div(
                            [
                                html.I(className="fas fa-info-circle me-2"),
                                "Utiliza estos valores exactos en la columna 'categoria' del archivo Excel.",
                            ]
                        ),
                        color="info",
                        className="mt-3",
                    ),
                ]
            ),
            dbc.ModalFooter(
                dbc.Button(
                    "Cerrar",
                    id="prescription-categories-modal-close",
                    className="ms-auto",
                )
            ),
        ],
        id="prescription-categories-modal",
        size="lg",
        is_open=False,
    )
