"""
Layout de administración unificado con gestión de usuarios, catálogos y sistema.
Issue #485: Reorganización 7→4 tabs para mejor UX.

ESTRUCTURA ACTUAL (Issue #485):
- Tab USUARIOS: Gestión de usuarios y farmacias (Ag Grid, filtros, CRUD)
- Tab CATÁLOGOS EXTERNOS: CIMA/Nomenclator + Prescripción (datos importados)
- Tab VENTA LIBRE: Sin Enriquecer + Duplicados + ML Monitoring + Validar NECESIDAD (datos enriquecidos)
- Tab SISTEMA: Herramientas + Base de Datos (infraestructura)

NOTAS:
- create_operation_feedback_toast fue removido → usar toast_manager centralizado
- system_status_banner ya está en app.py layout principal (no duplicar)
- create_database_panel() y create_tools_panel() eliminados (Issue #485)
"""

import logging

import dash_bootstrap_components as dbc
from dash import dcc, html

logger = logging.getLogger(__name__)

from components.base import BaseButton, BaseCard
from layouts.admin_prescription_panel import (
    create_prescription_categories_modal,
    create_prescription_panel,
)


def create_admin_layout() -> html.Div:
    """
    Layout principal para administradores con panel unificado.

    Issue #485: Reorganización 7→4 tabs principales:
    - Tab USUARIOS: Gestión de usuarios y farmacias
    - Tab CATÁLOGOS EXTERNOS: CIMA/Nomenclator + Prescripción
    - Tab VENTA LIBRE: Sin Enriquecer + Duplicados + ML Monitoring + Validar NECESIDAD
    - Tab SISTEMA: Herramientas + Base de Datos
    """
    return html.Div(
        [
            # Contenido principal de administración
            # (Banner de estado del sistema ya está en app.py layout principal)
            html.Div(
                [
                    # Título de la sección (simplificado para evitar padding excesivo)
                    html.Div(
                        [
                            html.H1(
                                [
                                    html.I(className="fas fa-user-shield me-3 text-primary"),
                                    "Panel de Administración Unificado",
                                ],
                                className="mb-2",
                            ),
                            html.P(
                                [
                                    "Gestión completa del sistema: Usuarios, Base de Datos y Herramientas. ",
                                    html.Strong(
                                        "Acceso restringido a administradores."
                                    ),
                                ],
                                className="lead text-muted mb-4",
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Tabs de administración REORGANIZADOS (Issue #485: 7→4 tabs)
                    # Estructura: Usuarios | Catálogos Externos | Venta Libre | Sistema
                    dcc.Tabs(
                        id="admin-tabs",
                        value="users",  # Default to users tab
                        children=[
                            dcc.Tab(
                                label="👥 Usuarios",
                                value="users",
                                className="admin-tab",
                            ),
                            dcc.Tab(
                                label="📥 Catálogos Externos",
                                value="catalogs",
                                className="admin-tab",
                            ),
                            dcc.Tab(
                                label="🏷️ Venta Libre",
                                value="venta-libre",
                                className="admin-tab",
                            ),
                            dcc.Tab(
                                label="🔧 Sistema",
                                value="system",
                                className="admin-tab",
                            ),
                        ],
                        className="custom-tabs mb-4",
                    ),
                    # Contenido de cada tab
                    html.Div(id="admin-tab-content"),
                ],
                # Nota: NO usar className="main-content" porque main-content-container
                # en app.py ya aplica marginLeft:280px. El CSS .main-content añadiría
                # OTRO margin-left:280px causando doble espacio entre sidebar y contenido.
            ),
            # Modales y componentes globales
            create_dangerous_operations_modal(),  # Mantener legacy modal
            create_user_management_modal(),  # NEW: Modal para crear/editar usuarios
            create_delete_user_confirmation_modal(),  # NEW: Confirmación de eliminación
            create_restore_user_confirmation_modal(),  # NEW: Confirmación de restauración
            create_prescription_categories_modal(),  # Issue #16 Fase 2: Modal de categorías de prescripción

            # Stores para gestión de estado
            dcc.Store(id="admin-session-store", data={"user_role": "admin", "authenticated": True}),
            dcc.Store(id="admin-settings-store", data={}),
            dcc.Store(id="selected-dangerous-operation", data={}),
            dcc.Store(
                id="admin-last-poll-time", storage_type="memory", data=None
            ),  # Issue #237: Timestamp del último polling

            # Issue #300 - Fase 2: Stores intermedios para desacoplar toast notifications
            dcc.Store(id="sync-operations-toast-store", storage_type="memory", data=None),
            dcc.Store(id="scheduled-sync-toast-store", storage_type="memory", data=None),
            dcc.Store(id="conflict-resolution-toast-store", storage_type="memory", data=None),

            # Stores intermedios para dangerous tools (DASH002 fix - REGLA #11)
            dcc.Store(id="dangerous-modal-state-store", storage_type="memory", data=None),
            dcc.Store(id="dangerous-confirmation-state-store", storage_type="memory", data=None),

            # Issue #283: Cache store (populated by callback, not pre-fetch)
            # El pre-fetch no funciona porque el usuario no está autenticado al cargar layout
            dcc.Store(
                id="admin-catalog-cache",
                data=None,  # Vacío inicialmente, se llena en el primer polling
                storage_type="memory",
            ),

            # NEW: Stores para gestión de usuarios (Issue #348 FASE 3)
            dcc.Store(id="admin-users-data-store", storage_type="memory", data=None),
            dcc.Store(id="admin-selected-user-store", storage_type="memory", data=None),
            dcc.Store(id="admin-user-form-mode-store", storage_type="memory", data="create"),  # "create" or "edit"

            # NEW: Store para preservar datos del formulario al cambiar entre tabs (Issue #365 FIX)
            dcc.Store(id="admin-user-form-data-store", storage_type="memory", data=None),

            # NEW: Store para resultados de operaciones de usuarios
            dcc.Store(id="admin-user-operation-result", storage_type="memory", data=None),

            # Issue #458 M6: ML Monitoring stores - MOVED TO app.py (REGLA #0.5 Skeleton Pattern)
            # Stores are now in global skeleton (app.py) to avoid "nonexistent object" errors at callback validation
            # Issue #477: duplicates-* stores moved to app.py skeleton

            # Issue #237 - Fase 2: Polling Inteligente
            # Intervalo dinámico: 5s durante sync activa, 30s en reposo
            dcc.Interval(id="admin-stats-interval", interval=30000, n_intervals=0),  # 30 segundos por defecto
            # Intervalo rápido para actualizar countdown (1 segundo)
            dcc.Interval(id="admin-countdown-interval", interval=1000, n_intervals=0),  # 1 segundo
        ],
        className="admin-layout",
    )


def create_users_panel() -> html.Div:
    """
    Panel de gestión de usuarios y farmacias (Issue #348 FASE 3 - Tab USERS).

    Componentes:
    - Toolbar con botón "Crear Usuario"
    - Filtros: Role, Subscription, Include Deleted
    - Tabla Ag Grid con usuarios y farmacias
    - Acciones: Edit, Delete, Restore
    - Dashboard de storage usage
    """
    return dbc.Container(
        [
            # Toolbar
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.H4(
                                [
                                    html.I(className="fas fa-users me-2 text-primary"),
                                    "Gestión de Usuarios y Farmacias",
                                ],
                                className="mb-0",
                            )
                        ],
                        width="auto",
                        className="flex-grow-1",
                    ),
                    dbc.Col(
                        [
                            BaseButton(
                                children=html.Div([
                                    html.I(className="fas fa-plus me-2"),
                                    "Crear Usuario",
                                ]),
                                id="admin-create-user-btn",
                                variant="primary",
                                size="sm",
                                className="d-flex align-items-center",
                            )
                        ],
                        width="auto",
                    ),
                ],
                className="mb-4 align-items-center",
            ),

            # Filtros
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Label("Rol", className="form-label small"),
                            dbc.Select(
                                id="admin-user-role-filter",
                                options=[
                                    {"label": "Todos", "value": "all"},
                                    {"label": "Admin", "value": "admin"},
                                    {"label": "Usuario", "value": "user"},
                                ],
                                value="all",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=3,
                    ),
                    dbc.Col(
                        [
                            html.Label("Suscripción", className="form-label small"),
                            dbc.Select(
                                id="admin-user-subscription-filter",
                                options=[
                                    {"label": "Todos", "value": "all"},
                                    {"label": "Free", "value": "free"},
                                    {"label": "Pro", "value": "pro"},
                                    {"label": "Max", "value": "max"},
                                ],
                                value="all",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=3,
                    ),
                    dbc.Col(
                        [
                            html.Label("Estado", className="form-label small"),
                            dbc.Checklist(
                                id="admin-user-include-deleted",
                                options=[{"label": " Incluir eliminados", "value": "include_deleted"}],
                                value=[],
                                inline=True,
                                switch=True,
                                className="mt-2",
                            ),
                        ],
                        width=12,
                        md=3,
                    ),
                    dbc.Col(
                        [
                            html.Label(" ", className="form-label small"),  # Spacer
                            BaseButton(
                                children=html.Div([
                                    html.I(className="fas fa-sync me-2"),
                                    "Actualizar",
                                ]),
                                id="admin-refresh-users-btn",
                                variant="secondary",
                                outline=True,
                                size="sm",
                                className="w-100 d-flex align-items-center justify-content-center",
                            ),
                        ],
                        width=12,
                        md=3,
                    ),
                ],
                className="mb-4",
            ),

            # User table (Ag Grid - a implementar en callback)
            html.Div(
                id="admin-users-table-container",
                children=[
                    dbc.Alert(
                        html.Div([
                            html.I(className="fas fa-info-circle me-2"),
                            "Cargando usuarios...",
                        ]),
                        color="info",
                        className="text-center",
                    )
                ],
            ),

            # Storage Usage Dashboard
            html.Hr(className="my-4"),
            html.H5(
                [
                    html.I(className="fas fa-hdd me-2 text-warning"),
                    "Uso de Almacenamiento por Farmacia",
                ],
                className="mb-3",
            ),
            html.Div(
                id="admin-storage-usage-chart",
                children=[
                    dbc.Alert(
                        "El gráfico de almacenamiento se cargará con los datos de los usuarios.",
                        color="light",
                        className="text-center text-muted",
                    )
                ],
            ),
        ],
        fluid=True,
    )


# =============================================================================
# LEGACY PANELS REMOVED (Issue #485)
# create_database_panel() y create_tools_panel() eliminados
# Funcionalidad ahora en create_catalogs_panel() y create_system_panel()
# =============================================================================


def create_manual_review_panel() -> html.Div:
    """
    Panel de Manual Review para productos sin enriquecer (Issue #448).

    Funcionalidad separada de Tools tab para mejor organización:
    - Estadísticas de productos sin enriquecer
    - Tabla interactiva con filtros y paginación
    - Exportación a CSV para análisis offline
    """
    return dbc.Container(
        [
            # Título y descripción
            html.H5(
                [
                    html.I(className="fas fa-search me-2 text-warning"),
                    "Productos Sin Enriquecer",
                ],
                className="mb-3",
            ),
            dbc.Alert(
                html.Div([
                    html.I(className="fas fa-info-circle me-2"),
                    "Productos con ventas pero sin enriquecimiento (CIMA/Nomenclator). ",
                    "Útil para análisis offline y clasificación de venta libre.",
                ]),
                color="info",
                className="mb-4",
            ),

            # SECCIÓN 1: ESTADÍSTICAS
            html.H6("Resumen General", className="mb-3 text-muted"),
            dbc.Row(
                [
                    # Stats cards
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-box-open me-2 text-warning"),
                                                    html.Strong("Productos: "),
                                                    html.Span("--", id="manual-review-products-count", className="text-warning fs-5"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-chart-line me-2 text-primary"),
                                                    html.Strong("Ventas: "),
                                                    html.Span("--", id="manual-review-sales-count"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-euro-sign me-2 text-success"),
                                                    html.Strong("Importe: "),
                                                    html.Span("--", id="manual-review-total-amount"),
                                                ],
                                            ),
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=12,
                        md=4,
                        className="mb-3",
                    ),
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.H6("Por Tipo de Código", className="text-muted mb-3"),
                                            html.Div(
                                                [
                                                    dbc.Badge("CN", color="primary", className="me-2"),
                                                    html.Span("--", id="manual-review-cn-count"),
                                                    html.Small(" (6-7 dígitos)", className="text-muted ms-1"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.Div(
                                                [
                                                    dbc.Badge("EAN", color="info", className="me-2"),
                                                    html.Span("--", id="manual-review-ean-count"),
                                                    html.Small(" (13 dígitos)", className="text-muted ms-1"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.Div(
                                                [
                                                    dbc.Badge("INTERNAL", color="secondary", className="me-2"),
                                                    html.Span("--", id="manual-review-internal-count"),
                                                    html.Small(" (otros)", className="text-muted ms-1"),
                                                ],
                                            ),
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=12,
                        md=4,
                        className="mb-3",
                    ),
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.H6("Exportar CSV", className="text-muted mb-3"),
                                            html.Label("Ventas mínimas", className="form-label small"),
                                            dbc.Input(
                                                id="manual-review-min-sales",
                                                type="number",
                                                min=1,
                                                value=1,
                                                size="sm",
                                                className="mb-3",
                                            ),
                                            BaseButton(
                                                children=html.Div([
                                                    html.I(className="fas fa-download me-2"),
                                                    "Descargar CSV",
                                                ]),
                                                id="manual-review-export-btn",
                                                variant="primary",
                                                className="w-100 d-flex align-items-center justify-content-center",
                                            ),
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=12,
                        md=4,
                        className="mb-3",
                    ),
                ],
                className="mb-4",
            ),

            # Download component for CSV export
            dcc.Download(id="manual-review-download"),

            html.Hr(className="my-4"),

            # SECCIÓN 2: FILTROS Y CONTROLES
            html.H6("Filtros y Búsqueda", className="mb-3 text-muted"),
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Label("Tipo de Código", className="form-label small"),
                            dbc.Select(
                                id="manual-review-code-type-filter",
                                options=[
                                    {"label": "Todos", "value": "all"},
                                    {"label": "CN (Código Nacional)", "value": "CN"},
                                    {"label": "EAN (EAN-13)", "value": "EAN"},
                                    {"label": "INTERNAL (Códigos internos)", "value": "INTERNAL"},
                                ],
                                value="all",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=3,
                        className="mb-3",
                    ),
                    dbc.Col(
                        [
                            html.Label("Buscar", className="form-label small"),
                            dbc.Input(
                                id="manual-review-search",
                                type="text",
                                placeholder="Buscar por nombre o código...",
                                size="sm",
                                debounce=True,
                            ),
                        ],
                        width=12,
                        md=3,
                        className="mb-3",
                    ),
                    dbc.Col(
                        [
                            html.Label("Ordenar por", className="form-label small"),
                            dbc.Select(
                                id="manual-review-order-by",
                                options=[
                                    {"label": "Importe (desc)", "value": "amount_desc"},
                                    {"label": "Ventas (desc)", "value": "sales_desc"},
                                    {"label": "Nombre (asc)", "value": "name_asc"},
                                ],
                                value="amount_desc",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=3,
                        className="mb-3",
                    ),
                    dbc.Col(
                        [
                            html.Label("Resultados", className="form-label small"),
                            dbc.Select(
                                id="manual-review-page-size",
                                options=[
                                    {"label": "25", "value": "25"},
                                    {"label": "50", "value": "50"},
                                    {"label": "100", "value": "100"},
                                ],
                                value="50",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=3,
                        className="mb-3",
                    ),
                ],
                className="mb-3",
            ),

            # SECCIÓN 3: TABLA INTERACTIVA
            html.H6("Productos", className="mb-3 text-muted"),
            dbc.Spinner(
                html.Div(id="manual-review-table-container"),
                color="primary",
                size="sm",
            ),

            # Paginación
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                id="manual-review-pagination-info",
                                className="text-muted small",
                            )
                        ],
                        width=12,
                        md=6,
                        className="d-flex align-items-center",
                    ),
                    dbc.Col(
                        [
                            dbc.ButtonGroup(
                                [
                                    BaseButton(
                                        children=html.I(className="fas fa-chevron-left"),
                                        id="manual-review-prev-page",
                                        variant="secondary",
                                        outline=True,
                                        size="sm",
                                        disabled=True,
                                    ),
                                    html.Span(
                                        id="manual-review-current-page",
                                        className="btn btn-sm btn-outline-secondary disabled mx-2",
                                        style={"minWidth": "80px", "textAlign": "center"},
                                    ),
                                    BaseButton(
                                        children=html.I(className="fas fa-chevron-right"),
                                        id="manual-review-next-page",
                                        variant="secondary",
                                        outline=True,
                                        size="sm",
                                        disabled=True,
                                    ),
                                ],
                                className="float-end",
                            )
                        ],
                        width=12,
                        md=6,
                        className="text-end",
                    ),
                ],
                className="mt-3",
            ),
        ],
        fluid=True,
    )


def create_ml_monitoring_panel() -> html.Div:
    """
    Panel de ML Monitoring para clasificador NECESIDAD (Issue #458 M6).

    Funcionalidades:
    - KPIs del clasificador (entropía, outliers, confianza, pendientes)
    - Alertas activas con acknowledge/resolve
    - Distribución de categorías
    - Productos de alto riesgo para validación prioritaria
    - Histórico de tendencias
    """
    return dbc.Container(
        [
            # Título y descripción
            html.H5(
                [
                    html.I(className="fas fa-brain me-2 text-primary"),
                    "Monitoreo del Clasificador NECESIDAD",
                ],
                className="mb-3",
            ),
            dbc.Alert(
                html.Div([
                    html.I(className="fas fa-info-circle me-2"),
                    "Métricas de salud del clasificador ML para productos venta libre. ",
                    "Monitorea entropía de distribución, tasa de outliers y confianza del modelo.",
                ]),
                color="info",
                className="mb-4",
            ),

            # SECCIÓN 1: KPIs PRINCIPALES
            html.H6("Métricas del Clasificador", className="mb-3 text-muted"),
            dbc.Row(
                [
                    # KPI: Entropía de Clusters
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-chart-pie me-2 text-primary"),
                                                    html.Strong("Entropía"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.H3(
                                                "--",
                                                id="ml-monitoring-entropy",
                                                className="text-primary mb-1",
                                            ),
                                            html.Small(
                                                "Uniformidad distribución (0-1)",
                                                className="text-muted",
                                            ),
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # KPI: Tasa de Outliers
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-exclamation-triangle me-2 text-warning"),
                                                    html.Strong("Outliers"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.H3(
                                                "--",
                                                id="ml-monitoring-outlier-rate",
                                                className="text-warning mb-1",
                                            ),
                                            html.Small(
                                                "Productos fuera de cluster",
                                                className="text-muted",
                                            ),
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # KPI: Confianza Promedio
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-check-circle me-2 text-success"),
                                                    html.Strong("Confianza ML"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.H3(
                                                "--",
                                                id="ml-monitoring-confidence",
                                                className="text-success mb-1",
                                            ),
                                            html.Small(
                                                "Promedio del modelo",
                                                className="text-muted",
                                            ),
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # KPI: Pendientes Validación
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-user-clock me-2 text-info"),
                                                    html.Strong("Pendientes"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.H3(
                                                "--",
                                                id="ml-monitoring-pending",
                                                className="text-info mb-1",
                                            ),
                                            html.Small(
                                                "Requieren validación humana",
                                                className="text-muted",
                                            ),
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # Issue #465: KPI Precision (basada en correcciones humanas)
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-bullseye me-2", id="ml-monitoring-precision-icon"),
                                                    html.Strong("Precisión"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.H3(
                                                "--",
                                                id="ml-monitoring-precision",
                                                className="mb-1",
                                            ),
                                            html.Small(
                                                [
                                                    "Accuracy vs humanos ",
                                                    dbc.Badge("--", id="ml-monitoring-precision-status", className="ms-1"),
                                                ],
                                                className="text-muted",
                                            ),
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # Issue #466: KPI Drift (cambio semanal en distribución)
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-exchange-alt me-2", id="ml-monitoring-drift-icon"),
                                                    html.Strong("Drift"),
                                                ],
                                                className="mb-2",
                                            ),
                                            html.H3(
                                                "--",
                                                id="ml-monitoring-drift",
                                                className="mb-1",
                                            ),
                                            html.Small(
                                                [
                                                    "Cambio semana/semana ",
                                                    dbc.Badge("--", id="ml-monitoring-drift-status", className="ms-1"),
                                                ],
                                                className="text-muted",
                                            ),
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=6,
                        lg=3,
                        className="mb-3",
                    ),
                ],
                className="mb-4",
            ),

            # Issue #458 F7: KPI Coverage destacado
            dbc.Row(
                [
                    dbc.Col(
                        [
                            BaseCard(
                                children=[
                                    dbc.CardBody(
                                        [
                                            dbc.Row(
                                                [
                                                    dbc.Col(
                                                        [
                                                            html.Div(
                                                                [
                                                                    html.I(className="fas fa-tasks me-2 text-primary"),
                                                                    html.Strong("Coverage de Clasificación", className="fs-5"),
                                                                ],
                                                                className="mb-2",
                                                            ),
                                                            html.Small(
                                                                "Porcentaje de productos con categoría NECESIDAD asignada",
                                                                className="text-muted",
                                                            ),
                                                        ],
                                                        width=12,
                                                        lg=6,
                                                    ),
                                                    dbc.Col(
                                                        [
                                                            html.Div(
                                                                [
                                                                    html.Span(
                                                                        "--",
                                                                        id="ml-monitoring-coverage",
                                                                        className="text-primary display-6 fw-bold",
                                                                    ),
                                                                    html.Span(" %", className="text-primary fs-4"),
                                                                ],
                                                                className="text-end",
                                                            ),
                                                            html.Small(
                                                                [
                                                                    html.Span("--", id="ml-monitoring-coverage-classified"),
                                                                    " / ",
                                                                    html.Span("--", id="ml-monitoring-coverage-total"),
                                                                    " productos",
                                                                ],
                                                                className="text-muted d-block text-end",
                                                            ),
                                                        ],
                                                        width=12,
                                                        lg=6,
                                                        className="d-flex flex-column justify-content-center",
                                                    ),
                                                ],
                                                className="align-items-center",
                                            ),
                                        ]
                                    )
                                ],
                                shadow="sm",
                            )
                        ],
                        width=12,
                        className="mb-3",
                    ),
                ],
                className="mb-4",
            ),

            # Botón refresh y timestamp
            dbc.Row(
                [
                    dbc.Col(
                        [
                            BaseButton(
                                children=html.Div([
                                    html.I(className="fas fa-sync-alt me-2"),
                                    "Actualizar Métricas",
                                ]),
                                id="ml-monitoring-refresh-btn",
                                variant="primary",
                                outline=True,
                                size="sm",
                            ),
                            html.Small(
                                id="ml-monitoring-last-update",
                                className="text-muted ms-3",
                            ),
                        ],
                        className="mb-4",
                    ),
                ],
            ),

            html.Hr(className="my-4"),

            # SECCIÓN 2: ALERTAS ACTIVAS
            html.H6(
                [
                    html.I(className="fas fa-bell me-2"),
                    "Alertas Activas",
                ],
                className="mb-3 text-muted",
            ),
            dbc.Spinner(
                html.Div(id="ml-monitoring-alerts-container"),
                color="primary",
                size="sm",
            ),

            html.Hr(className="my-4"),

            # SECCIÓN 3: DISTRIBUCIÓN Y PRODUCTOS ALTO RIESGO
            dbc.Row(
                [
                    # Distribución de Categorías
                    dbc.Col(
                        [
                            html.H6(
                                [
                                    html.I(className="fas fa-chart-bar me-2"),
                                    "Distribución por Categoría",
                                ],
                                className="mb-3 text-muted",
                            ),
                            dbc.Spinner(
                                dcc.Graph(
                                    id="ml-monitoring-distribution-chart",
                                    config={"displayModeBar": False, "responsive": True},
                                    style={"height": "300px"},
                                ),
                                color="primary",
                                size="sm",
                            ),
                        ],
                        width=12,
                        lg=6,
                        className="mb-4",
                    ),
                    # Productos Alto Riesgo
                    dbc.Col(
                        [
                            html.H6(
                                [
                                    html.I(className="fas fa-exclamation-circle me-2"),
                                    "Productos Alto Riesgo (Top 10)",
                                ],
                                className="mb-3 text-muted",
                            ),
                            dbc.Spinner(
                                html.Div(id="ml-monitoring-high-risk-table"),
                                color="primary",
                                size="sm",
                            ),
                        ],
                        width=12,
                        lg=6,
                        className="mb-4",
                    ),
                ],
            ),

            html.Hr(className="my-4"),

            # SECCIÓN 4: HISTÓRICO DE TENDENCIAS
            html.H6(
                [
                    html.I(className="fas fa-chart-line me-2"),
                    "Tendencias (30 días)",
                ],
                className="mb-3 text-muted",
            ),
            dbc.Spinner(
                dcc.Graph(
                    id="ml-monitoring-trend-chart",
                    config={"displayModeBar": False, "responsive": True},
                    style={"height": "250px"},
                ),
                color="primary",
                size="sm",
            ),
        ],
        fluid=True,
    )


def create_dangerous_operations_modal() -> dbc.Modal:
    """
    Modal de confirmación para operaciones peligrosas (legacy - mantener compatibilidad).
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                [html.H4([html.I(className="fas fa-exclamation-triangle me-2 text-danger"), "Confirmación Requerida"])]
            ),
            dbc.ModalBody(
                [
                    html.Div(id="dangerous-operation-description"),
                    html.Hr(),
                    html.Div(id="dangerous-operation-confirmation-instruction"),
                    dbc.Input(
                        id="dangerous-operation-confirmation",
                        placeholder="Escriba la confirmación exacta...",
                        className="mb-3",
                    ),
                    dbc.FormText("Esta operación NO se puede deshacer.", color="danger"),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button("Cancelar", id="dangerous-operation-cancel", color="secondary", outline=True),
                    dbc.Button(
                        html.Div([html.I(className="fas fa-skull-crossbones me-2"), "EJECUTAR"]),
                        id="dangerous-operation-execute",
                        color="danger",
                        disabled=True,
                    ),
                ]
            ),
        ],
        id="dangerous-operations-modal",
        size="lg",
    )


def create_user_management_modal() -> dbc.Modal:
    """
    Modal unificado para crear y editar usuarios (Issue #348 FASE 3).

    El modal cambia dinámicamente entre modo "crear" y "editar" según el store
    admin-user-form-mode-store.
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                html.H4(
                    [
                        html.I(className="fas fa-user-plus me-2", id="admin-user-modal-icon"),
                        html.Span("Crear Usuario", id="admin-user-modal-title"),
                    ]
                )
            ),
            dbc.ModalBody(
                [
                    # Tabs para User y Pharmacy
                    dbc.Tabs(
                        id="admin-user-form-tabs",
                        active_tab="user-tab",
                        children=[
                            dbc.Tab(label="👤 Usuario", tab_id="user-tab"),
                            dbc.Tab(label="🏥 Farmacia", tab_id="pharmacy-tab"),
                        ],
                        className="mb-3",
                    ),

                    # Form content (dinámico según tab activo)
                    html.Div(id="admin-user-form-content"),

                    # Errores del formulario
                    html.Div(id="admin-user-form-errors", className="mt-3"),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button("Cancelar", id="admin-user-modal-cancel", color="secondary", outline=True),
                    dbc.Button(
                        html.Div([
                            html.I(className="fas fa-save me-2", id="admin-user-modal-save-icon"),
                            html.Span("Guardar", id="admin-user-modal-save-text"),
                        ]),
                        id="admin-user-modal-save",
                        color="primary",
                        disabled=False,
                    ),
                ]
            ),
        ],
        id="admin-user-modal",
        size="lg",
        is_open=False,
    )


def create_delete_user_confirmation_modal() -> dbc.Modal:
    """
    Modal de confirmación para eliminar usuario (soft delete - GDPR compliant).
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                html.H4([
                    html.I(className="fas fa-exclamation-triangle me-2 text-danger"),
                    "Confirmar Eliminación",
                ])
            ),
            dbc.ModalBody(
                [
                    html.P(
                        [
                            "¿Está seguro de que desea eliminar el usuario ",
                            html.Strong(id="admin-delete-user-name"),
                            "?",
                        ],
                        className="mb-3",
                    ),
                    dbc.Alert(
                        html.Div([
                            html.I(className="fas fa-info-circle me-2"),
                            "Esta es una eliminación lógica (soft delete). ",
                            "El usuario puede ser restaurado posteriormente desde la lista de usuarios eliminados.",
                        ]),
                        color="info",
                        className="mb-3",
                    ),
                    html.P(
                        [
                            html.Strong("Consecuencias:"),
                        ],
                        className="mb-2",
                    ),
                    html.Ul(
                        [
                            html.Li("El usuario no podrá iniciar sesión"),
                            html.Li("Los datos de ventas y configuración se conservan"),
                            html.Li("La farmacia asociada permanece en el sistema"),
                            html.Li("El usuario puede ser restaurado por un administrador"),
                        ],
                        className="small text-muted",
                    ),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button("Cancelar", id="admin-delete-user-modal-cancel", color="secondary", outline=True),
                    dbc.Button(
                        html.Div([
                            html.I(className="fas fa-trash me-2"),
                            "Eliminar Usuario",
                        ]),
                        id="admin-delete-user-modal-confirm",
                        color="danger",
                    ),
                ]
            ),
        ],
        id="admin-delete-user-modal",
        size="md",
        is_open=False,
    )


def create_restore_user_confirmation_modal() -> dbc.Modal:
    """
    Modal de confirmación para restaurar usuario eliminado.
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                html.H4([
                    html.I(className="fas fa-undo me-2 text-success"),
                    "Confirmar Restauración",
                ])
            ),
            dbc.ModalBody(
                [
                    html.P(
                        [
                            "¿Desea restaurar el usuario ",
                            html.Strong(id="admin-restore-user-name"),
                            "?",
                        ],
                        className="mb-3",
                    ),
                    dbc.Alert(
                        html.Div([
                            html.I(className="fas fa-check-circle me-2"),
                            "El usuario restaurado podrá iniciar sesión inmediatamente con sus credenciales anteriores.",
                        ]),
                        color="success",
                    ),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button("Cancelar", id="admin-restore-user-modal-cancel", color="secondary", outline=True),
                    dbc.Button(
                        html.Div([
                            html.I(className="fas fa-undo me-2"),
                            "Restaurar Usuario",
                        ]),
                        id="admin-restore-user-modal-confirm",
                        color="success",
                    ),
                ]
            ),
        ],
        id="admin-restore-user-modal",
        size="md",
        is_open=False,
    )


def create_duplicates_panel() -> html.Div:
    """
    Panel de gestión de duplicados VentaLibre (Issue #477).

    Funcionalidad:
    - Estadísticas de duplicados (pendientes, mergeados, diferentes)
    - Lista de grupos de duplicados potenciales
    - Acciones: Merge, Marcar como Diferentes, Skip
    - Modal de confirmación para merge
    """
    return dbc.Container(
        [
            # Título y descripción
            html.H5(
                [
                    html.I(className="fas fa-clone me-2 text-info"),
                    "Gestión de Duplicados - Venta Libre",
                ],
                className="mb-3",
            ),
            dbc.Alert(
                html.Div([
                    html.I(className="fas fa-info-circle me-2"),
                    "Revisión de productos potencialmente duplicados en el catálogo de venta libre. ",
                    "Los productos con tokens idénticos (100% similitud) se muestran para revisión manual.",
                ]),
                color="info",
                className="mb-4",
            ),

            # SECCIÓN 1: ESTADÍSTICAS
            html.H6("Estadísticas", className="mb-3 text-muted"),
            dbc.Row(
                [
                    dbc.Col(
                        dbc.Card(
                            dbc.CardBody(
                                [
                                    html.Div(
                                        [
                                            html.I(className="fas fa-clock me-2 text-warning"),
                                            html.Strong("Pendientes: "),
                                            html.Span("--", id="duplicates-pending-count", className="text-warning fs-5"),
                                        ],
                                        className="mb-2",
                                    ),
                                    html.Div(
                                        [
                                            html.I(className="fas fa-compress-arrows-alt me-2 text-success"),
                                            html.Strong("Mergeados: "),
                                            html.Span("--", id="duplicates-merged-count"),
                                        ],
                                        className="mb-2",
                                    ),
                                    html.Div(
                                        [
                                            html.I(className="fas fa-not-equal me-2 text-secondary"),
                                            html.Strong("Diferentes: "),
                                            html.Span("--", id="duplicates-different-count"),
                                        ],
                                        className="mb-2",
                                    ),
                                    html.Div(
                                        [
                                            html.I(className="fas fa-percentage me-2 text-info"),
                                            html.Strong("Cobertura tokens: "),
                                            html.Span("--", id="duplicates-coverage-pct"),
                                            html.Span("%", className="text-muted"),
                                        ],
                                    ),
                                ]
                            ),
                            className="shadow-sm",
                        ),
                        width=12,
                        md=6,
                        lg=4,
                    ),
                    dbc.Col(
                        [
                            dbc.Button(
                                html.Span([
                                    html.I(className="fas fa-sync me-2"),
                                    "Actualizar",
                                ]),
                                id="duplicates-refresh-btn",
                                color="primary",
                                outline=True,
                                className="me-2",
                            ),
                            html.Small(
                                id="duplicates-last-update",
                                className="text-muted",
                            ),
                        ],
                        width=12,
                        md=6,
                        lg=8,
                        className="d-flex align-items-center justify-content-md-end mt-3 mt-md-0",
                    ),
                ],
                className="mb-4",
            ),

            # SECCIÓN 2: FILTROS Y ORDENACIÓN (Issue #480)
            html.H6("Filtros y Ordenación", className="mb-3 text-muted"),
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Label("Marca", className="small text-muted"),
                            dbc.Select(
                                id="duplicates-brand-filter",
                                options=[{"label": "Todas las marcas", "value": ""}],
                                value="",
                                className="form-select-sm",
                            ),
                        ],
                        width=6,
                        md=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Label("Ordenar por", className="small text-muted"),
                            dbc.Select(
                                id="duplicates-sort-by",
                                options=[
                                    {"label": "Similitud", "value": "similarity"},
                                    {"label": "Ventas", "value": "sales"},
                                    {"label": "Fecha", "value": "date"},
                                ],
                                value="similarity",
                                className="form-select-sm",
                            ),
                        ],
                        width=6,
                        md=2,
                    ),
                    dbc.Col(
                        [
                            dbc.Label("Orden", className="small text-muted"),
                            dbc.Select(
                                id="duplicates-sort-order",
                                options=[
                                    {"label": "Descendente", "value": "desc"},
                                    {"label": "Ascendente", "value": "asc"},
                                ],
                                value="desc",
                                className="form-select-sm",
                            ),
                        ],
                        width=6,
                        md=2,
                    ),
                    dbc.Col(
                        [
                            dbc.Label("Fecha desde", className="small text-muted"),
                            dbc.Input(
                                id="duplicates-date-from",
                                type="date",
                                className="form-control-sm",
                            ),
                        ],
                        width=6,
                        md=2,
                    ),
                    dbc.Col(
                        [
                            dbc.Label(" ", className="small text-muted d-block"),  # Spacer
                            dbc.Button(
                                html.Span([
                                    html.I(className="fas fa-filter me-1"),
                                    "Aplicar",
                                ]),
                                id="duplicates-apply-filters-btn",
                                color="primary",
                                size="sm",
                                className="w-100",
                            ),
                        ],
                        width=6,
                        md=2,
                        className="d-flex flex-column",
                    ),
                    dbc.Col(
                        [
                            dbc.Label(" ", className="small text-muted d-block"),  # Spacer
                            dbc.Button(
                                html.Span([
                                    html.I(className="fas fa-times me-1"),
                                    "Limpiar",
                                ]),
                                id="duplicates-clear-filters-btn",
                                color="secondary",
                                size="sm",
                                outline=True,
                                className="w-100",
                            ),
                        ],
                        width=6,
                        md=1,
                        className="d-flex flex-column",
                    ),
                ],
                className="mb-4 g-2",
            ),

            # SECCIÓN 2.5: BÚSQUEDA MANUAL (Issue #475)
            html.H6(
                [
                    html.I(className="fas fa-search me-2 text-primary"),
                    "Búsqueda Manual de Duplicados",
                ],
                className="mb-3 text-muted",
            ),
            dbc.Card(
                dbc.CardBody(
                    [
                        dbc.Row(
                            [
                                dbc.Col(
                                    [
                                        dbc.InputGroup(
                                            [
                                                dbc.InputGroupText(
                                                    html.I(className="fas fa-search")
                                                ),
                                                dbc.Input(
                                                    id="duplicates-search-input",
                                                    placeholder="Buscar producto similar (mín. 3 caracteres)...",
                                                    type="text",
                                                    debounce=True,
                                                ),
                                                dbc.InputGroupText(
                                                    [
                                                        dbc.Select(
                                                            id="duplicates-search-threshold",
                                                            options=[
                                                                {"label": "60%", "value": "0.6"},
                                                                {"label": "70%", "value": "0.7"},
                                                                {"label": "80%", "value": "0.8"},
                                                            ],
                                                            value="0.6",
                                                            className="form-select-sm border-0",
                                                            style={"width": "70px"},
                                                        ),
                                                    ],
                                                    className="p-0",
                                                ),
                                                dbc.Button(
                                                    html.Span([
                                                        html.I(className="fas fa-search me-1"),
                                                        "Buscar",
                                                    ]),
                                                    id="duplicates-search-btn",
                                                    color="primary",
                                                ),
                                            ],
                                            size="lg",
                                        ),
                                        html.Small(
                                            "Busca productos similares en el catálogo para comparar manualmente.",
                                            className="text-muted d-block mt-1",
                                        ),
                                    ],
                                    width=12,
                                ),
                            ],
                        ),
                        # Resultados de búsqueda
                        html.Div(
                            id="duplicates-search-results",
                            className="mt-3",
                        ),
                    ]
                ),
                className="mb-4 shadow-sm",
            ),

            # SECCIÓN 3: LISTA DE GRUPOS
            html.H6("Grupos de Duplicados Potenciales", className="mb-3 text-muted"),
            html.Div(
                id="duplicates-groups-container",
                children=[
                    dbc.Alert(
                        html.Span([
                            html.I(className="fas fa-spinner fa-spin me-2"),
                            "Cargando grupos de duplicados...",
                        ]),
                        color="light",
                    )
                ],
            ),

            # PAGINACIÓN
            dbc.Row(
                [
                    dbc.Col(
                        html.Span(id="duplicates-pagination-info", className="text-muted"),
                        width="auto",
                    ),
                    dbc.Col(
                        dbc.ButtonGroup(
                            [
                                dbc.Button(
                                    html.Span([html.I(className="fas fa-chevron-left me-1"), "Anterior"]),
                                    id="duplicates-prev-page",
                                    color="secondary",
                                    size="sm",
                                    outline=True,
                                    disabled=True,
                                ),
                                dbc.Button(
                                    id="duplicates-current-page",
                                    color="light",
                                    size="sm",
                                    disabled=True,
                                    children="Página 1",
                                ),
                                dbc.Button(
                                    html.Span(["Siguiente", html.I(className="fas fa-chevron-right ms-1")]),
                                    id="duplicates-next-page",
                                    color="secondary",
                                    size="sm",
                                    outline=True,
                                    disabled=True,
                                ),
                            ],
                        ),
                        width="auto",
                    ),
                ],
                className="justify-content-between align-items-center mt-3",
            ),

            # MODAL DE CONFIRMACIÓN DE MERGE
            dbc.Modal(
                [
                    dbc.ModalHeader(
                        html.H5([
                            html.I(className="fas fa-compress-arrows-alt me-2 text-success"),
                            "Confirmar Merge",
                        ])
                    ),
                    dbc.ModalBody(
                        [
                            html.P(
                                "Esta acción fusionará los productos seleccionados. "
                                "Las ventas serán reasignadas al producto primario.",
                                className="text-muted mb-3",
                            ),
                            html.Div(id="duplicates-merge-preview"),
                        ]
                    ),
                    dbc.ModalFooter(
                        [
                            dbc.Button("Cancelar", id="duplicates-merge-cancel", color="secondary", outline=True),
                            dbc.Button(
                                html.Span([html.I(className="fas fa-check me-2"), "Confirmar Merge"]),
                                id="duplicates-merge-confirm",
                                color="success",
                            ),
                        ]
                    ),
                ],
                id="duplicates-merge-modal",
                size="lg",
                is_open=False,
            ),
        ],
        fluid=True,
        className="py-4",
    )


# =============================================================================
# PANELES COMBINADOS (Issue #485: Reorganización 7→4 tabs)
# =============================================================================


def create_catalogs_panel() -> html.Div:
    """
    Panel combinado de Catálogos Externos (Issue #485).

    Combina en sub-tabs:
    - CIMA/Nomenclator: Sincronización de catálogos externos
    - Prescripción: Gestión de clasificación de productos prescripción

    Estos son datos que IMPORTAMOS de fuentes externas (Ministerio, AEMPS).
    """
    from layouts.admin_prescription_panel import create_prescription_panel

    return dbc.Container(
        [
            # Header
            html.H5(
                [
                    html.I(className="fas fa-cloud-download-alt me-2 text-primary"),
                    "Catálogos Externos",
                ],
                className="mb-2",
            ),
            html.P(
                "Datos importados de fuentes externas: CIMA, Nomenclátor, Referencias de Prescripción.",
                className="text-muted mb-4",
            ),

            # Sub-tabs
            dbc.Tabs(
                id="catalogs-sub-tabs",
                active_tab="cima-nomenclator",
                children=[
                    dbc.Tab(
                        label="CIMA / Nomenclátor",
                        tab_id="cima-nomenclator",
                        children=[
                            html.Div(
                                _create_cima_nomenclator_section(),
                                className="pt-4",
                            )
                        ],
                    ),
                    dbc.Tab(
                        label="Prescripción",
                        tab_id="prescription",
                        children=[
                            html.Div(
                                create_prescription_panel(),
                                className="pt-4",
                            )
                        ],
                    ),
                ],
                className="nav-pills",
            ),
        ],
        fluid=True,
        className="py-3",
    )


def _create_cima_nomenclator_section() -> html.Div:
    """Sección interna de CIMA/Nomenclátor (extraída de database_panel)."""
    return html.Div([
        # Stats cards
        dbc.Row(
            [
                # CIMA
                dbc.Col(
                    [
                        BaseCard(
                            children=[
                                dbc.CardHeader("CIMA", className="fw-bold bg-light"),
                                dbc.CardBody(
                                    [
                                        html.Div(
                                            [
                                                html.I(className="fas fa-capsules me-2 text-primary"),
                                                html.Strong("Productos: "),
                                                html.Span(
                                                    "--",
                                                    id="admin-cima-count",
                                                    className="text-primary fs-5",
                                                ),
                                            ],
                                            className="mb-2",
                                        ),
                                        html.Div(
                                            [
                                                html.I(className="fas fa-clock me-2 text-muted"),
                                                html.Strong("Última sync: "),
                                                html.Span(
                                                    "--",
                                                    id="admin-cima-last-sync",
                                                    className="text-muted",
                                                ),
                                            ],
                                            className="mb-2",
                                        ),
                                        html.Div(
                                            [
                                                html.I(className="fas fa-info-circle me-2"),
                                                html.Strong("Estado: "),
                                                html.Span("--", id="admin-cima-status"),
                                            ]
                                        ),
                                    ]
                                ),
                            ],
                            shadow="sm",
                        )
                    ],
                    width=12,
                    lg=6,
                    className="mb-3",
                ),
                # Nomenclátor
                dbc.Col(
                    [
                        BaseCard(
                            children=[
                                dbc.CardHeader("Nomenclátor", className="fw-bold bg-light"),
                                dbc.CardBody(
                                    [
                                        html.Div(
                                            [
                                                html.I(className="fas fa-book-medical me-2 text-info"),
                                                html.Strong("Productos: "),
                                                html.Span(
                                                    "--",
                                                    id="admin-nomenclator-count",
                                                    className="text-info fs-5",
                                                ),
                                            ],
                                            className="mb-2",
                                        ),
                                        html.Div(
                                            [
                                                html.I(className="fas fa-clock me-2 text-muted"),
                                                html.Strong("Última sync: "),
                                                html.Span(
                                                    "--",
                                                    id="admin-nomenclator-last-sync",
                                                    className="text-muted",
                                                ),
                                            ],
                                            className="mb-2",
                                        ),
                                        html.Div(
                                            [
                                                html.I(className="fas fa-info-circle me-2"),
                                                html.Strong("Estado: "),
                                                html.Span("--", id="admin-nomenclator-status"),
                                            ]
                                        ),
                                    ]
                                ),
                            ],
                            shadow="sm",
                        )
                    ],
                    width=12,
                    lg=6,
                    className="mb-3",
                ),
            ],
            className="mb-4",
        ),

        # Botones de sincronización
        html.Hr(),
        html.H6("Operaciones de Sincronización", className="mb-3"),
        dbc.Row(
            [
                dbc.Col(
                    [
                        dcc.Loading(
                            id="sync-buttons-loading",
                            type="circle",
                            color="#0d6efd",
                            children=[
                                dbc.ButtonGroup(
                                    [
                                        dbc.Button(
                                            [
                                                html.I(className="fas fa-sync me-2", id="sync-cima-icon"),
                                                html.Span("Sync CIMA", id="sync-cima-text"),
                                            ],
                                            id="admin-sync-cima-button",
                                            color="primary",
                                            outline=True,
                                            disabled=False,
                                        ),
                                        dbc.Button(
                                            [
                                                html.I(className="fas fa-sync me-2", id="sync-nomenclator-icon"),
                                                html.Span("Sync Nomenclátor", id="sync-nomenclator-text"),
                                            ],
                                            id="admin-sync-nomenclator-button",
                                            color="info",
                                            outline=True,
                                            disabled=False,
                                        ),
                                        dbc.Button(
                                            [
                                                html.I(className="fas fa-sync-alt me-2", id="sync-all-icon"),
                                                html.Span("Sync Completo", id="sync-all-text"),
                                            ],
                                            id="admin-sync-all-button",
                                            color="success",
                                            outline=True,
                                            disabled=False,
                                        ),
                                    ],
                                    className="w-100",
                                )
                            ],
                        ),
                        html.Small(
                            "Las sincronizaciones pueden tardar entre 30s y 3min.",
                            className="text-muted d-block mt-2"
                        ),
                    ]
                )
            ],
            className="mb-4",
        ),
    ])


def create_classification_panel() -> html.Div:
    """
    Panel de clasificación NECESIDAD para productos VentaLibre (Issue #449).

    Interfaz de Human-in-the-Loop para:
    - Aprobar clasificaciones automáticas del ML
    - Corregir clasificaciones incorrectas
    - Marcar productos como outliers
    - Saltar productos para revisión posterior

    Usa la API de feedback existente (/api/v1/feedback/*).
    """
    return dbc.Container(
        [
            # Título y descripción
            html.H5(
                [
                    html.I(className="fas fa-tasks me-2 text-success"),
                    "Clasificación NECESIDAD",
                ],
                className="mb-3",
            ),
            dbc.Alert(
                html.Div([
                    html.I(className="fas fa-info-circle me-2"),
                    "Revisión y corrección de clasificaciones automáticas. ",
                    "Prioridad P1 = 'otros', P2 = baja confianza (<0.6), P3 = resto pendientes.",
                ]),
                color="info",
                className="mb-4",
            ),

            # SECCIÓN 1: KPIs
            html.H6("Métricas de Clasificación", className="mb-3 text-muted"),
            dbc.Row(
                [
                    # KPI: Cobertura
                    dbc.Col(
                        BaseCard(
                            children=[
                                dbc.CardBody(
                                    [
                                        html.Div(
                                            [
                                                html.I(className="fas fa-percentage me-2 text-primary"),
                                                html.Strong("Cobertura"),
                                            ],
                                            className="mb-2",
                                        ),
                                        html.H3(
                                            "--",
                                            id="classification-coverage-pct",
                                            className="text-primary mb-1",
                                        ),
                                        html.Small(
                                            "Productos clasificados",
                                            className="text-muted",
                                        ),
                                    ]
                                )
                            ],
                            shadow="sm",
                        ),
                        width=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # KPI: P1 (otros)
                    dbc.Col(
                        BaseCard(
                            children=[
                                dbc.CardBody(
                                    [
                                        html.Div(
                                            [
                                                html.I(className="fas fa-exclamation-circle me-2 text-danger"),
                                                html.Strong("P1 - 'Otros'"),
                                            ],
                                            className="mb-2",
                                        ),
                                        html.H3(
                                            "--",
                                            id="classification-p1-count",
                                            className="text-danger mb-1",
                                        ),
                                        html.Small(
                                            "Categoría genérica",
                                            className="text-muted",
                                        ),
                                    ]
                                )
                            ],
                            shadow="sm",
                        ),
                        width=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # KPI: P2 (baja confianza)
                    dbc.Col(
                        BaseCard(
                            children=[
                                dbc.CardBody(
                                    [
                                        html.Div(
                                            [
                                                html.I(className="fas fa-question-circle me-2 text-warning"),
                                                html.Strong("P2 - Baja Conf."),
                                            ],
                                            className="mb-2",
                                        ),
                                        html.H3(
                                            "--",
                                            id="classification-p2-count",
                                            className="text-warning mb-1",
                                        ),
                                        html.Small(
                                            "Confianza < 60%",
                                            className="text-muted",
                                        ),
                                    ]
                                )
                            ],
                            shadow="sm",
                        ),
                        width=6,
                        lg=3,
                        className="mb-3",
                    ),
                    # KPI: Accuracy
                    dbc.Col(
                        BaseCard(
                            children=[
                                dbc.CardBody(
                                    [
                                        html.Div(
                                            [
                                                html.I(className="fas fa-check-double me-2 text-success"),
                                                html.Strong("Accuracy"),
                                            ],
                                            className="mb-2",
                                        ),
                                        html.H3(
                                            "--",
                                            id="classification-accuracy-pct",
                                            className="text-success mb-1",
                                        ),
                                        html.Small(
                                            "Correcciones humanas",
                                            className="text-muted",
                                        ),
                                    ]
                                )
                            ],
                            shadow="sm",
                        ),
                        width=6,
                        lg=3,
                        className="mb-3",
                    ),
                ],
                className="mb-4",
            ),

            # SECCIÓN 2: FILTROS Y ACCIONES
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Label("Prioridad:", className="me-2 text-muted"),
                            dbc.Select(
                                id="classification-priority-filter",
                                options=[
                                    {"label": "Todos", "value": "all"},
                                    {"label": "P1 - 'Otros'", "value": "p1"},
                                    {"label": "P2 - Baja confianza", "value": "p2"},
                                    {"label": "P3 - Pendientes", "value": "p3"},
                                ],
                                value="all",
                                style={"width": "180px", "display": "inline-block"},
                            ),
                        ],
                        width="auto",
                        className="d-flex align-items-center",
                    ),
                    dbc.Col(
                        [
                            dbc.Button(
                                html.Span([
                                    html.I(className="fas fa-sync me-2"),
                                    "Actualizar",
                                ]),
                                id="classification-refresh-btn",
                                color="primary",
                                outline=True,
                                size="sm",
                                className="me-2",
                            ),
                            html.Small(
                                id="classification-last-update",
                                className="text-muted",
                            ),
                        ],
                        width="auto",
                    ),
                ],
                className="mb-4 align-items-center",
            ),

            # SECCIÓN 3: TABLA DE PRODUCTOS
            html.H6("Productos Pendientes de Revisión", className="mb-3 text-muted"),
            html.Div(
                id="classification-table-container",
                children=[
                    dbc.Alert(
                        html.Span([
                            html.I(className="fas fa-spinner fa-spin me-2"),
                            "Cargando productos...",
                        ]),
                        color="light",
                        className="text-center py-5",
                    ),
                ],
            ),

            # SECCIÓN 4: PAGINACIÓN
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.ButtonGroup(
                                [
                                    dbc.Button(
                                        html.I(className="fas fa-chevron-left"),
                                        id="classification-prev-page",
                                        color="secondary",
                                        outline=True,
                                        size="sm",
                                        disabled=True,
                                    ),
                                    dbc.Button(
                                        html.Span([
                                            "Página ",
                                            html.Span("1", id="classification-current-page"),
                                        ]),
                                        color="light",
                                        size="sm",
                                        disabled=True,
                                    ),
                                    dbc.Button(
                                        html.I(className="fas fa-chevron-right"),
                                        id="classification-next-page",
                                        color="secondary",
                                        outline=True,
                                        size="sm",
                                        disabled=True,
                                    ),
                                ],
                            ),
                            html.Small(
                                id="classification-pagination-info",
                                className="ms-3 text-muted",
                            ),
                        ],
                        className="d-flex align-items-center justify-content-center",
                    ),
                ],
                className="mt-3",
            ),
        ],
        fluid=True,
        className="py-3",
    )


def create_venta_libre_panel() -> html.Div:
    """
    Panel combinado de Venta Libre (Issue #485).

    Combina en sub-tabs:
    - Sin Enriquecer: Productos sin enriquecer para clasificación manual
    - Validar NECESIDAD: Validación human-in-the-loop de clasificaciones
    - Duplicados: Detección y merge de duplicados VL
    - ML Monitoring: Métricas del clasificador

    Estos son datos que ENRIQUECEMOS con nuestro motor de clasificación.
    """
    return dbc.Container(
        [
            # Header
            html.H5(
                [
                    html.I(className="fas fa-tags me-2 text-warning"),
                    "Venta Libre - Clasificación",
                ],
                className="mb-2",
            ),
            html.P(
                "Productos enriquecidos con nuestro motor de clasificación: revisión manual, duplicados, monitoreo ML.",
                className="text-muted mb-4",
            ),

            # Sub-tabs
            dbc.Tabs(
                id="venta-libre-sub-tabs",
                active_tab="manual-review",
                children=[
                    dbc.Tab(
                        label="🔍 Sin Enriquecer",
                        tab_id="manual-review",
                        children=[
                            html.Div(
                                id="vl-manual-review-content",
                                children=create_manual_review_panel(),
                                className="pt-4",
                            )
                        ],
                    ),
                    dbc.Tab(
                        label="🔄 Duplicados",
                        tab_id="duplicates",
                        children=[
                            html.Div(
                                id="vl-duplicates-content",
                                children=create_duplicates_panel(),
                                className="pt-4",
                            )
                        ],
                    ),
                    dbc.Tab(
                        label="📊 ML Monitoring",
                        tab_id="ml-monitoring",
                        children=[
                            html.Div(
                                id="vl-ml-monitoring-content",
                                children=create_ml_monitoring_panel(),
                                className="pt-4",
                            )
                        ],
                    ),
                    dbc.Tab(
                        label="✅ Validar NECESIDAD",
                        tab_id="classification",
                        children=[
                            html.Div(
                                id="vl-classification-content",
                                children=create_classification_panel(),
                                className="pt-4",
                            )
                        ],
                    ),
                    dbc.Tab(
                        label="🔑 Keywords",
                        tab_id="keywords",
                        children=[
                            html.Div(
                                id="vl-keywords-content",
                                children=create_keywords_panel(),
                                className="pt-4",
                            )
                        ],
                    ),
                    dbc.Tab(
                        label="🔄 Aliases",
                        tab_id="aliases",
                        children=[
                            html.Div(
                                id="vl-aliases-content",
                                children=create_aliases_panel(),
                                className="pt-4",
                            )
                        ],
                    ),
                    dbc.Tab(
                        label="🏷️ Brand Aliases",
                        tab_id="brand-aliases",
                        children=[
                            html.Div(
                                id="vl-brand-aliases-content",
                                children=create_brand_aliases_panel(),
                                className="pt-4",
                            )
                        ],
                    ),
                    dbc.Tab(
                        label="🔗 Grupos Curados",
                        tab_id="curated-groups",
                        children=[
                            html.Div(
                                id="vl-curated-groups-content",
                                children=create_curated_groups_panel(),
                                className="pt-4",
                            )
                        ],
                    ),
                ],
                className="nav-pills",
            ),
        ],
        fluid=True,
        className="py-3",
    )


def create_keywords_panel() -> html.Div:
    """
    Panel de gestión de keywords para clasificador NECESIDAD (Issue #449).

    Permite añadir, editar y eliminar keywords dinámicamente desde la UI
    sin necesidad de modificar código Python ni hacer deploy.

    Tipos de keywords:
    - keyword: Palabras genéricas (anticaida -> caida_cabello)
    - brand: Marcas (olistic -> caida_cabello)
    - blacklist: Exclusiones (regalo -> interno_no_venta)
    """
    return dbc.Container(
        [
            # Header
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.H5(
                                [
                                    html.I(className="fas fa-key me-2 text-warning"),
                                    "Gestión de Keywords",
                                ],
                                className="mb-2",
                            ),
                            html.P(
                                [
                                    "Añade, edita o elimina keywords para el clasificador NECESIDAD. ",
                                    html.Strong("Los cambios toman efecto inmediatamente."),
                                ],
                                className="text-muted mb-0",
                            ),
                        ],
                        width=12,
                        lg=8,
                    ),
                    dbc.Col(
                        [
                            dbc.ButtonGroup(
                                [
                                    BaseButton(
                                        children=html.Span([html.I(className="fas fa-plus me-2"), "Añadir Keyword"]),
                                        id="keywords-add-btn",
                                        variant="primary",
                                        size="sm",
                                    ),
                                    BaseButton(
                                        children=html.Span([html.I(className="fas fa-magic me-2"), "Aplicar a Productos"]),
                                        id="keywords-apply-btn",
                                        variant="success",
                                        outline=True,
                                        size="sm",
                                    ),
                                    BaseButton(
                                        children=html.I(className="fas fa-sync-alt"),
                                        id="keywords-refresh-btn",
                                        variant="secondary",
                                        outline=True,
                                        size="sm",
                                    ),
                                ],
                                className="float-end",
                            ),
                        ],
                        width=12,
                        lg=4,
                        className="text-end",
                    ),
                ],
                className="mb-4",
            ),

            # Filtros
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Label("Tipo", className="small"),
                            dbc.Select(
                                id="keywords-type-filter",
                                options=[
                                    {"label": "Todos", "value": "all"},
                                    {"label": "Keywords", "value": "keyword"},
                                    {"label": "Marcas", "value": "brand"},
                                    {"label": "Blacklist", "value": "blacklist"},
                                ],
                                value="all",
                                size="sm",
                            ),
                        ],
                        width=6,
                        md=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Label("Categoría", className="small"),
                            dbc.Select(
                                id="keywords-category-filter",
                                options=[{"label": "Todas", "value": "all"}],
                                value="all",
                                size="sm",
                            ),
                        ],
                        width=6,
                        md=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Label("Buscar", className="small"),
                            dbc.Input(
                                id="keywords-search-input",
                                type="text",
                                placeholder="Buscar keyword...",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=4,
                    ),
                    dbc.Col(
                        [
                            dbc.Label(" ", className="small d-block"),
                            dbc.Checklist(
                                id="keywords-show-inactive",
                                options=[{"label": "Mostrar inactivos", "value": "show_inactive"}],
                                value=[],
                                switch=True,
                            ),
                        ],
                        width=12,
                        md=2,
                    ),
                ],
                className="mb-3",
            ),

            # Stats
            html.Div(id="keywords-stats-container", className="mb-3"),

            # Table
            html.Div(id="keywords-table-container"),

            # Pagination
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Span(
                                "--",
                                id="keywords-pagination-info",
                                className="text-muted small",
                            ),
                        ],
                        width=6,
                    ),
                    dbc.Col(
                        [
                            dbc.ButtonGroup(
                                [
                                    dbc.Button(
                                        html.I(className="fas fa-chevron-left"),
                                        id="keywords-prev-page",
                                        color="secondary",
                                        size="sm",
                                        outline=True,
                                    ),
                                    dbc.Button(
                                        html.I(className="fas fa-chevron-right"),
                                        id="keywords-next-page",
                                        color="secondary",
                                        size="sm",
                                        outline=True,
                                    ),
                                ],
                                className="float-end",
                            ),
                        ],
                        width=6,
                    ),
                ],
                className="mt-3",
            ),

            # Stores moved to app.py skeleton (REGLA #0.5)
            # keywords-page-store, keywords-data-store, keywords-edit-id-store are in app.py

            # Modal para añadir/editar keyword
            _create_keyword_modal(),

            # Modal para preview
            _create_keyword_preview_modal(),

            # Modal para aplicar keywords a productos (Issue #449 Phase 3)
            _create_keyword_apply_modal(),
        ],
        fluid=True,
        className="py-3",
    )


def _create_keyword_modal() -> dbc.Modal:
    """Modal para crear/editar keyword."""
    return dbc.Modal(
        [
            dbc.ModalHeader(
                dbc.ModalTitle(id="keywords-modal-title", children="Añadir Keyword"),
                close_button=True,
            ),
            dbc.ModalBody(
                [
                    dbc.Form(
                        [
                            dbc.Row(
                                [
                                    dbc.Col(
                                        [
                                            dbc.Label("Keyword *"),
                                            dbc.Input(
                                                id="keywords-form-keyword",
                                                type="text",
                                                placeholder="Ej: anticaida, olistic, regalo",
                                            ),
                                            dbc.FormText(
                                                "Se convertirá a minúsculas automáticamente.",
                                                className="text-muted",
                                            ),
                                        ],
                                        width=12,
                                        md=6,
                                    ),
                                    dbc.Col(
                                        [
                                            dbc.Label("Tipo *"),
                                            dbc.Select(
                                                id="keywords-form-type",
                                                options=[
                                                    {"label": "Keyword", "value": "keyword"},
                                                    {"label": "Marca", "value": "brand"},
                                                    {"label": "Blacklist", "value": "blacklist"},
                                                ],
                                                value="keyword",
                                            ),
                                        ],
                                        width=12,
                                        md=6,
                                    ),
                                ],
                                className="mb-3",
                            ),
                            dbc.Row(
                                [
                                    dbc.Col(
                                        [
                                            dbc.Label("Categoría NECESIDAD *"),
                                            dbc.Select(
                                                id="keywords-form-category",
                                                options=[],
                                                placeholder="Seleccionar categoría...",
                                            ),
                                        ],
                                        width=12,
                                        md=8,
                                    ),
                                    dbc.Col(
                                        [
                                            dbc.Label("Prioridad"),
                                            dbc.Input(
                                                id="keywords-form-priority",
                                                type="number",
                                                min=1,
                                                max=1000,
                                                value=100,
                                            ),
                                            dbc.FormText(
                                                "Mayor = más prioridad (default: 100).",
                                                className="text-muted",
                                            ),
                                        ],
                                        width=12,
                                        md=4,
                                    ),
                                ],
                                className="mb-3",
                            ),
                            dbc.Row(
                                [
                                    dbc.Col(
                                        [
                                            dbc.Label("Notas (opcional)"),
                                            dbc.Textarea(
                                                id="keywords-form-notes",
                                                placeholder="Notas adicionales...",
                                                rows=2,
                                            ),
                                        ],
                                        width=12,
                                    ),
                                ],
                                className="mb-3",
                            ),
                            html.Div(id="keywords-preview-container"),
                        ]
                    ),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        "Preview",
                        id="keywords-preview-btn",
                        color="info",
                        outline=True,
                        className="me-auto",
                    ),
                    dbc.Button(
                        "Cancelar",
                        id="keywords-modal-cancel-btn",
                        color="secondary",
                        outline=True,
                    ),
                    dbc.Button(
                        "Guardar",
                        id="keywords-modal-save-btn",
                        color="primary",
                    ),
                ]
            ),
        ],
        id="keywords-modal",
        is_open=False,
        size="lg",
    )


def _create_keyword_preview_modal() -> dbc.Modal:
    """Modal para preview de impacto de keyword."""
    return dbc.Modal(
        [
            dbc.ModalHeader(dbc.ModalTitle("Preview de Impacto")),
            dbc.ModalBody(id="keywords-preview-modal-body"),
            dbc.ModalFooter(
                dbc.Button(
                    "Cerrar",
                    id="keywords-preview-modal-close",
                    color="secondary",
                )
            ),
        ],
        id="keywords-preview-modal",
        is_open=False,
    )


def _create_keyword_apply_modal() -> dbc.Modal:
    """
    Modal para aplicar keywords a productos existentes (Issue #449 Phase 3).

    Permite previsualizar y aplicar keywords a productos de ProductCatalogVentaLibre.
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                dbc.ModalTitle("Aplicar Keywords a Productos"),
                close_button=True,
            ),
            dbc.ModalBody(
                [
                    dbc.Alert(
                        html.Span([
                            html.I(className="fas fa-info-circle me-2"),
                            "Esta acción escaneará todos los productos y aplicará los keywords activos ",
                            "para actualizar sus categorías NECESIDAD.",
                        ]),
                        color="info",
                        className="mb-3",
                    ),
                    # Preview results container
                    html.Div(id="keywords-apply-preview-container"),
                    # Note: keywords-apply-results-store is in global skeleton (app.py) per REGLA #0.5
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        html.Span([html.I(className="fas fa-eye me-2"), "Previsualizar"]),
                        id="keywords-apply-preview-btn",
                        color="info",
                        outline=True,
                        className="me-2",
                    ),
                    dbc.Button(
                        html.Span([html.I(className="fas fa-check me-2"), "Aplicar Cambios"]),
                        id="keywords-apply-confirm-btn",
                        color="success",
                        disabled=True,  # Enabled after preview
                    ),
                    dbc.Button(
                        "Cerrar",
                        id="keywords-apply-close-btn",
                        color="secondary",
                    ),
                ]
            ),
        ],
        id="keywords-apply-modal",
        is_open=False,
        size="lg",
    )


def create_system_panel() -> html.Div:
    """
    Panel de Sistema (Issue #485).

    Combina en sub-tabs:
    - Herramientas: Dangerous operations, audit logs
    - Base de Datos: Backups, stats, mantenimiento

    Operaciones de infraestructura y mantenimiento.
    """
    from components.admin_danger_zone import create_danger_zone_panel

    return dbc.Container(
        [
            # Header
            html.H5(
                [
                    html.I(className="fas fa-cogs me-2 text-secondary"),
                    "Sistema",
                ],
                className="mb-2",
            ),
            html.P(
                "Operaciones de infraestructura: herramientas administrativas, backups, estadísticas.",
                className="text-muted mb-4",
            ),

            # Sub-tabs
            dbc.Tabs(
                id="system-sub-tabs",
                active_tab="tools",
                children=[
                    dbc.Tab(
                        label="🔧 Herramientas",
                        tab_id="tools",
                        children=[
                            html.Div(
                                _create_tools_section(create_danger_zone_panel),
                                className="pt-4",
                            )
                        ],
                    ),
                    dbc.Tab(
                        label="💾 Base de Datos",
                        tab_id="database",
                        children=[
                            html.Div(
                                _create_database_section(),
                                className="pt-4",
                            )
                        ],
                    ),
                ],
                className="nav-pills",
            ),

            # Stores necesarios
            dcc.Store(id="sync-operation-feedback"),
        ],
        fluid=True,
        className="py-3",
    )


def _create_tools_section(create_danger_zone_panel) -> html.Div:
    """Sección de herramientas (dangerous ops + audit logs)."""
    return html.Div([
        # Panel moderno con AdminDangerZone
        create_danger_zone_panel(panel_id="admin-danger-zone"),

        # Audit Logs
        html.Hr(className="my-4"),
        html.H6(
            [
                html.I(className="fas fa-clipboard-list me-2 text-primary"),
                "Registro de Auditoría",
            ],
            className="mb-3",
        ),
        dbc.Row(
            [
                dbc.Col(
                    [
                        html.Label("Filtrar por acción", className="form-label small"),
                        dbc.Select(
                            id="admin-audit-action-filter",
                            options=[
                                {"label": "Todas", "value": "all"},
                                {"label": "Crear Usuario", "value": "create_user"},
                                {"label": "Editar Usuario", "value": "update_user"},
                                {"label": "Eliminar Usuario", "value": "delete_user"},
                                {"label": "Restaurar Usuario", "value": "restore_user"},
                                {"label": "Crear Backup", "value": "create_backup"},
                                {"label": "Operación Peligrosa", "value": "dangerous_operation"},
                            ],
                            value="all",
                            size="sm",
                        ),
                    ],
                    width=12,
                    md=4,
                ),
                dbc.Col(
                    [
                        html.Label("Límite", className="form-label small"),
                        dbc.Select(
                            id="admin-audit-limit-filter",
                            options=[
                                {"label": "50", "value": "50"},
                                {"label": "100", "value": "100"},
                                {"label": "500", "value": "500"},
                            ],
                            value="100",
                            size="sm",
                        ),
                    ],
                    width=12,
                    md=4,
                ),
                dbc.Col(
                    [
                        html.Label(" ", className="form-label small"),
                        BaseButton(
                            children=html.Div([
                                html.I(className="fas fa-sync me-2"),
                                "Actualizar Logs",
                            ]),
                            id="admin-refresh-audit-logs-btn",
                            variant="secondary",
                            outline=True,
                            size="sm",
                            className="w-100 d-flex align-items-center justify-content-center",
                        ),
                    ],
                    width=12,
                    md=4,
                ),
            ],
            className="mb-3",
        ),
        html.Div(
            id="admin-audit-logs-container",
            children=[
                dbc.Alert(
                    "Los logs de auditoría se cargarán aquí",
                    color="light",
                    className="text-center text-muted",
                )
            ],
        ),
    ])


def _create_database_section() -> html.Div:
    """Sección de base de datos (backups + stats)."""
    return html.Div([
        # Backups
        html.H6(
            [
                html.I(className="fas fa-database me-2 text-success"),
                "Backups y Restauración",
            ],
            className="mb-3",
        ),
        dbc.Row(
            [
                dbc.Col(
                    [
                        BaseButton(
                            children=html.Div([
                                html.I(className="fas fa-plus me-2"),
                                "Crear Backup",
                            ]),
                            id="admin-create-backup-btn",
                            variant="success",
                            outline=True,
                            className="w-100 mb-2 d-flex align-items-center justify-content-center",
                        ),
                        html.Small(
                            "Límite: 3 backups por hora",
                            className="text-muted d-block text-center",
                        ),
                    ],
                    width=12,
                    md=4,
                ),
                dbc.Col(
                    [
                        html.Div(
                            id="admin-backup-list-container",
                            children=[
                                dbc.Alert(
                                    "Los backups se mostrarán aquí",
                                    color="light",
                                    className="mb-0",
                                )
                            ],
                        )
                    ],
                    width=12,
                    md=8,
                ),
            ],
            className="mb-4",
        ),

        # Stats BD
        html.Hr(className="my-4"),
        html.H6(
            [
                html.I(className="fas fa-chart-bar me-2 text-info"),
                "Estadísticas de Base de Datos",
            ],
            className="mb-3",
        ),
        html.Div(
            id="admin-database-stats-container",
            children=[
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                BaseCard(
                                    children=[
                                        dbc.CardBody(
                                            [
                                                html.H6("Tamaño Total", className="text-muted mb-2"),
                                                html.H3("--", id="admin-db-total-size", className="text-info mb-0"),
                                            ]
                                        )
                                    ],
                                    shadow="sm",
                                )
                            ],
                            width=12,
                            md=4,
                            className="mb-3",
                        ),
                        dbc.Col(
                            [
                                BaseCard(
                                    children=[
                                        dbc.CardBody(
                                            [
                                                html.H6("Registros Totales", className="text-muted mb-2"),
                                                html.H3("--", id="admin-db-total-records", className="text-primary mb-0"),
                                            ]
                                        )
                                    ],
                                    shadow="sm",
                                )
                            ],
                            width=12,
                            md=4,
                            className="mb-3",
                        ),
                        dbc.Col(
                            [
                                BaseCard(
                                    children=[
                                        dbc.CardBody(
                                            [
                                                html.H6("Conexiones Activas", className="text-muted mb-2"),
                                                html.H3("--", id="admin-db-active-connections", className="text-success mb-0"),
                                            ]
                                        )
                                    ],
                                    shadow="sm",
                                )
                            ],
                            width=12,
                            md=4,
                            className="mb-3",
                        ),
                    ]
                )
            ],
        ),
    ])


def create_aliases_panel() -> html.Div:
    """
    Panel de gestión de category aliases para clasificador NECESIDAD (Issue #459).

    Los aliases normalizan la salida del clasificador a nombres esperados en BD.
    Ejemplo: "aftas_llagas" (clasificador) → "aftas" (nombre en BD)

    Diferente de Keywords:
    - Keywords: Pattern matching en INPUT (nombre producto)
    - Aliases: Normalización de OUTPUT (categoría clasificada)
    """
    return dbc.Container(
        [
            # Header
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.H5(
                                [
                                    html.I(className="fas fa-exchange-alt me-2 text-info"),
                                    "Gestión de Aliases",
                                ],
                                className="mb-2",
                            ),
                            html.P(
                                [
                                    "Mapea categorías del clasificador a nombres esperados en BD. ",
                                    html.Strong("Los cambios toman efecto inmediatamente."),
                                ],
                                className="text-muted mb-0",
                            ),
                        ],
                        width=12,
                        lg=8,
                    ),
                    dbc.Col(
                        [
                            dbc.ButtonGroup(
                                [
                                    BaseButton(
                                        children=html.Span([html.I(className="fas fa-plus me-2"), "Añadir Alias"]),
                                        id="aliases-add-btn",
                                        variant="primary",
                                        size="sm",
                                    ),
                                    BaseButton(
                                        children=html.I(className="fas fa-sync-alt"),
                                        id="aliases-refresh-btn",
                                        variant="secondary",
                                        outline=True,
                                        size="sm",
                                    ),
                                ],
                                className="float-end",
                            ),
                        ],
                        width=12,
                        lg=4,
                        className="text-end",
                    ),
                ],
                className="mb-4",
            ),

            # Filtros
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Label("Buscar", className="small"),
                            dbc.Input(
                                id="aliases-search-input",
                                type="text",
                                placeholder="Buscar en source o target...",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=6,
                    ),
                    dbc.Col(
                        [
                            dbc.Label(" ", className="small d-block"),
                            dbc.Checklist(
                                id="aliases-show-inactive",
                                options=[{"label": "Mostrar inactivos", "value": "show_inactive"}],
                                value=[],
                                switch=True,
                            ),
                        ],
                        width=12,
                        md=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Label(" ", className="small d-block"),
                            html.Span(
                                id="aliases-last-update",
                                className="text-muted small",
                            ),
                        ],
                        width=12,
                        md=3,
                        className="text-end pt-2",
                    ),
                ],
                className="mb-3",
            ),

            # Stats badges
            html.Div(id="aliases-stats-container", className="mb-3"),

            # Table
            html.Div(id="aliases-table-container"),

            # Pagination
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Span(
                                id="aliases-pagination-info",
                                className="text-muted small",
                            ),
                        ],
                        width=6,
                    ),
                    dbc.Col(
                        [
                            dbc.ButtonGroup(
                                [
                                    dbc.Button(
                                        html.I(className="fas fa-chevron-left"),
                                        id="aliases-prev-page",
                                        size="sm",
                                        outline=True,
                                        color="secondary",
                                    ),
                                    dbc.Button(
                                        html.I(className="fas fa-chevron-right"),
                                        id="aliases-next-page",
                                        size="sm",
                                        outline=True,
                                        color="secondary",
                                    ),
                                ],
                                className="float-end",
                            ),
                        ],
                        width=6,
                        className="text-end",
                    ),
                ],
                className="mt-3",
            ),

            # Modal para añadir/editar alias
            _create_alias_modal(),
        ],
        fluid=True,
        className="py-3",
    )


def _create_alias_modal() -> dbc.Modal:
    """Modal para crear/editar category alias."""
    return dbc.Modal(
        [
            dbc.ModalHeader(
                dbc.ModalTitle(id="aliases-modal-title", children="Añadir Alias"),
                close_button=True,
            ),
            dbc.ModalBody(
                [
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Categoría Origen (clasificador) *"),
                                    dbc.Input(
                                        id="aliases-form-source",
                                        type="text",
                                        placeholder="ej: aftas_llagas",
                                    ),
                                    dbc.FormText("Lo que devuelve el clasificador"),
                                ],
                                width=12,
                                className="mb-3",
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Categoría Destino (BD) *"),
                                    dbc.Input(
                                        id="aliases-form-target",
                                        type="text",
                                        placeholder="ej: aftas",
                                    ),
                                    dbc.FormText("Lo que espera la base de datos"),
                                ],
                                width=12,
                                className="mb-3",
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Razón / Notas"),
                                    dbc.Textarea(
                                        id="aliases-form-reason",
                                        placeholder="Motivo del alias...",
                                        rows=2,
                                    ),
                                ],
                                width=12,
                                className="mb-3",
                            ),
                        ]
                    ),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        "Cancelar",
                        id="aliases-modal-cancel-btn",
                        color="secondary",
                        outline=True,
                    ),
                    dbc.Button(
                        "Guardar",
                        id="aliases-modal-save-btn",
                        color="primary",
                    ),
                ]
            ),
        ],
        id="aliases-modal",
        is_open=False,
        centered=True,
    )


def create_brand_aliases_panel() -> html.Div:
    """
    Panel de gestión de brand aliases para corrección de marcas detectadas.

    Permite dos acciones:
    - ALIAS: Fusionar marcas (ej: "oralkin" → "kin")
    - EXCLUDE: Excluir falsos positivos (ej: "clorhexidina" no es marca)

    Diferente de CategoryAlias:
    - CategoryAlias: Normaliza OUTPUT del clasificador NECESIDAD
    - BrandAlias: Normaliza OUTPUT del BrandDetectionService
    """
    return dbc.Container(
        [
            # Header
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.H5(
                                [
                                    html.I(className="fas fa-tags me-2 text-warning"),
                                    "Gestión de Brand Aliases",
                                ],
                                className="mb-2",
                            ),
                            html.P(
                                [
                                    "Corrige marcas detectadas: fusiona variantes o excluye falsos positivos. ",
                                    html.Strong("Los cambios toman efecto inmediatamente en nuevos procesamientos."),
                                ],
                                className="text-muted mb-0",
                            ),
                        ],
                        width=12,
                        lg=8,
                    ),
                    dbc.Col(
                        [
                            dbc.ButtonGroup(
                                [
                                    BaseButton(
                                        children=html.Span([html.I(className="fas fa-plus me-2"), "Añadir"]),
                                        id="brand-aliases-add-btn",
                                        variant="primary",
                                        size="sm",
                                    ),
                                    BaseButton(
                                        children=html.I(className="fas fa-sync-alt"),
                                        id="brand-aliases-refresh-btn",
                                        variant="secondary",
                                        outline=True,
                                        size="sm",
                                    ),
                                ],
                                className="float-end",
                            ),
                        ],
                        width=12,
                        lg=4,
                        className="text-end",
                    ),
                ],
                className="mb-4",
            ),

            # Filtros
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Label("Buscar", className="small"),
                            dbc.Input(
                                id="brand-aliases-search-input",
                                type="text",
                                placeholder="Buscar marca origen o destino...",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=4,
                    ),
                    dbc.Col(
                        [
                            dbc.Label("Acción", className="small"),
                            dbc.Select(
                                id="brand-aliases-action-filter",
                                options=[
                                    {"label": "Todas", "value": ""},
                                    {"label": "🔄 Alias (fusionar)", "value": "alias"},
                                    {"label": "🚫 Excluir (no es marca)", "value": "exclude"},
                                ],
                                value="",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Label(" ", className="small d-block"),
                            dbc.Checklist(
                                id="brand-aliases-show-inactive",
                                options=[{"label": "Mostrar inactivos", "value": "show_inactive"}],
                                value=[],
                                switch=True,
                            ),
                        ],
                        width=12,
                        md=2,
                    ),
                    dbc.Col(
                        [
                            dbc.Label(" ", className="small d-block"),
                            html.Span(
                                id="brand-aliases-last-update",
                                className="text-muted small",
                            ),
                        ],
                        width=12,
                        md=3,
                        className="text-end pt-2",
                    ),
                ],
                className="mb-3",
            ),

            # Stats badges
            html.Div(id="brand-aliases-stats-container", className="mb-3"),

            # Table
            html.Div(id="brand-aliases-table-container"),

            # Pagination
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Span(
                                id="brand-aliases-pagination-info",
                                className="text-muted small",
                            ),
                        ],
                        width=6,
                    ),
                    dbc.Col(
                        [
                            dbc.ButtonGroup(
                                [
                                    dbc.Button(
                                        html.I(className="fas fa-chevron-left"),
                                        id="brand-aliases-prev-page",
                                        size="sm",
                                        outline=True,
                                        color="secondary",
                                    ),
                                    dbc.Button(
                                        html.I(className="fas fa-chevron-right"),
                                        id="brand-aliases-next-page",
                                        size="sm",
                                        outline=True,
                                        color="secondary",
                                    ),
                                ],
                                className="float-end",
                            ),
                        ],
                        width=6,
                        className="text-end",
                    ),
                ],
                className="mt-3",
            ),

            # Modal para añadir/editar brand alias
            _create_brand_alias_modal(),

            # Store para paginación y edición
            dcc.Store(id="brand-aliases-current-page", data=0),
            dcc.Store(id="brand-aliases-edit-id", data=None),
        ],
        fluid=True,
        className="py-3",
    )


def _create_brand_alias_modal() -> dbc.Modal:
    """Modal para crear/editar brand alias."""
    return dbc.Modal(
        [
            dbc.ModalHeader(
                dbc.ModalTitle(id="brand-aliases-modal-title", children="Añadir Brand Alias"),
                close_button=True,
            ),
            dbc.ModalBody(
                [
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Marca Detectada (origen) *"),
                                    dbc.Input(
                                        id="brand-aliases-form-source",
                                        type="text",
                                        placeholder="ej: oralkin",
                                    ),
                                    dbc.FormText("La marca que detecta BrandDetectionService"),
                                ],
                                width=12,
                                className="mb-3",
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Acción *"),
                                    dbc.Select(
                                        id="brand-aliases-form-action",
                                        options=[
                                            {"label": "🔄 Alias (fusionar con otra marca)", "value": "alias"},
                                            {"label": "🚫 Excluir (no es una marca)", "value": "exclude"},
                                        ],
                                        value="alias",
                                    ),
                                    dbc.FormText("Alias fusiona, Excluir elimina de detección"),
                                ],
                                width=12,
                                className="mb-3",
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Marca Destino (target) *"),
                                    dbc.Input(
                                        id="brand-aliases-form-target",
                                        type="text",
                                        placeholder="ej: kin",
                                    ),
                                    dbc.FormText("La marca correcta a usar"),
                                ],
                                id="brand-aliases-target-container",
                                width=12,
                                className="mb-3",
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Razón / Notas"),
                                    dbc.Textarea(
                                        id="brand-aliases-form-reason",
                                        placeholder="Motivo del alias...",
                                        rows=2,
                                    ),
                                ],
                                width=12,
                                className="mb-3",
                            ),
                        ]
                    ),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        "Cancelar",
                        id="brand-aliases-modal-cancel-btn",
                        color="secondary",
                        outline=True,
                    ),
                    dbc.Button(
                        "Guardar",
                        id="brand-aliases-modal-save-btn",
                        color="primary",
                    ),
                ]
            ),
        ],
        id="brand-aliases-modal",
        is_open=False,
        centered=True,
    )


def create_curated_groups_panel() -> html.Div:
    """
    Panel de gestión de Grupos Intercambiables Curados (Issue #521).

    Los grupos curados son grupos de productos intercambiables creados
    y mantenidos manualmente por el administrador. Sustituyen al clustering
    dinámico (ADR-004).

    Funcionalidades:
    - CRUD de grupos (crear, editar, eliminar, toggle activo)
    - Gestión de miembros (añadir/eliminar productos por EAN/CN)
    - Filtros por L1, estado, origen (legacy/manual)
    - Estadísticas de productos y marcas
    """
    return dbc.Container(
        [
            # Header
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.H5(
                                [
                                    html.I(className="fas fa-layer-group me-2 text-primary"),
                                    "Grupos Intercambiables Curados",
                                ],
                                className="mb-2",
                            ),
                            html.P(
                                [
                                    "Gestiona grupos de productos intercambiables curados manualmente. ",
                                    html.Strong("Sustituye al clustering dinámico."),
                                ],
                                className="text-muted mb-0",
                            ),
                        ],
                        width=12,
                        lg=8,
                    ),
                    dbc.Col(
                        [
                            dbc.ButtonGroup(
                                [
                                    BaseButton(
                                        children=html.Span([html.I(className="fas fa-plus me-2"), "Nuevo Grupo"]),
                                        id="curated-groups-add-btn",
                                        variant="primary",
                                        size="sm",
                                    ),
                                    BaseButton(
                                        children=html.I(className="fas fa-sync-alt"),
                                        id="curated-groups-refresh-btn",
                                        variant="secondary",
                                        outline=True,
                                        size="sm",
                                    ),
                                ],
                                className="float-end",
                            ),
                        ],
                        width=12,
                        lg=4,
                        className="text-end",
                    ),
                ],
                className="mb-4",
            ),

            # Stats badges
            html.Div(id="curated-groups-stats-container", className="mb-3"),

            # Filtros
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Label("Buscar", className="small"),
                            dbc.Input(
                                id="curated-groups-search-input",
                                type="text",
                                placeholder="Buscar por nombre o descripción...",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=4,
                    ),
                    dbc.Col(
                        [
                            dbc.Label("Categoría L1", className="small"),
                            dbc.Select(
                                id="curated-groups-l1-filter",
                                options=[{"label": "Todas", "value": ""}],
                                value="",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Label("Origen", className="small"),
                            dbc.Select(
                                id="curated-groups-source-filter",
                                options=[
                                    {"label": "Todos", "value": ""},
                                    {"label": "Manual", "value": "manual_curated"},
                                    {"label": "Legacy", "value": "legacy_clustering"},
                                ],
                                value="",
                                size="sm",
                            ),
                        ],
                        width=12,
                        md=2,
                    ),
                    dbc.Col(
                        [
                            dbc.Label(" ", className="small d-block"),
                            dbc.Checklist(
                                id="curated-groups-show-inactive",
                                options=[{"label": "Mostrar inactivos", "value": "show_inactive"}],
                                value=[],
                                switch=True,
                            ),
                        ],
                        width=12,
                        md=3,
                    ),
                ],
                className="mb-3",
            ),

            # Table de grupos
            html.Div(id="curated-groups-table-container"),

            # Pagination
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Span(
                                id="curated-groups-pagination-info",
                                className="text-muted small",
                            ),
                        ],
                        width=6,
                    ),
                    dbc.Col(
                        [
                            dbc.ButtonGroup(
                                [
                                    dbc.Button(
                                        html.I(className="fas fa-chevron-left"),
                                        id="curated-groups-prev-page",
                                        size="sm",
                                        outline=True,
                                        color="secondary",
                                    ),
                                    dbc.Button(
                                        html.I(className="fas fa-chevron-right"),
                                        id="curated-groups-next-page",
                                        size="sm",
                                        outline=True,
                                        color="secondary",
                                    ),
                                ],
                                className="float-end",
                            ),
                        ],
                        width=6,
                        className="text-end",
                    ),
                ],
                className="mt-3",
            ),

            # Modal para crear/editar grupo
            _create_curated_group_modal(),

            # Modal para gestionar miembros
            _create_curated_group_members_modal(),

            # Store para estado de paginación y grupo seleccionado
            dcc.Store(id="curated-groups-pagination-store", data={"page": 1, "page_size": 20}),
            dcc.Store(id="curated-groups-selected-group-store", data=None),

            # Confirmation dialog for delete
            dcc.ConfirmDialog(
                id="curated-groups-delete-confirm",
                message="¿Estás seguro de eliminar este grupo? Esta acción eliminará también todos los productos del grupo.",
            ),
            dcc.Store(id="curated-groups-delete-group-id", data=None),
        ],
        fluid=True,
        className="py-3",
    )


def _create_curated_group_modal() -> dbc.Modal:
    """Modal para crear/editar grupo curado."""
    return dbc.Modal(
        [
            dbc.ModalHeader(
                dbc.ModalTitle(id="curated-groups-modal-title", children="Nuevo Grupo"),
                close_button=True,
            ),
            dbc.ModalBody(
                [
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Label("Nombre del Grupo *"),
                                    dbc.Input(
                                        id="curated-groups-form-name",
                                        type="text",
                                        placeholder="ej: Ibuprofeno 600mg",
                                        maxLength=100,
                                    ),
                                    dbc.FormText("Nombre descriptivo del grupo intercambiable"),
                                ],
                                width=12,
                                className="mb-3",
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Descripción"),
                                    dbc.Textarea(
                                        id="curated-groups-form-description",
                                        placeholder="Descripción detallada del grupo...",
                                        rows=2,
                                        maxLength=500,
                                    ),
                                ],
                                width=12,
                                className="mb-3",
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Categoría L1"),
                                    dbc.Select(
                                        id="curated-groups-form-l1",
                                        options=[{"label": "Sin asignar", "value": ""}],
                                        value="",
                                    ),
                                ],
                                width=12,
                                md=6,
                                className="mb-3",
                            ),
                            dbc.Col(
                                [
                                    dbc.Label("Subcategoría L2"),
                                    dbc.Input(
                                        id="curated-groups-form-l2",
                                        type="text",
                                        placeholder="ej: analgesicos",
                                        maxLength=50,
                                    ),
                                ],
                                width=12,
                                md=6,
                                className="mb-3",
                            ),
                        ]
                    ),
                    # Hidden store for edit mode
                    dcc.Store(id="curated-groups-edit-id", data=None),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        "Cancelar",
                        id="curated-groups-modal-cancel-btn",
                        color="secondary",
                        outline=True,
                    ),
                    dbc.Button(
                        "Guardar",
                        id="curated-groups-modal-save-btn",
                        color="primary",
                    ),
                ]
            ),
        ],
        id="curated-groups-modal",
        is_open=False,
        centered=True,
        size="lg",
    )


def _create_curated_group_members_modal() -> dbc.Modal:
    """Modal para gestionar miembros de un grupo curado."""
    return dbc.Modal(
        [
            dbc.ModalHeader(
                dbc.ModalTitle(id="curated-members-modal-title", children="Miembros del Grupo"),
                close_button=True,
            ),
            dbc.ModalBody(
                [
                    # Info del grupo
                    html.Div(id="curated-members-group-info", className="mb-3"),

                    # Formulario añadir miembro
                    dbc.Card(
                        dbc.CardBody(
                            [
                                html.H6("Añadir Producto", className="mb-3"),
                                dbc.Row(
                                    [
                                        dbc.Col(
                                            [
                                                dbc.Label("EAN13", className="small"),
                                                dbc.Input(
                                                    id="curated-members-form-ean",
                                                    type="text",
                                                    placeholder="8437000000000",
                                                    maxLength=13,
                                                    size="sm",
                                                ),
                                            ],
                                            width=12,
                                            md=3,
                                        ),
                                        dbc.Col(
                                            [
                                                dbc.Label("Cód. Nacional", className="small"),
                                                dbc.Input(
                                                    id="curated-members-form-cn",
                                                    type="text",
                                                    placeholder="123456",
                                                    maxLength=20,
                                                    size="sm",
                                                ),
                                            ],
                                            width=12,
                                            md=3,
                                        ),
                                        dbc.Col(
                                            [
                                                dbc.Label("Nombre (opcional)", className="small"),
                                                dbc.Input(
                                                    id="curated-members-form-name",
                                                    type="text",
                                                    placeholder="Nombre producto",
                                                    maxLength=255,
                                                    size="sm",
                                                ),
                                            ],
                                            width=12,
                                            md=4,
                                        ),
                                        dbc.Col(
                                            [
                                                dbc.Label(" ", className="small d-block"),
                                                dbc.Button(
                                                    html.Span([html.I(className="fas fa-plus me-1"), "Añadir"]),
                                                    id="curated-members-add-btn",
                                                    color="primary",
                                                    size="sm",
                                                ),
                                            ],
                                            width=12,
                                            md=2,
                                        ),
                                    ]
                                ),
                            ]
                        ),
                        className="mb-3",
                    ),

                    # Lista de miembros
                    html.Div(id="curated-members-list-container"),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        "Cerrar",
                        id="curated-members-modal-close-btn",
                        color="secondary",
                    ),
                ]
            ),
        ],
        id="curated-members-modal",
        is_open=False,
        centered=True,
        size="xl",
    )


# Los callbacks ahora están modularizados en frontend/callbacks/admin/
# Ver callbacks/admin/users.py para callbacks de gestión de usuarios (Issue #348 FASE 3)
# Ver callbacks/admin/catalog_management.py para callbacks de sync CIMA/Nomenclator
# Ver callbacks/admin/admin_tabs.py para callbacks de navegación de tabs (Issue #485)
# Ver callbacks/admin/duplicates.py para callbacks de duplicados VentaLibre (Issue #477)
# Ver callbacks/admin/aliases.py para callbacks de category aliases (Issue #459)
# Ver callbacks/admin/curated_groups.py para callbacks de grupos curados (Issue #521)
