# DASH DETAILED PATTERNS - xFarma Frontend
## Patrones Avanzados y Ejemplos Completos

> **Propósito**: Documentación exhaustiva de patrones Dash, auth management, timing issues y callbacks avanzados.
>
> **Para referencia rápida**: Ver `frontend/CLAUDE.md`

---

## Índice

- [Skeleton Pattern (REGLA #0.5)](#skeleton-pattern-regla-05)
- [Timing Issues & Guards (Issue #330)](#timing-issues--guards-issue-330)
- [Callback Patterns Avanzados](#callback-patterns-avanzados)
- [Auth Patterns Frontend](#auth-patterns-frontend)
- [Performance Optimization](#performance-optimization)

---

## Skeleton Pattern (REGLA #0.5)

### Propósito

Estructura de layout obligatoria para páginas multi-página en Dash que previene timing issues y mejora UX con estados de carga.

### Problema que Resuelve

En apps Dash multi-página con routing manual:
- Callbacks se registran globalmente al inicio
- Validación de callbacks ocurre antes de renderizar páginas
- Dash no sabe qué componentes existirán en qué pathname
- **Resultado**: "A nonexistent object was used in an Output" errors

### Arquitectura del Skeleton

```
┌─────────────────────────────────────────┐
│         app.py (Skeleton Global)        │
│  ┌───────────────────────────────────┐  │
│  │  dcc.Location(id="url")           │  │ ← Routing
│  │  html.Div(id="page-content")      │  │ ← Container dinámico
│  │  dcc.Store(id="auth-state")       │  │ ← 15+ stores globales
│  └───────────────────────────────────┘  │
└─────────────────────────────────────────┘
                    │
                    ▼
         ┌──────────────────────┐
         │ Callback routing     │
         │ Input: url.pathname  │
         └──────────────────────┘
                    │
      ┌─────────────┼─────────────┐
      ▼             ▼             ▼
┌─────────┐  ┌──────────┐  ┌──────────┐
│ /home   │  │ /upload  │  │ /generic │
└─────────┘  └──────────┘  └──────────┘
```

### Implementación Paso a Paso

#### Paso 1: Crear Skeleton Global (app.py)

```python
# app.py
import dash
from dash import dcc, html
import dash_bootstrap_components as dbc

app = dash.Dash(
    __name__,
    external_stylesheets=[dbc.themes.BOOTSTRAP],
    suppress_callback_exceptions=True,  # ← CRÍTICO para multi-página
    title="xFarma - Gestión Farmacéutica"
)

app.layout = html.Div([
    # 1️⃣ Routing
    dcc.Location(id='url', refresh=False),

    # 2️⃣ Global Stores (siempre presentes)
    dcc.Store(id='auth-state', storage_type='memory', data={"authenticated": False}),
    dcc.Store(id='auth-tokens-store', storage_type='local'),  # Tokens encriptados
    dcc.Store(id='partners-selection-store', storage_type='session'),
    dcc.Store(id='laboratory-cache-store', storage_type='local'),
    dcc.Store(id='upload-state-store', storage_type='session'),
    # ... 10+ stores más

    # 3️⃣ Container Dinámico (se reemplaza según pathname)
    html.Div(id='page-content')
])

server = app.server  # Para Gunicorn
```

**Por qué funciona**:
- ✅ Todos los stores están SIEMPRE en el DOM
- ✅ Callbacks pueden referenciar stores sin importar pathname
- ✅ `suppress_callback_exceptions=True` permite componentes dinámicos

#### Paso 2: Crear Layout de Página con Skeleton Local

```python
# layouts/upload.py
import dash_bootstrap_components as dbc
from dash import html, dcc

def create_layout():
    """
    Layout de página de upload con skeleton local.

    REGLA #0.5: SIEMPRE incluir estructura base completa.
    """
    return dbc.Container([
        # 🏗️ SKELETON LOCAL: Estructura siempre presente
        html.H1("Subir Archivo ERP", className="mb-4"),

        html.Div(id='upload-instructions', children=[
            html.P("Selecciona archivo CSV de tu ERP farmacéutico"),
            html.Ul([
                html.Li("Formatos soportados: Farmatic, Farmanager, Nixfarma, Unycop, IOFWin"),
                html.Li("Tamaño máximo: 10 MB"),
                html.Li("Encoding: Latin-1 o UTF-8")
            ])
        ]),

        # Container de upload (crítico: siempre presente)
        html.Div(id='upload-container', children=[
            dcc.Upload(
                id='upload-data',
                children=html.Div([
                    '📁 Arrastra archivo o ',
                    html.A('Selecciona')
                ]),
                className='upload-area'
            )
        ]),

        # Progress container (inicialmente vacío, pero DOM existe)
        html.Div(id='upload-progress-container', children=[]),

        # Results container (inicialmente vacío, pero DOM existe)
        html.Div(id='upload-results-container', children=[])

    ], fluid=True)
```

**Skeleton Local - Elementos Obligatorios**:
1. ✅ Título de página (contexto para usuario)
2. ✅ Instrucciones (siempre visibles)
3. ✅ Containers de componentes dinámicos (vacíos pero existentes)
4. ✅ IDs consistentes (matching con callbacks)

#### Paso 3: Routing Callback

```python
# app.py (continuación)
from dash import Input, Output
from layouts import upload, dashboard, generics, auth, homepage

@app.callback(
    Output('page-content', 'children'),
    Input('url', 'pathname')
)
def display_page(pathname):
    """
    Routing central: decide qué layout renderizar.

    IMPORTANTE: Los callbacks de las páginas están registrados ANTES
    de que este callback se ejecute.
    """
    if pathname == '/home':
        return homepage.create_layout()
    elif pathname == '/upload':
        return upload.create_layout()
    elif pathname == '/dashboard':
        return dashboard.create_layout()
    elif pathname == '/generics':
        return generics.create_layout()
    elif pathname in ['/login', '/register', '/oauth-callback']:
        return auth.create_layout(pathname)
    else:
        # Default: redirigir a login
        return auth.create_layout('/login')
```

#### Paso 4: Callback con Guards

```python
# callbacks/upload.py
from dash import callback, Input, Output, State, ctx
from dash.exceptions import PreventUpdate
import logging

logger = logging.getLogger(__name__)

@callback(
    Output('upload-progress-container', 'children'),
    Input('upload-data', 'contents'),
    [
        State('upload-data', 'filename'),
        State('auth-state', 'data')
    ],
    prevent_initial_call=True  # ← CRÍTICO: no ejecutar en page load
)
def handle_upload_progress(contents, filename, auth_state):
    """
    Callback con guards completos para prevenir timing issues.
    """
    # 🛡️ GUARD #1: Verificar trigger
    if not ctx.triggered or ctx.triggered[0]['value'] is None:
        logger.debug("[handle_upload_progress] No trigger - PreventUpdate")
        raise PreventUpdate

    # 🛡️ GUARD #2: Verificar pathname (solo ejecutar en /upload)
    # Nota: pathname no está en Inputs/States, pero podemos inferir
    # desde la existencia de 'upload-data' (solo visible en /upload)

    # 🛡️ GUARD #3: Verificar autenticación
    if not auth_state or not auth_state.get('authenticated'):
        logger.warning("[handle_upload_progress] User not authenticated")
        raise PreventUpdate

    # 🛡️ GUARD #4: Verificar datos
    if not contents or not filename:
        logger.warning("[handle_upload_progress] Missing contents or filename")
        raise PreventUpdate

    # ✅ Todos los guards pasaron - procesar upload
    logger.info(f"[handle_upload_progress] Processing {filename}")

    # Lógica de upload...
    return html.Div([
        dbc.Progress(value=50, striped=True, animated=True),
        html.P(f"Procesando {filename}...")
    ])
```

### Verificación del Skeleton

#### Checklist de Implementación

- [ ] `suppress_callback_exceptions=True` en app.py
- [ ] Todos los dcc.Store en skeleton global (app.py)
- [ ] Layouts de páginas retornan estructura completa (no fragmentos)
- [ ] Todos los IDs de Output/Input existen en DOM (aunque vacíos)
- [ ] Callbacks tienen `prevent_initial_call=True`
- [ ] Guards validan trigger, pathname, auth, datos

#### Testing del Skeleton

```python
# tests/unit/test_skeleton_pattern.py
def test_global_stores_exist(dash_duo):
    """Verificar que stores globales existen en DOM"""
    from app import app

    dash_duo.start_server(app)

    # Verificar existencia de stores
    assert dash_duo.find_element('#auth-state')
    assert dash_duo.find_element('#auth-tokens-store')
    assert dash_duo.find_element('#partners-selection-store')
    # ... verificar 15+ stores

def test_page_content_container_exists(dash_duo):
    """Verificar que container dinámico existe"""
    from app import app

    dash_duo.start_server(app)
    assert dash_duo.find_element('#page-content')

def test_upload_containers_exist_on_page(dash_duo):
    """Verificar que containers de upload existen cuando se carga la página"""
    from app import app

    dash_duo.start_server(app)
    dash_duo.wait_for_page(url=f"{dash_duo.server_url}/upload", timeout=4)

    # Verificar skeleton local
    assert dash_duo.find_element('#upload-container')
    assert dash_duo.find_element('#upload-progress-container')
    assert dash_duo.find_element('#upload-results-container')
```

### Troubleshooting

#### Error: "A nonexistent object was used in an Output"

**Causa**: Callback referencia componente que no está en DOM actual.

**Solución**:
1. ✅ Agregar componente al skeleton global (si es store) o skeleton local
2. ✅ Verificar que layout de página retorna estructura completa
3. ✅ Usar `prevent_initial_call=True` en callback
4. ✅ Agregar guards de validación

#### Error: Callbacks no se ejecutan después de navegación

**Causa**: Guards demasiado restrictivos o falta trigger.

**Solución**:
```python
# ❌ INCORRECTO: Guard sin logging
if not data:
    raise PreventUpdate

# ✅ CORRECTO: Guard con logging
if not data:
    logger.debug(f"[{callback_name}] Missing data - PreventUpdate")
    raise PreventUpdate
```

Logging ayuda a debuggear por qué callbacks no se ejecutan.

---

## Timing Issues & Guards (Issue #330)

### Contexto del Problema

Apps Dash multi-página con routing manual tienen un **timing challenge fundamental**:

```
Timeline de Ejecución:
T0: app.py importa layouts y callbacks
T1: Callbacks se registran globalmente
T2: Dash valida todos los callbacks (busca Inputs/Outputs en DOM)
T3: Usuario navega a /upload
T4: Layout de /upload se renderiza
T5: Callback se ejecuta (datos ahora disponibles)
```

**Problema**: Validación en T2 ocurre ANTES de renderizar layout en T4.

### Estrategias Anti-Timing

#### 1. suppress_callback_exceptions=True (Nivel App)

```python
app = dash.Dash(
    __name__,
    suppress_callback_exceptions=True  # ← Desactiva validación estricta
)
```

**Qué hace**:
- ✅ Permite callbacks con Inputs/Outputs que no existen aún
- ✅ Confía en que componentes existirán cuando callback se ejecute
- ⚠️ Requiere guards para prevenir errores runtime

**Cuándo usar**: Siempre en apps multi-página con routing manual.

#### 2. prevent_initial_call=True (Nivel Callback)

```python
@callback(
    Output('result', 'children'),
    Input('button', 'n_clicks'),
    prevent_initial_call=True  # ← No ejecutar en page load
)
def my_callback(n_clicks):
    return f"Clicked {n_clicks} times"
```

**Qué hace**:
- ✅ Callback NO se ejecuta cuando componentes se renderizan
- ✅ Solo se ejecuta cuando Input cambia después de page load
- ✅ Previene errores de "None is not valid" en inputs iniciales

**Cuándo usar**: Casi siempre (excepto callbacks de inicialización).

#### 3. Guard Pattern (Nivel Callback)

```python
@callback(...)
def my_callback(...):
    # GUARD #1: Verificar trigger
    if not ctx.triggered:
        raise PreventUpdate

    # GUARD #2: Verificar datos
    if data is None or data == []:
        raise PreventUpdate

    # GUARD #3: Verificar contexto (pathname, auth, etc.)
    if not is_correct_context():
        raise PreventUpdate

    # ✅ Guards pasados - ejecutar lógica
    return process_data(data)
```

**Tipos de Guards**:

| Guard | Qué Valida | Cuándo Usar |
|-------|------------|-------------|
| **Trigger** | `ctx.triggered` no vacío | Siempre (primero) |
| **Datos** | Inputs no son None/vacíos | Cuando datos son requeridos |
| **Auth** | Usuario autenticado | Callbacks que llaman APIs |
| **Pathname** | Pathname correcto | Callbacks específicos a página |
| **Store** | Store tiene datos | Callbacks que dependen de stores |

#### 4. dcc.Store Global (Nivel Arquitectura)

```python
# app.py - Stores SIEMPRE presentes
dcc.Store(id='auth-state', storage_type='memory'),
dcc.Store(id='upload-state', storage_type='session'),
dcc.Store(id='cache', storage_type='local')
```

**Ventajas**:
- ✅ Stores existen independiente de pathname
- ✅ Callbacks pueden leer/escribir stores sin timing issues
- ✅ Estado persiste entre navegaciones (según storage_type)

**Storage Types**:
- `memory`: Se pierde con F5, no funciona en multi-worker
- `session`: Persiste con F5 en misma pestaña
- `local`: Persiste entre sesiones, límite ~5-10MB

#### 5. dcc.Loading (Nivel UX)

```python
dcc.Loading(
    id="loading-upload",
    type="default",  # "graph", "cube", "circle", "dot", "default"
    children=[
        html.Div(id='upload-results')
    ]
)
```

**Qué hace**:
- ✅ Muestra spinner mientras callback ejecuta
- ✅ Mejora UX durante operaciones lentas
- ✅ Previene múltiples clicks (desactiva inputs)

**Cuándo usar**: Callbacks con operaciones >1 segundo (API calls, uploads, etc.).

### Ejemplo Completo: Callback con Todos los Guards

```python
from dash import callback, Input, Output, State, ctx, no_update
from dash.exceptions import PreventUpdate
from utils.auth_helpers import is_user_authenticated
from utils.request_coordinator import request_coordinator
import logging

logger = logging.getLogger(__name__)

@callback(
    [
        Output('generic-analysis-results', 'children'),
        Output('generic-analysis-loading', 'data')
    ],
    Input('analyze-generics-btn', 'n_clicks'),
    [
        State('partners-selection-store', 'data'),
        State('auth-state', 'data'),
        State('date-range', 'start_date'),
        State('date-range', 'end_date')
    ],
    prevent_initial_call=True  # ← GUARD Nivel 0: No ejecutar en page load
)
def perform_generic_analysis(
    n_clicks,
    partners_store,
    auth_state,
    start_date,
    end_date
):
    """
    Análisis de oportunidades de genéricos con guards completos.

    Guards implementados:
    - Trigger validation
    - Auth validation
    - Data validation
    - Store validation
    - BD-first pattern (REGLA #16)
    """
    # 🛡️ GUARD #1: Verificar trigger
    if not ctx.triggered:
        logger.debug("[perform_generic_analysis] No trigger - PreventUpdate")
        raise PreventUpdate

    triggered_id = ctx.triggered[0]['prop_id'].split('.')[0]

    if triggered_id != 'analyze-generics-btn':
        logger.debug(f"[perform_generic_analysis] Wrong trigger: {triggered_id}")
        raise PreventUpdate

    # 🛡️ GUARD #2: Verificar n_clicks (button pressed)
    if not n_clicks or n_clicks == 0:
        logger.debug("[perform_generic_analysis] No clicks - PreventUpdate")
        raise PreventUpdate

    # 🛡️ GUARD #3: Verificar autenticación
    if not is_user_authenticated(auth_state):
        logger.warning("[perform_generic_analysis] User not authenticated")
        return (
            html.Div("Debes iniciar sesión para usar esta función", className="alert alert-warning"),
            {"loading": False}
        )

    # 🛡️ GUARD #4: Verificar fechas
    if not start_date or not end_date:
        logger.warning("[perform_generic_analysis] Missing date range")
        return (
            html.Div("Selecciona rango de fechas", className="alert alert-warning"),
            {"loading": False}
        )

    # 🛡️ GUARD #5: BD-first pattern (REGLA #16 - NO confiar en store)
    logger.info("[perform_generic_analysis] Reading partners from BD (BD-first pattern)")
    selected_partners, success = get_selected_partners_from_db(auth_state)

    if not success:
        logger.error("[perform_generic_analysis] Failed to fetch partners from DB")
        return (
            html.Div("Error cargando laboratorios. Intenta de nuevo.", className="alert alert-danger"),
            {"loading": False}
        )

    if not selected_partners or len(selected_partners) == 0:
        logger.warning("[perform_generic_analysis] No partners selected in DB")
        return (
            html.Div("Selecciona al menos un laboratorio en Configuración", className="alert alert-warning"),
            {"loading": False}
        )

    # ✅ TODOS LOS GUARDS PASARON - Ejecutar análisis
    logger.info(f"[perform_generic_analysis] Starting analysis with {len(selected_partners)} partners")

    try:
        # API call con partners desde BD (no desde store)
        response = request_coordinator.make_request(
            method="POST",
            endpoint="/api/v1/generic-opportunities",
            json_data={
                "partner_ids": selected_partners,
                "start_date": start_date,
                "end_date": end_date
            }
        )

        if response.status_code != 200:
            logger.error(f"[perform_generic_analysis] API error: {response.status_code}")
            return (
                html.Div(f"Error del servidor: {response.status_code}", className="alert alert-danger"),
                {"loading": False}
            )

        data = response.json()

        # Renderizar resultados
        results_component = render_generic_analysis_results(data)

        logger.info("[perform_generic_analysis] Analysis completed successfully")
        return (results_component, {"loading": False})

    except Exception as e:
        logger.exception("[perform_generic_analysis] Exception during analysis")
        return (
            html.Div(f"Error inesperado: {str(e)}", className="alert alert-danger"),
            {"loading": False}
        )

# Helper para BD-first pattern
def get_selected_partners_from_db(auth_state):
    """
    Obtiene partners seleccionados desde BD (fuente de verdad única).

    Returns:
        tuple: (selected_partners: list[str], success: bool)
    """
    if not auth_state or 'user' not in auth_state:
        return [], False

    pharmacy_id = auth_state['user'].get('pharmacy_id')

    if not pharmacy_id:
        return [], False

    try:
        response = request_coordinator.make_request(
            method="GET",
            endpoint=f"/api/v1/pharmacy-partners/{pharmacy_id}"
        )

        if response.status_code == 200:
            data = response.json()
            return data.get('selected_partners', []), True
        else:
            return [], False

    except Exception as e:
        logger.exception(f"[get_selected_partners_from_db] Exception: {str(e)}")
        return [], False
```

### Debugging Timing Issues

#### Logging Estructurado

```python
# utils/logging_config.py
import logging

def setup_callback_logging():
    """Configurar logging para callbacks Dash"""
    logging.basicConfig(
        level=logging.DEBUG,
        format='%(asctime)s [%(levelname)s] %(name)s: %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S'
    )

    # Logger específico para callbacks
    callback_logger = logging.getLogger('callbacks')
    callback_logger.setLevel(logging.DEBUG)

    return callback_logger
```

**Usar en callbacks**:
```python
logger = logging.getLogger('callbacks.upload')

@callback(...)
def my_callback(...):
    logger.debug(f"[my_callback] Started - trigger: {ctx.triggered_id}")
    logger.debug(f"[my_callback] Inputs: {input1}, {input2}")
    logger.debug(f"[my_callback] States: auth={auth_state}")

    # ... lógica

    logger.info(f"[my_callback] Completed successfully")
```

#### Browser Console

Errores de timing aparecen en browser console (F12):

```
Warning: A nonexistent object was used in an Output of a callback.
The id of this object is "upload-progress-container" and the property is "children".
```

**Cómo interpretar**:
- ⚠️ Es un warning, NO un error crítico
- 🔍 Indica que callback se registró antes de que componente existiera
- ✅ Si funcionalidad sigue funcionando, es transitorio (OK)
- ❌ Si funcionalidad NO funciona, revisar skeleton pattern

---

## Callback Patterns Avanzados

### Cascading Callbacks

**Problema**: Callback B depende del resultado de Callback A.

**Solución**: Chain de callbacks usando stores intermedios.

```python
# Callback A: Fetch data
@callback(
    Output('data-store', 'data'),
    Input('fetch-btn', 'n_clicks')
)
def fetch_data(n_clicks):
    if not n_clicks:
        raise PreventUpdate

    data = api_call()  # Fetch data
    return data

# Callback B: Process data (se dispara cuando data-store cambia)
@callback(
    Output('processed-store', 'data'),
    Input('data-store', 'data')
)
def process_data(data):
    if not data:
        raise PreventUpdate

    processed = process(data)
    return processed

# Callback C: Display results (se dispara cuando processed-store cambia)
@callback(
    Output('results', 'children'),
    Input('processed-store', 'data')
)
def display_results(processed):
    if not processed:
        return "No data"

    return render_results(processed)
```

**Flujo**:
```
fetch-btn.n_clicks (change)
  ↓
fetch_data() → data-store.data (change)
  ↓
process_data() → processed-store.data (change)
  ↓
display_results() → results.children (update)
```

### Store Patterns Complejos

#### Pattern 1: Cache Persistente con TTL

```python
import time

@callback(
    Output('cache-store', 'data', allow_duplicate=True),
    Input('initialize-cache-btn', 'n_clicks'),
    State('cache-store', 'data'),
    prevent_initial_call=True
)
def initialize_cache(n_clicks, cache_data):
    """Inicializar cache con TTL de 7 días"""
    if not n_clicks:
        raise PreventUpdate

    # Verificar si cache está fresco
    if cache_data:
        cache_timestamp = cache_data.get('timestamp', 0)
        cache_age_days = (time.time() - cache_timestamp) / 86400

        if cache_age_days < 7:
            logger.info(f"Cache fresco ({cache_age_days:.1f} días) - skip")
            raise PreventUpdate

    # Cache stale o vacío - regenerar
    logger.info("Regenerando cache (TTL expirado o vacío)")
    data = fetch_from_api()

    return {
        'data': data,
        'timestamp': time.time()
    }
```

#### Pattern 2: Store Compositional (Multiple Sources)

```python
@callback(
    Output('unified-store', 'data'),
    [
        Input('source1-store', 'data'),
        Input('source2-store', 'data'),
        Input('source3-store', 'data')
    ]
)
def merge_stores(source1, source2, source3):
    """Merge múltiples stores en uno unificado"""
    if not all([source1, source2, source3]):
        raise PreventUpdate

    return {
        'partners': source1.get('selected', []),
        'dates': source2.get('range', {}),
        'filters': source3.get('active', [])
    }
```

---

## Auth Patterns Frontend

### Token Management (CRÍTICO)

**REGLA**: NUNCA leer `auth-tokens-store` directamente (tokens encriptados).

#### Arquitectura de Autenticación

```
┌──────────────────────────────────────────────────────────┐
│                    Backend API                           │
│  /api/v1/auth/login → LoginResponse (tokens + user)     │
└──────────────────────┬───────────────────────────────────┘
                       │
                       ▼
┌──────────────────────────────────────────────────────────┐
│              Frontend Auth Manager                       │
│  ┌────────────────────────────────────────────────────┐  │
│  │ 1. Encriptar tokens (Fernet)                       │  │
│  │ 2. Guardar en localStorage (auth-tokens-store)     │  │
│  │ 3. Actualizar auth_manager singleton               │  │
│  └────────────────────────────────────────────────────┘  │
└──────────────────────┬───────────────────────────────────┘
                       │
        ┌──────────────┼──────────────┐
        ▼              ▼              ▼
┌─────────────┐ ┌─────────────┐ ┌─────────────┐
│ Callback A  │ │ Callback B  │ │ Callback C  │
│ auth_manager│ │ auth_manager│ │ auth_manager│
│.get_token() │ │.get_token() │ │.get_token() │
└─────────────┘ └─────────────┘ └─────────────┘
```

#### Flujo de Login (Issue #398)

```python
# 1️⃣ Usuario hace login
POST /api/v1/auth/login
{
    "email": "user@pharmacy.com",
    "password": "password123"
}

# 2️⃣ Backend retorna tokens + user info
{
    "user": {
        "id": "uuid",
        "email": "user@pharmacy.com",
        "role": "admin",
        "pharmacy_id": "uuid"
    },
    "access_token": "jwt_token",
    "refresh_token": "jwt_refresh",
    "token_type": "bearer"
}

# 3️⃣ Frontend: callbacks/auth_callbacks.py → handle_login()
def handle_login(n_clicks, email, password, remember_me):
    result = auth_manager.login(email, password)  # Llama al backend

    if result["success"]:
        user_info = result["data"].get("user")  # ← Ya viene del backend (0 HTTP extra)

        # Guardar tokens encriptados + user info en localStorage
        tokens_data = {
            "tokens": encrypted_tokens,  # Encriptados con Fernet
            "storage_type": "local",
            "user": user_info  # ← Cacheado para próxima sesión
        }

        # Auth state inmediato (sin race condition)
        auth_state = {"authenticated": True, "user": user_info}

        return dcc.Location(pathname="/home"), tokens_data, auth_state, ""
```

**Beneficios (Issue #398)**:
- ✅ 50% reducción en API calls (2→1 por login)
- ✅ Eliminación de race condition (100-500ms)
- ✅ Auth state inmediato (sin flash de "not authenticated")

#### Uso de auth_manager en Callbacks

```python
# ❌ ANTI-PATRÓN: Leer auth-tokens-store directamente
@callback(
    Output('result', 'children'),
    State('auth-tokens-store', 'data')
)
def my_callback(tokens_data):
    # ❌ Esta clave NO existe (tokens están encriptados)
    token = tokens_data["access_token"]  # KeyError

    headers = {"Authorization": f"Bearer {token}"}
    response = requests.get(f"{API_URL}/endpoint", headers=headers)
    return response.json()

# ✅ PATRÓN CORRECTO: Usar auth_manager
from utils.auth import auth_manager

@callback(
    Output('result', 'children')
)
def my_callback():
    # ✅ Token ya desencriptado
    token = auth_manager.get_access_token()

    if not token:
        return "No autenticado"

    headers = {"Authorization": f"Bearer {token}"}
    response = requests.get(f"{API_URL}/endpoint", headers=headers)
    return response.json()
```

### Auth Guard (Issue #187)

**Propósito**: Renovar tokens automáticamente antes de que expiren.

#### Configuración

```python
# utils/auth.py
class AuthManager:
    TOKEN_REFRESH_INTERVAL = 12 * 60  # 12 minutos (tokens expiran en 15 min)
    TOKEN_EXPIRY = 15 * 60  # 15 minutos
    REFRESH_TOKEN_EXPIRY = 7 * 24 * 60 * 60  # 7 días
```

#### Flujo de Auto-Renovación

```
Timeline:
T0:   Login → access_token (expira T0+15min), refresh_token (expira T0+7días)
T12:  Auth Guard se dispara (12 min después)
      → Llama /api/v1/auth/refresh
      → Obtiene nuevo access_token (expira T12+15min)
      → Actualiza auth_manager
T24:  Auth Guard se dispara de nuevo
      → Renueva access_token
      ... (ciclo continúa)
T7d:  refresh_token expira
      → Próximo refresh falla con 401
      → Auth Guard logout automático
      → Redirige a /login
```

#### Implementación

```python
# callbacks/auth_guard.py
from dash import callback, Input, Output, State
from dash.exceptions import PreventUpdate
import logging

logger = logging.getLogger(__name__)

@callback(
    Output('auth-state', 'data', allow_duplicate=True),
    Input('auth-guard-interval', 'n_intervals'),
    State('auth-state', 'data'),
    prevent_initial_call=True
)
def auto_refresh_token(n_intervals, auth_state):
    """
    Auth Guard: Renueva token automáticamente cada 12 minutos.

    Triggers: dcc.Interval(id='auth-guard-interval', interval=12*60*1000)
    """
    # GUARD: Solo ejecutar si autenticado
    if not auth_state or not auth_state.get('authenticated'):
        logger.debug("[auto_refresh_token] User not authenticated - skip")
        raise PreventUpdate

    # Intentar refresh
    logger.info("[auto_refresh_token] Attempting token refresh")

    success = auth_manager.refresh_token()

    if success:
        logger.info("[auto_refresh_token] Token refreshed successfully")
        # Auth state no cambia (sigue autenticado)
        raise PreventUpdate
    else:
        # Refresh falló (refresh_token expiró)
        logger.warning("[auto_refresh_token] Token refresh failed - logging out")
        auth_manager.logout()
        return {"authenticated": False, "user": None}
```

**Componente Interval en app.py**:
```python
# app.py
dcc.Interval(
    id='auth-guard-interval',
    interval=12 * 60 * 1000,  # 12 minutos en ms
    n_intervals=0
)
```

### Verificación Proactiva (Issue #302)

**Problema**: Callbacks con `dcc.Interval` y API calls SIN verificación de auth causan loops de errores 401.

**Solución**: Verificar autenticación ANTES de hacer API calls.

```python
# ❌ ANTI-PATRÓN: API call sin verificar auth
@callback(
    Output('data-store', 'data'),
    Input('refresh-interval', 'n_intervals')
)
def auto_refresh_data(n_intervals):
    # ❌ Usuario podría no estar autenticado
    response = requests.get(f"{API_URL}/data", headers={"Authorization": "Bearer {token}"})
    # → 401 → Retry loop → 100+ requests/min → Rate limiting
    return response.json()

# ✅ PATRÓN CORRECTO: Verificación proactiva
from utils.auth_helpers import is_user_authenticated

@callback(
    Output('data-store', 'data'),
    Input('refresh-interval', 'n_intervals'),
    State('auth-state', 'data')
)
def auto_refresh_data(n_intervals, auth_state):
    # ✅ Verificar ANTES de API call
    if not is_user_authenticated(auth_state):
        logger.debug("[auto_refresh_data] User not authenticated - skip API call")
        raise PreventUpdate

    # ✅ AHORA SÍ: Usuario autenticado
    token = auth_manager.get_access_token()
    response = requests.get(f"{API_URL}/data", headers={"Authorization": f"Bearer {token}"})

    if response.status_code == 401:
        # Token expiró - Auth Guard se encargará
        logger.warning("[auto_refresh_data] 401 - Token expired")
        raise PreventUpdate

    return response.json()
```

### Multi-Worker Token Restoration (Issue #442) ⭐ CRÍTICO

**Problema**: En Render con Gunicorn multi-worker, cada worker tiene su propio `auth_manager` singleton.

```
┌─────────────────────────────────────────────────────────────────────┐
│                     Render Production                                │
│  ┌────────────┐  ┌────────────┐  ┌────────────┐  ┌────────────┐    │
│  │  Worker 1  │  │  Worker 2  │  │  Worker 3  │  │  Worker 4  │    │
│  │ auth_mgr   │  │ auth_mgr   │  │ auth_mgr   │  │ auth_mgr   │    │
│  │ ✅ tokens  │  │ ❌ empty   │  │ ❌ empty   │  │ ❌ empty   │    │
│  └────────────┘  └────────────┘  └────────────┘  └────────────┘    │
│        ↑               ↑               ↑               ↑            │
│        │               │               │               │            │
│  sync_auth ran    API callback    API callback    API callback     │
│  HERE             runs HERE       runs HERE       runs HERE        │
│                   → 401 Error     → 401 Error     → 401 Error      │
└─────────────────────────────────────────────────────────────────────┘
```

**Síntomas**:
- 401 Unauthorized en API calls después de login exitoso
- "User: unknown" en logs de Render
- Intermitente (depende de qué worker procesa cada request)

**Solución**: Restaurar tokens desde `auth-tokens-store` ANTES de cada API call.

```python
# ❌ ANTI-PATRÓN: Callback sin token restoration (funciona local, falla en Render)
@callback(
    Output('result', 'children'),
    Input('save-btn', 'n_clicks'),
    State('auth-state', 'data'),  # ← Solo auth check
    prevent_initial_call=True
)
def save_data(n_clicks, auth_state):
    if not is_user_authenticated(auth_state):
        raise PreventUpdate

    # ❌ Este worker puede NO tener tokens en auth_manager
    response = request_coordinator.make_request("/api/v1/data", method="PUT")
    # → 401 si sync_auth_context corrió en otro worker
    return response

# ✅ PATRÓN CORRECTO: Token restoration en multi-worker
@callback(
    Output('result', 'children'),
    Input('save-btn', 'n_clicks'),
    [
        State('auth-state', 'data'),
        State('auth-tokens-store', 'data'),  # ← NUEVO: Tokens encriptados
    ],
    prevent_initial_call=True
)
def save_data(n_clicks, auth_state, auth_tokens):
    if not is_user_authenticated(auth_state):
        raise PreventUpdate

    # ✅ Restaurar tokens en ESTE worker antes de API call
    if auth_tokens and "tokens" in auth_tokens:
        from utils.auth import auth_manager
        auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])
        logger.debug("[CALLBACK] Tokens restored from auth-tokens-store")

    # ✅ Ahora auth_manager tiene los tokens
    response = request_coordinator.make_request("/api/v1/data", method="PUT")
    return response
```

**Cuándo aplicar** (REGLA #7.6):
- ✅ TODO callback que use `request_coordinator.make_request()`
- ✅ TODO callback que use `auth_manager.get_access_token()`
- ✅ TODO callback que haga peticiones HTTP autenticadas
- ❌ NO necesario en callbacks que solo lean/escriban stores locales

**Archivos conocidos que necesitan este patrón**:
| Archivo | Estado | Prioridad |
|---------|--------|-----------|
| `settings.py` (save_profile, save_pharmacy) | ✅ Corregido (Issue #442) | Alta |
| `storage_usage.py` (fetch_storage_usage) | ✅ Corregido | Alta |
| `prescription.py` (4 callbacks) | ✅ Corregido (Issue #442) | Media |
| `admin/sync_operations.py` (3 callbacks) | ✅ Corregido (Issue #442) | Media |
| `admin/dangerous_tools.py` (1 callback) | ✅ Corregido (Issue #442) | Baja |
| `admin/catalog_management.py` (1 callback) | ✅ Corregido (Issue #442) | Baja |
| `catalog_callbacks.py` (2 callbacks) | ✅ Corregido (Issue #442) | Media |

**Helper de Verificación**:
```python
# utils/auth_helpers.py
def is_user_authenticated(auth_state):
    """
    Verifica si usuario está autenticado.

    Args:
        auth_state (dict): Estado de autenticación desde dcc.Store

    Returns:
        bool: True si autenticado, False en caso contrario
    """
    if not auth_state:
        return False

    if not isinstance(auth_state, dict):
        return False

    return auth_state.get("authenticated", False)
```

### Orden de Registro de Callbacks (CRÍTICO)

```python
# app.py (ORDEN OBLIGATORIO)

# 1️⃣ PRIMERO: Auth Context (desencripta tokens)
from callbacks.auth_context import register_auth_context_callbacks
register_auth_context_callbacks(app)

# 2️⃣ SEGUNDO: Auth Guard (renueva tokens, logout)
from callbacks.auth_guard import register_auth_guard_callbacks
register_auth_guard_callbacks(app)

# 3️⃣ TERCERO: Auth General (login, logout manual, OAuth)
from callbacks.auth_callbacks import register_auth_callbacks
register_auth_callbacks(app)

# 4️⃣ DESPUÉS: Otros callbacks (dashboard, upload, etc.)
from callbacks.dashboard import register_dashboard_callbacks
register_dashboard_callbacks(app)
# ... más callbacks
```

**Por qué este orden**:
- ✅ `auth_context` debe ejecutarse PRIMERO para desencriptar tokens
- ✅ `auth_guard` debe estar listo para detectar 401s y renovar
- ✅ `auth_callbacks` depende de auth_manager inicializado
- ✅ Otros callbacks asumen que auth ya está configurado

---

## Performance Optimization

### 1. Lazy Loading de Componentes

```python
# ❌ ANTI-PATRÓN: Importar todo al inicio
from components import component1, component2, ..., component50

# ✅ PATRÓN CORRECTO: Importar solo cuando se usa
def create_layout():
    from components.heavy_component import HeavyComponent

    return html.Div([
        HeavyComponent()
    ])
```

### 2. Memoization de Callbacks

```python
from functools import lru_cache

@lru_cache(maxsize=100)
def expensive_computation(input_data):
    """Función cara que se cachea"""
    # ... lógica compleja
    return result

@callback(...)
def my_callback(data):
    # Usar función cacheada
    result = expensive_computation(tuple(data))  # tuple para hashability
    return result
```

### 3. Pagination en Grandes Datasets

```python
@callback(
    Output('table', 'children'),
    [
        Input('page-number', 'value'),
        Input('page-size', 'value')
    ],
    State('full-data-store', 'data')
)
def paginate_table(page, page_size, full_data):
    """Paginar datos para mejorar performance"""
    if not full_data:
        return []

    start = (page - 1) * page_size
    end = start + page_size

    page_data = full_data[start:end]

    return create_table(page_data)
```

### 4. Background Callbacks (Dash Enterprise)

```python
# Nota: Requiere Dash Enterprise o dash-tools
from dash import callback, background

@callback(
    Output('result', 'children'),
    Input('long-task-btn', 'n_clicks'),
    background=True,  # ← Ejecuta en background
    running=[
        (Output('long-task-btn', 'disabled'), True, False)
    ],
    cancel=[Input('cancel-btn', 'n_clicks')]
)
def long_running_task(n_clicks):
    """Tarea larga que no bloquea UI"""
    import time

    for i in range(100):
        time.sleep(0.1)
        # ... procesamiento

    return "Completado"
```

---

## Referencias

- **frontend/CLAUDE.md**: Quick reference y reglas condensadas
- **frontend/docs/CALLBACK_PATTERNS.md**: Patrones específicos de callbacks
- **docs/DETAILED_RULES.md**: Reglas críticas del proyecto (backend + frontend)
- **Dash Documentation**: https://dash.plotly.com

---

> **DASH_DETAILED_PATTERNS.md** - Patrones avanzados Dash para xFarma
> Last updated: 2025-11-13 (Created during frontend/CLAUDE.md optimization)
