# Patrones de Callbacks en xFarma - Guía Completa

> 📚 **Prerequisito**: Leer `DASH_FRAMEWORK_GUIDE.md` para entender los conceptos fundamentales de Dash antes de trabajar con callbacks.

## ⚠️ REGLAS CRÍTICAS PARA CALLBACKS

### 1. **ANTES DE CREAR/MODIFICAR UN CALLBACK**
- [ ] Listar TODOS los IDs que usará el callback (Inputs, Outputs, States)
- [ ] Verificar que CADA ID existe en el layout actual
- [ ] Si eliminas un componente del layout, buscar TODOS los callbacks que lo usan
- [ ] Ejecutar: `python frontend/utils/validate_callbacks.py`

### 2. **DÓNDE APARECEN LOS ERRORES**

| Tipo de Error | Dónde Aparece | Cómo Detectarlo |
|--------------|---------------|-----------------|
| **ID no existe** | ❌ **NAVEGADOR** (consola JS) | Abrir DevTools (F12) |
| **Import Python** | ✅ Logs servidor | `docker logs` |
| **Callback duplicado** | ✅ Logs servidor | `docker logs` |
| **Error en función** | ✅ Logs servidor | `docker logs` |

> **IMPORTANTE**: Los errores de callbacks con IDs faltantes NO aparecen en logs del servidor, solo en el NAVEGADOR.

### 3. **COMANDOS DE VERIFICACIÓN**

```bash
# ANTES de hacer commit - Ejecutar SIEMPRE:

# 1. Verificar que todos los IDs existen
python frontend/utils/validate_callbacks.py

# 2. Generar reporte detallado
python frontend/utils/validate_callbacks.py --report

# 3. Buscar un ID específico en todo el código
grep -r "id-del-componente" frontend/

# 4. Verificar en el navegador (F12 → Console)
```

## Principios de Diseño

1. **Single Source of Truth**: Cada output debe tener un único callback responsable
2. **Store Intermedio**: Usar stores para coordinar múltiples fuentes de datos
3. **allow_duplicate con Propósito**: Solo usar cuando es el patrón correcto
4. **Verificación Obligatoria**: Siempre validar IDs antes de commit
5. **Interval Reset Pattern**: Resetear `n_intervals` para carga inmediata (respeta REGLA #11)

## 🔄 Patrón: Interval Reset Trigger (Issue #301)

**Problema**: Necesitas polling periódico + carga inmediata al cambiar estado (tab activo, filtro, navegación).

**Anti-patrón** ❌:
```python
# NO HACER: Múltiples callbacks escuchando el mismo Input (viola REGLA #11)
@callback(Output('data', 'children'), Input('my-tabs', 'value'))
def callback1(): ...

@callback(Output('interval', 'disabled'), Input('my-tabs', 'value'))  # Conflicto!
def callback2(): ...
# Resultado: callback2 "consume" el evento, callback1 no se dispara
```

**Solución** ✅:
```python
# Callback de control (maneja el trigger original)
@callback(
    [Output('content', 'children'),
     Output('my-interval', 'disabled'),
     Output('my-interval', 'n_intervals')],  # 🔑 CLAVE: Añadir n_intervals
    Input('my-tabs', 'value')
)
def control_callback(active_tab):
    is_active = active_tab == 'target-tab'
    content = render_tab_content(active_tab)

    # 🔑 Resetear n_intervals cuando se activa el tab
    n_reset = 0 if is_active else dash.no_update

    return content, not is_active, n_reset

# Callback de datos (escucha el interval)
@callback(
    Output('data', 'children'),
    Input('my-interval', 'n_intervals')  # Se dispara con reset Y polling
)
def data_callback(n_intervals):
    # Este callback se ejecuta:
    # 1. Inmediatamente cuando n_intervals resetea a 0 (tab activo)
    # 2. Cada X segundos por el polling natural del interval
    return fetch_and_display_data()
```

**Por qué funciona**:
- Resetear `n_intervals` a 0 dispara el callback como si el interval hubiera "ticked"
- El callback de datos **no necesita saber** qué causó el cambio (tab change vs polling)
- Respeta REGLA #11: cada Input tiene un solo callback propietario

**Casos de uso en xFarma**:
1. **Tabs con datos API** (`admin_tabs.py` + `catalog_management.py`):
   - Usuario cambia a "Gestión de Catálogo" → reset → datos cargan < 1s
   - Sigue en el tab → polling cada 30s

2. **Filtros en dashboards**:
   - Usuario selecciona período → reset interval → datos actualizan inmediatamente
   - Polling continúa con los filtros actuales

3. **Páginas con auto-refresh**:
   - `/upload` con progreso de subida
   - `/generics` con análisis en background
   - Navegas a la página → reset → carga inmediata
   - Sigues en la página → polling continuo

**Cuándo usar**:
- ✅ Tienes `dcc.Interval` para polling
- ✅ Necesitas carga inmediata al cambiar estado
- ✅ El callback que maneja el estado puede controlar el interval

**Cuándo NO usar**:
- ❌ No hay polling (solo trigger único) → usa Input directo
- ❌ Callbacks completamente independientes
- ❌ Interval nunca necesita "reseteo" (solo polling pasivo)

**Ventajas**:
- ✅ Respeta REGLA #11 (One Input One Callback)
- ✅ Más directo que Store intermedio
- ✅ Sin race conditions
- ✅ Fácil de entender y mantener

**Ejemplo real**: `frontend/callbacks/admin/admin_tabs.py` (líneas 37, 75)

## ✅ Casos Válidos para allow_duplicate=True

### 1. Toast Notifications (`toast-trigger-store`)
**Patrón**: Publisher-Subscriber (Pub-Sub)  
**Razón**: Múltiples módulos necesitan enviar notificaciones de forma independiente  
**Ejemplo**:
```python
# Desde catalog_callbacks.py
Output("toast-trigger-store", "data", allow_duplicate=True)

# Desde upload.py
Output("toast-trigger-store", "data", allow_duplicate=True)

# Desde config.py
Output("toast-trigger-store", "data", allow_duplicate=True)
```
**Alternativa costosa**: Un callback central monstruoso que maneje todos los casos de notificación

### 2. Modales de Confirmación
**Patrón**: Multiple Triggers  
**Razón**: Diferentes acciones pueden abrir/cerrar el mismo modal  
**Ejemplos**:
- `delete-confirmation-modal.is_open`
- `sync-status-modal.is_open`
- `catalog-progress-modal.is_open`

### 3. System Alerts (`system-alerts.children`)
**Patrón**: Event Aggregation  
**Razón**: Errores y alertas pueden originarse desde cualquier módulo  
**Uso**: Cada módulo puede reportar sus propios errores sin acoplamiento

### 4. Progress Stores
**Patrón**: Progress Reporting  
**Razón**: Diferentes procesos reportan su progreso de forma independiente  
**Ejemplo**: `sidebar-progress-store.data`

## ❌ Casos NO Válidos para allow_duplicate=True

### 1. Contenido Principal de Componentes
**Problema**: Race conditions, parpadeos, comportamiento impredecible  
**Mal ejemplo**:
```python
# homepage.py
Output("catalog-control-panel-container", "children")

# catalog_callbacks.py  
Output("catalog-control-panel-container", "children")  # ERROR!
```

**Solución Correcta - Store Intermedio**:
```python
# Múltiples fuentes actualizan un store
Input("dashboard-data") → catalog-panel-unified-store
Input("catalog-interval") → catalog-panel-unified-store

# Un único callback renderiza
catalog-panel-unified-store → Output("catalog-control-panel-container", "children")
```

### 2. Datos Estructurados
**Problema**: Inconsistencia de datos, difícil debugging  
**Solución**: Single source of truth con store centralizado

## Arquitectura de Refactorización Implementada

### Caso: catalog-control-panel-container

```mermaid
graph LR
    A[dashboard-data-store] --> D[catalog-panel-unified-store]
    B[catalog-status-store] --> D
    C[catalog-info-interval] --> D
    D --> E[catalog-control-panel-container]
```

**Beneficios**:
- ✅ Un único punto de renderizado
- ✅ Fácil debugging (revisar el store)
- ✅ Sin race conditions
- ✅ Prioridad clara de fuentes de datos

## Guía de Decisión

```
¿Múltiples callbacks necesitan actualizar el mismo output?
    │
    ├─ NO → Usar callback normal
    │
    └─ SÍ → ¿Es un patrón Pub-Sub (notificaciones, eventos)?
            │
            ├─ SÍ → allow_duplicate=True es válido
            │
            └─ NO → ¿Son triggers para abrir/cerrar UI (modales)?
                    │
                    ├─ SÍ → allow_duplicate=True puede ser válido
                    │
                    └─ NO → Usar Store Intermedio
```

## Implementación del Store Intermedio

### 1. Definir el Store
```python
dcc.Store(
    id="component-unified-store",
    data={
        "source": None,      # Identificar origen de la actualización
        "data": {},          # Datos combinados
        "last_update": None  # Timestamp para debugging
    }
)
```

### 2. Callbacks de Actualización
```python
@callback(
    Output("component-unified-store", "data"),
    [Input("source-1", "data"),
     Input("source-2", "data")],
    State("component-unified-store", "data")
)
def update_unified_store(source1, source2, current_store):
    ctx = dash.callback_context
    trigger = ctx.triggered[0]["prop_id"].split(".")[0]
    
    new_store = current_store.copy() if current_store else {}
    new_store["source"] = trigger
    new_store["last_update"] = datetime.now().isoformat()
    
    # Combinar datos según la lógica de negocio
    # ...
    
    return new_store
```

### 3. Callback de Renderizado
```python
@callback(
    Output("component-container", "children"),
    Input("component-unified-store", "data")
)
def render_component(unified_data):
    if not unified_data:
        return create_default_component()
    
    # Renderizar basado en datos unificados
    return create_component(unified_data["data"])
```

## 📝 CHECKLIST PARA CAMBIOS EN UI

### Al ELIMINAR un componente:
1. [ ] Buscar el ID en TODOS los archivos: `grep -r "id-del-componente" frontend/`
2. [ ] Eliminar/actualizar TODOS los callbacks que lo usan
3. [ ] Ajustar los returns de callbacks (número de valores)

### Al AÑADIR un componente:
1. [ ] Definir el ID en el layout PRIMERO
2. [ ] Luego crear los callbacks
3. [ ] Verificar que no hay IDs duplicados
4. [ ] Ejecutar `validate_callbacks.py` antes de commit

### Al RENOMBRAR un ID:
1. [ ] Cambiar en el layout
2. [ ] Buscar y reemplazar en TODOS los callbacks
3. [ ] Verificar con grep que no quedó ninguna referencia vieja

## 🚨 ERRORES COMUNES Y SOLUCIONES

### Error: "A nonexistent object was used in an Output"
**Causa**: El ID no existe en el layout actual  
**Solución**:
1. Copiar el ID exacto del error
2. Buscarlo en layouts: `grep -r "id-exacto" frontend/`
3. Si no existe, añadirlo o eliminar el callback

### Error: "Duplicate callback outputs"
**Causa**: Dos callbacks intentan actualizar el mismo Output  
**Solución**:
1. Buscar el Output duplicado
2. Usar `allow_duplicate=True` o combinar callbacks
3. Mejor aún: usar Store Intermedio (ver arriba)

### Error: Callback no se ejecuta
**Causa**: El componente está comentado o dentro de un condicional  
**Solución**:
1. Verificar que el componente se renderiza siempre
2. No usar IDs dentro de componentes condicionales

## Mantenimiento

### Al añadir nuevos callbacks:
1. Verificar si el output ya existe con `validate_callbacks.py`
2. Si existe, evaluar si es un caso válido para allow_duplicate
3. Si no es válido, usar store intermedio
4. Documentar la decisión en este archivo

### Herramientas de Validación:
- `frontend/utils/validate_callbacks.py`: Detecta callbacks huérfanos y IDs faltantes
- `frontend/utils/find_duplicate_outputs.py`: Encuentra outputs duplicados

### Workflow obligatorio:
1. **ESCRIBIR** el componente con su ID
2. **VERIFICAR** que aparece en el layout
3. **ENTONCES** crear el callback
4. **PROBAR** en el navegador (F12 → Console)

## 📊 Patrón Avanzado: Intermediate Store Decoupling (REGLA #11)

### Problema: Múltiples Callbacks con Outputs Duplicados

Cuando múltiples callbacks necesitan escribir al mismo output, especialmente `toast-trigger-store`, surgen errores DASH002 que violan REGLA #11.

**Síntoma**:
```bash
DuplicateCallbackOutput: You have already assigned a callback to the output
with ID "toast-trigger-store" and property "data". An output can only have
a single callback function assigned to it.
```

### Solución: Intermediate Store Decoupling Pattern

Este patrón usa stores intermedios para desacoplar la lógica de negocio de la escritura final al output compartido.

**Arquitectura**:
```mermaid
graph TD
    A[Callback 1] --> D[Store Intermedio 1]
    B[Callback 2] --> E[Store Intermedio 2]
    C[Callback 3] --> F[Store Intermedio 3]
    D --> G[Unified Callback]
    E --> G
    F --> G
    G --> H[toast-trigger-store]
```

### Implementación Completa (Ejemplo Real: sync_operations.py)

#### Paso 1: Crear Stores Intermedios en Layout

```python
# frontend/layouts/admin.py (líneas 90-92)
dcc.Store(id='sync-operations-toast-store', storage_type='memory'),
dcc.Store(id='scheduled-sync-toast-store', storage_type='memory'),
dcc.Store(id='conflict-resolution-toast-store', storage_type='memory')
```

**Tip**: Usar `storage_type='memory'` para toasts (no necesitan persistencia).

#### Paso 2: Callbacks Escriben a Stores Intermedios

```python
# frontend/callbacks/admin/sync_operations.py

@app.callback(
    Output('sync-operations-toast-store', 'data'),  # ← Store intermedio
    Input('admin-sync-cima-button', 'n_clicks'),
    prevent_initial_call=True
)
def handle_sync_operation(n_clicks):
    """Lógica de negocio + toast al store intermedio."""
    try:
        # Ejecutar sincronización CIMA
        result = perform_sync()

        # ✅ Escribir toast al store intermedio (NO directamente a toast-trigger-store)
        return success_toast("Sincronización completada", "Éxito")
    except Exception as e:
        return error_toast(f"Error: {str(e)}", "Error")


@app.callback(
    Output('scheduled-sync-toast-store', 'data'),  # ← Otro store intermedio
    Input('enable-scheduled-sync-switch', 'value'),
    prevent_initial_call=True
)
def handle_scheduled_sync(enabled):
    """Otra lógica + toast al store intermedio."""
    if enabled:
        return info_toast("Sincronización programada activada", "Info")
    return warning_toast("Sincronización programada desactivada", "Aviso")
```

#### Paso 3: Callback Unificado con Helper Reutilizable

```python
# frontend/callbacks/admin/sync_operations.py

from utils.callback_helpers import unified_toast_trigger  # ← Helper reutilizable

@app.callback(
    Output('toast-trigger-store', 'data'),  # ← UN SOLO callback escribe aquí
    [
        Input('sync-operations-toast-store', 'data'),
        Input('scheduled-sync-toast-store', 'data'),
        Input('conflict-resolution-toast-store', 'data')
    ],
    prevent_initial_call=True
)
def unified_toast_trigger_callback(sync_toast, scheduled_toast, conflict_toast):
    """
    Callback unificado usando helper reutilizable.
    Cumple REGLA #11: Un solo callback escribe a toast-trigger-store.
    """
    return unified_toast_trigger(sync_toast, scheduled_toast, conflict_toast)
```

**Helper Reutilizable** (`frontend/utils/callback_helpers.py`):
```python
def unified_toast_trigger(*store_inputs):
    """
    Detecta cuál store fue actualizado y retorna su valor.
    Thread-safe mediante ctx.triggered.
    """
    if not ctx.triggered:
        return no_update

    # Detectar cuál store fue actualizado
    triggered_value = ctx.triggered[0]['value']

    if triggered_value:
        return triggered_value

    return no_update
```

### Ventajas del Patrón

| Beneficio | Descripción |
|-----------|-------------|
| ✅ **Cumple REGLA #11** | Un solo callback escribe a toast-trigger-store |
| ✅ **Desacoplamiento** | Lógica de negocio separada de notificaciones |
| ✅ **Escalable** | Agregar nuevo toast = agregar nuevo Input al callback unificado |
| ✅ **Thread-safe** | ctx.triggered detecta fuente exacta, sin race conditions |
| ✅ **Reutilizable** | Helper `unified_toast_trigger()` usado en múltiples módulos |
| ✅ **Testeable** | Cada callback de negocio testeable independientemente |

### Cuándo Usar Este Patrón

**✅ SÍ usar cuando**:
- Múltiples callbacks necesitan escribir al mismo output
- El output es un store compartido (ej: `toast-trigger-store`, `url.pathname`)
- La lógica de negocio es compleja y merece callbacks separados
- Necesitas escalabilidad (agregar más fuentes fácilmente)

**❌ NO usar cuando**:
- Solo un callback escribe al output (usar callback normal)
- El output es contenido visual crítico (considerar store + callback de renderizado único)
- La lógica es tan simple que cabe en un solo callback sin complejidad

### Archivos de Referencia

- **Implementación completa**: `frontend/callbacks/admin/sync_operations.py`
- **Helper reutilizable**: `frontend/utils/callback_helpers.py`
- **Stores intermedios**: `frontend/layouts/admin.py` (líneas 90-92)
- **Issue relacionado**: #300 (Fix DASH002 - Outputs duplicados)

## 📝 Patrón: Callbacks Separados para Contenido Dinámico por Tabs

### Problema: Dynamic Tab Content con Unified Callbacks

Cuando se usa **dynamic content replacement** (como en `/admin` o `/ajustes`), un callback reemplaza todo el contenido del tab activo. Si usas un callback unificado que escucha múltiples botones de diferentes tabs, obtendrás errores:

```
Error: A nonexistent object was used in an Input of a Dash callback.
The id of this object is 'save-pharmacy-btn'
```

**Causa**: El callback unificado intenta escuchar `save-pharmacy-btn` pero ese botón NO existe en el DOM cuando el tab "Perfil" está activo (solo existe cuando el tab "Farmacia" está renderizado).

### Solución: Separar Callbacks por Tab

**❌ ANTIPATRÓN - Callback Unificado**:
```python
@app.callback(
    [Output('profile-save-feedback', 'children'),
     Output('pharmacy-save-feedback', 'children'),
     Output('preferences-save-feedback', 'children')],
    [Input('save-profile-btn', 'n_clicks'),      # ← Error cuando tab != Perfil
     Input('save-pharmacy-btn', 'n_clicks'),     # ← Error cuando tab != Farmacia
     Input('save-preferences-btn', 'n_clicks')], # ← Error cuando tab != Preferencias
    [State(...), State(...), ...],
    prevent_initial_call=True
)
def save_settings_unified(profile_clicks, pharmacy_clicks, preferences_clicks, ...):
    trigger_id = ctx.triggered[0]['prop_id'].split('.')[0]

    if trigger_id == 'save-profile-btn':
        # Guardar perfil
        ...
```

**Problema**: Cuando renderizas el tab "Farmacia", el componente `save-profile-btn` desaparece del DOM, pero el callback sigue intentando escucharlo → Error "nonexistent object".

**✅ PATRÓN CORRECTO - Callbacks Separados**:
```python
# Callback 1: Solo componentes del tab Perfil
@app.callback(
    [Output('profile-save-feedback', 'children', allow_duplicate=True),
     Output('toast-trigger-store', 'data', allow_duplicate=True)],
    Input('save-profile-btn', 'n_clicks'),  # ← Solo escucha botón de Perfil
    [State('profile-full-name', 'value'),
     State('profile-phone', 'value'),
     ...],
    prevent_initial_call=True
)
def save_profile_settings(n_clicks, full_name, phone, ...):
    if n_clicks is None or n_clicks == 0:  # ✅ Validación robusta
        return no_update, no_update

    feedback, toast = _save_profile_logic(full_name, phone, ...)
    return feedback, toast


# Callback 2: Solo componentes del tab Farmacia
@app.callback(
    [Output('pharmacy-save-feedback', 'children', allow_duplicate=True),
     Output('toast-trigger-store', 'data', allow_duplicate=True)],
    Input('save-pharmacy-btn', 'n_clicks'),  # ← Solo escucha botón de Farmacia
    [State('pharmacy-name', 'value'),
     State('pharmacy-code', 'value'),
     ...],
    prevent_initial_call=True
)
def save_pharmacy_settings(n_clicks, pharmacy_name, pharmacy_code, ...):
    if n_clicks is None or n_clicks == 0:  # ✅ Validación robusta
        return no_update, no_update

    feedback, toast = _save_pharmacy_logic(pharmacy_name, pharmacy_code, ...)
    return feedback, toast


# Callback 3: Solo componentes del tab Preferencias
@app.callback(
    [Output('preferences-save-feedback', 'children', allow_duplicate=True),
     Output('toast-trigger-store', 'data', allow_duplicate=True)],
    Input('save-preferences-btn', 'n_clicks'),  # ← Solo escucha botón de Preferencias
    [State('refresh-interval', 'value'),
     State('system-notifications', 'value'),
     ...],
    prevent_initial_call=True
)
def save_preferences_settings(n_clicks, refresh_interval, system_notifications, ...):
    if n_clicks is None or n_clicks == 0:  # ✅ Validación robusta
        return no_update, no_update

    feedback, toast = _save_preferences_logic(refresh_interval, system_notifications, ...)
    return feedback, toast
```

### Características Clave del Patrón

1. **Callbacks independientes**: Cada tab tiene su propio callback
2. **Un Input por callback**: Cumple REGLA #11 (no duplicar Inputs)
3. **allow_duplicate=True**: Necesario porque múltiples callbacks escriben a `toast-trigger-store`
4. **Validación robusta de n_clicks**: `if n_clicks is None or n_clicks == 0:` (NO usar `if not n_clicks:`)
5. **Helper functions**: Extraer lógica de negocio a funciones auxiliares (`_save_profile_logic()`)

### Por qué `allow_duplicate=True` es Correcto Aquí

- ✅ **toast-trigger-store es Pub-Sub**: Múltiples módulos envían notificaciones independientemente
- ✅ **Feedback específico por tab**: Cada tab escribe a su propio `*-save-feedback`
- ✅ **Sin race conditions**: Cada callback se ejecuta solo cuando su tab está activo

### Validación de n_clicks: Patrón Robusto

**❌ INCORRECTO**:
```python
if not n_clicks:
    return no_update
```
**Problema**: `if not 0:` evalúa a `True`, causando comportamiento inesperado en el primer click.

**✅ CORRECTO**:
```python
if n_clicks is None or n_clicks == 0:
    return no_update
```
**Beneficio**: Validación explícita que maneja correctamente `None` (inicial) y `0` (sin clicks).

### Cuándo Usar Este Patrón

**✅ SÍ usar cuando**:
- Página con tabs y **dynamic content replacement**
- Cada tab tiene botones/inputs que desaparecen del DOM al cambiar de tab
- Callbacks necesitan escuchar componentes que no siempre existen
- Sigues el patrón de `/admin` o `/ajustes`

**❌ NO usar cuando**:
- Usas **show/hide pattern** (componentes siempre en DOM, solo cambia `style={'display': 'none'}`)
- Todos los componentes están siempre presentes en el layout
- Usas tabs sin reemplazo dinámico de contenido

### Alternativa: Show/Hide Pattern

Si prefieres evitar callbacks separados, usa el show/hide pattern:

```python
# Todos los tabs siempre en el DOM
html.Div([
    html.Div(id='tab-perfil-content', style={'display': 'block'}),  # ← Visible
    html.Div(id='tab-farmacia-content', style={'display': 'none'}),  # ← Oculto
    html.Div(id='tab-preferencias-content', style={'display': 'none'}),  # ← Oculto
])

# Callback cambia display, NO reemplaza contenido
@app.callback(
    [Output('tab-perfil-content', 'style'),
     Output('tab-farmacia-content', 'style'),
     Output('tab-preferencias-content', 'style')],
    Input('settings-tabs', 'active_tab')
)
def toggle_tab_visibility(active_tab):
    if active_tab == 'perfil':
        return {'display': 'block'}, {'display': 'none'}, {'display': 'none'}
    elif active_tab == 'farmacia':
        return {'display': 'none'}, {'display': 'block'}, {'display': 'none'}
    else:
        return {'display': 'none'}, {'display': 'none'}, {'display': 'block'}
```

**Trade-off**:
- ✅ **Ventaja**: Callback unificado funciona (componentes siempre existen)
- ❌ **Desventaja**: Mayor uso de memoria (3 tabs siempre renderizados)

### Archivos de Referencia

- **Implementación completa**: `frontend/callbacks/settings.py` (líneas 124-198)
- **Layout dinámico**: `frontend/pages/settings_page.py`
- **Patrón show/hide**: `frontend/layouts/admin.py` (usa dbc.Tabs con dynamic content)
- **Commit relacionado**: 34d97fc (Fix callbacks separados para tabs dinámicos)

## 📌 REGLA DE ORO

> **"NUNCA confíes en que un ID existe - SIEMPRE verifica"**

## 🛡️ Patrón: Non-Blocking Dependency Triggers (Commit fd7bc07)

**Cuándo usar**: Callback necesita un store como trigger (para reactividad) pero NO depende de su valor para ejecutar.

**Problema resuelto**: Race conditions donde un callback se bloquea innecesariamente cuando una dependencia tiene error.

### Ejemplo: Partners Dropdown (Issue #X - Commit fd7bc07)

**Síntoma**:
```python
# ❌ ANTES: Partners dropdown vacío en carga de página
# Callback se bloqueaba cuando context-store tenía error
```

**Root Cause**:
```python
@callback(
    [...],
    Input("context-store", "data"),  # Trigger necesario
    [...]
)
def load_partners_dropdown(context_data, ...):
    # ❌ PROBLEMA: Guard bloqueaba ejecución
    if context_data and context_data.get("error"):
        raise PreventUpdate  # Partners nunca cargan!
```

**Solución** ✅:
```python
@callback(
    [...],
    [
        Input("refresh-partners-list-btn", "n_clicks"),
        Input("url", "pathname"),
        Input("auth-ready", "data"),
        Input("context-store", "data"),  # ✅ Mantener como trigger
    ],
    [...]
)
def load_partners_dropdown(refresh_clicks, pathname, auth_ready, context_data, ...):
    """
    ✅ FIX: context-store como TRIGGER pero NO como DEPENDENCY.
    Partners cargan INDEPENDIENTEMENTE del estado de context (REGLA #12).
    """
    # Guard 1: Pathname (critical)
    if pathname != "/generics":
        raise PreventUpdate

    # Guard 2: Auth ready (critical)
    if not auth_ready:
        raise PreventUpdate

    # Guard 3: User authenticated (critical)
    if not is_user_authenticated(auth_state):
        raise PreventUpdate

    # ✅ NEW: Log but DON'T block execution
    if context_data and context_data.get("error"):
        logger.info(f"Context tiene error, pero continuando carga de partners (independientes)")
        # NO raise PreventUpdate - continue execution

    # ✅ Partners ahora cargan exitosamente
    partners = fetch_partners_from_db()
    return partners
```

### Arquitectura del Patrón

**Principio clave**: Distinguir entre **trigger** (dispara callback) y **dependency** (valor necesario para ejecutar).

```python
# Categorizar Inputs según su propósito:

# 1. TRIGGERS PUROS (no dependency):
Input("context-store", "data")  # Dispara callback, pero valor NO importa

# 2. DEPENDENCIES PURAS (también trigger):
Input("url", "pathname")  # Dispara callback Y valor es crítico

# 3. TRIGGER + SOFT DEPENDENCY:
Input("auth-ready", "data")  # Dispara callback, valor importa pero no bloquea
```

### Cuándo Aplicar Este Patrón

✅ **Usar cuando**:
- Callback necesita dispararse cuando store X cambia (trigger)
- PERO la lógica del callback NO depende del valor de store X
- Store X puede tener errores que NO afectan el resultado del callback
- Existe arquitectura que permite independencia (ej: REGLA #12 - Auto-Init Partners)

❌ **NO usar cuando**:
- Callback REQUIERE el valor del store para funcionar correctamente
- Error en el store indica que el callback no debería ejecutarse
- No existe fallback para obtener datos necesarios

### Implementación Step-by-Step

1. **Identificar el problema**:
   ```python
   # Síntoma: Callback no se ejecuta cuando debería
   # Causa: Guard bloqueante con PreventUpdate
   ```

2. **Analizar dependencias**:
   ```python
   # ¿El callback REALMENTE necesita este valor?
   # ¿Puede obtener datos de otra fuente (ej: BD)?
   ```

3. **Refactorizar guards**:
   ```python
   # ANTES:
   if dependency_data and dependency_data.get("error"):
       raise PreventUpdate  # ❌ Bloquea

   # DESPUÉS:
   if dependency_data and dependency_data.get("error"):
       logger.info(f"Dependency tiene error, pero continuando ejecución")
       # ✅ Log pero no bloquea
   ```

4. **Agregar trigger logging**:
   ```python
   trigger = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else "initial"
   logger.debug(f"[callback_name] Triggered by: {trigger}")
   ```

5. **Agregar unit test de regresión**:
   ```python
   def test_callback_with_dependency_error():
       """
       Verify callback executes even when dependency has error.
       Regression test for commit fd7bc07.
       """
       dependency_data = {"error": "Dependency load failed"}
       # ... simulate callback logic
       assert should_block == False  # Callback should NOT block
   ```

### Beneficios

✅ **Resiliencia**: Callback funciona incluso si dependencias fallan
✅ **UX mejorada**: Datos cargan más rápido sin esperar dependencias
✅ **Debugging**: Logs explícitos de qué disparó el callback
✅ **Testing**: Tests unitarios verifican comportamiento sin mocks complejos

### Consideraciones

⚠️ **Performance**: Evaluar si agregar más Inputs aumenta frecuencia de ejecución
⚠️ **Lógica alternativa**: Asegurar que existe fuente de datos alternativa (ej: BD)
⚠️ **Documentación**: Actualizar docstring explicando por qué Input NO es dependency

### Casos de Uso en xFarma

1. **Partners Dropdown** (`callbacks/generics.py:load_partners_dropdown`):
   - Trigger: `context-store` (para reactividad en navegación)
   - Dependency: Base de datos (REGLA #16 - BD-first pattern)
   - Fix: Commit fd7bc07

2. **Context Loading** (futuro):
   - Puede aplicarse a otros callbacks que usen `context-store` como trigger

### Testing

```python
# Test 1: Verificar que callback NO bloquea con error
def test_callback_with_context_error():
    context_data = {"error": "Context load failed"}
    # Should NOT raise PreventUpdate
    assert should_block == False

# Test 2: Verificar trigger mechanism
def test_trigger_mechanism():
    # Verify Input("context-store") is present
    assert "context-store.data" in inputs

# Test 3: Verificar datos cargan correctamente
def test_data_loads_successfully():
    # Despite context error, data should load from DB
    assert len(partners) == 4
```

### Referencias

- **Issue**: Partners dropdown vacío en página load
- **Commit**: fd7bc07 - Fix partners dropdown race condition
- **REGLA #12**: Auto-Init Partners (independencia de context)
- **REGLA #16**: BD-First Pattern (database as source of truth)
- **Tests**: `frontend/tests/unit/test_generics_callbacks.py::test_load_partners_dropdown_with_context_error`

---

## 🔒 Patrón: Input vs State con Guards (Race Condition Prevention)

**Problema**: Callbacks con guards que bloquean ejecución usando State en lugar de Input.

### Antipatrón

```python
# ❌ INCORRECTO: auth-ready como State
@app.callback(
    Output("data-store", "data"),
    [Input("url", "pathname")],
    [State("auth-ready", "data")],  # ← State NO dispara callback
    prevent_initial_call=False,
)
def load_data(pathname, auth_ready):
    # Guard bloquea si auth no sincronizó
    if not auth_ready:
        raise PreventUpdate

    # Cargar datos...
    return fetch_data()
```

**Flujo del problema:**
1. Usuario navega → Callback se dispara por `pathname`
2. `auth_ready` es `False` (auth no sincronizó aún)
3. Guard bloquea con `PreventUpdate` → **NO carga datos**
4. Auth sincroniza → `auth_ready` cambia a `True`
5. **PERO** callback NO se vuelve a disparar (auth-ready es State)
6. **Resultado**: Datos nunca se cargan → UI permanece vacía

### Solución: Cambiar State → Input

```python
# ✅ CORRECTO: auth-ready como Input
@app.callback(
    Output("data-store", "data"),
    [Input("url", "pathname"), Input("auth-ready", "data")],  # ← Input
    [State("auth-state", "data")],
    prevent_initial_call=False,
)
def load_data(pathname, auth_ready, auth_state):
    # Guard sigue bloqueando si necesario
    if not auth_ready:
        raise PreventUpdate

    # Pero ahora callback SE DISPARA cuando auth_ready cambia a True
    return fetch_data()
```

**Beneficios:**
- ✅ Callback se dispara cuando auth sincroniza exitosamente
- ✅ UI se carga correctamente después de autenticación
- ✅ Sin race conditions entre navegación y auth

### Cuándo Aplicar Este Patrón

**Usar Input (no State) cuando:**
- ✅ Hay un guard que bloquea basado en ese valor
- ✅ El callback DEBE ejecutarse cuando ese valor cambia
- ✅ El valor es una señal de "ready" o "available"

**Usar State cuando:**
- ✅ Solo necesitas leer el valor (no necesitas disparar)
- ✅ Quieres evitar re-ejecuciones innecesarias
- ✅ El valor cambia frecuentemente pero no afecta lógica

### Ejemplo Real: Panel de Contexto Vacío

**Issue**: Panel de contexto vacío en `/generics` en Render
**Causa**: `auth-ready` era State → Callback no se disparaba cuando auth sincronizaba
**Fix**: Commit `858f097` - Cambiar auth-ready a Input

```python
# Antes (problemático)
[State("auth-ready", "data")]  # ← State

# Después (corregido)
[Input("auth-ready", "data")]  # ← Input
```

---

## 📦 Patrón: Fuente de Verdad en Payloads (Stale Store Prevention)

**Problema**: Usar stores desactualizados para construir payloads de persistencia.

### Antipatrón

```python
# ❌ INCORRECTO: Usar store que puede estar stale
@app.callback(
    Output("selection-store", "data"),
    Input("dropdown", "value"),
    State("selection-store", "data"),
)
def save_selection(dropdown_value, current_store):
    selected_items = dropdown_value

    # ❌ MAL: available_items puede estar desactualizado
    available_items = current_store.get("available", [])

    # Payload enviado al backend
    persist_to_backend(selected_items, available_items)
```

**Problema:**
- `current_store["available"]` puede quedar stale después de refresh/navegación
- Si el store perdió items, el payload al backend tampoco los incluye
- Backend interpreta items faltantes como "deseleccionados"
- **Resultado**: Items desaparecen sin que el usuario los haya eliminado

### Solución: Usar Context Data (Fuente de Verdad)

```python
# ✅ CORRECTO: Usar context_data (fuente de verdad)
@app.callback(
    Output("selection-store", "data"),
    Input("dropdown", "value"),
    [State("selection-store", "data"), State("context-store", "data")],
)
def save_selection(dropdown_value, current_store, context_data):
    selected_items = dropdown_value

    # ✅ BIEN: Usar context_data (siempre actualizado)
    if context_data and "items_in_universe" in context_data:
        available_items = [
            item["name"]
            for item in context_data["items_in_universe"]
            if isinstance(item, dict) and "name" in item
        ]
        logger.debug(f"Usando {len(available_items)} items desde context (fuente de verdad)")
    else:
        # Fallback a store solo si context no disponible
        available_items = current_store.get("available", [])
        logger.warning(f"context_data no disponible, usando store (puede estar stale)")

    # Payload completo enviado al backend
    persist_to_backend(selected_items, available_items)
```

**Beneficios:**
- ✅ Payload siempre incluye lista completa actualizada
- ✅ Solo items explícitamente deseleccionados se marcan como no seleccionados
- ✅ Sin desapariciones misteriosas de items

### Jerarquía de Fuentes de Verdad

```python
# 1. MEJOR: Context data (universo completo)
context_data["items_in_universe"]  # ✅ Siempre actualizado

# 2. FALLBACK: Store (puede estar stale)
current_store.get("available", [])  # ⚠️ Usar solo si context no disponible

# 3. EVITAR: Hardcodear lista
available_items = ["Item1", "Item2"]  # ❌ No escalable
```

### Ejemplo Real: Partners Desapareciendo

**Issue**: Usuario elimina EXELTIS → También desaparece ARISTO (sin haberlo eliminado)
**Causa**: `current_store["available"]` stale usado para payload
**Fix**: Commit `935bf9c` - Usar `context_data["laboratories_in_universe"]`

**Antes (problemático):**
```python
available_partners = current_store.get("available", [])  # ← Stale (6 items)
persist_partners_selection(pharmacy_id, selected_partners, available_partners)
# Backend marca ARISTO como no seleccionado (no estaba en lista)
```

**Después (corregido):**
```python
if context_data and "laboratories_in_universe" in context_data:
    available_partners = [
        lab["laboratory_name"]
        for lab in context_data["laboratories_in_universe"]
        if isinstance(lab, dict) and "laboratory_name" in lab
    ]  # ← Fuente de verdad (7 items completos)
```

### Cuándo Aplicar Este Patrón

**Usar context_data cuando:**
- ✅ Construyes payloads para persistencia
- ✅ Necesitas lista completa del universo de datos
- ✅ El store puede quedar desactualizado

**Usar store cuando:**
- ✅ Solo UI temporal (no persistencia)
- ✅ Datos específicos de sesión
- ✅ Performance crítico (context_data muy grande)

---

## 📝 Patrón: Multi-Tab Form Persistence (Issue #417)

### Problema

En formularios multi-tab, los datos ingresados se pierden al cambiar entre pestañas porque los inputs se renderizan dinámicamente según el tab activo:

```python
# ❌ ANTI-PATRÓN: Inputs dinámicos sin persistencia
@app.callback(
    Output("tab-content", "children"),
    Input("tabs", "active_tab")
)
def render_tab_content(active_tab):
    if active_tab == "tab-1":
        return dbc.Input(id="email-input")  # ← Se destruye al cambiar tab
    else:
        return dbc.Input(id="pharmacy-input")  # ← Se destruye al volver
```

**Consecuencia**: Al cambiar de tab "Usuario" → "Farmacia" → "Usuario", los valores ingresados desaparecen.

### Solución: Auto-Save Pattern con dcc.Store

**Arquitectura**:
1. **Store global** en skeleton (`app.py`) con `storage_type="memory"`
2. **Auto-save callback** que captura todos los cambios de inputs
3. **Populate callback** que restaura valores desde store con fallback priority
4. **Save callback** que lee del store en lugar de States individuales
5. **Cleanup** del store en momentos estratégicos (modal open, cancel, save)

### Implementación Completa

#### 1. Store en Skeleton (REGLA #0.5)

```python
# frontend/app.py (skeleton global)
app.layout = dbc.Container([
    # ... otros componentes ...

    # CRITICAL: Store para formulario multi-tab
    dcc.Store(
        id="admin-user-form-data-store",
        storage_type="memory",  # Session-only, no persistent
        data=None
    ),
])
```

**Storage Types**:
- ✅ `"memory"`: Datos se pierden al cerrar tab/refrescar (recomendado para formularios temporales)
- ⚠️ `"session"`: Persiste con F5 pero se pierde al cerrar tab
- ❌ `"local"`: Persiste indefinidamente (NO usar para formularios temporales)

#### 2. Auto-Save Callback

```python
# frontend/callbacks/admin/users.py
@app.callback(
    Output("admin-user-form-data-store", "data"),
    [
        Input("user-email-input", "value"),
        Input("user-password-input", "value"),
        Input("user-rol-dropdown", "value"),
        Input("user-is-active-switch", "value"),
        # ... resto de inputs (13 total en este caso)
        Input("pharmacy-name-input", "value"),
        Input("pharmacy-erp-dropdown", "value"),
    ],
    prevent_initial_call=True  # CRITICAL: No ejecutar en page load
)
def save_form_data_on_change(
    email, password, rol, is_active,
    pharmacy_name, erp_type
):
    """
    Auto-save: Guarda TODOS los valores del formulario en el store.

    Se ejecuta cada vez que cambia cualquier input del formulario.
    Preserva datos al navegar entre tabs.

    Note:
        - prevent_initial_call=True: Evita ejecución en page load
        - Todos los parámetros pueden ser None (inputs no inicializados)
    """
    # Guardar estructura completa con valores actuales
    return {
        "email": email,
        "password": password,
        "rol": rol,
        "is_active": is_active,
        "pharmacy_name": pharmacy_name,
        "erp_type": erp_type,
        # ... resto de campos
    }
```

**Características clave**:
- ✅ Un callback, múltiples Inputs (cumple REGLA #11)
- ✅ Se ejecuta en CADA cambio de cualquier input
- ✅ Preserva estructura completa del formulario
- ✅ `prevent_initial_call=True` evita ejecución innecesaria

#### 3. Populate Callback con Fallback Priority

```python
@app.callback(
    [
        Output("user-email-input", "value"),
        Output("user-password-input", "value"),
        Output("user-rol-dropdown", "value"),
        # ... resto de outputs
    ],
    Input("admin-selected-user-store", "data"),  # Trigger: usuario seleccionado
    State("admin-user-form-data-store", "data"),  # Estado: datos guardados
)
def populate_user_form(selected_user, saved_form_data):
    """
    Populate: Restaura valores del formulario con prioridad:
    1. saved_form_data (datos guardados en auto-save)
    2. selected_user (usuario seleccionado para editar)
    3. defaults (valores por defecto)

    Note:
        Prioridad asegura que datos ingresados NO se pierdan
        al cambiar entre tabs o seleccionar otro usuario.
    """
    # PRIORITY #1: Datos guardados en el store (auto-save)
    if saved_form_data:
        return (
            saved_form_data.get("email", ""),
            saved_form_data.get("password", ""),
            saved_form_data.get("rol", "user"),
            # ... resto de campos desde saved_form_data
        )

    # PRIORITY #2: Usuario seleccionado para editar
    if selected_user:
        return (
            selected_user.get("email", ""),
            "",  # Password nunca se pre-llena
            selected_user.get("role", "user"),
            # ... resto de campos desde selected_user
        )

    # PRIORITY #3: Valores por defecto (modo creación)
    return (
        "",  # email
        "",  # password
        "user",  # rol por defecto
        # ... resto de defaults
    )
```

**Fallback Priority Cascade**:
```
saved_form_data (auto-save)
    ↓ (si None)
selected_user (usuario seleccionado)
    ↓ (si None)
defaults (valores por defecto)
```

#### 4. Save Callback Simplificado

```python
@app.callback(
    [
        Output("admin-user-form-data-store", "data", allow_duplicate=True),
        Output("admin-users-data-store", "data", allow_duplicate=True),
        # ... otros outputs
    ],
    Input("save-user-btn", "n_clicks"),
    [
        State("admin-user-form-data-store", "data"),  # ← ÚNICO State necesario
        State("auth-state", "data"),
        State("admin-user-form-mode-store", "data"),  # "create" o "edit"
    ],
    prevent_initial_call=True
)
def save_user(n_clicks, form_data, auth_state, mode):
    """
    Save: Lee del store en lugar de 13 States individuales.

    Simplificación:
        ANTES: 13 States (email, password, rol, ...)
        AHORA: 1 State (form_data con todo)
    """
    if not n_clicks or not form_data:
        raise PreventUpdate

    # Verificación auth
    from utils.auth_helpers import is_user_authenticated
    if not is_user_authenticated(auth_state):
        raise PreventUpdate

    # Generar username desde email (si necesario)
    username = form_data.get("email", "").split("@")[0] if form_data.get("email") else ""

    # Construir payload para backend
    payload = {
        "email": form_data.get("email"),
        "username": username,  # Auto-generado
        "password": form_data.get("password"),
        "role": form_data.get("rol", "user"),
        # ... resto de campos desde form_data
    }

    # API call
    if mode == "create":
        response = requests.post(f"{BACKEND_URL}/api/v1/admin/users", ...)
    else:
        response = requests.put(f"{BACKEND_URL}/api/v1/admin/users/{user_id}", ...)

    if response.status_code in [200, 201]:
        # Limpiar store después de guardar exitosamente
        return (
            None,  # Limpiar form_data store
            {"reload": True},  # Trigger reload de tabla usuarios
            # ... otros outputs
        )
    else:
        # Mantener datos en store si falla (usuario puede reintentar)
        return (
            no_update,  # NO limpiar store
            no_update,
            # ...
        )
```

**Simplificación**:
- ✅ **ANTES**: 13 States individuales → 13 parámetros en función
- ✅ **AHORA**: 1 State (form_data) → 1 parámetro + dict access

#### 5. Store Cleanup Strategy

```python
# A. Limpiar al abrir modal de creación
@app.callback(
    Output("admin-user-form-data-store", "data"),
    Input("create-user-btn", "n_clicks"),
    prevent_initial_call=True
)
def open_create_user_modal(n_clicks):
    """Limpiar store al abrir modal de creación."""
    if not n_clicks:
        raise PreventUpdate
    return None  # Limpiar store

# B. Limpiar al cancelar modal
@app.callback(
    Output("admin-user-form-data-store", "data", allow_duplicate=True),
    Input("cancel-user-modal-btn", "n_clicks"),
    prevent_initial_call=True
)
def cancel_user_modal(n_clicks):
    """Limpiar store al cancelar modal."""
    if not n_clicks:
        raise PreventUpdate
    return None  # Limpiar store

# C. Limpiar al guardar exitosamente (ver callback save_user arriba)
```

**Estrategia de limpieza**:
- ✅ **Modal Open**: Evita datos residuales de operaciones previas
- ✅ **Modal Cancel**: Evita datos huérfanos
- ✅ **Save Success**: Previene re-submit accidental

### Ventajas del Patrón

| Aspecto | Antes (Inputs Dinámicos) | Después (Auto-Save) |
|---------|--------------------------|---------------------|
| **Persistencia** | ❌ Datos se pierden al cambiar tab | ✅ Datos persisten entre tabs |
| **Complejidad Save** | ❌ 13 States individuales | ✅ 1 State con dict |
| **Race Conditions** | ❌ Inputs pueden no existir | ✅ Store siempre disponible |
| **UX** | ❌ Usuario re-ingresa datos | ✅ Flujo natural sin re-ingreso |
| **Código** | ❌ Callbacks complejos | ✅ Callbacks simplificados |

### Cuándo Usar Este Patrón

**✅ Usar auto-save pattern cuando:**
- Formularios multi-tab con inputs dinámicos
- Riesgo de pérdida de datos al navegar
- Muchos campos (>5) que requieren coordinación
- UX crítico para usuarios (evitar re-ingreso)

**❌ NO usar cuando:**
- Formulario simple de 1-2 campos (overhead innecesario)
- Datos sensibles que NO deben persistir (usar `prevent_initial_call` + cleanup agresivo)
- Performance crítico con >50 campos (considerar debounce)

### Troubleshooting

**Problema**: Store no limpia después de guardar
```python
# ❌ INCORRECTO: Olvidar limpiar en success path
if response.status_code == 201:
    return {"reload": True}, no_update  # Store NO limpiado

# ✅ CORRECTO: Limpiar explícitamente
if response.status_code == 201:
    return {"reload": True}, None  # Store limpiado
```

**Problema**: Datos se pierden después de F5
```python
# Usar storage_type="session" si necesitas persistencia con F5
dcc.Store(id="form-store", storage_type="session")
```

**Problema**: Auto-save se ejecuta en page load
```python
# SIEMPRE usar prevent_initial_call=True en auto-save
@app.callback(..., prevent_initial_call=True)
```

### Referencias

- **Implementación completa**: `frontend/callbacks/admin/users.py` (Issue #417)
- **Tests**: `frontend/tests/callbacks/admin/test_users_multitab_persistence.py` (18 tests)
- **Skeleton pattern**: `frontend/CLAUDE.md` REGLA #0.5
- **Storage types**: [Dash dcc.Store docs](https://dash.plotly.com/dash-core-components/store)

---

## 🔄 Patrón: Filtros Múltiples Integrados (Issue #415)

### Problema
Múltiples filtros (fecha, empleados, partners) deben actualizar el mismo análisis de forma coordinada.

### Solución: Apply Button + State Pattern

```python
@app.callback(
    Output("analysis-store", "data"),
    [
        Input("partners-selection-store", "data"),  # Trigger automático
        Input("url", "pathname"),                   # Trigger navegación
        Input("discount-slider", "value"),          # Trigger slider
        Input("apply-btn", "n_clicks"),             # Trigger manual botón
    ],
    [
        State("date-range", "start_date"),          # Filtro fecha (State)
        State("date-range", "end_date"),
        State("employee-filter", "value"),          # Filtro empleados (State)
        State("auth-state", "data"),
    ],
    prevent_initial_call=False,
)
def perform_analysis(partners, pathname, discount, apply_clicks, start_date, end_date, employees, auth_state):
    """
    Callback con múltiples triggers + filtros como State.

    Diseño:
    - Inputs: Disparan automáticamente (partners, slider, navegación, botón)
    - States: Se leen pero NO disparan (fechas, empleados)
    - El botón "Aplicar" permite aplicar filtros manualmente
    """
    # Guard: Solo ejecutar en página correcta + autenticado
    if pathname != "/mi-pagina" or not is_user_authenticated(auth_state):
        raise PreventUpdate

    # Calcular período desde filtros de fecha
    period_months = 12  # Default
    if start_date and end_date:
        delta = (end_date - start_date).days
        period_months = max(1, min(24, delta // 30))

    # Construir payload con todos los filtros
    payload = {
        "partners": partners,
        "period_months": period_months,
        "discount": discount,
        "employee_ids": employees or [],
    }

    return api_client.post("/api/v1/analysis/...", data=payload)
```

### Cuándo Usar

| Escenario | Patrón Recomendado |
|-----------|-------------------|
| Filtro cambia → análisis actualiza inmediatamente | Input |
| Filtro cambia → usuario confirma con botón | State + Apply Button |
| Múltiples filtros dependientes | State para todos + Apply Button |
| Filtro costoso (requiere API call) | State + Apply Button |

### Referencias

- **Implementación**: `frontend/callbacks/generics.py` - `perform_dynamic_analysis()`
- **Issue**: #415 (Mejora página /generics)
- **Componentes filtros**: `frontend/components/filters/`

---

## 📊 Patrón: Measures API (Reference Implementation - Issue #471)

El tab de Inventario en `/ventalibre` y `/prescription` es la **reference implementation** del patrón Measures API (REGLA #19).

### Arquitectura

```
Backend (Measures)                    Frontend (Callbacks)
┌─────────────────────┐              ┌─────────────────────────┐
│ backend/app/        │              │ frontend/callbacks/     │
│   measures/         │   API Calls  │   inventory/            │
│   ├── base.py       │◄────────────│   ├── kpis.py (POST)   │
│   ├── inventory_*   │   GET/POST  │   ├── abc.py (GET)     │
│   ├── rotation_*    │   /calculate│   └── table.py (GET)   │
│   └── profitability_│             │                         │
└─────────────────────┘              └─────────────────────────┘
                                              │
                                              ▼
                                     ┌─────────────────────────┐
                                     │ frontend/components/    │
                                     │   inventory/            │
                                     │   ├── kpi_cards.py     │
                                     │   ├── abc_chart.py     │
                                     │   ├── rotation_table.py│
                                     │   └── tab_layout.py    │
                                     └─────────────────────────┘
```

### Callbacks Consumiendo Measures API

**1. Múltiples medidas en una llamada (KPIs)**:
```python
# frontend/callbacks/inventory/kpis.py

@app.callback(
    [Output(f"{id_prefix}-inv-kpi-stock-value", "children"), ...],
    [Input("url", "pathname"), Input(f"{id_prefix}-tabs", "active_tab")],
    [State("auth-state", "data"), State(f"{id_prefix}-inv-product-type", "data")],
    prevent_initial_call=True,
)
def update_inventory_kpis(pathname, active_tab, auth_state, product_type):
    # Guard auth (REGLA #7: Security First)
    if not is_user_authenticated(auth_state):
        raise PreventUpdate

    # Solo cargar cuando tab activo
    if active_tab != "tab-inventario":
        raise PreventUpdate

    # ✅ CORRECTO: POST /calculate/multiple para múltiples KPIs
    response = request_coordinator.make_request(
        "POST",
        f"{BACKEND_URL}/api/v1/measures/calculate/multiple",
        json={
            "measure_names": ["stock_value", "rotation_index", "gmroi", "dead_stock_value"],
            "filters": {"pharmacy_id": str(pharmacy_id)},
        },
        timeout=30,
    )

    data = response.json()
    results = data.get("results", {})
    # Extraer y formatear cada medida...
```

**2. Una medida específica (ABC Classification)**:
```python
# frontend/callbacks/inventory/abc.py

# ✅ CORRECTO: GET /calculate/{measure_name} para una medida
response = request_coordinator.make_request(
    "GET",
    f"{BACKEND_URL}/api/v1/measures/calculate/abc_classification",
    params={"pharmacy_id": str(pharmacy_id)},
    timeout=30,
)

data = response.json()
result = data.get("value", {})  # Nota: "value" no "result"
```

### Endpoints de Measures API

| Endpoint | Método | Uso |
|----------|--------|-----|
| `/api/v1/measures/` | GET | Lista todas las medidas disponibles |
| `/api/v1/measures/calculate/{name}` | GET | Calcular una medida con query params |
| `/api/v1/measures/calculate/multiple` | POST | Calcular múltiples medidas con body JSON |
| `/api/v1/measures/dashboard/{pharmacy_id}` | GET | Dashboard predefinido (KPIs principales) |

### Medidas Disponibles (Issue #470)

| Categoría | Medida | Descripción |
|-----------|--------|-------------|
| **Inventario** | `stock_value` | Valor total del stock a coste |
| **Inventario** | `dead_stock_value` | Stock sin venta >180 días |
| **Inventario** | `stock_coverage_days` | Días de cobertura |
| **Rotación** | `rotation_index` | Índice anual (veces/año) |
| **Rotación** | `days_inventory` | Días promedio en inventario |
| **Rotación** | `abc_classification` | Clasificación Pareto A/B/C |
| **Rentabilidad** | `roi_product` | ROI por producto (%) |
| **Rentabilidad** | `gmroi` | Gross Margin Return on Investment |
| **Rentabilidad** | `contribution_margin` | Margen de contribución (%) |

### Componentes Reutilizables

El tab de inventario demuestra componentes reutilizables con prefijo parametrizable:

```python
# frontend/layouts/ventalibre.py

from components.inventory import create_inventory_tab

# Tab 3: Inventario (reutilizable)
dbc.Tab(
    label="Inventario",
    tab_id="tab-inventario",
    children=create_inventory_tab(
        id_prefix="ventalibre",       # Prefijo único para IDs
        product_type="venta_libre",   # Filtro de tipo de producto
        title="Inventario Parafarmacia",
        description="Análisis de rotación y rentabilidad...",
    ),
),
```

Los callbacks se registran para ambos prefijos (`prescription`, `ventalibre`):
```python
# frontend/callbacks/inventory/__init__.py

def register_inventory_callbacks(app):
    register_inventory_kpis_callbacks(app)
    register_inventory_abc_callbacks(app)
    register_inventory_table_callbacks(app)
```

### Cuándo Usar Este Patrón

| Escenario | Patrón Recomendado |
|-----------|-------------------|
| Nuevo dashboard con KPIs | Usar Measures API + este patrón |
| Cálculos ad-hoc en callbacks | ❌ Migrar a medida en backend |
| Múltiples KPIs con mismos filtros | POST `/calculate/multiple` |
| Una medida específica | GET `/calculate/{measure_name}` |

### Referencias

- **ADR**: `docs/architecture/ADR-003-measures-filters-powerbi-pattern.md`
- **Backend Measures**: `backend/app/measures/`
- **Componentes**: `frontend/components/inventory/`
- **Callbacks**: `frontend/callbacks/inventory/`
- **Issue**: #471 (Dashboard Rentabilidad/Rotación)

---

## Referencias
- [Dash Callback Documentation](https://dash.plotly.com/duplicate-callback-outputs)
- [Dash Pattern Matching Callbacks](https://dash.plotly.com/pattern-matching-callbacks)
- **xFarma Callback Helpers**: `frontend/utils/callback_helpers.py`
