# API Endpoints Documentation

Complete reference of backend API endpoints for xFarma frontend developers.
Last updated: 2026-01-07 (Issue #491 - VentaLibre Time Series/YoY/Contributors)

**Total endpoints: 192** (NEW: 3 evolution endpoints - Issue #491)

**Actualización importante (PR #107)**: Sistema de roles simplificado a solo `admin` y `user`. Registro público eliminado.

## 🔒 Security & Rate Limiting

### Rate Limits
- **Default**: 100 requests per minute per IP
- **Upload endpoints**: 10 requests per minute
- **Data-intensive endpoints**: 30 requests per minute
- **Bulk operations**: 5 concurrent requests maximum

### File Upload Security
- **Max file size**: 100MB
- **Allowed types**: `.csv`, `.txt`, `.xls`, `.xlsx`
- **Validation**: File type verification, virus scanning
- **Storage**: Temporary with automatic cleanup after processing

## 🚨 Error Response Format

All endpoints return consistent error responses:

```json
{
  "error": {
    "code": 400,
    "message": "Descripción general del error",
    "details": "Información adicional o campos específicos",
    "timestamp": "2024-01-15T10:30:00Z"
  }
}
```

### Common Error Codes
- `400 Bad Request`: Parámetros inválidos o faltantes
- `401 Unauthorized`: Token JWT inválido o expirado
- `403 Forbidden`: Sin permisos para la operación
- `404 Not Found`: Recurso no encontrado
- `409 Conflict`: Conflicto con el estado actual
- `422 Unprocessable Entity`: Validación de datos fallida
- `429 Too Many Requests`: Límite de tasa excedido
- `500 Internal Server Error`: Error del servidor
- `503 Service Unavailable`: Servicio temporalmente no disponible

### Validation Error Format
```json
{
  "error": {
    "code": 422,
    "message": "Error de validación",
    "details": [
      {
        "field": "start_date",
        "message": "La fecha debe estar en formato YYYY-MM-DD",
        "value": "2024-13-45"
      },
      {
        "field": "limit",
        "message": "El límite debe ser entre 1 y 1000",
        "value": 5000
      }
    ]
  }
}
```

## 🚀 Performance Optimizations (January 2025)

### Recent Improvements
- **Cache Implementation**: 30-second cache for `/api/system/status` endpoint
- **Reduced Polling**: Frontend intervals optimized (5s → 30s) reducing server load by 85%
- **New Lightweight Endpoint**: `/health/simple` for fast connectivity checks (~45ms)
- **Retry Logic**: Automatic retry with exponential backoff in API client
- **Query Optimization**: Indexed database queries for faster response times

### Recommended Intervals
- Health checks: 30 seconds
- System status: 30 seconds
- Catalog updates: 15 seconds during sync
- Dashboard refresh: 30 seconds

## 📋 Table of Contents

<details>
<summary><strong>Click to expand/collapse TOC</strong></summary>

### Core APIs
1. [Health & Monitoring](#1-health--monitoring) - `/health`, `/metrics`
2. [System Status & Sync](#2-system-status--sync) - Estado del sistema, sincronizaciones
3. [File Upload](#3-file-upload) - Upload de archivos ERP con progreso
4. [Re-enrichment](#4-re-enrichment) - Re-enriquecimiento de datos

### Analytics & Analysis
5. [Sales Analysis](#5-sales-analysis) - Análisis de ventas
6. [Measures (Power BI Style)](#6-measures-power-bi-style) - Métricas calculadas
7. [Generic Analysis](#7-generic-analysis) - Análisis de genéricos
8. [Pharmacy Partners](#8-pharmacy-partners) ⭐ **ACTUALIZADO** - Gestión con auto-inicialización
9. [Partner Analysis (Redesign)](#9-partner-analysis-redesign) ⭐ - Análisis avanzado partners
10. [Laboratory Mapping](#10-laboratory-mapping) ⭐ - Mapeo de laboratorios

### Management
11. [Pharmacy Management](#11-pharmacy-management) - Gestión de farmacias
    - 11.1 [Pharmacy Employees](#111-pharmacy-employees-new---issue-402--pro-feature) 🆕 🔒 **PRO Feature** - Employee filtering (Issue #402)
12. [Generic Opportunities](#12-generic-opportunities) - Oportunidades de genéricos

### Security & Admin
13. [Authentication](#13-authentication) ⚠️ **ACTUALIZADO** - Login, JWT, OAuth
    - 13.1 [Invitaciones (Sistema de Onboarding)](#131-invitaciones-sistema-de-onboarding) 🆕 - Sistema de invitaciones admin
    - 13.2 [Sistema de Roles y Permisos](#132-sistema-de-roles-y-permisos-simplificado) 🆕 - admin/user roles
14. [Admin Operations](#14-admin-operations) - Operaciones administrativas
15. [System Unified](#15-system-unified-new) 🆕 - APIs unificadas de sistema
16. [Admin Tools](#16-admin-tools-new) 🆕 - Herramientas admin centralizadas
17. [Admin - User Management](#17-admin---user-management-new) 🆕 **Issue #348 FASE 2** - Gestión de usuarios y farmacias
18. [Admin - Database Management](#18-admin---database-management-new) 🆕 **Issue #348 FASE 2** - Backups y administración de BD

### OTC Taxonomy
22. [Intercambiable Groups](#22-intercambiable-groups-new---issue-446) 🆕 **Issue #446** - Grupos intercambiables de venta libre

</details>

### 🔗 Quick Links
- [Most Used Endpoints](#-quick-reference)
- [Authentication Flow](#authentication-flow)
- [Error Response Format](#-error-response-format)
- [Changelog](#-changelog-de-documentación)

---

## 1. Health & Monitoring

### GET /health/simple ⭐
**Lightweight health check** - optimized for frequent monitoring.

**Response Time:** ~40-50ms (only DB connectivity check)

**Response:**
```json
{
  "status": "healthy",
  "service": "xfarma-backend"
}
```

### GET /health
Complete health check with external services status.

**Response Time:** ~200-500ms (includes external service checks)

**Response:**
```json
{
  "status": "healthy",
  "database": "healthy",
  "redis": "not_implemented",
  "external_data": "healthy",
  "timezone": {
    "name": "Europe/Madrid",
    "offset": "+01:00",
    "is_dst": false
  },
  "server_time_madrid": "2025-01-15T12:30:00+01:00"
}
```

### GET /metrics ⭐ NEW
**Prometheus metrics endpoint** - Observabilidad para monitoreo externo (Issue #114 Fase 3).

**Purpose:** Expone métricas del sistema en formato Prometheus para Grafana/monitoreo externo.

**Authentication:** None (público para scrapers de Prometheus)

**Rate Limit:** 60 requests/min (protección contra abuso)

**Response Format:** Prometheus text format
```
# HELP cima_stalls_total Total number of CIMA sync stalls detected
# TYPE cima_stalls_total counter
cima_stalls_total{environment="production",is_render="true"} 3

# HELP cima_recovery_attempts_total Total number of CIMA sync recovery attempts
# TYPE cima_recovery_attempts_total counter
cima_recovery_attempts_total{environment="production",is_render="true",success="true"} 5
cima_recovery_attempts_total{environment="production",is_render="true",success="false"} 1

# HELP cima_heartbeat_age_seconds Age of the last CIMA sync heartbeat in seconds
# TYPE cima_heartbeat_age_seconds gauge
cima_heartbeat_age_seconds{environment="production",is_render="true"} 45.3

# HELP cima_recovery_duration_seconds Duration of CIMA sync recovery operations
# TYPE cima_recovery_duration_seconds histogram
cima_recovery_duration_seconds_bucket{environment="production",is_render="true",le="1"} 2
cima_recovery_duration_seconds_bucket{environment="production",is_render="true",le="5"} 4
cima_recovery_duration_seconds_bucket{environment="production",is_render="true",le="10"} 5
cima_recovery_duration_seconds_count{environment="production",is_render="true"} 5
cima_recovery_duration_seconds_sum{environment="production",is_render="true"} 23.4

# HELP cima_circuit_breaker_state Current state of the CIMA circuit breaker
# TYPE cima_circuit_breaker_state gauge
cima_circuit_breaker_state{environment="production",is_render="true"} 0
```

**Métricas disponibles:**
- `cima_stalls_total`: Total de stalls detectados en sincronización CIMA
- `cima_recovery_attempts_total{success}`: Intentos de recovery (exitosos/fallidos)
- `cima_heartbeat_age_seconds`: Edad del último heartbeat (segundos)
- `cima_recovery_duration_seconds`: Duración de operaciones de recovery (histogram)
- `cima_circuit_breaker_state`: Estado del circuit breaker (0=CLOSED, 1=OPEN, 2=HALF_OPEN)
- `rate_limit_rejections_total`: Rechazos por límite de tasa
- `active_requests_gauge`: Requests activos en el sistema

**Use Cases:**
- Dashboards Grafana para monitoreo CIMA sync
- Alertas de Prometheus para stalls prolongados
- Análisis histórico de performance del sistema
- Debugging de resiliencia CIMA (circuit breaker patterns)

**Ejemplo de uso con curl:**
```bash
curl http://localhost:8000/metrics
```

---

## 2. System Status & Sync

### GET /api/system/status ⭐
**Endpoint unificado** que combina toda la información del sistema: estado, inicialización, catálogo y salud.

**Performance:** Implementa cache de 30 segundos para reducir carga en base de datos.

**Query Parameters:**
- `include_details` (bool): Incluir información detallada. Default: true

**Response:**
```json
{
  "status": "ready",
  "message": "Sistema completamente operativo",
  "progress": 100,
  "timestamp": "2024-01-15T10:30:00Z",
  
  "initialization": {
    "required": false,
    "overall_status": "ready",
    "overall_progress": 100,
    "overall_message": "Sistema completamente inicializado",
    "catalog_empty": false,
    "auto_initialize": false,
    "components": {
      "CATALOG": {
        "status": "ready",
        "progress": 100,
        "message": "Catálogo actualizado"
      },
      "CIMA": {
        "status": "ready",
        "progress": 100,
        "message": "CIMA sincronizado"
      }
    }
  },
  
  "catalog": {
    "total_products": 25000,
    "with_cima": 20000,
    "with_nomenclator": 15000,
    "distribution": {
      "only_cima": 10000,
      "only_nomenclator": 5000,
      "both_sources": 10000,
      "no_sources": 0
    },
    "sync_status": {
      "SINCRONIZADO": 24000,
      "BAJA": 1000
    },
    "coverage_percentage": 92.5
  },
  
  "sync": {
    "last_updates": {
      "cima": {
        "last_sync": "2024-01-15T08:00:00Z",
        "status": "ready"
      },
      "nomenclator": {
        "last_sync": "2024-01-15T08:00:00Z",
        "status": "ready"
      }
    },
    "in_progress": false,
    "available_sources": ["cima", "nomenclator"]
  },
  
  "health": {
    "database": true,
    "catalog": true,
    "redis": false
  },
  
  "actions_available": {
    "initialize_catalog": false,
    "sync_nomenclator": true,
    "sync_cima": true,
    "calculate_groups": true
  }
}
```

### POST /api/system/sync
**Endpoint unificado** para toda sincronización del sistema.

**Authentication**: Requires admin user (JWT token with admin role)
**Rate Limit**: 3 requests/hour

**⚠️ IMPORTANTE - Comportamiento 100% Síncrono** (actualizado 2025-11-02):

| Target | Status en Response | Duración Típica | Notas |
|--------|-------------------|-----------------|-------|
| `nomenclator` | `completed` | 30-60s | Solo nomenclator (67k productos) |
| `cima` | `completed` | 90-150s | Solo CIMA con chunking (render: 300, local: 1000) |
| `all` | `completed` | 120-210s | Nomenclator + CIMA secuencial |

**Cambio importante**: Desde 2025-11-02, **TODAS** las sincronizaciones ejecutan **síncronamente** (no background tasks).
- ✅ **Garantiza ejecución**: No se pierden en Render con múltiples workers
- ✅ **Logs visibles**: Duración y progreso en logs de Render inmediatamente
- ✅ **UI bloqueante**: Frontend ya tiene spinner, usuario espera respuesta
- ⚠️ **Timeouts**: Gunicorn configurado con 180s timeout (suficiente para < 210s típicos)

**Request:**
```json
{
  "target": "all",  // "all", "catalog", "cima", "nomenclator"
  "force": false    // Forzar sync aunque no sea necesaria
}
```

**Response (target="nomenclator"):**
```json
{
  "status": "completed",
  "sync_id": "sync_20241102_103000",
  "target": "nomenclator",
  "components": ["NOMENCLATOR"],
  "message": "Sincronización de nomenclator completada",
  "result": {
    "products_synced": 67000,
    "errors": 0
  },
  "duration_seconds": 45.23
}
```

**Response (target="cima"):**
```json
{
  "status": "completed",
  "sync_id": "sync_20241102_103000",
  "target": "cima",
  "components": ["CIMA"],
  "message": "Sincronización de CIMA completada",
  "result": {
    "products_synced": 67000,
    "chunks_processed": 224,
    "errors": 0
  },
  "duration_seconds": 135.78
}
```

**Response (target="all"):**
```json
{
  "status": "completed",
  "sync_id": "sync_20241102_103000",
  "target": "all",
  "components": ["NOMENCLATOR", "CIMA"],
  "message": "Sincronización completa (nomenclator + CIMA) completada",
  "result": {
    "nomenclator": {
      "products_synced": 67000,
      "errors": 0
    },
    "cima": {
      "products_synced": 67000,
      "chunks_processed": 224,
      "errors": 0
    }
  },
  "duration_seconds": 181.01,
  "step_durations": {
    "nomenclator": 45.23,
    "cima": 135.78
  }
}
```

**Razón del cambio (commits ce85a08, 2025-11-02)**:
- Background tasks pueden perderse en Render con múltiples workers
- Sincronización síncrona garantiza ejecución completa antes de retornar
- UI ya tiene spinner, usuario espera comportamiento bloqueante
- Logs de duración permiten monitorear si se acercan al timeout (180s)

### GET /api/system/sync/{sync_id}
Obtener estado de una sincronización específica.

**Response:**
```json
{
  "sync_id": "sync_20240115_103000",
  "status": "in_progress",
  "overall_progress": 65,
  "components": [
    {
      "component": "CIMA",
      "status": "initializing",
      "progress": 45,
      "message": "Descargando productos..."
    }
  ],
  "timestamp": "2024-01-15T10:35:00Z"
}
```

### POST /api/system/sync/cancel
**🔧 NUEVO** - Cancelar sincronizaciones activas o enganchadas.

**Query Parameters:**
- `component` (optional): Componente específico a cancelar (cima, nomenclator, catalog)
- `force` (bool): Forzar cancelación incluso si está progresando activamente

**Request:** (Sin body)

**Response:**
```json
{
  "status": "cancelled",
  "cancelled_count": 1,
  "cancelled_components": [
    {
      "component": "catalog",
      "was_progressing": false,
      "runtime_minutes": 65
    }
  ],
  "force": false,
  "timestamp": "2024-01-15T10:35:00Z"
}
```

### GET /api/system/sync/cleanup
**🔧 NUEVO** - Limpieza automática de sincronizaciones enganchadas.

**Query Parameters:**
- `timeout_minutes` (int): Timeout en minutos para considerar una sincronización enganchada (default: 60, min: 5, max: 480)

**Response:**
```json
{
  "cleaned_count": 1,
  "cleaned_components": [
    {
      "component": "catalog",
      "started_at": "2024-01-15T09:30:00Z",
      "runtime_minutes": 65
    }
  ],
  "timeout_minutes": 60,
  "timestamp": "2024-01-15T10:35:00Z"
}
```

**⚠️ Gestión Automática de Timeouts**
- El sistema ejecuta limpieza automática cada 30 minutos
- Timeout por defecto: 60 minutos
- Sincronizaciones enganchadas se marcan como `ERROR` automáticamente
- Logs detallados para debugging y auditoría

### GET /api/v1/system/health/watchdog ⭐
**🔧 NUEVO (Issue #114)** - Health check mejorado del monitor CIMA con circuit breaker y resiliencia.

**NOTA**: El endpoint mantiene el nombre "watchdog" por compatibilidad histórica, pero usa internamente el servicio **CimaSyncMonitor** (Issue #109). El término "watchdog" es legacy - el monitor actual implementa circuit breaker, heartbeat tracking, y auto-recovery.

**Purpose**: Endpoint especializado para monitorear la salud del sistema de sincronización CIMA con información detallada sobre circuit breaker, heartbeat tracking y auto-recovery.

**Query Parameters**: None

**Response:**
```json
{
  "status": "healthy",
  "monitoring_active": true,
  "redis_available": true,
  "circuit_breaker_state": "CLOSED",
  "recovery_attempts_today": 2,
  "last_recovery_successful": true,
  "heartbeat": {
    "phase": "processing",
    "processed": 25000,
    "total": 67000,
    "progress_percentage": 37.31,
    "timestamp": "2025-10-04T11:30:00Z"
  },
  "time_since_heartbeat_seconds": 15,
  "is_stalled": false,
  "circuit_breaker": {
    "state": "CLOSED",
    "failure_count": 0,
    "success_count": 10,
    "failure_threshold": 5,
    "can_attempt": true,
    "time_in_current_state_seconds": 3600
  },
  "heartbeat_tracker": {
    "redis_available": true,
    "has_heartbeat": true,
    "storage_backend": "redis"
  },
  "auto_recovery_enabled": true,
  "last_check": "2025-10-04T11:30:00Z"
}
```

**Status Values:**
- `healthy`: Sistema operando normalmente, heartbeat reciente, circuit breaker cerrado
- `degraded`: Heartbeat antiguo (>10 min), recovery attempts recientes, o circuit breaker en HALF_OPEN
- `unhealthy`: Monitoreo inactivo, circuit breaker abierto, o stall detectado

**Circuit Breaker States:**
- `CLOSED`: Funcionamiento normal, permitiendo operaciones
- `OPEN`: Múltiples fallos detectados, bloqueando operaciones temporalmente
- `HALF_OPEN`: En período de prueba después de timeout

**Use Cases:**
- Dashboard de salud del sistema
- Alertas automáticas de degradación
- Debugging de problemas de sincronización CIMA
- Monitoreo de resiliencia del sistema

**Example Usage:**
```javascript
// Frontend monitoring dashboard
const watchdogHealth = await apiClient.get('/api/v1/system/health/watchdog');

if (watchdogHealth.status === 'unhealthy') {
  showAlert('Sistema de sincronización CIMA degradado');
} else if (watchdogHealth.circuit_breaker_state === 'OPEN') {
  showWarning('Circuit breaker activo - auto-recovery en progreso');
}
```

### GET /api/v1/system/cache/stats ⭐
**🔧 NUEVO (Issue #194)** - Redis cache performance statistics for monitoring.

**Purpose**: Monitor enrichment cache performance including hit rates, total requests, and error counts for capacity planning and debugging.

**Query Parameters**: None

**Authentication**: None (public monitoring endpoint)

**Response:**
```json
{
  "status": "success",
  "cache_stats": {
    "enabled": true,
    "hits": 15234,
    "misses": 2156,
    "errors": 12,
    "invalidations": 45,
    "hit_rate": 87.59,
    "total_requests": 17390
  },
  "health": {
    "redis_available": true,
    "cache_enabled": true
  },
  "timestamp": "2025-10-14T10:30:00Z"
}
```

**Response (Redis unavailable):**
```json
{
  "status": "error",
  "cache_stats": {
    "enabled": false,
    "hits": 0,
    "misses": 0,
    "errors": 0,
    "invalidations": 0,
    "hit_rate": 0,
    "total_requests": 0
  },
  "health": {
    "redis_available": false,
    "cache_enabled": false
  },
  "error": "Redis connection failed",
  "timestamp": "2025-10-14T10:30:00Z"
}
```

**Metrics Explained:**
- `hits`: Number of successful cache lookups (data found in cache)
- `misses`: Number of cache misses (data not in cache, fetched from DB)
- `errors`: Number of cache operation errors (connection failures, timeouts)
- `invalidations`: Number of cache keys explicitly invalidated
- `hit_rate`: Percentage of requests served from cache (hits / total_requests × 100)
- `total_requests`: Total cache operations (hits + misses)

**Use Cases:**
- Monitor cache effectiveness (target hit rate: >80%)
- Capacity planning (identify need for cache size increase)
- Debug performance issues (high miss rate indicates poor caching strategy)
- Alert on cache unavailability (errors > threshold)

**Performance Targets:**
- Hit rate: >80% for optimal performance
- Errors: <1% of total requests
- Response time: <50ms

**Example Usage:**
```javascript
// Monitor cache health in admin dashboard
const cacheStats = await apiClient.get('/api/v1/system/cache/stats');

if (!cacheStats.health.redis_available) {
  showAlert('Redis cache no disponible - rendimiento degradado');
} else if (cacheStats.cache_stats.hit_rate < 80) {
  showWarning(`Cache hit rate bajo: ${cacheStats.cache_stats.hit_rate}%`);
} else {
  showSuccess(`Cache funcionando óptimamente: ${cacheStats.cache_stats.hit_rate}% hit rate`);
}

// Display metrics
console.log(`Cache hits: ${cacheStats.cache_stats.hits.toLocaleString()}`);
console.log(`Cache misses: ${cacheStats.cache_stats.misses.toLocaleString()}`);
console.log(`Total requests: ${cacheStats.cache_stats.total_requests.toLocaleString()}`);
```

---

## 2.5 ERP Sync Management (Pivot 2026) 🆕

Endpoints for managing ERP database synchronization in local pharmacy installations.

**Note**: These endpoints are for local single-tenant installations. No pharmacy_id required as each installation syncs its ONE pharmacy.

### GET /api/v1/erp-sync/status ⭐

Get current ERP sync status and configuration.

**Response:**
```json
{
  "status": "syncing",
  "sync_enabled": true,
  "last_sync_completed_at": "2026-01-10T15:30:00Z",
  "last_sale_timestamp": "2026-01-10T15:28:00Z",
  "last_sync_records": 142,
  "last_sync_duration_seconds": 5,
  "total_syncs": 48,
  "total_records_synced": 5840,
  "consecutive_errors": 0,
  "last_error": null,
  "erp_type": "farmanager",
  "erp_host": "localhost",
  "erp_database": "gesql",
  "sync_interval_minutes": 15
}
```

**Status Values:**
- `not_initialized`: First sync not yet run
- `idle`: Waiting for next scheduled sync
- `syncing`: Sync in progress
- `error`: Last sync failed (check `last_error`)

---

### POST /api/v1/erp-sync/run ⭐

Trigger manual ERP sync.

**Query Parameters:**
- `force_full` (bool, default: false): If true, do full sync ignoring last_sale_timestamp

**Response (Success):**
```json
{
  "success": true,
  "records_synced": 142,
  "records_skipped": 0,
  "duration_seconds": 5,
  "error": null,
  "skipped": false,
  "reason": null
}
```

**Response (Already Syncing):**
```json
{
  "success": false,
  "records_synced": 0,
  "records_skipped": 0,
  "duration_seconds": 0,
  "error": null,
  "skipped": true,
  "reason": "Sync already in progress"
}
```

**Error Codes:**
- 500: Sync failed (check `error` field for details)

**Frontend Usage (ERPStatusBanner retry button):**
```javascript
// Callback handle_erp_retry_sync in homepage.py
const response = await fetch('/api/v1/erp-sync/run', { method: 'POST' });
const result = await response.json();

if (result.success) {
  showSuccess(`Sincronizados ${result.records_synced} registros`);
} else if (result.skipped) {
  showWarning(result.reason);
} else {
  showError(`Error: ${result.error}`);
}
```

---

### POST /api/v1/erp-sync/run-async

Trigger ERP sync in background (non-blocking).

**Query Parameters:**
- `force_full` (bool, default: false): If true, do full sync

**Response:**
```json
{
  "status": "started",
  "message": "Sync started in background. Check /status for progress."
}
```

---

### POST /api/v1/erp-sync/reset-errors

Reset error state after manual intervention (e.g., fixing ERP connection).

**Use Case**: When ERP connection is fixed and you want to clear consecutive errors and re-enable automatic sync.

**Response:**
```json
{
  "success": true,
  "message": "Error state reset. Automatic sync re-enabled."
}
```

---

### POST /api/v1/erp-sync/enable

Enable automatic ERP sync.

**Response:**
```json
{
  "status": "ok",
  "message": "Sync enabled"
}
```

---

### POST /api/v1/erp-sync/disable

Disable automatic ERP sync.

**Response:**
```json
{
  "status": "ok",
  "message": "Sync disabled"
}
```

---

### GET /api/v1/erp-sync/config

Get ERP configuration (from environment). Password is hidden for security.

**Response:**
```json
{
  "erp_type": "farmanager",
  "host": "localhost",
  "port": 3306,
  "database": "gesql",
  "charset": "utf8",
  "sync_interval_minutes": 15,
  "initial_sync_days": 90,
  "batch_size": 500,
  "user": "kaifarma",
  "password_configured": true
}
```

---

## 3. File Upload

### POST /api/v1/upload/
Upload pharmacy sales file.

**Request:**
- Content-Type: `multipart/form-data`
- Body: `file` (binary)

**Query Parameters:**
- `validate_only` (bool): Solo validar sin procesar
- `force_format` (string): Forzar formato específico (farmatic, farmanager)

**Response:**
```json
{
  "file_id": "123e4567-e89b-12d3-a456-426614174000",
  "filename": "ventas_enero.csv",
  "status": "processing",
  "rows_processed": 0,
  "total_rows": 1500,
  "detected_format": "farmatic",
  "encoding": "latin-1",
  "file_size_mb": 2.5,
  "estimated_processing_time": 30
}
```

### POST /api/v1/upload/inventory 🆕
Upload pharmacy inventory file (Issue #476).

**Description:**
Carga archivos de inventario desde ERPs farmacéuticos. Similar al upload de ventas pero sin enriquecimiento CIMA.

**Request:**
- Content-Type: `multipart/form-data`
- Body: `file` (binary) - CSV exportado desde ERP

**Query Parameters:**
- `pharmacy_id` (string UUID, optional): ID de la farmacia. Auto-detectado si no se proporciona.
- `erp_type` (string, optional): Tipo de ERP - "farmanager" (default).
- `snapshot_date` (string, optional): Fecha del inventario en formato YYYY-MM-DD. Default: hoy.

**Auth**: Required (JWT Bearer Token)

**Response:**
```json
{
  "message": "Inventario en cola para procesamiento.",
  "upload_id": "123e4567-e89b-12d3-a456-426614174000",
  "status": "queued",
  "filename": "Informe_Articulos.csv",
  "file_type": "inventory_farmanager",
  "snapshot_date": "2024-12-25"
}
```

**Notes:**
- Detección de duplicados por (pharmacy_id, snapshot_date) - reemplaza si existe
- Datos guardados en tabla `inventory_snapshots`

### GET /api/v1/upload/status/{upload_id}
Get upload processing status with enrichment progress tracking.

**Purpose**: Obtener estado actual del procesamiento de un archivo de ventas subido, incluyendo progreso detallado del enriquecimiento automático con datos CIMA/nomenclator.

**Path Parameters:**
- `upload_id` (UUID): ID del archivo de ventas subido

**Response:**
```json
{
  "file_id": "123e4567-e89b-12d3-a456-426614174000",
  "filename": "ventas_enero_2024.csv",
  "status": "completed",
  "upload_date": "2024-10-14T10:30:00Z",
  "rows_processed": 10000,
  "total_rows": 10000,
  "processing_notes": "Procesados 10,000 de 10,000 registros. Enriquecimiento completado.",
  "errors": [],
  "warnings": ["5 productos no encontrados en catálogos"],

  "enrichment_status": "processing",
  "enrichment_progress": {
    "total": 10000,
    "processed": 4500,
    "enriched": 4000,
    "manual_review": 300,
    "failed": 200,
    "percentage": 45.0,
    "phase": "Matching por código nacional...",
    "matches_by_code": 3500,
    "matches_by_name": 300,
    "matches_by_ean": 200
  }
}
```

#### Campos de Progreso de Enriquecimiento (Fase 1 - Issue #252)

**`enrichment_status`** (string):
- Estado actual del proceso de enriquecimiento automático
- **Valores posibles:**
  - `"pending"`: Enriquecimiento no iniciado aún
  - `"processing"`: Enriquecimiento en proceso activo
  - `"completed"`: Enriquecimiento finalizado exitosamente
  - `"failed"`: Enriquecimiento falló

**`enrichment_progress`** (object | null):
- Estadísticas detalladas del enriquecimiento en tiempo real
- `null` si `enrichment_status == "pending"`
- **Campos disponibles:**
  - `total` (int): Total de registros de venta a enriquecer
  - `processed` (int): Registros ya procesados
  - `enriched` (int): Exitosamente enriquecidos con datos CIMA/nomenclator
  - `manual_review` (int): Productos que requieren revisión manual
  - `failed` (int): Registros que fallaron en el enriquecimiento
  - `percentage` (float): Porcentaje de completitud (0.0 - 100.0)
  - `phase` (string): Descripción legible de la fase actual
    - Ejemplos: `"Iniciando enriquecimiento..."`, `"Matching por código nacional..."`, `"Matching por EAN..."`, `"Matching por nombre (fuzzy)..."`, `"Finalizando..."`
  - `matches_by_code` (int): Coincidencias exitosas por código nacional
  - `matches_by_ean` (int): Coincidencias exitosas por código EAN
  - `matches_by_name` (int): Coincidencias exitosas por matching de nombre (fuzzy)

**Nota sobre `matches_by_*`:**
- La suma de `matches_by_code + matches_by_ean + matches_by_name` = `enriched`
- `manual_review` NO se cuenta en `enriched` (requiere intervención humana)
- `failed` son registros que fallaron completamente y no están en ninguna categoría

#### Ejemplos de Response por Estado

**Estado: Enriquecimiento Pendiente**
```json
{
  "upload_id": "123e4567-e89b-12d3-a456-426614174000",
  "filename": "ventas_enero_2024.csv",
  "status": "completed",
  "enrichment_status": "pending",
  "enrichment_progress": null
}
```

**Estado: Enriquecimiento en Proceso (45% completo)**
```json
{
  "upload_id": "123e4567-e89b-12d3-a456-426614174000",
  "filename": "ventas_enero_2024.csv",
  "status": "completed",
  "enrichment_status": "processing",
  "enrichment_progress": {
    "total": 10000,
    "processed": 4500,
    "enriched": 4000,
    "manual_review": 300,
    "failed": 200,
    "percentage": 45.0,
    "phase": "Matching por código nacional...",
    "matches_by_code": 3500,
    "matches_by_name": 300,
    "matches_by_ean": 200
  }
}
```

**Estado: Enriquecimiento Completado (100%)**
```json
{
  "upload_id": "123e4567-e89b-12d3-a456-426614174000",
  "filename": "ventas_enero_2024.csv",
  "status": "completed",
  "enrichment_status": "completed",
  "enrichment_progress": {
    "total": 10000,
    "processed": 10000,
    "enriched": 9500,
    "manual_review": 300,
    "failed": 200,
    "percentage": 100.0,
    "phase": "Completado",
    "matches_by_code": 8000,
    "matches_by_name": 1000,
    "matches_by_ean": 500
  }
}
```

**Estado: Enriquecimiento Fallido**
```json
{
  "upload_id": "123e4567-e89b-12d3-a456-426614174000",
  "filename": "ventas_enero_2024.csv",
  "status": "completed",
  "enrichment_status": "failed",
  "enrichment_progress": {
    "total": 10000,
    "processed": 2000,
    "enriched": 1500,
    "manual_review": 100,
    "failed": 400,
    "percentage": 20.0,
    "phase": "Error durante enriquecimiento",
    "matches_by_code": 1200,
    "matches_by_name": 200,
    "matches_by_ean": 100
  }
}
```

**Frontend Usage:**
```javascript
// Polling para seguir progreso de enriquecimiento
async function pollEnrichmentStatus(uploadId) {
  const poll = async () => {
    const response = await fetch(`/api/v1/upload/status/${uploadId}`);
    const data = await response.json();

    // Actualizar UI con progreso
    if (data.enrichment_progress) {
      updateProgressBar(data.enrichment_progress.percentage);
      updatePhaseLabel(data.enrichment_progress.phase);
      updateMatchStats({
        byCode: data.enrichment_progress.matches_by_code,
        byEAN: data.enrichment_progress.matches_by_ean,
        byName: data.enrichment_progress.matches_by_name
      });
    }

    // Verificar estado final
    if (data.enrichment_status === 'completed') {
      onEnrichmentComplete(data);
    } else if (data.enrichment_status === 'failed') {
      onEnrichmentError(data);
    } else if (data.enrichment_status === 'processing') {
      setTimeout(poll, 2000); // Poll cada 2 segundos
    }
  };

  poll();
}
```

### GET /api/v1/upload/history
Get upload history for the pharmacy.

**Query Parameters:**
- `limit` (int): Number of records to return
- `offset` (int): Pagination offset

### GET /api/v1/upload/status/summary 🆕
Get summary of all uploads with zombie detection.

**Response:**
```json
{
  "pharmacy_id": "11111111-1111-1111-1111-111111111111",
  "status_counts": {
    "completed": 45,
    "processing": 2,
    "error": 3,
    "pending": 0
  },
  "potential_zombies": {
    "count": 2,
    "details": [
      {
        "upload_id": "bd9382a8-41f4-4869-b78a-017a3aa86b06",
        "filename": "farmanager_sales_sample.csv",
        "hours_in_processing": 1.5,
        "processing_started_at": "2025-09-15T10:37:06"
      }
    ]
  },
  "timestamp": "2025-09-15T12:00:00"
}
```

### POST /api/v1/upload/cleanup/zombies 🆕
**Clean up zombie file uploads** (archivos atascados en estado "processing").

Detecta y limpia archivos que llevan demasiado tiempo en estado "processing" (zombies).
Útil para resolver uploads estancados por fallos del sistema, timeouts, o crashes del worker.

**Auth**: Required (JWT Bearer Token)
**RBAC**:
- Admin: Limpia zombies de TODAS las farmacias (scope: global)
- User: Limpia zombies solo de SU farmacia (scope: pharmacy_<uuid>)

**Query Parameters:**
- `max_age_hours` (float, default: 1.0): Horas en estado "processing" antes de considerar zombie

**Request Example:**
```bash
curl -X POST "http://localhost:8000/api/v1/upload/cleanup/zombies?max_age_hours=2.0" \
  -H "Authorization: Bearer eyJhbGciOiJIUzI1NiIs..."
```

**Response Fields:**
- `success` (boolean): Indica si la operación fue exitosa
- `zombies_cleaned` (integer): Número de archivos zombies limpiados
- `cutoff_time` (string ISO8601): Timestamp límite usado para detección de zombies
- `max_age_hours` (float): Parámetro de antigüedad usado en la operación
- `zombie_details` (array): Detalles de cada zombie limpiado (id, filename, pharmacy_id, time_in_processing_hours)
- `message` (string): Mensaje descriptivo de la operación
- `scope` (string): **[NUEVO v0f1dacb]** Indica el alcance de la operación para auditoría
  - `"global"` - Usuario admin limpió zombies de todas las farmacias
  - `"pharmacy_<uuid>"` - Usuario no-admin limpió solo su propia farmacia (UUID de la farmacia)

**Response Example (Admin - Global Scope):**
```json
{
  "success": true,
  "zombies_cleaned": 3,
  "cutoff_time": "2025-10-26T07:30:00Z",
  "max_age_hours": 1.0,
  "zombie_details": [
    {
      "upload_id": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
      "filename": "ventas_enero.csv",
      "pharmacy_id": "11111111-1111-1111-1111-111111111111",
      "time_in_processing_hours": 2.5,
      "processing_started_at": "2025-10-26T05:00:00Z"
    },
    {
      "upload_id": "b2c3d4e5-f6a7-8901-bcde-f12345678901",
      "filename": "ventas_febrero.xlsx",
      "pharmacy_id": "22222222-2222-2222-2222-222222222222",
      "time_in_processing_hours": 1.8,
      "processing_started_at": "2025-10-26T05:42:00Z"
    }
  ],
  "message": "Se limpiaron 3 archivos zombies",
  "scope": "global"
}
```

**Response Example (Non-Admin - Pharmacy Scope):**
```json
{
  "success": true,
  "zombies_cleaned": 1,
  "cutoff_time": "2025-10-26T08:15:00Z",
  "max_age_hours": 1.0,
  "zombie_details": [
    {
      "upload_id": "c3d4e5f6-a7b8-9012-cdef-123456789012",
      "filename": "ventas_marzo.csv",
      "pharmacy_id": "33333333-3333-3333-3333-333333333333",
      "time_in_processing_hours": 1.2,
      "processing_started_at": "2025-10-26T07:03:00Z"
    }
  ],
  "message": "Se limpiaron 1 archivos zombies",
  "scope": "pharmacy_33333333-3333-3333-3333-333333333333"
}
```

**Response Example (No Zombies Found):**
```json
{
  "success": true,
  "zombies_cleaned": 0,
  "cutoff_time": "2025-10-26T09:00:00Z",
  "max_age_hours": 1.0,
  "zombie_details": [],
  "message": "Se limpiaron 0 archivos zombies",
  "scope": "pharmacy_44444444-4444-4444-4444-444444444444"
}
```

**Error Responses:**
- `401 Unauthorized`: Token JWT inválido o expirado
- `500 Internal Server Error`: Error durante la operación de limpieza

**Security Notes:**
- El campo `scope` permite auditoría precisa de operaciones de limpieza
- Admins pueden limpiar zombies de cualquier farmacia (útil para mantenimiento global)
- Users solo pueden limpiar zombies de su propia farmacia (seguridad por pharmacy_id)
- Operación registrada en logs con `[CLEANUP]` tag para trazabilidad

**Use Cases:**
- Admin: Limpieza periódica de zombies de todas las farmacias
- User: Auto-recuperación de uploads fallidos sin intervención de soporte
- Monitoring: Campo `scope` facilita auditoría de quién limpió qué y cuándo

**Related Endpoints:**
- `GET /api/v1/upload/status/summary` - Ver resumen de uploads incluyendo zombies detectados
- `GET /api/v1/upload/list` - Listar uploads con filtros de estado

### GET /api/v1/upload/enrichment/progress
Get enrichment progress for uploads.

### POST /api/v1/upload/enrichment/trigger
Manually trigger enrichment process.

### GET /api/v1/upload/{upload_id}/delete-preview 🆕
**Preview de eliminación de upload** - Ver qué datos se eliminarán sin ejecutar la acción.

**Auth**: Required (JWT Bearer Token)
**Security**: Usuario solo puede ver previews de uploads de su propia farmacia.

**Path Parameters:**
- `upload_id` (string UUID): ID del upload a previsualizar

**Response:**
```json
{
  "upload_id": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
  "filename": "ventas_enero_2024.csv",
  "uploaded_at": "2024-01-15T10:30:00Z",
  "sales_count": 15420,
  "enrichment_count": 15420,
  "date_range": {
    "from": "01/01/2024",
    "to": "31/01/2024"
  },
  "total_amount": 125430.50
}
```

**Error Responses:**
- `401 Unauthorized`: Token JWT inválido o expirado
- `404 Not Found`: Upload no encontrado o no pertenece a la farmacia del usuario

### DELETE /api/v1/upload/{upload_id} 🆕
**Eliminar upload y datos asociados** - Elimina permanentemente un upload y todos sus datos de ventas.

**Auth**: Required (JWT Bearer Token)
**Security**: Usuario solo puede eliminar uploads de su propia farmacia.

**ADVERTENCIA**: Esta operación es IRREVERSIBLE. Elimina:
- El registro del upload
- Todos los datos de ventas asociados (SalesData)
- Todos los datos de enriquecimiento (SalesEnrichment)

La eliminación usa CASCADE configurado en los modelos de base de datos.

**Path Parameters:**
- `upload_id` (string UUID): ID del upload a eliminar

**Response:**
```json
{
  "success": true,
  "message": "Upload 'ventas_enero_2024.csv' eliminado correctamente",
  "deleted": {
    "upload_id": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
    "filename": "ventas_enero_2024.csv",
    "sales_records": 15420,
    "enrichment_records": 15420
  }
}
```

**Error Responses:**
- `401 Unauthorized`: Token JWT inválido o expirado
- `404 Not Found`: Upload no encontrado o no pertenece a la farmacia del usuario
- `500 Internal Server Error`: Error durante la eliminación

**Audit**: La operación se registra en audit_log con:
- `action`: DELETE
- `resource_type`: file_upload
- `details`: filename, sales_deleted, enrichment_deleted, pharmacy_id

**Frontend Usage:**
```python
# 1. Obtener preview antes de mostrar confirmación
response = backend_client.get_upload_delete_preview(upload_id)
if response.success:
    show_confirmation_modal(response.data)

# 2. Ejecutar eliminación tras confirmación del usuario
response = backend_client.delete_upload(upload_id)
if response.success:
    show_success_toast(f"Eliminado: {response.data['deleted']['sales_records']} ventas")
    refresh_upload_history()
```

---

## 4. Re-enrichment

### GET /api/reenrichment/status/{pharmacy_id} ⭐
**Check re-enrichment recommendations** for a specific pharmacy.

**Path Parameters:**
- `pharmacy_id` (string): Pharmacy UUID

**Response:**
```json
{
  "pharmacy_id": "11111111-1111-1111-1111-111111111111",
  "total_sales": 21677,
  "needs_update": 0,
  "products_affected": 0,
  "reasons": [],
  "recommendation": "Datos actualizados"
}
```

### POST /api/reenrichment/execute/{pharmacy_id}
**Execute re-enrichment** for a specific pharmacy.

**Path Parameters:**
- `pharmacy_id` (string): Pharmacy UUID

**Response:**
```json
{
  "status": "no_action_needed",
  "message": "No hay datos que necesiten actualización"
}
```

**Response (when re-enrichment needed):**
```json
{
  "status": "initiated",
  "message": "Re-enriquecimiento iniciado para 150 registros",
  "products_affected": 25,
  "estimated_time_seconds": 300
}
```

### POST /api/reenrichment/execute-all
**Execute global re-enrichment** for all pharmacies.

**Response:**
```json
{
  "status": "initiated",
  "message": "Re-enriquecimiento global iniciado",
  "check_progress_at": "/api/v1/reenrichment/progress"
}
```

### GET /api/reenrichment/catalog-changes
**Get recent catalog changes** that might trigger re-enrichment needs.

**Query Parameters:**
- `since_hours` (int, default: 24): Hours to look back for changes

**Response:**
```json
{
  "since": "2025-01-14T12:00:00",
  "total_updated": 156,
  "sample_products": [
    {
      "codigo_nacional": "123456",
      "nombre": "PARACETAMOL 500MG 20 COMP",
      "actualizado": "2025-01-15T10:30:00",
      "fuentes": "nomenclator,cima"
    }
  ]
}
```

### POST /api/reenrichment/failed-records/{pharmacy_id}
**Re-enrich failed records** for a specific pharmacy.

**Path Parameters:**
- `pharmacy_id` (string): Pharmacy UUID

**Query Parameters:**
- `limit` (int, default: 100): Maximum number of records to process

**Response:**
```json
{
  "status": "initiated",
  "message": "Reenriquecimiento de registros fallidos iniciado para farmacia 11111111-1111-1111-1111-111111111111",
  "limit": 100,
  "estimated_time_seconds": 4
}
```

### POST /api/reenrichment/failed-records-sync/{pharmacy_id}
**Synchronously re-enrich failed records** (for testing or small batches).

**Path Parameters:**
- `pharmacy_id` (string): Pharmacy UUID

**Query Parameters:**
- `limit` (int, default: 10): Maximum number of records to process

**Response:**
```json
{
  "status": "completed",
  "pharmacy_id": "11111111-1111-1111-1111-111111111111",
  "stats": {
    "processed": 10,
    "reenriched": 8,
    "cache_hits": 5,
    "cache_misses": 3
  },
  "success_rate": "80.0%"
}
```

### POST /api/reenrichment/scale/{pharmacy_id}
**Large-scale enrichment** optimized for 200K-400K records.

**Path Parameters:**
- `pharmacy_id` (string): Pharmacy UUID

**Query Parameters:**
- `batch_size` (int, default: 1000): Batch size for processing

**Response:**
```json
{
  "status": "initiated",
  "message": "Enriquecimiento iniciado para 250000 registros",
  "batch_size": 1000,
  "estimated_chunks": 251,
  "estimated_time_minutes": 500,
  "monitor_progress_at": "/api/v1/enrichment/progress/11111111-1111-1111-1111-111111111111"
}
```

### POST /api/reenrichment/scale-sync/{pharmacy_id}
**Synchronous large-scale enrichment** (for testing with smaller batches).

**Path Parameters:**
- `pharmacy_id` (string): Pharmacy UUID

**Query Parameters:**
- `batch_size` (int, default: 500): Batch size for processing

**Response:**
```json
{
  "status": "completed",
  "pharmacy_id": "11111111-1111-1111-1111-111111111111",
  "stats": {
    "processed": 500,
    "enriched": 475,
    "cache_hits": 400,
    "cache_misses": 75,
    "total_chunks": 1
  },
  "performance": {
    "success_rate": "95.0%",
    "cache_efficiency": "84.2%",
    "chunks_processed": 1,
    "avg_records_per_chunk": 500
  }
}
```

---

## 5. Sales Analysis

### GET /api/sales/pharmacies
Get available pharmacies for the current user.

### GET /api/sales/summary ⭐
Get sales summary with filters.

**Query Parameters:**
- `start_date` (required): Start date (YYYY-MM-DD)
- `end_date` (required): End date (YYYY-MM-DD)
- `group_by` (optional): Grouping (day, week, month, quarter)
- `include_enriched_only` (bool): Solo productos enriquecidos
- `laboratory` (string): Filtrar por laboratorio
- `category` (string): Filtrar por categoría

**Response:**
```json
{
  "total_sales": 50000.00,
  "total_units": 1500,
  "unique_products": 450,
  "enrichment_rate": 92.5,
  "period_comparison": {
    "sales_change": 15.5,
    "units_change": 8.2,
    "vs_period": "2023-12-01 to 2023-12-31"
  },
  "daily_sales": [
    {
      "date": "2024-01-15",
      "sales": 2500.00,
      "units": 75,
      "transactions": 45
    }
  ]
}
```

### GET /api/sales/trends
Get sales trends analysis.

### GET /api/sales/top-products ⭐
Get top selling products.

**Query Parameters:**
- `start_date`: Start date (YYYY-MM-DD)
- `end_date`: End date (YYYY-MM-DD)
- `limit`: Number of products (default: 10, max: 100)
- `order_by`: Sort by (sales, units, margin)

### GET /api/sales/categories
Get sales by categories.

### GET /api/sales/laboratories
Get top selling laboratories.

### GET /api/sales/enriched/summary/{pharmacy_id}
Get enriched sales summary for specific pharmacy.

### GET /api/sales/enriched/generic-opportunities/{pharmacy_id}
Get generic medication opportunities.

### GET /api/sales/enriched/therapeutic-analysis/{pharmacy_id}
Get therapeutic analysis based on ATC codes.

### GET /api/sales/enrichment/status/{pharmacy_id}
Get enrichment status for pharmacy data.

### GET /api/sales/date-range/{pharmacy_id}
Get available date range for pharmacy data.

---

## 5. Measures (Power BI Style)

> **REGLA #19**: Usar este patrón para todos los dashboards de analytics.
> Ver: `CLAUDE.md` REGLA #19, `docs/architecture/ADR-003-measures-filters-powerbi-pattern.md`

### GET /api/v1/measures/
List all available measures (26 medidas disponibles).

**Categorías de medidas:**
| Categoría | Medidas | Descripción |
|-----------|---------|-------------|
| **Core** (7) | `total_ventas`, `total_unidades`, `num_transacciones`, `ticket_promedio`, `margen_bruto`, `precio_promedio`, `unidades_por_transaccion` | Ventas básicas |
| **Temporal** (4) | `mat`, `ytd`, `mom`, `qoq` | Moving Annual Total, Year-to-Date, Month-over-Month, Quarter-over-Quarter |
| **Genéricos** (6) | `generic_savings_opportunity`, `partner_lab_sales`, `generic_vs_branded_ratio`, `homogeneous_group_sales`, `therapeutic_category_sales`, `monthly_partner_trend` | Análisis de genéricos |
| **Inventario** (3) | `stock_value`, `dead_stock_value`, `stock_coverage_days` | Valor stock, stock muerto, días cobertura (Issue #470) |
| **Rotación** (3) | `rotation_index`, `days_inventory`, `abc_classification` | Índice rotación, días inventario, clasificación ABC (Issue #470) |
| **Rentabilidad** (3) | `roi_product`, `gmroi`, `contribution_margin` | ROI, GMROI, margen contribución (Issue #470) |

**Response:**
```json
{
  "measures": [
    {
      "name": "stock_value",
      "display_name": "Valor de Inventario",
      "category": "Inventario",
      "unit": "€",
      "description": "Valor total del inventario a precio de coste"
    }
  ]
}
```

### GET /api/v1/measures/calculate/{measure_name}
Calculate specific measure with FilterContext.

**Query Parameters (FilterContext):**
| Parámetro | Tipo | Requerido | Descripción |
|-----------|------|-----------|-------------|
| `pharmacy_id` | UUID | ✅ | ID de la farmacia |
| `start_date` | date | ❌ | Fecha inicio (YYYY-MM-DD) |
| `end_date` | date | ❌ | Fecha fin (YYYY-MM-DD) |
| `laboratories` | list | ❌ | IDs de laboratorios |
| `therapeutic_categories` | list | ❌ | Categorías terapéuticas |
| `product_type` | string | ❌ | "medicamento", "venta_libre", "veterinario" |
| `is_generic` | bool | ❌ | Solo genéricos (true/false) |
| `min_sales_count` | int | ❌ | Mínimo de ventas para incluir |
| `snapshot_date` | date | ❌ | Fecha del snapshot de inventario (Issue #470) |
| `min_stock` | int | ❌ | Stock mínimo para filtrar (Issue #470) |
| `max_stock` | int | ❌ | Stock máximo para filtrar (Issue #470) |

**Ejemplo - Ventas totales de genéricos en Q1 2025:**
```bash
GET /api/v1/measures/calculate/total_sales?pharmacy_id=xxx&start_date=2025-01-01&end_date=2025-03-31&is_generic=true
```

**Response:**
```json
{
  "measure": "total_sales",
  "value": 45230.50,
  "unit": "€",
  "period": {"start": "2025-01-01", "end": "2025-03-31"},
  "filters_applied": ["is_generic=true"]
}
```

### POST /api/v1/measures/calculate/multiple
Calculate multiple measures at once (más eficiente que llamadas individuales).

**Request:**
```json
{
  "measures": ["total_sales", "total_units", "avg_ticket"],
  "pharmacy_id": "uuid",
  "start_date": "2025-01-01",
  "end_date": "2025-03-31",
  "filters": {
    "product_type": "medicamento",
    "laboratories": ["LAB001", "LAB002"]
  }
}
```

**Response:**
```json
{
  "results": {
    "total_sales": {"value": 45230.50, "unit": "€"},
    "total_units": {"value": 1523, "unit": "unidades"},
    "avg_ticket": {"value": 29.70, "unit": "€"}
  },
  "period": {"start": "2025-01-01", "end": "2025-03-31"}
}
```

### GET /api/v1/measures/dashboard/{pharmacy_id}
Get dashboard with predefined measures (KPIs principales).

### DELETE /api/v1/measures/cache
Clear measures cache (admin only).

---

## 6. Generic Analysis

### GET /api/generic-analysis/dashboard/{pharmacy_id} ⭐
Get complete generic analysis dashboard.

**Response:**
```json
{
  "summary": {
    "total_sales": 100000.00,
    "generic_sales": 35000.00,
    "generic_percentage": 35.0,
    "potential_savings": 5000.00
  },
  "top_opportunities": [...],
  "trend": {...},
  "therapeutic_distribution": [...]
}
```

### GET /api/generic-analysis/savings-opportunities/{pharmacy_id}
Get detailed savings opportunities.

### GET /api/generic-analysis/homogeneous-groups/{pharmacy_id}
Analysis by homogeneous groups.

### GET /api/generic-analysis/partners-trend/{pharmacy_id}
Get partners trend analysis.

### GET /api/generic-analysis/therapeutic-distribution/{pharmacy_id}
Get therapeutic distribution analysis.

### GET /api/v1/generic-analysis/context-treemap/{pharmacy_id} ⭐
Treemap de distribución de ventas usando medidas Power BI-style.

**Issue #472**: Migración a patrón Measures. Usa `TherapeuticCategorySales` internamente.

**Query Parameters:**
- `start_date` (optional): Fecha inicio (YYYY-MM-DD)
- `end_date` (optional): Fecha fin (YYYY-MM-DD)

**Response:**
```json
{
  "treemap_data": {
    "labels": ["N", "R", "A", "M"],
    "parents": ["", "", "", ""],
    "values": [15000.0, 12000.0, 8000.0, 5000.0],
    "text": ["37.5%", "30.0%", "20.0%", "12.5%"],
    "colors": ["#FF9800", "#607D8B", "#4CAF50", "#009688"]
  },
  "summary": {
    "total_categories": 4,
    "total_sales": 40000.0
  },
  "measure_used": "therapeutic_category_sales",
  "generado_en": "2025-12-25T10:30:00Z"
}
```

**Colores por Categoría ATC:**
- A (Digestivo): Verde #4CAF50
- N (Sistema Nervioso): Naranja #FF9800
- R (Respiratorio): Gris azulado #607D8B
- M (Musculoesquelético): Teal #009688
- (Ver endpoint para lista completa de 14 categorías)

### POST /api/generic-analysis/calculate-multiple-generic-measures
Calculate multiple generic measures.

---

## 8. Pharmacy Partners

### POST /api/pharmacy-partners/initialize/{pharmacy_id}
Auto-initialize partners based on sales data.

### GET /api/v1/pharmacy-partners/{pharmacy_id} ⭐
**Get all partners for pharmacy** with automatic initialization.

**🆕 Auto-Inicialización** (REGLA #12 desde 2025-10-20):
- **Primera llamada para farmacia nueva**: Inicializa automáticamente 56 laboratorios genéricos
- **Top 8 laboratorios auto-seleccionados**: Basados en ventas de últimos 13 meses
- **Re-inicialización automática**: Si no hay partners seleccionados, re-inicializa con sugerencias
- **Transparente**: No requiere acción manual del usuario

**Path Parameters:**
- `pharmacy_id` (UUID): ID de la farmacia

**Query Parameters:**
- `force_refresh` (bool, optional): Forzar re-inicialización aunque existan partners (default: false)

**Response:**
```json
{
  "partners": [
    {
      "id": "uuid",
      "pharmacy_id": "uuid",
      "laboratory_code": "111",
      "laboratory_name": "CINFA S.A.",
      "is_selected": true,
      "is_auto_suggested": true,
      "suggestion_rank": 1,
      "suggestion_reason": "Top 1 por volumen de ventas - €45,678 (últimos 13 meses)",
      "total_sales_last_13m": 45678.50,
      "product_count": 150,
      "created_at": "2025-10-20T10:00:00Z",
      "updated_at": "2025-10-20T10:00:00Z"
    },
    {
      "id": "uuid",
      "pharmacy_id": "uuid",
      "laboratory_code": "426",
      "laboratory_name": "NORMON S.A.",
      "is_selected": true,
      "is_auto_suggested": true,
      "suggestion_rank": 2,
      "suggestion_reason": "Top 2 por volumen de ventas - €38,234 (últimos 13 meses)",
      "total_sales_last_13m": 38234.20,
      "product_count": 125,
      "created_at": "2025-10-20T10:00:00Z",
      "updated_at": "2025-10-20T10:00:00Z"
    }
  ],
  "total_count": 56,
  "selected_count": 8,
  "auto_initialized": true,
  "initialization_timestamp": "2025-10-20T10:00:00Z"
}
```

**Logs Esperados (Auto-Inicialización):**
```
[AUTO_INIT] Inicializando partners automáticamente para farmacia {pharmacy_id}
Initialized 56 partners for pharmacy {pharmacy_id}
[AUTO_INIT] Partners inicializados exitosamente: 8 auto-sugeridos
```

**Comportamiento Detallado:**

1. **Nueva Farmacia (sin partners):**
   - Primera llamada GET detecta `count == 0`
   - Inicializa 56 laboratorios genéricos del catálogo
   - Calcula Top 8 por ventas últimos 13 meses
   - Marca Top 8 con `is_selected=true`, `is_auto_suggested=true`, ranking 1-8
   - Retorna partners ya inicializados

2. **Farmacia Existente (partners sin seleccionar):**
   - Detecta `selected_count == 0`
   - Re-ejecuta auto-inicialización si `force_refresh=true`
   - Caso sin `force_refresh`: Retorna partners existentes sin cambios

3. **Farmacia Existente (con partners seleccionados):**
   - Retorna partners existentes sin auto-inicialización
   - `force_refresh=true` permite re-calcular sugerencias

**Criterios de Auto-Selección (Top 8):**
- Laboratorios con >50% productos genéricos en catálogo
- Ordenados por `total_sales_last_13m` (descendente)
- Primeros 8 marcados como `is_selected=true`
- `suggestion_rank` = 1 (mayor ventas) a 8 (menor ventas)
- `suggestion_reason` incluye ventas formateadas

**Frontend Usage:**
```javascript
// Obtener partners con auto-inicialización transparente
const response = await fetch(`${BACKEND_URL}/api/v1/pharmacy-partners/${pharmacyId}`, {
  headers: { 'Authorization': `Bearer ${token}` }
});
const data = await response.json();

// Primera vez para nueva farmacia:
// data.auto_initialized === true
// data.selected_count === 8
// data.partners.filter(p => p.is_auto_suggested).length === 8

// Forzar re-inicialización (útil si datos cambiaron)
const refreshResponse = await fetch(
  `${BACKEND_URL}/api/v1/pharmacy-partners/${pharmacyId}?force_refresh=true`,
  { headers: { 'Authorization': `Bearer ${token}` } }
);
```

**Referencia:**
- **REGLA #12** en `CLAUDE.md` - Auto-Inicialización de Partners
- **Issue**: Partners vacíos en /generics (2025-10-19)

### PUT /api/v1/pharmacy-partners/{pharmacy_id}/selection
Update selected partners.

**Request:**
```json
{
  "selected_laboratories": ["CINFA", "NORMON", "KERN"]
}
```

### GET /api/v1/pharmacy-partners/{pharmacy_id}/selected ⭐
**Get only selected partners** with automatic initialization.

**🆕 Auto-Inicialización** (REGLA #12 desde 2025-10-20):
- **Idéntico comportamiento** a `GET /api/v1/pharmacy-partners/{pharmacy_id}`
- **Filtro adicional**: Solo retorna partners con `is_selected=true`
- **Caso sin seleccionados**: Auto-inicializa y retorna Top 8 sugeridos

**Path Parameters:**
- `pharmacy_id` (UUID): ID de la farmacia

**Query Parameters:**
- `force_refresh` (bool, optional): Forzar re-inicialización (default: false)

**Response:**
```json
{
  "selected_partners": [
    {
      "id": "uuid",
      "laboratory_code": "111",
      "laboratory_name": "CINFA S.A.",
      "is_selected": true,
      "is_auto_suggested": true,
      "suggestion_rank": 1,
      "suggestion_reason": "Top 1 por volumen de ventas - €45,678 (últimos 13 meses)",
      "total_sales_last_13m": 45678.50,
      "product_count": 150
    }
  ],
  "selected_count": 8,
  "auto_initialized": true,
  "initialization_timestamp": "2025-10-20T10:00:00Z"
}
```

**Logs Esperados (Auto-Inicialización):**
```
[AUTO_INIT] Inicializando partners automáticamente para farmacia {pharmacy_id}
Initialized 56 partners for pharmacy {pharmacy_id}
[AUTO_INIT] Partners inicializados exitosamente: 8 auto-sugeridos
```

**Frontend Usage:**
```javascript
// Obtener solo partners seleccionados (auto-inicializa si vacío)
const response = await fetch(
  `${BACKEND_URL}/api/v1/pharmacy-partners/${pharmacyId}/selected`,
  { headers: { 'Authorization': `Bearer ${token}` } }
);
const data = await response.json();

// Primera vez para nueva farmacia:
// data.selected_count === 8 (Top 8 auto-seleccionados)
// data.auto_initialized === true

// Uso típico en componente /generics
if (data.selected_count === 0) {
  showWarning('No hay partners seleccionados - Por favor, seleccione laboratorios');
} else {
  renderGenericAnalysis(data.selected_partners);
}
```

**Ventaja vs Endpoint GET All:**
- **Optimizado**: Solo retorna partners relevantes para análisis de genéricos
- **Menor payload**: Reduce tamaño de response (8 vs 56 items típicamente)
- **UX mejorado**: Elimina filtrado manual en frontend

----

## 9. Partner Analysis (Redesign) ⭐ **NEW**

**Purpose**: Advanced partner analysis with Power BI-style architecture for the redesigned Partners Panel.

**Key Features**:
- Fixed substitutable universe context (cacheable)
- Dynamic analysis with selected partners (real-time)
- Temporal drill-down (quarter → month → fortnight)
- Detailed homogeneous group analysis

### GET /api/v1/analysis/substitutable-universe/{pharmacy_id} ⭐
Get fixed context of substitutable universe for pharmacy.

**Parameters**:
- `pharmacy_id` (path): Pharmacy UUID
- `period_months` (query): Analysis period 1-24 months (default: 12)

### POST /api/v1/analysis/partner-dynamic/{pharmacy_id} ⭐
Calculate dynamic analysis based on selected partners.

**Query Parameters** (Optional - P1.3 Pagination):
- `limit` (int, optional): Número máximo de grupos homogéneos a retornar. Si no se especifica, retorna todos los grupos.
- `offset` (int, optional, default: 0): Número de grupos a saltar (para paginación).

**Request Body**: `{"selected_partners": ["CINFA", "NORMON S.A."], "period_months": 12}`

**Note**: Los grupos se ordenan por ahorro potencial descendente antes de aplicar limit/offset.

### GET /api/v1/analysis/temporal-breakdown/{pharmacy_id} ⭐
Get temporal breakdown for drill-down analysis.

**Parameters**:
- `partner_codes` (List[str], required): Códigos de partners seleccionados (ej: `["111", "426"]`)
- `level` (str, optional): Nivel temporal - `quarter` (default), `month`, `fortnight`
- `period_months` (int, optional): Período de análisis en meses (1-24, default: 12)
- `homogeneous_code` (str, optional): **NUEVO (Issue #346)** - Código de conjunto homogéneo para filtrar (ej: `"1160.0"`). Si se proporciona, solo incluye productos de ese conjunto homogéneo.

**Use Case (Issue #346)**: Drill-down desde gráfico de conjuntos homogéneos → evolución temporal de un conjunto específico.

**Response** (con filtro aplicado):
```json
{
  "pharmacy_id": "uuid",
  "selected_partner_codes": ["111", "426"],
  "temporal_level": "quarter",
  "homogeneous_filter": "1160.0",  // Presente solo si filtro aplicado
  "analysis_period": {
    "start_date": "2024-01-01",
    "end_date": "2024-12-31",
    "months": 12
  },
  "temporal_data": [
    {
      "period_start": "2024-01-01T00:00:00",
      "period_label": "Q1 2024",
      "total_units": 300,
      "partner_units": 60,
      "opportunity_units": 240,
      "savings_base": 720.50,
      "partner_penetration": 20.0
    }
  ],
  "summary": {
    "periods_count": 4,
    "avg_partner_penetration": 18.5,
    "total_opportunity_units": 1200,
    "total_savings_base": 3500.75
  }
}
```

### GET /api/v1/analysis/homogeneous-detail/{pharmacy_id}/{homogeneous_code} ⭐
Get detailed analysis of specific homogeneous group.

**Parameters**: `homogeneous_code`, `partners`, `period_months`

**Business Logic**: Only homogeneous groups with generic alternatives. Dynamic filtering by partner availability.

### GET /api/v1/analysis/context-treemap/{pharmacy_id} ⭐ **NEW (Issue #415)**
Get hierarchical context data for sales treemap visualization.

**Query Parameters:**
- `start_date` (string, optional): Analysis start date (YYYY-MM-DD format)
- `end_date` (string, optional): Analysis end date (YYYY-MM-DD format)
- `employee_names` (List[str], optional): Filter by employee names (PRO feature, Issue #402)

**Response Structure:**
```json
{
  "pharmacy_id": "uuid",
  "treemap_data": {
    "labels": ["Total Ventas", "Sustituible", "No Sustituible", "Analizable", "No Analizable", "Ya Partners", "Oportunidad"],
    "parents": ["", "Total Ventas", "Total Ventas", "Sustituible", "Sustituible", "Analizable", "Analizable"],
    "values": [100000, 45000, 55000, 30000, 15000, 18000, 12000],
    "text": ["EUR100,000 (100%)", "EUR45,000 (45%)", ...],
    "colors": ["#6c757d", "#0d6efd", "#adb5bd", "#17a2b8", "#6c757d", "#28a745", "#ffc107"]
  },
  "hierarchy": {
    "name": "Total Ventas",
    "value": 100000,
    "percentage": 100.0,
    "children": [
      {
        "name": "Sustituible",
        "value": 45000,
        "percentage": 45.0,
        "children": [...]
      }
    ]
  },
  "summary": {
    "total_sales": 100000,
    "substitutable_sales": 45000,
    "substitutable_percentage": 45.0,
    "analyzable_sales": 30000,
    "analyzable_percentage": 66.67,
    "partner_sales": 18000,
    "partner_percentage": 60.0,
    "opportunity_sales": 12000,
    "opportunity_percentage": 40.0
  },
  "period": {
    "start_date": "2024-01-01",
    "end_date": "2024-12-31"
  }
}
```

**Business Logic:**
- **Total Ventas**: All pharmacy sales in period
- **Sustituible**: Products with homogeneous groups that have at least one generic available
- **Analizable**: Substitutable products where selected partners have products available
- **Ya Partners**: Sales of selected partners' products within analyzable universe
- **Oportunidad**: Non-partner sales within analyzable universe (conversion potential)

**Use Cases:**
- Context visualization for /generics page (Issue #415)
- Understanding sales breakdown by substitution potential
- Identifying opportunity segments for partner negotiation

---

## 10. Laboratory Mapping ⭐ **NEW**

**Purpose**: High-performance laboratory code ↔ name mapping system optimized for Spanish pharmaceutical nomenclator.

**Key Features**:
- **Production-ready**: Sub-100ms response times with Redis caching
- **Secure**: Input validation, rate limiting, SQL injection protection
- **Paginated**: Optimized for large datasets (1000+ laboratories)
- **Cache-optimized**: 24-hour TTL with automatic invalidation
- **Spanish-focused**: Real pharmaceutical laboratory names (CINFA, NORMON, TEVA, etc.)

**Rate Limiting**: 60 requests/minute, 500 requests/hour per IP
**Cache**: Redis with 24h TTL, automatic catalog-based invalidation

### GET /api/v1/laboratory-mapping/codes-to-names ⭐
**Convert laboratory codes to names** - Primary mapping endpoint.

**Query Parameters:**
- `codes` (List[str], optional): Specific codes to map (max 20)
  - Format: 1-4 digits (e.g., "111", "426", "1079")
  - Example: `?codes=111,426,1079`
- `page` (int, optional): Page number for pagination (1-10000)
- `per_page` (int, optional): Items per page (1-200)
- `envelope` (bool, optional): Return with pagination metadata (default: false)

**Response (Traditional Format):**
```json
{
  "111": "CINFA S.A.",
  "426": "NORMON S.A.",
  "1079": "TEVA PHARMA, S.L.U",
  "644": "SANDOZ FARMACEUTICA, S.A",
  "863": "KERN PHARMA, S.L."
}
```

**Response (Envelope Format with `envelope=true`):**
```json
{
  "items": {
    "111": "CINFA S.A.",
    "426": "NORMON S.A.",
    "1079": "TEVA PHARMA, S.L.U"
  },
  "pagination": {
    "total": 1247,
    "page": 2,
    "per_page": 50,
    "total_pages": 25,
    "has_next": true,
    "has_previous": true
  }
}
```

**Usage Examples:**
```javascript
// Get specific laboratory codes
const response = await fetch('/api/v1/laboratory-mapping/codes-to-names?codes=111,426,1079');
const mappings = await response.json();
console.log(mappings["111"]); // "CINFA S.A."

// Paginated access to all laboratories
const paginatedResponse = await fetch('/api/v1/laboratory-mapping/codes-to-names?page=1&per_page=50&envelope=true');
const data = await paginatedResponse.json();
console.log(`Total laboratories: ${data.pagination.total}`);
```

**Performance:**
- Cached response: ~15ms
- Cache miss: ~85ms
- Pagination: ~95ms with metadata

**Error Responses:**
- `400`: Invalid codes format or pagination params
- `413`: Request too large (>100KB)
- `422`: Validation errors (invalid codes format)
- `429`: Rate limit exceeded
- `500`: Internal server error

### GET /api/v1/laboratory-mapping/names-to-codes
**Convert laboratory names to codes** - Reverse mapping endpoint.

**Query Parameters:**
- `names` (List[str], optional): Specific names to map (max 20)
  - Max length: 200 characters each
  - XSS protection and sanitization applied
  - Example: `?names=CINFA S.A.,NORMON S.A.`
- `page` (int, optional): Page number (1-10000)
- `per_page` (int, optional): Items per page (1-200)
- `envelope` (bool, optional): Return with pagination metadata

**Response:**
```json
{
  "CINFA S.A.": "111",
  "NORMON S.A.": "426",
  "TEVA PHARMA, S.L.U": "1079",
  "SANDOZ FARMACEUTICA, S.A": "644",
  "KERN PHARMA, S.L.": "863"
}
```

**Usage Examples:**
```javascript
// Convert laboratory names back to codes
const response = await fetch('/api/v1/laboratory-mapping/names-to-codes?names=CINFA S.A.,NORMON S.A.');
const mappings = await response.json();
console.log(mappings["CINFA S.A."]); // "111"
```

### GET /api/v1/laboratory-mapping/generic-laboratories
**Get generic laboratories only** - Filtered mapping for generic drug analysis.

**Query Parameters:**
- `page` (int): Page number (default: 1, range: 1-10000)
- `per_page` (int): Items per page (default: 50, range: 1-200)
- `search` (str, optional): Search term (max 100 chars)
- `envelope` (bool, optional): Return with pagination metadata

**Response:**
```json
{
  "111": "CINFA S.A.",
  "426": "NORMON S.A.",
  "863": "KERN PHARMA, S.L.",
  "644": "SANDOZ FARMACEUTICA, S.A",
  "1079": "TEVA PHARMA, S.L.U"
}
```

**Usage Examples:**
```javascript
// Get first page of generic laboratories
const response = await fetch('/api/v1/laboratory-mapping/generic-laboratories?page=1&per_page=20');
const generics = await response.json();

// Search generic laboratories
const searchResponse = await fetch('/api/v1/laboratory-mapping/generic-laboratories?search=CINFA&envelope=true');
const searchResults = await searchResponse.json();
console.log(`Found ${searchResults.pagination.total} results`);
```

**Specialized for:**
- Partner selection components
- Generic analysis dashboards
- Laboratory filtering in analytics

### GET /api/v1/laboratory-mapping/cache-stats
**Get cache performance statistics** - Monitoring endpoint.

**Response:**
```json
{
  "hits": 1250,
  "misses": 200,
  "total_operations": 1450,
  "hit_rate_percent": 86.21,
  "error_rate_percent": 0.0,
  "avg_response_time_ms": 12.5,
  "cache_size_estimate_mb": 2.3,
  "redis_connected_clients": 5,
  "redis_used_memory_human": "15.2M",
  "redis_hit_rate_percent": 89.4,
  "timestamp": "2025-01-21T10:30:00Z"
}
```

**Monitoring Usage:**
```javascript
// Monitor cache performance
const stats = await fetch('/api/v1/laboratory-mapping/cache-stats').then(r => r.json());
if (stats.hit_rate_percent < 80) {
  console.warn('Laboratory mapping cache hit rate below target');
}
```

### POST /api/v1/laboratory-mapping/invalidate-cache
**Invalidate laboratory mapping cache** - Administrative endpoint.

**Response:**
```json
{
  "invalidated_keys": 45,
  "message": "Cache de laboratory mappings invalidado exitosamente",
  "timestamp": "2025-01-21T10:30:00Z"
}
```

**Administrative Usage:**
```javascript
// Force cache refresh after catalog updates
const result = await fetch('/api/v1/laboratory-mapping/invalidate-cache', {
  method: 'POST',
  headers: { 'Authorization': `Bearer ${adminToken}` }
});
const response = await result.json();
console.log(`Invalidated ${response.invalidated_keys} cache entries`);
```

**Common Laboratory Codes (Spanish Market):**
- `111`: CINFA S.A. (Generic leader)
- `426`: NORMON S.A. (Major generic manufacturer)
- `863`: KERN PHARMA, S.L. (Specialty generics)
- `644`: SANDOZ FARMACEUTICA, S.A (International generics)
- `1079`: TEVA PHARMA, S.L.U (Global generic leader)

**Security Features:**
- Input validation with regex patterns
- XSS protection and sanitization
- SQL injection prevention
- Rate limiting per IP
- Request size limits (100KB max)
- Security headers enforcement

**Performance Characteristics:**
- Cache hit: 15-25ms average
- Cache miss: 85-120ms average
- Pagination: 95-150ms with metadata
- Memory usage: ~2-5MB cache size
- Target hit rate: >80%

---

## 11. Pharmacy Management

### GET /api/pharmacy/{pharmacy_id}
Get pharmacy details.

**Response:**
```json
{
  "id": "uuid",
  "name": "Farmacia Central",
  "code": "12345",
  "address": "Calle Mayor 1",
  "city": "Madrid",
  "subscription_status": "active"
}
```

### PUT /api/pharmacy/{pharmacy_id}
Update pharmacy details.

### GET /api/pharmacy/by-code/{pharmacy_code}
Get pharmacy by code.

---

## 11.1 Pharmacy Employees (NEW - Issue #402) 🔒 PRO Feature

Employee filtering system for generic analysis. Allows PRO+ users to filter sales data and analysis by specific employees.

**Feature Status**: ✅ Implemented (Backend + Frontend)
**Required Plan**: PRO or MAX
**Related Issues**: #402

### GET /api/v1/employees/{pharmacy_id}

Get list of unique employees for a pharmacy based on sales data.

**Access Control**:
- ✅ JWT authentication required
- ✅ PRO+ subscription plan required (enforced by `@require_subscription_plan("pro")` decorator)
- ✅ User must own the pharmacy (REGLA #10: 1:1 relationship)

**Parameters**:
- `pharmacy_id` (path, UUID, required): ID of the pharmacy

**Response** (200 OK):
```json
{
  "employees": [
    {
      "name": "María García",
      "sales_count": 1250,
      "last_sale_date": "2025-11-05"
    },
    {
      "name": "Juan Pérez",
      "sales_count": 890,
      "last_sale_date": "2025-11-04"
    },
    {
      "name": "__sin_empleado__",
      "sales_count": 320,
      "last_sale_date": "2025-11-03"
    }
  ],
  "total_count": 3
}
```

**Special Values**:
- `__sin_empleado__`: Represents sales without an assigned employee (COALESCE pattern)

**Error Responses**:
- `401 Unauthorized`: No JWT token provided or invalid token
- `403 Forbidden`: User plan doesn't have access (FREE plan user attempting PRO feature)
  ```json
  {
    "detail": "Esta funcionalidad requiere plan PRO o superior. Plan actual: FREE. Por favor actualice su suscripción para acceder a esta funcionalidad."
  }
  ```
- `404 Not Found`: Pharmacy not found or user doesn't own the pharmacy
- `500 Internal Server Error`: Database error

**Frontend Usage**:
```python
# Frontend callback example
import requests
from frontend.utils.config import BACKEND_URL
from frontend.utils.auth import auth_manager

@callback(
    Output("employee-dropdown", "options"),
    Input("url", "pathname"),
    State("auth-state", "data")
)
def load_employees(pathname, auth_state):
    if pathname != "/generics":
        raise PreventUpdate

    from frontend.utils.auth_helpers import is_user_authenticated
    if not is_user_authenticated(auth_state):
        raise PreventUpdate

    # Get pharmacy_id from auth_state
    pharmacy_id = auth_state.get("user", {}).get("pharmacy_id")
    if not pharmacy_id:
        return []

    token = auth_manager.get_access_token()

    try:
        response = requests.get(
            f"{BACKEND_URL}/api/v1/employees/{pharmacy_id}",
            headers={"Authorization": f"Bearer {token}"},
            timeout=10
        )
        response.raise_for_status()
        data = response.json()

        return [
            {"label": emp["name"], "value": emp["name"]}
            for emp in data["employees"]
        ]
    except requests.RequestException as e:
        logger.error(f"Failed to load employees: {e}")
        return []
```

**Integration with Generic Analysis**:

This endpoint provides the employee list for filtering in generic analysis endpoints:

1. **GET /api/v1/generic-opportunities** - Now accepts `selected_employee_names` parameter
2. **POST /api/v1/partner-analysis/dynamic** - Now accepts `selected_employee_names` parameter
3. **POST /api/v1/partner-analysis/temporal** - Now accepts `selected_employee_names` parameter

**Backend Implementation Details**:
- **Service**: `backend/app/services/employee_service.py`
- **Endpoint**: `backend/app/api/employees.py`
- **Schema**: `backend/app/schemas/employee.py`
- **Plan Enforcement**: `backend/app/api/deps.py` - `require_subscription_plan()` decorator
- **Database Index**: `employee_name` indexed for performance (migration `33d327f16952`)

**Security Logging**:
All plan enforcement denials are logged with security metadata for audit trail:
```python
logger.warning(
    f"[PLAN_ENFORCEMENT] Usuario {user.email} (plan: {user_plan}) "
    f"intentó acceder a feature {min_plan}+",
    extra={
        "user_id": str(user.id),
        "user_email": user.email,
        "user_plan": user_plan,
        "required_plan": min_plan,
        "security_event": "subscription_plan_denied"
    }
)
```

**Related Documentation**:
- Plan enforcement decorator: `backend/app/api/deps.py` lines 379-442
- Employee service: `backend/app/services/employee_service.py`
- Schema definitions: `backend/app/schemas/employee.py`
- Tests: `backend/tests/test_deps.py` (plan enforcement tests), `backend/tests/api/test_employees.py`

---

## 12. Generic Opportunities

### GET /api/pharmacies/{pharmacy_id}/generic-opportunities
Get detailed generic opportunities analysis.

### POST /api/pharmacies/{pharmacy_id}/generic-opportunities/recalculate
Force recalculation of generic opportunities.

### GET /api/pharmacies/{pharmacy_id}/partners
Get partners for specific pharmacy (alternative endpoint).

### PUT /api/pharmacies/{pharmacy_id}/partners
Update partners for specific pharmacy (alternative endpoint).

---

## 13. Authentication

**⚠️ IMPORTANTE**: Sistema simplificado a solo 2 roles (admin/user) desde PR #107.
- **Eliminado**: Registro público (`POST /auth/register`)
- **Nuevo**: Sistema de invitaciones exclusivamente administrado por admins
- **Roles disponibles**: `admin` y `user` únicamente

### ❌ POST /api/v1/register (ELIMINADO)
**DEPRECADO desde Issue #85 / PR #107**

El registro público de usuarios ha sido eliminado y reemplazado por el sistema de invitaciones.

**Migración:**
- Los usuarios existentes mantienen sus cuentas
- Nuevos usuarios solo mediante invitaciones creadas por administradores
- Ver sección "Invitaciones" para el nuevo flujo

**Razón de eliminación:**
- Seguridad: Control total sobre quién accede al sistema
- Compliance: Trazabilidad completa de onboarding
- Gestión: Administradores aprueban cada nuevo usuario

### POST /api/v1/login
User login with email/password.

**Rate Limit:** 10/minute per IP

**Request:**
```json
{
  "email": "user@pharmacy.com",
  "password": "password123"
}
```

**Response:**
```json
{
  "user": {
    "id": "uuid",
    "email": "user@pharmacy.com",
    "username": "username",
    "full_name": "Juan Pérez",
    "role": "user",
    "is_active": true,
    "is_verified": true
  },
  "access_token": "eyJ...",
  "refresh_token": "eyJ...",
  "token_type": "bearer"
}
```

### POST /api/v1/refresh
Refresh access token using refresh token.

**Request:**
```json
{
  "refresh_token": "eyJ..."
}
```

**Response:**
```json
{
  "access_token": "eyJ...",
  "token_type": "bearer"
}
```

### GET /api/v1/me
Get current authenticated user profile.

**Headers Required:** `Authorization: Bearer {access_token}`

**Response:**
```json
{
  "id": "uuid",
  "email": "user@pharmacy.com",
  "username": "username",
  "full_name": "Juan Pérez",
  "role": "user",
  "pharmacy_id": "uuid",
  "is_active": true,
  "is_verified": true,
  "created_at": "2024-01-15T10:30:00Z",
  "last_login": "2024-01-20T14:00:00Z"
}
```

### PUT /api/v1/me
Update current user profile.

**Headers Required:** `Authorization: Bearer {access_token}`

**Request:**
```json
{
  "full_name": "Juan Pérez García",
  "phone": "+34600111222",
  "notify_uploads": true,
  "notify_errors": true,
  "notify_analysis": false
}
```

### POST /api/v1/verify-email
Verify user email with verification token.

**Request:**
```json
{
  "user_id": "uuid",
  "verification_token": "token_string"
}
```

### POST /api/v1/forgot-password
Initiate password reset process.

**Rate Limit:** 3/hour per IP

**Note:** Currently disabled. Email service configuration required.

**Request:**
```json
{
  "email": "user@pharmacy.com"
}
```

### POST /api/v1/reset-password
Reset password with reset token.

**Note:** Currently disabled. Email service configuration required.

**Request:**
```json
{
  "reset_token": "token_string",
  "new_password": "NewSecurePassword123"
}
```

### GET /api/v1/oauth/providers
Get available OAuth providers and their login URLs.

**Response:**
```json
{
  "google_enabled": false,
  "microsoft_enabled": false,
  "google_url": null,
  "microsoft_url": null
}
```

### GET /api/v1/oauth/google
Initiate Google OAuth login (redirects to Google).

**Note:** Requires OAuth configuration in environment variables.

### GET /api/v1/oauth/google/callback
Google OAuth callback (handled by backend).

### GET /api/v1/oauth/microsoft
Initiate Microsoft OAuth login (redirects to Microsoft).

**Note:** Requires OAuth configuration in environment variables.

### GET /api/v1/oauth/microsoft/callback
Microsoft OAuth callback (handled by backend).

### POST /api/v1/oauth/exchange
Exchange OAuth state key for actual tokens.

**Request:**
```json
{
  "state_key": "secure_state_key_from_oauth_callback"
}
```

**Response:**
```json
{
  "access_token": "eyJ...",
  "refresh_token": "eyJ...",
  "token_type": "bearer"
}
```

### POST /api/v1/logout
User logout (client should discard tokens).

### GET /api/v1/auth/storage-usage ⭐ NEW
**Get storage usage for current authenticated user** (Issue #420: FREE tier experience)

**Authentication**: Requiere JWT token

**Response Success (200):**
```json
{
  "total_used_mb": 45.23,
  "limit_mb": 100,
  "percentage": 45.2,
  "plan": "free",
  "warning_threshold": 80
}
```

**Response Fields:**
- `total_used_mb` (float): Total MB used by file uploads
- `limit_mb` (int): Storage limit based on subscription plan (FREE: 100MB, PRO: 1GB, MAX: 10GB)
- `percentage` (float): Usage percentage (0-100), capped at 100
- `plan` (string): Current subscription plan (`free`, `pro`, `max`)
- `warning_threshold` (int): Percentage threshold for warning display (default: 80)

**Usage Notes:**
- Calculates storage from `FileUpload` table for user's pharmacy
- Only counts non-deleted files (`is_deleted=False`)
- Used in `/settings/subscription` page for storage meter

---

## 13.1 Invitaciones (Sistema de Onboarding)

**Propósito**: Gestión de invitaciones para nuevos usuarios del sistema.

**Seguridad**:
- Solo administradores pueden crear invitaciones
- Tokens seguros con expiración
- Audit logs automáticos

### POST /api/v1/invitations/create ⭐
**Crear nueva invitación** (Admin-only)

**Authentication**: Requiere JWT token con rol `admin`
**Rate Limit**: 10/hour

**Query Parameters:**
- `pharmacy_id` (UUID, required): ID de la farmacia para el nuevo usuario

**Request Body:**
```json
{
  "email": "nuevo.usuario@farmacia.com",
  "full_name": "María García López",
  "role": "user",
  "dni_nie": "12345678Z",
  "message": "Bienvenido al equipo de Farmacia Central",
  "expires_in_days": 7
}
```

**Response Success (201):**
```json
{
  "id": "uuid",
  "email": "nuevo.usuario@farmacia.com",
  "full_name": "María García López",
  "role": "user",
  "pharmacy_id": "uuid",
  "token": "secure_token_string",
  "status": "pending",
  "invited_by_id": "admin_uuid",
  "expires_at": "2025-10-10T12:00:00Z",
  "created_at": "2025-10-03T12:00:00Z"
}
```

**Response Error (403) - No admin:**
```json
{
  "error": {
    "code": 403,
    "message": "No tienes permisos para crear invitaciones",
    "details": "Solo administradores pueden crear invitaciones",
    "timestamp": "2025-10-03T12:00:00Z"
  }
}
```

**Response Error (400) - Role inválido:**
```json
{
  "error": {
    "code": 400,
    "message": "Rol inválido",
    "details": "Los roles permitidos son: admin, user",
    "timestamp": "2025-10-03T12:00:00Z"
  }
}
```

### GET /api/v1/invitations/list/{pharmacy_id} ⭐
**Listar invitaciones de una farmacia** (Admin-only)

**Authentication**: Requiere JWT token con rol `admin`

**Path Parameters:**
- `pharmacy_id` (UUID): ID de la farmacia

**Query Parameters:**
- `status` (string, optional): Filtrar por estado (`pending`, `accepted`, `expired`, `cancelled`)
- `limit` (int, optional): Límite de resultados (1-500, default: 100)

**Response Success (200):**
```json
{
  "invitations": [
    {
      "id": "uuid",
      "email": "usuario1@farmacia.com",
      "full_name": "Juan Pérez",
      "role": "user",
      "status": "pending",
      "expires_at": "2025-10-10T12:00:00Z",
      "created_at": "2025-10-03T12:00:00Z"
    },
    {
      "id": "uuid",
      "email": "usuario2@farmacia.com",
      "full_name": "Ana López",
      "role": "user",
      "status": "accepted",
      "accepted_at": "2025-10-04T15:30:00Z",
      "created_at": "2025-10-02T10:00:00Z"
    }
  ],
  "total": 2,
  "pending_count": 1,
  "accepted_count": 1
}
```

**Response Error (403) - No admin o farmacia incorrecta:**
```json
{
  "error": {
    "code": 403,
    "message": "No tienes acceso a las invitaciones de esta farmacia",
    "details": "Solo administradores de esta farmacia pueden ver invitaciones",
    "timestamp": "2025-10-03T12:00:00Z"
  }
}
```

### POST /api/v1/invitations/validate
**Validar token de invitación** (Público)

**Authentication**: No requiere autenticación

**Request:**
```json
{
  "token": "secure_token_string"
}
```

**Response Success (200):**
```json
{
  "valid": true,
  "email": "nuevo.usuario@farmacia.com",
  "full_name": "María García López",
  "role": "user",
  "pharmacy_name": "Farmacia Central",
  "expires_at": "2025-10-10T12:00:00Z",
  "message": "Invitación válida"
}
```

**Response Error (400) - Token expirado:**
```json
{
  "valid": false,
  "message": "La invitación ha expirado"
}
```

### POST /api/v1/invitations/accept
**Aceptar invitación y crear usuario** (Público)

**Authentication**: No requiere autenticación

**Request:**
```json
{
  "token": "secure_token_string",
  "username": "mgarcia",
  "password": "SecurePassword123!"
}
```

**Response Success (201):**
```json
{
  "user": {
    "id": "uuid",
    "email": "nuevo.usuario@farmacia.com",
    "username": "mgarcia",
    "full_name": "María García López",
    "role": "user",
    "pharmacy_id": "uuid",
    "is_active": true,
    "is_verified": true
  },
  "access_token": "eyJ...",
  "refresh_token": "eyJ...",
  "token_type": "bearer",
  "message": "Usuario creado exitosamente"
}
```

**Response Error (400) - Contraseña débil:**
```json
{
  "error": {
    "code": 400,
    "message": "La contraseña no cumple los requisitos de seguridad",
    "details": "Mínimo 8 caracteres, una mayúscula, una minúscula, un número",
    "timestamp": "2025-10-03T12:00:00Z"
  }
}
```

### DELETE /api/v1/invitations/cancel/{invitation_id}
**Cancelar invitación pendiente** (Admin-only)

**Authentication**: Requiere JWT token con rol `admin`

**Path Parameters:**
- `invitation_id` (UUID): ID de la invitación a cancelar

**Response Success (200):**
```json
{
  "message": "Invitación cancelada exitosamente",
  "invitation_id": "uuid",
  "status": "cancelled",
  "cancelled_at": "2025-10-03T12:00:00Z"
}
```

**Response Error (403) - No admin:**
```json
{
  "error": {
    "code": 403,
    "message": "No tienes permisos para cancelar invitaciones",
    "details": "Solo administradores pueden cancelar invitaciones",
    "timestamp": "2025-10-03T12:00:00Z"
  }
}
```

### POST /api/v1/invitations/resend/{invitation_id}
**Reenviar invitación** (Admin-only)

**Authentication**: Requiere JWT token con rol `admin`

**Path Parameters:**
- `invitation_id` (UUID): ID de la invitación a reenviar

**Response Success (200):**
```json
{
  "message": "Invitación actualizada. El email se enviará cuando el servicio esté activo",
  "expires_at": "2025-10-10T12:00:00Z"
}
```

**Note**: El servicio de email está pendiente de configuración. Por ahora solo extiende la expiración.

---

## 13.2 Sistema de Roles y Permisos (Simplificado)

**Actualización PR #107**: Sistema simplificado de 4 roles a solo 2 roles.

### Roles Disponibles

#### Admin (`admin`)
**Acceso completo al sistema**

**Permisos**:
- `view_system_status`: Ver estado del sistema y métricas
- `trigger_external_sync`: Sincronizar con CIMA/nomenclator
- `manage_users`: Gestionar usuarios (crear invitaciones, activar/desactivar)
- `manage_catalog`: Gestionar catálogo de productos
- `view_all_data`: Ver datos de todas las farmacias
- `delete_any`: Eliminar cualquier registro del sistema
- **Además**: Todos los permisos de `user`

**Endpoints exclusivos**:
- POST `/api/v1/invitations/create`
- GET `/api/v1/invitations/list/{pharmacy_id}`
- DELETE `/api/v1/invitations/cancel/{invitation_id}`
- POST `/api/v1/invitations/resend/{invitation_id}`
- GET `/api/v1/admin/users`
- POST `/api/v1/system/sync`
- DELETE `/api/v1/admin/delete-all-data`

#### User (`user`)
**Acceso estándar para farmacéuticos**

**Permisos**:
- `view`: Ver datos de su farmacia
- `upload`: Subir archivos de ventas
- `edit_own`: Editar sus propios datos
- `view_own_reports`: Ver sus propios informes
- `trigger_enrichment`: Activar enriquecimiento de datos
- `view_data_freshness`: Ver estado de actualización de catálogos

**Endpoints accesibles**:
- POST `/api/v1/upload/`
- GET `/api/sales/summary`
- GET `/api/generic-analysis/dashboard/{pharmacy_id}`
- GET `/api/pharmacy-partners/{pharmacy_id}`
- PUT `/api/v1/me` (solo su perfil)

### Roles Deprecados (Issue #85)

Los siguientes roles fueron eliminados y migrados automáticamente:

- ~~`manager`~~ → Migrado a `user`
- ~~`viewer`~~ → Migrado a `user`

**Migración automática**: La migración `706632bf152b` actualizó todos los usuarios existentes.

### Validación de Permisos

**En el backend** (`User.has_permission()`):
```python
# Verificar permisos antes de operaciones críticas
if not current_user.has_permission("manage_users"):
    raise HTTPException(
        status_code=403,
        detail="No tienes permisos para gestionar usuarios"
    )
```

**Superusers**:
- Los usuarios con `is_superuser=True` tienen todos los permisos
- Bypass de validación de roles
- Solo asignar a administradores de confianza

### Error Responses de Permisos

**403 Forbidden - Usuario sin permisos suficientes:**
```json
{
  "error": {
    "code": 403,
    "message": "No tienes permisos para realizar esta operación",
    "details": "Se requiere rol 'admin' o permiso 'manage_users'",
    "required_permission": "manage_users",
    "current_role": "user",
    "timestamp": "2025-10-03T12:00:00Z"
  }
}
```

**401 Unauthorized - Token inválido:**
```json
{
  "error": {
    "code": 401,
    "message": "Token de autenticación inválido o expirado",
    "details": "Por favor, inicia sesión nuevamente",
    "timestamp": "2025-10-03T12:00:00Z"
  }
}
```

---

## 14. Admin Operations

### DELETE /api/v1/admin/delete-all-data ⚠️
**DANGEROUS**: Delete all data for a pharmacy.

**Authentication**: Requiere JWT token con rol `admin`

**Query Parameters:**
- `pharmacy_id`: Pharmacy ID to delete data for
- `confirm`: Must be "true" to execute

### GET /api/v1/admin/statistics
Get system statistics.

**Response:**
```json
{
  "total_pharmacies": 10,
  "total_uploads": 500,
  "total_sales_records": 1000000,
  "catalog_products": 25000,
  "database_size_mb": 1024
}
```

### POST /api/v1/admin/vacuum-database
Run VACUUM on database for maintenance.

### POST /api/v1/admin/reindex-tables
**Reindex database tables** - Rebuilds indexes for better performance.

**Authentication**: Requiere JWT token con rol `admin` + Permission `MANAGE_DATABASE`

**Purpose**: Ejecuta REINDEX en todas las tablas principales del sistema para optimizar performance de queries.

**Response:**
```json
{
  "tables_reindexed": 15,
  "time_ms": 1234,
  "tables": [
    "sales_data",
    "sales_enrichment",
    "product_catalog",
    "users",
    "pharmacies"
  ]
}
```

**Cuándo usar:**
- Después de cargas masivas de datos (uploads grandes)
- Cuando queries son más lentas de lo esperado
- Como parte de mantenimiento preventivo mensual
- Nunca durante horas pico de uso

**Performance:**
- Tiempo típico: 1-5 segundos (depende de tamaño de BD)
- Bloquea temporalmente tablas (evitar en producción con usuarios activos)

### POST /api/v1/admin/check-nomenclator
Manually trigger nomenclator check.

### GET /api/v1/admin/catalog/stats ⭐ NEW (Issue #185)
**Get comprehensive catalog statistics** - For admin panel monitoring.

**Authentication**: Requiere JWT token con rol `admin`

**Purpose**: Provee estadísticas completas del catálogo CIMA para el panel de administración.

**Cache** (Issue #194): Redis con TTL de 5 minutos. Reduce carga en DB en ~98% con polling frecuente.
- **Invalidación automática** tras:
  - Sincronización CIMA (`catalog_maintenance_service`)
  - Re-enriquecimiento masivo (`reenrichment_service`)
  - Actualización nomenclator
- **Fallback**: Si Redis no disponible, consulta DB directamente
- ⚠️ **Limitación conocida**: Breve race condition después de sincronización CIMA donde stats cacheados inmediatamente pueden estar desactualizados (máx 5 min de staleness debido a TTL)

**Response Time:**
- Con cache hit: ~10-50ms ⚡
- Con cache miss: ~200-500ms

**Response:**
```json
{
  "total_products": 67000,
  "data_sources": {
    "only_nomenclator": 5000,
    "only_cima": 10000,
    "both_sources": 50000,
    "no_sources": 2000,
    "with_nomenclator": 55000,
    "with_cima": 60000
  },
  "sync_status": {
    "synced": 65000,
    "pending": 2000
  },
  "enrichment_rate": 0.746,
  "last_update": "2025-10-06T15:30:00Z",
  "nomenclator_local": {
    "count": 67000,
    "last_update": "2025-10-06T10:00:00Z"
  },
  "homogeneous_groups_master": {
    "count": 8000
  },
  "system_sync_dates": {
    "nomenclator": "2025-10-06T10:00:00Z",
    "cima": "2025-10-06T14:00:00Z",
    "catalog": "2025-10-06T15:30:00Z"
  }
}
```

**Error Responses:**
- `401 Unauthorized`: Token JWT inválido o expirado
- `403 Forbidden`: Usuario no tiene rol admin
- `422 Unprocessable Entity`: Datos del catálogo inválidos
- `500 Internal Server Error`: Error al recuperar estadísticas

**Frontend Usage:**
```python
import requests

response = requests.get(
    f"{BACKEND_URL}/api/v1/admin/catalog/stats",
    headers={"Authorization": f"Bearer {jwt_token}"}
)

if response.status_code == 200:
    stats = response.json()
    total_products = stats["total_products"]
    enrichment_rate = stats["enrichment_rate"]  # Already calculated by backend
    print(f"Enrichment rate: {enrichment_rate:.2%}")
```

### GET /api/v1/system/catalog/sync-history ⭐ NEW (Issue #349)
**Get catalog synchronization history** - Track CIMA/nomenclator sync operations for admin panel.

**Authentication**: Requires JWT token with `admin` role

**Purpose**: Provides historical data of catalog synchronization operations for monitoring and troubleshooting.

**Query Parameters:**
- `limit`: Number of records to return (1-100, default: 50)
- `offset`: Number of records to skip for pagination (default: 0)
- `sync_type`: Filter by sync type ("cima" or "nomenclator", optional)
- `status`: Filter by sync status ("success", "failure", or "partial", optional)

**Response:**
```json
{
  "history": [
    {
      "sync_date": "2025-10-31T10:30:00Z",
      "sync_type": "cima",
      "status": "success",
      "records_updated": 67234,
      "duration_seconds": 45.2,
      "triggered_by": "automatic",
      "error_message": null
    },
    {
      "sync_date": "2025-10-30T08:15:00Z",
      "sync_type": "nomenclator",
      "status": "failure",
      "records_updated": 0,
      "duration_seconds": 10.5,
      "triggered_by": "manual",
      "error_message": "Connection timeout to API"
    }
  ],
  "total_count": 45
}
```

**Example:**
```python
# Get recent sync history
response = requests.get(
    f"{BACKEND_URL}/api/v1/system/catalog/sync-history",
    headers={"Authorization": f"Bearer {jwt_token}"},
    params={"limit": 20}
)

# Get only failed syncs
response = requests.get(
    f"{BACKEND_URL}/api/v1/system/catalog/sync-history",
    headers={"Authorization": f"Bearer {jwt_token}"},
    params={"status": "failure"}
)
```

**Response Time:** ~50-100ms

**Notes:**
- Results are ordered by sync_date descending (most recent first)
- Sync events are automatically logged by `catalog_maintenance_service` during sync operations
- Useful for debugging sync issues and monitoring catalog update frequency

### POST /api/v1/admin/catalog/clear-checkpoint/{component} ⭐ NEW
**Clear stuck sync checkpoint** - Use when sync got stuck but data is complete.

**Authentication**: Requires JWT token with `admin_catalog_manage` permission

**Path Parameters:**
- `component`: Component name (`cima`, `nomenclator`, `catalog`)

**Response:**
```json
{
  "status": "success",
  "message": "Checkpoint cleared for CIMA",
  "component": "CIMA",
  "previous_state": {
    "status": "ERROR",
    "checkpoint_page": 333,
    "checkpoint_data": "{\"stall_recovery\": true, ...}"
  },
  "new_status": "READY"
}
```

**Use Cases:**
- False positive STALL detection at end of data
- Sync completed but checkpoint stuck in ERROR state
- Recovery from stuck sync without re-downloading all data

**Example:**
```python
response = requests.post(
    f"{BACKEND_URL}/api/v1/admin/catalog/clear-checkpoint/cima",
    headers={"Authorization": f"Bearer {jwt_token}"}
)
```

**Response Time:** ~50ms

### GET /api/v1/admin/atc-coverage ⭐ NEW (Issue #517)
**Get ATC code coverage statistics** - Monitor ATC enrichment progress.

**Authentication**: Requires JWT token with `view_system_stats` permission

**Response:**
```json
{
  "total_cima_products": 68682,
  "products_with_atc": 41904,
  "products_without_atc": 26778,
  "coverage_percentage": 61.0,
  "target_percentage": 60.0,
  "target_reached": true
}
```

**Response Time:** ~100ms

### POST /api/v1/admin/atc-backfill ⭐ NEW (Issue #517)
**Trigger ATC backfill from CIMA API** - Fetch ATC codes for products missing them.

**Authentication**: Requires JWT token with `admin_catalog_manage` permission

**Request Body:**
```json
{
  "batch_size": 100,          // 10-500, products per batch
  "concurrent_requests": 5,   // 1-20, parallel API requests
  "incremental": false,       // true = only last 7 days
  "max_products": 5000        // 1-50000, limit total products
}
```

**Response:**
```json
{
  "status": "completed",
  "processed": 500,
  "successful": 487,
  "failed": 8,
  "skipped": 5,
  "duration_seconds": 45.2,
  "coverage": {
    "total_cima_products": 68682,
    "products_with_atc": 42391,
    "coverage_percentage": 61.7,
    "target_reached": true
  }
}
```

**Response Time:** Variable (depends on max_products, ~100ms per product)

### GET /api/v1/admin/manual-review/stats ⭐ NEW (Issue #447)
**Get statistics for products pending manual review** - Products with sales but no enrichment.

**Authentication**: Requires JWT token with `admin` role

**Query Parameters:**
- `pharmacy_id` (optional, UUID): Filter by specific pharmacy

**Response:**
```json
{
  "total_products": 4491,
  "total_sales": 18234,
  "total_amount": "156789.45",
  "by_code_type": {
    "CN": 4116,
    "EAN": 375,
    "INTERNAL": 0
  }
}
```

**Code Types (stats - simplified):**
- `CN`: National Code (6-7 digits)
- `EAN`: European Article Number (13 digits)
- `INTERNAL`: Other formats

**Response Time:** ~200ms

### GET /api/v1/admin/manual-review/products ⭐ NEW (Issue #447)
**List products pending manual review** - Paginated list with sales metrics and code classification.

**Authentication**: Requires JWT token with `admin` role

**Query Parameters:**
- `pharmacy_id` (optional, UUID): Filter by pharmacy
- `min_sales` (optional, int): Minimum sale count to include
- `page` (optional, int, default=1): Page number
- `page_size` (optional, int, default=100, max=500): Results per page

**Response:**
```json
{
  "products": [
    {
      "product_code": "6543215",
      "product_name": "FRENADOL COMPLEX 10 SOBRES",
      "code_type": "CN_OTC",
      "checksum_valid": true,
      "in_catalog": false,
      "pharmacy_count": 3,
      "sale_count": 45,
      "total_units": 127,
      "total_amount": "1234.56",
      "avg_price": "9.72",
      "first_sale": "2024-01-15",
      "last_sale": "2024-12-10"
    }
  ],
  "total_count": 4491,
  "page": 1,
  "page_size": 100,
  "stats": {
    "total_products": 4491,
    "total_sales": 18234,
    "total_amount": "156789.45",
    "by_code_type": {"CN": 4116, "EAN": 375, "INTERNAL": 0}
  }
}
```

**Code Types (detailed with checksum validation):**
| Type | Description | Checksum | In Catalog |
|------|-------------|----------|------------|
| `CN_CATALOGED` | CN válido, existe en CIMA/Nomenclator | ✅ | ✅ |
| `CN_OTC` | CN válido, NO en catálogos (parafarmacia/OTC) | ✅ | ❌ |
| `EAN_CATALOGED` | EAN-13 válido, existe en catálogos | ✅ | ✅ |
| `EAN_OTC` | EAN-13 válido, NO en catálogos | ✅ | ❌ |
| `CN_UNVERIFIED` | 6 dígitos sin validar (puede ser CN legacy o código interno) | ✅ | ❌ |
| `INTERNAL` | Código interno farmacia | ❌ | - |

**Notes:**
- Products are sorted by `total_amount` descending (most valuable first)
- `checksum_valid=true` + `in_catalog=false` = Likely OTC/parafarmacia (Frenadol, Be+, etc.)
- Use `min_sales` filter to focus on high-volume products

**Response Time:** ~500ms (depends on page_size)

### GET /api/v1/admin/manual-review/export ⭐ NEW (Issue #447)
**Export products pending manual review to CSV** - Download for offline analysis with code classification.

**Authentication**: Requires JWT token with `admin` role

**Query Parameters:**
- `format` (optional, "csv"|"json", default="csv"): Export format
- `pharmacy_id` (optional, UUID): Filter by pharmacy
- `min_sales` (optional, int): Minimum sale count to include

**Response:** CSV file download (StreamingResponse)

**CSV Columns:**
```
product_code,product_name,code_type,checksum_valid,in_catalog,pharmacy_count,sale_count,total_units,total_amount,avg_price,first_sale,last_sale
```

**Example:**
```python
import requests

response = requests.get(
    f"{BACKEND_URL}/api/v1/admin/manual-review/export",
    params={"min_sales": 5},  # Only products with 5+ sales
    headers={"Authorization": f"Bearer {jwt_token}"}
)

# Save to file
with open("manual_review_products.csv", "wb") as f:
    f.write(response.content)
```

**Use Cases:**
- Offline analysis of unclassified products for xFarma_libre taxonomy
- Data exploration before ML clustering (Issue #446)
- Quality review of products pending enrichment
- Identifying OTC products (`CN_OTC`, `EAN_OTC`) for category assignment

**Analysis Tips:**
- `code_type=CN_OTC` or `EAN_OTC`: Products with valid official codes not in CIMA/Nomenclator → parafarmacia candidates
- `code_type=INTERNAL`: Pharmacy-specific codes, services → "Servicios" category
- `pharmacy_count > 1`: Products shared across pharmacies → likely universal codes
- `pharmacy_count = 1`: Unique to one pharmacy → likely internal SKU

**Response Time:** ~1-5s (depends on data volume)

---

## 22. Admin - ML Classifier Monitoring ⭐ NEW (Issue #458)

**Ruta base**: `/api/v1/admin/classifier`

Endpoints para monitoreo de salud del clasificador NECESIDAD (productos venta libre).

### GET /api/v1/admin/classifier/metrics ⭐

Obtener métricas actuales del clasificador.

**Headers:** `Authorization: Bearer {jwt_token}`

**Response:**
```json
{
  "success": true,
  "metrics": {
    "timestamp": "2025-12-21T10:30:00Z",
    "cluster_entropy": 0.85,
    "outlier_rate": 0.03,
    "outlier_count": 150,
    "total_with_umap": 5000,
    "avg_ml_confidence": 0.78,
    "validation_pending": 234,
    "model_decay_7d": 0.02,
    "coverage": 0.92,
    "classified_count": 4600,
    "total_count": 5000,
    "alerts": []
  }
}
```

**Métricas incluidas:**
- `cluster_entropy`: Uniformidad distribución (0-1, mayor = más uniforme)
- `outlier_rate`: Porcentaje de productos fuera de clusters
- `avg_ml_confidence`: Confianza promedio del modelo
- `validation_pending`: Productos pendientes de validación humana
- `coverage`: Porcentaje de productos clasificados (clasificados / total)
- `model_decay_7d`: Degradación del modelo últimos 7 días

### GET /api/v1/admin/classifier/alerts ⭐

Obtener alertas activas del clasificador.

**Headers:** `Authorization: Bearer {jwt_token}`

**Query params:**
- `include_resolved`: bool (default: false)

**Response:**
```json
{
  "success": true,
  "data": {
    "alerts": [
      {
        "id": "alert-123",
        "severity": "warning",
        "title": "Outlier rate elevated",
        "message": "Outlier rate (5.2%) exceeds threshold (5%)",
        "created_at": "2025-12-21T09:00:00Z"
      }
    ]
  }
}
```

**Severities:** `critical`, `warning`, `info`

### GET /api/v1/admin/classifier/distribution ⭐

Obtener distribución de productos por categoría NECESIDAD.

**Headers:** `Authorization: Bearer {jwt_token}`

**Response:**
```json
{
  "success": true,
  "data": {
    "distribution": [
      {
        "category": "Higiene",
        "count": 1250,
        "percentage": 0.25,
        "avg_confidence": 0.82
      },
      {
        "category": "Dermocosmética",
        "count": 980,
        "percentage": 0.196,
        "avg_confidence": 0.75
      }
    ]
  }
}
```

### GET /api/v1/admin/classifier/high-risk ⭐

Obtener productos de alto riesgo (outliers, baja confianza).

**Headers:** `Authorization: Bearer {jwt_token}`

**Query params:**
- `limit`: int (default: 10, max: 100)

**Response:**
```json
{
  "success": true,
  "data": {
    "products": [
      {
        "id": "prod-456",
        "product_name": "Crema hidratante X",
        "ml_category": "Dermocosmética",
        "ml_confidence": 0.45,
        "z_score": 3.2,
        "reason": "Low confidence + high z-score"
      }
    ]
  }
}
```

### GET /api/v1/admin/classifier/metrics/history ⭐

Obtener historial de métricas para gráficas de tendencias.

**Headers:** `Authorization: Bearer {jwt_token}`

**Query params:**
- `days`: int (default: 30, max: 90)
- `metrics`: string (comma-separated, e.g., "cluster_entropy,outlier_rate")

**Response:**
```json
{
  "success": true,
  "data": {
    "history": {
      "cluster_entropy": [
        {"timestamp": "2025-12-20", "value": 0.84},
        {"timestamp": "2025-12-21", "value": 0.85}
      ],
      "outlier_rate": [
        {"timestamp": "2025-12-20", "value": 0.032},
        {"timestamp": "2025-12-21", "value": 0.030}
      ]
    }
  }
}
```

**Use Cases:**
- ML model health monitoring dashboard
- Proactive detection of classifier degradation
- Identification of products needing human validation
- Trend analysis for model retraining decisions

### GET /api/test
Test endpoint for development.

### GET /api/timezone
Test timezone configuration.

---

## 23. Admin - VentaLibre Duplicates Management ⭐ NEW (Issue #477, #480)

**Base URL**: `/api/v1/admin/duplicates`

**Authentication**: Requires JWT token with `admin_catalog_manage` permission

### GET /api/v1/admin/duplicates/pending

List groups of potential duplicate products pending admin review.

**Query Parameters**:
| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `page` | int | 1 | Page number (≥1) |
| `page_size` | int | 20 | Items per page (1-100) |
| `status` | string | "pending_review" | Filter by status: pending_review, confirmed_different, merged |
| `brand` | string | null | Filter by detected brand (case-insensitive partial match) |
| `pharmacy_id` | UUID | null | Filter groups containing products from this pharmacy |
| `date_from` | string | null | Filter by first seen date (YYYY-MM-DD) |
| `sort_by` | string | "similarity" | Sort by: similarity, sales, date |
| `sort_order` | string | "desc" | Sort order: asc, desc |

**Response**:
```json
{
  "groups": [
    {
      "group_id": "uuid",
      "products": [
        {
          "id": "uuid",
          "product_name_display": "FRENADOL COMPLEX 10 SOBRES",
          "product_name_normalized": "frenadol complex",
          "cn_codes": ["123456"],
          "ean13": "8470001234567",
          "total_sales_count": 150,
          "pharmacies_count": 3,
          "ml_category": "RESFRIADO",
          "ml_confidence": 0.95,
          "human_verified": false,
          "first_seen_at": "2025-01-15T10:30:00Z"
        }
      ],
      "similarity_score": 1.0,
      "tokens": "frenadol complex 10 sobres",
      "suggested_primary_id": "uuid",
      "created_at": "2025-01-15T10:30:00Z"
    }
  ],
  "total_groups": 45,
  "total_products_affected": 120,
  "page": 1,
  "page_size": 20,
  "has_more": true
}
```

### GET /api/v1/admin/duplicates/stats

Get statistics about duplicates in the VentaLibre catalog.

**Response**:
```json
{
  "total_with_tokens": 12500,
  "total_without_tokens": 350,
  "pending_review": 45,
  "confirmed_different": 120,
  "merged": 85,
  "coverage_pct": 97.3
}
```

### GET /api/v1/admin/duplicates/filter-options ⭐ NEW (Issue #480)

Get available filter options for the duplicates UI (brands dropdown).

**Response**:
```json
{
  "brands": ["FRENADOL", "GELOCATIL", "NUROFEN", ...],
  "statuses": ["pending_review", "confirmed_different", "merged"]
}
```

### POST /api/v1/admin/duplicates/merge

Merge secondary products into a primary product.

**Request Body**:
```json
{
  "primary_id": "uuid-of-product-to-keep",
  "secondary_ids": ["uuid-1", "uuid-2"]
}
```

**Response**:
```json
{
  "success": true,
  "primary_id": "uuid",
  "merged_count": 2,
  "sales_redirected": 150,
  "message": "Successfully merged 2 products into primary. 150 sales redirected."
}
```

### POST /api/v1/admin/duplicates/reject

Mark products as different (not duplicates).

**Request Body**:
```json
{
  "product_ids": ["uuid-1", "uuid-2"],
  "group_id": "uuid-optional"
}
```

**Response**:
```json
{
  "success": true,
  "products_updated": 2,
  "message": "Marked 2 products as different (not duplicates)"
}
```

### POST /api/v1/admin/duplicates/skip

Skip/postpone decision on a duplicate group.

**Request Body**:
```json
{
  "group_id": "uuid-of-group"
}
```

**Response**:
```json
{
  "success": true,
  "message": "Skipped group with 3 products"
}
```

---

## 24. Admin - Brand Aliases Management ⭐ NEW

Sistema de corrección de marcas para BrandDetectionService.

**Base URL**: `/api/v1/admin/brand-aliases`

### Conceptos

| Acción | Descripción | Ejemplo |
|--------|-------------|---------|
| `alias` | Fusionar marca A con marca B | "oralkin" → "kin" |
| `exclude` | Excluir falso positivo (no es marca) | "clorhexidina" → (excluir) |

### GET /api/v1/admin/brand-aliases

Lista brand aliases con filtros y estadísticas.

**Query Parameters**:
| Parámetro | Tipo | Descripción |
|-----------|------|-------------|
| `is_active` | bool | Filtrar por estado activo/inactivo |
| `action` | string | Filtrar por acción: "alias" o "exclude" |
| `search` | string | Buscar en source_brand o target_brand |
| `limit` | int | Máximo resultados (default: 100) |
| `offset` | int | Offset para paginación |

**Response**:
```json
{
  "items": [
    {
      "id": 1,
      "source_brand": "oralkin",
      "target_brand": "kin",
      "action": "alias",
      "is_active": true,
      "reason": "Misma marca, nombre diferente",
      "usage_count": 245,
      "last_used_at": "2026-01-06T10:30:00Z",
      "created_at": "2026-01-06T09:00:00Z",
      "affected_count": 42
    }
  ],
  "total": 5,
  "offset": 0,
  "limit": 100,
  "stats": {
    "total": 5,
    "active": 4,
    "inactive": 1,
    "aliases": 4,
    "excludes": 1,
    "total_usage": 1250,
    "most_used": "oralkin",
    "least_used": "ivb wellness"
  }
}
```

### GET /api/v1/admin/brand-aliases/stats

Obtiene estadísticas de brand aliases.

**Response**: Igual que `stats` del endpoint anterior.

### POST /api/v1/admin/brand-aliases

Crea un nuevo brand alias.

**Request Body**:
```json
{
  "source_brand": "oralkin",
  "target_brand": "kin",
  "action": "alias",
  "reason": "Misma marca con nombre diferente"
}
```

**Nota**: `target_brand` es obligatorio para `action: "alias"`, ignorado para `action: "exclude"`.

**Response**: `201 Created` con el alias creado.

### PUT /api/v1/admin/brand-aliases/{id}

Actualiza un brand alias existente.

**Request Body** (campos opcionales):
```json
{
  "source_brand": "new_source",
  "target_brand": "new_target",
  "action": "exclude",
  "reason": "Actualizado"
}
```

**Response**: `200 OK` con el alias actualizado.

### DELETE /api/v1/admin/brand-aliases/{id}

Elimina un brand alias.

**Query Parameters**:
| Parámetro | Tipo | Default | Descripción |
|-----------|------|---------|-------------|
| `hard` | bool | false | Si true, elimina permanentemente. Si false, soft delete (desactiva). |

**Response**: `200 OK`

### POST /api/v1/admin/brand-aliases/{id}/toggle

Alterna el estado activo/inactivo de un brand alias.

**Response**: `200 OK` con el alias actualizado.

### POST /api/v1/admin/brand-aliases/{id}/reprocess ⭐

Re-procesa productos afectados por un brand alias.

**Descripción**: Actualiza todos los `SalesEnrichment` que tienen `detected_brand = source_brand`:
- Si `action = "alias"`: Renombra a `target_brand`
- Si `action = "exclude"`: Establece `detected_brand = NULL`

**Response**:
```json
{
  "alias_id": 1,
  "source_brand": "oralkin",
  "target_brand": "kin",
  "action": "alias",
  "updated_count": 42,
  "message": "Renombrado 'oralkin' → 'kin' en 42 productos"
}
```

**Uso típico**: Después de crear un alias, usar este endpoint para aplicarlo a productos existentes.

---

## 🚀 Quick Reference

### Most Used Endpoints
1. `POST /api/v1/login` - User login (email/password o OAuth) ⭐
2. `POST /api/v1/upload/` - Upload sales files
3. `GET /api/system/status` - System status
4. `POST /api/v1/invitations/create` - Crear invitación (admin-only) 🆕
5. `GET /api/v1/laboratory-mapping/codes-to-names` - Laboratory code to name mapping
6. `GET /api/sales/summary` - Sales analytics
7. `GET /api/generic-analysis/dashboard/{pharmacy_id}` - Generic analysis
8. `GET /api/reenrichment/status/{pharmacy_id}` - Check re-enrichment needs

### Response Times (Expected)
- Simple queries: < 100ms
- Analytics endpoints: < 500ms
- File upload: < 30s for 10MB
- Catalog sync: 5-10 minutes
- Dashboard status: < 300ms (cached)

### Common Headers
```javascript
const headers = {
  'Authorization': `Bearer ${token}`,
  'Content-Type': 'application/json',
  'Accept': 'application/json',
  'X-Request-ID': generateRequestId(), // Para tracking
  'X-Client-Version': '1.0.0'
};
```

### Environment URLs
- **Development**: http://localhost:8000/api
- **Staging**: https://api-staging.xfarma.com/api
- **Production**: https://api.xfarma.com/api

### API Path
Todos los endpoints usan el prefijo `/api/`

---

## 📝 Notes for Frontend Developers

### Authentication Flow
```javascript
// 1. Login
const loginResponse = await fetch('/api/auth/login', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify({ email, password })
});

// 2. Store tokens
const { access_token, refresh_token } = await loginResponse.json();
localStorage.setItem('access_token', access_token);
localStorage.setItem('refresh_token', refresh_token);

// 3. Use in requests
const response = await fetch('/api/sales/summary', {
  headers: {
    'Authorization': `Bearer ${access_token}`
  }
});

// 4. Refresh when needed
if (response.status === 401) {
  const refreshResponse = await fetch('/api/auth/refresh-token', {
    method: 'POST',
    body: JSON.stringify({ refresh_token })
  });
  // Update tokens and retry
}
```

### File Upload with Progress
```javascript
function uploadFileWithProgress(file, onProgress) {
  return new Promise((resolve, reject) => {
    const formData = new FormData();
    formData.append('file', file);
    
    const xhr = new XMLHttpRequest();
    
    xhr.upload.addEventListener('progress', (event) => {
      if (event.lengthComputable) {
        const percentComplete = (event.loaded / event.total) * 100;
        onProgress(percentComplete);
      }
    });
    
    xhr.addEventListener('load', () => {
      if (xhr.status === 200) {
        resolve(JSON.parse(xhr.responseText));
      } else {
        reject(new Error(`Upload failed: ${xhr.status}`));
      }
    });
    
    xhr.open('POST', '/api/v1/upload/');
    xhr.setRequestHeader('Authorization', `Bearer ${getToken()}`);
    xhr.send(formData);
  });
}
```

### Polling Pattern for Long Operations
```javascript
async function pollStatus(uploadId) {
  const poll = async () => {
    const response = await fetch(`/api/v1/upload/status/${uploadId}`);
    const data = await response.json();
    
    if (data.status === 'processing') {
      setTimeout(poll, 2000); // Poll every 2 seconds
    } else if (data.status === 'completed') {
      onComplete(data);
    } else if (data.status === 'failed') {
      onError(data);
    }
  };
  
  poll();
}
```

### Error Handling Best Practices
```javascript
class APIError extends Error {
  constructor(response, data) {
    super(data?.error?.message || 'API Error');
    this.status = response.status;
    this.code = data?.error?.code;
    this.details = data?.error?.details;
  }
}

async function apiCall(url, options = {}) {
  try {
    const response = await fetch(url, {
      ...options,
      headers: {
        'Authorization': `Bearer ${getToken()}`,
        'Content-Type': 'application/json',
        ...options.headers
      }
    });
    
    const data = await response.json();
    
    if (!response.ok) {
      throw new APIError(response, data);
    }
    
    return data;
  } catch (error) {
    if (error instanceof APIError) {
      // Handle API errors
      console.error(`API Error ${error.status}: ${error.message}`);
      
      if (error.status === 401) {
        // Redirect to login
        window.location.href = '/login';
      } else if (error.status === 429) {
        // Rate limited - wait and retry
        await new Promise(resolve => setTimeout(resolve, 5000));
        return apiCall(url, options);
      }
    } else {
      // Network or other errors
      console.error('Network error:', error);
    }
    throw error;
  }
}
```

### Date Handling
```javascript
// Convert local date to API format
function toAPIDate(localDate) {
  return new Date(localDate).toISOString().split('T')[0];
}

// Parse API date to local
function fromAPIDate(apiDate) {
  return new Date(apiDate + 'T00:00:00');
}

// Format for display
function formatDate(date) {
  return new Intl.DateTimeFormat('es-ES').format(date);
}
```

---

## 15. System Unified (New)

### GET /api/v1/system-unified/health-summary ⭐

**Nueva API unificada para estado del sistema**

**Descripción**: Resumen de salud del sistema para diferentes audiencias

**Query Parameters**:
- `level` (string): Nivel de detalle
  - `basic`: Para farmacéuticos (estado simple)
  - `detailed`: Para administradores (métricas técnicas)
  - `diagnostic`: Para desarrolladores (información debug)

**Response Example (basic)**:
```json
{
  "success": true,
  "data": {
    "overall_status": "operational",
    "status_display": {
      "text": "OPERATIVO",
      "color": "success",
      "icon": "check-circle"
    },
    "catalog_ready": true,
    "total_products": 67000,
    "last_check": "2025-09-21T19:54:29.879557",
    "alerts": []
  },
  "message": "Resumen de salud (basic) obtenido exitosamente"
}
```

**Frontend Usage**:
```javascript
// Estado básico para farmacéuticos
const basicStatus = await apiClient.get('/api/v1/system-unified/health-summary?level=basic');

// Estado detallado para administradores
const detailedStatus = await apiClient.get('/api/v1/system-unified/health-summary?level=detailed');
```

### GET /api/v1/system-unified/status-complete

**Estado completo del sistema consolidado**

**Descripción**: Combina estado básico, métricas de rendimiento, componentes y actividad reciente

**Query Parameters**:
- `include_metrics` (bool): Incluir métricas de rendimiento (default: true)
- `include_components` (bool): Incluir estado de componentes (default: true)
- `include_recent_activity` (bool): Incluir actividad reciente (default: false)

**Response Example**:
```json
{
  "success": true,
  "data": {
    "status": "operational",
    "timestamp": "2025-09-21T19:54:29.879557",
    "version": "1.0.0",
    "environment": "development",
    "components": {
      "database": {
        "status": "operational",
        "catalog_products": 67000,
        "sales_records": 15000
      },
      "catalog": {
        "status": "operational",
        "total_products": 67000,
        "cima_products": 30000,
        "nomenclator_products": 37000
      }
    },
    "metrics": {
      "database_response_time_ms": 45.2,
      "memory_usage_mb": 245.7,
      "cpu_usage_percent": 15.3
    }
  },
  "message": "Estado del sistema obtenido exitosamente"
}
```

---

## 16. Admin Tools (New)

### GET /api/v1/admin-tools/dangerous-operations ⭐

**Lista operaciones peligrosas disponibles**

**Descripción**: Operaciones que requieren confirmación especial y pueden afectar datos

**Response Example**:
```json
{
  "success": true,
  "message": "Operaciones peligrosas listadas",
  "data": {
    "operations": [
      {
        "id": "delete_all_sales_data",
        "name": "Eliminar todos los datos de ventas",
        "description": "Elimina TODOS los registros de ventas de TODAS las farmacias",
        "risk_level": "CRITICAL",
        "confirmation_required": "ELIMINAR VENTAS",
        "estimated_time": "30 segundos",
        "reversible": false
      },
      {
        "id": "vacuum_database",
        "name": "VACUUM completo de base de datos",
        "description": "Optimiza y reorganiza toda la base de datos",
        "risk_level": "LOW",
        "confirmation_required": "VACUUM DB",
        "estimated_time": "10-15 minutos",
        "reversible": true
      }
    ]
  }
}
```

### POST /api/v1/admin-tools/dangerous-operations/{operation_id}/execute

**Ejecuta operación peligrosa después de verificar confirmación**

**Path Parameters**:
- `operation_id` (string): ID de la operación a ejecutar

**Request Body**:
```json
{
  "confirmation_text": "ELIMINAR VENTAS"
}
```

**Response Example**:
```json
{
  "success": true,
  "message": "Operación 'delete_all_sales_data' iniciada en segundo plano",
  "data": {
    "operation_id": "delete_all_sales_data",
    "started_at": "2025-09-21T19:54:29.879557",
    "status": "in_progress"
  }
}
```

### POST /api/v1/admin-tools/maintenance/clear-cache

**Limpia todas las cachés del sistema**

**Descripción**: Operación segura que limpia cachés en memoria y Redis

**Response Example**:
```json
{
  "success": true,
  "message": "Caché del sistema limpiada exitosamente",
  "data": {
    "cleared_at": "2025-09-21T19:54:29.879557"
  }
}
```

### POST /api/v1/admin-tools/maintenance/verify-connectivity

**Verifica conectividad de sistemas críticos**

**Descripción**: Prueba conexiones a base de datos, APIs externas y servicios

**Response Example**:
```json
{
  "success": true,
  "message": "Verificación de conectividad completada",
  "data": {
    "database": {
      "status": "operational",
      "response_time_ms": 25.4,
      "last_check": "2025-09-21T19:54:29.879557"
    },
    "cima_api": {
      "status": "operational",
      "response_time_ms": 156.7,
      "last_check": "2025-09-21T19:54:29.879557"
    },
    "redis": {
      "status": "operational",
      "response_time_ms": 3.2,
      "last_check": "2025-09-21T19:54:29.879557"
    }
  }
}
```

### GET /api/v1/admin-tools/system-analysis

**Análisis del sistema para identificar problemas**

**Query Parameters**:
- `analysis_type` (string): Tipo de análisis
  - `full`: Análisis completo del sistema
  - `performance`: Análisis de rendimiento
  - `data`: Análisis de datos e integridad
  - `security`: Análisis de seguridad

**Response Example**:
```json
{
  "success": true,
  "message": "Análisis del sistema (full) completado",
  "data": {
    "analysis_type": "full",
    "timestamp": "2025-09-21T19:54:29.879557",
    "performance": {
      "database_performance": "good",
      "memory_usage": "normal",
      "score": 90
    },
    "data": {
      "catalog_integrity": "good",
      "orphaned_records": 0,
      "score": 88
    },
    "security": {
      "vulnerabilities": [],
      "recommendations": [
        "Implementar autenticación de dos factores"
      ],
      "score": 82
    },
    "overall_score": 85
  }
}
```

**Frontend Usage**:
```javascript
// Listar operaciones peligrosas
const operations = await apiClient.get('/api/v1/admin-tools/dangerous-operations');

// Ejecutar operación con confirmación
const result = await apiClient.post('/api/v1/admin-tools/dangerous-operations/vacuum_database/execute', {
  confirmation_text: 'VACUUM DB'
});

// Análisis del sistema
const analysis = await apiClient.get('/api/v1/admin-tools/system-analysis?analysis_type=performance');
```

---

## 17. Admin - User Management (NEW)

**Ruta base**: `/api/v1/admin/users`

Endpoints para gestión completa de usuarios y farmacias con RBAC (Issue #348 FASE 2).

### POST /api/v1/admin/users ⭐

**Crear usuario con farmacia asociada** - Transacción atómica

**Permisos requeridos**: `MANAGE_USERS`

**Rate Limit**: 10 requests/minuto

**Request Body**:
```json
{
  "user": {
    "email": "farmacia.central@example.com",
    "full_name": "Farmacia Central",
    "password": "SecurePass123!",
    "role": "user",
    "subscription_plan": "free"
  },
  "pharmacy": {
    "name": "Farmacia Central",
    "nif": "B12345678",
    "address": "Calle Mayor 1",
    "city": "Madrid",
    "postal_code": "28001",
    "phone": "+34912345678"
  }
}
```

**Response** (201 Created):
```json
{
  "id": "550e8400-e29b-41d4-a716-446655440000",
  "email": "farmacia.central@example.com",
  "full_name": "Farmacia Central",
  "role": "user",
  "subscription_plan": "free",
  "is_active": true,
  "created_at": "2025-11-01T10:30:00Z",
  "pharmacy": {
    "id": "660e8400-e29b-41d4-a716-446655440001",
    "name": "Farmacia Central",
    "nif": "B12345678",
    "address": "Calle Mayor 1",
    "city": "Madrid",
    "postal_code": "28001",
    "phone": "+34912345678"
  }
}
```

**Error Responses**:
- `400 Bad Request`: Email ya existe, NIF duplicado, o farmacia ya tiene usuario (REGLA #10)
- `403 Forbidden`: Usuario sin permisos `MANAGE_USERS`
- `422 Unprocessable Entity`: Datos inválidos (password débil, email malformado, NIF inválido)
- `429 Too Many Requests`: Rate limit excedido

**Frontend Usage**:
```javascript
const response = await fetch(`${BACKEND_URL}/api/v1/admin/users`, {
  method: 'POST',
  headers: {
    'Authorization': `Bearer ${token}`,
    'Content-Type': 'application/json'
  },
  body: JSON.stringify({
    user: {
      email: 'farmacia.sol@example.com',
      full_name: 'Farmacia del Sol',
      password: 'P@ssw0rd123!',
      role: 'user',
      subscription_plan: 'pro'
    },
    pharmacy: {
      name: 'Farmacia del Sol',
      nif: 'B87654321',
      address: 'Avenida Libertad 45',
      city: 'Barcelona',
      postal_code: '08001',
      phone: '+34934567890'
    }
  })
});

if (response.ok) {
  const user = await response.json();
  console.log(`Usuario creado: ${user.email} → Farmacia: ${user.pharmacy.name}`);
}
```

---

### GET /api/v1/admin/users ⭐

**Listar usuarios con paginación y filtros**

**Permisos requeridos**: `VIEW_USERS`

**Query Parameters**:
- `skip` (int, default: 0): Offset para paginación
- `limit` (int, default: 100, max: 500): Límite de resultados
- `role` (string, opcional): Filtrar por rol (`admin`, `user`)
- `subscription` (string, opcional): Filtrar por plan (`free`, `pro`, `max`)
- `include_deleted` (bool, default: false): Incluir usuarios con soft delete

**Response** (200 OK):
```json
{
  "users": [
    {
      "id": "550e8400-e29b-41d4-a716-446655440000",
      "email": "farmacia.central@example.com",
      "full_name": "Farmacia Central",
      "role": "user",
      "subscription_plan": "pro",
      "is_active": true,
      "created_at": "2025-10-15T08:00:00Z",
      "deleted_at": null,
      "pharmacy": {
        "id": "660e8400-e29b-41d4-a716-446655440001",
        "name": "Farmacia Central",
        "nif": "B12345678",
        "city": "Madrid"
      }
    },
    {
      "id": "770e8400-e29b-41d4-a716-446655440002",
      "email": "farmacia.sol@example.com",
      "full_name": "Farmacia del Sol",
      "role": "user",
      "subscription_plan": "free",
      "is_active": false,
      "created_at": "2025-09-20T12:30:00Z",
      "deleted_at": "2025-10-30T18:45:00Z",
      "pharmacy": {
        "id": "880e8400-e29b-41d4-a716-446655440003",
        "name": "Farmacia del Sol",
        "nif": "B87654321",
        "city": "Barcelona"
      }
    }
  ],
  "total": 2,
  "skip": 0,
  "limit": 100
}
```

**Frontend Usage**:
```javascript
// Listar usuarios activos con plan Pro
const response = await fetch(
  `${BACKEND_URL}/api/v1/admin/users?role=user&subscription=pro&skip=0&limit=50`,
  {
    headers: { 'Authorization': `Bearer ${token}` }
  }
);

const { users, total } = await response.json();
console.log(`${total} usuarios encontrados (mostrando ${users.length})`);
```

---

### GET /api/v1/admin/users/{user_id}

**Obtener detalles de un usuario específico**

**Permisos requeridos**: `VIEW_USERS`

**Path Parameters**:
- `user_id` (UUID): ID del usuario

**Response** (200 OK):
```json
{
  "id": "550e8400-e29b-41d4-a716-446655440000",
  "email": "farmacia.central@example.com",
  "full_name": "Farmacia Central",
  "role": "user",
  "subscription_plan": "pro",
  "is_active": true,
  "created_at": "2025-10-15T08:00:00Z",
  "updated_at": "2025-10-28T14:20:00Z",
  "deleted_at": null,
  "pharmacy": {
    "id": "660e8400-e29b-41d4-a716-446655440001",
    "name": "Farmacia Central",
    "nif": "B12345678",
    "address": "Calle Mayor 1",
    "city": "Madrid",
    "postal_code": "28001",
    "phone": "+34912345678",
    "created_at": "2025-10-15T08:00:00Z"
  }
}
```

**Error Responses**:
- `403 Forbidden`: Usuario sin permisos `VIEW_USERS`
- `404 Not Found`: Usuario no encontrado

---

### PUT /api/v1/admin/users/{user_id}

**Actualizar usuario** - Solo role, subscription_plan, is_active

**Permisos requeridos**: `MANAGE_USERS`

**Rate Limit**: 10 requests/minuto

**Path Parameters**:
- `user_id` (UUID): ID del usuario

**Request Body** (todos los campos opcionales):
```json
{
  "role": "admin",
  "subscription_plan": "max",
  "is_active": true
}
```

**Response** (200 OK):
```json
{
  "id": "550e8400-e29b-41d4-a716-446655440000",
  "email": "farmacia.central@example.com",
  "full_name": "Farmacia Central",
  "role": "admin",
  "subscription_plan": "max",
  "is_active": true,
  "updated_at": "2025-11-01T16:45:00Z"
}
```

**Error Responses**:
- `400 Bad Request`: Intento de cambiar propio rol (prohibido)
- `403 Forbidden`: Usuario sin permisos `MANAGE_USERS`
- `404 Not Found`: Usuario no encontrado
- `429 Too Many Requests`: Rate limit excedido

**Frontend Usage**:
```javascript
// Promover usuario a admin
const response = await fetch(`${BACKEND_URL}/api/v1/admin/users/${userId}`, {
  method: 'PUT',
  headers: {
    'Authorization': `Bearer ${token}`,
    'Content-Type': 'application/json'
  },
  body: JSON.stringify({ role: 'admin' })
});

if (response.ok) {
  const updatedUser = await response.json();
  console.log(`Usuario promovido a ${updatedUser.role}`);
}
```

---

### DELETE /api/v1/admin/users/{user_id}

**Soft delete de usuario** - GDPR Article 17 compliant

**Permisos requeridos**: `MANAGE_USERS`

**Rate Limit**: 10 requests/minuto

**Path Parameters**:
- `user_id` (UUID): ID del usuario

**Response** (204 No Content)

**Error Responses**:
- `400 Bad Request`: Usuario ya eliminado
- `403 Forbidden`: Usuario sin permisos `MANAGE_USERS`
- `404 Not Found`: Usuario no encontrado
- `429 Too Many Requests`: Rate limit excedido

**Frontend Usage**:
```javascript
// Soft delete de usuario (GDPR compliant)
const response = await fetch(`${BACKEND_URL}/api/v1/admin/users/${userId}`, {
  method: 'DELETE',
  headers: { 'Authorization': `Bearer ${token}` }
});

if (response.status === 204) {
  console.log('Usuario eliminado (soft delete)');
}
```

---

### POST /api/v1/admin/users/{user_id}/restore

**Restaurar usuario desde soft delete**

**Permisos requeridos**: `MANAGE_USERS`

**Rate Limit**: 10 requests/minuto

**Path Parameters**:
- `user_id` (UUID): ID del usuario eliminado

**Response** (200 OK):
```json
{
  "id": "550e8400-e29b-41d4-a716-446655440000",
  "email": "farmacia.central@example.com",
  "full_name": "Farmacia Central",
  "role": "user",
  "subscription_plan": "pro",
  "is_active": true,
  "deleted_at": null,
  "updated_at": "2025-11-01T17:00:00Z"
}
```

**Error Responses**:
- `400 Bad Request`: Usuario no está eliminado
- `403 Forbidden`: Usuario sin permisos `MANAGE_USERS`
- `404 Not Found`: Usuario no encontrado
- `429 Too Many Requests`: Rate limit excedido

---

### GET /api/v1/admin/storage/usage ⭐

**Obtener uso de almacenamiento por farmacia**

**Permisos requeridos**: `VIEW_SYSTEM_STATS`

**Query Parameters**:
- `pharmacy_id` (UUID, opcional): Filtrar por farmacia específica

**Response** (200 OK):
```json
[
  {
    "pharmacy_id": "660e8400-e29b-41d4-a716-446655440001",
    "pharmacy_name": "Farmacia Central",
    "total_sales_records": 50000,
    "total_uploads": 120,
    "storage_mb": 1024.5,
    "storage_percentage": 10.0,
    "subscription_plan": "pro",
    "storage_limit_mb": 10240
  },
  {
    "pharmacy_id": "880e8400-e29b-41d4-a716-446655440003",
    "pharmacy_name": "Farmacia del Sol",
    "total_sales_records": 25000,
    "total_uploads": 60,
    "storage_mb": 512.2,
    "storage_percentage": 50.2,
    "subscription_plan": "free",
    "storage_limit_mb": 1024
  }
]
```

**Storage Limits por Plan**:
- **free**: 1 GB
- **pro**: 10 GB
- **max**: 100 GB

**Frontend Usage**:
```javascript
// Obtener uso de almacenamiento de todas las farmacias
const response = await fetch(`${BACKEND_URL}/api/v1/admin/storage/usage`, {
  headers: { 'Authorization': `Bearer ${token}` }
});

const storageStats = await response.json();

// Identificar farmacias cerca del límite
const nearLimit = storageStats.filter(s => s.storage_percentage > 80);
console.log(`${nearLimit.length} farmacias cerca del límite de almacenamiento`);
```

---

## 18. Admin - Database Management (NEW)

**Ruta base**: `/api/v1/admin/database`

Endpoints para gestión de backups, estadísticas y mantenimiento de base de datos (Issue #348 FASE 2).

### POST /api/v1/admin/database/backups ⭐

**Crear backup de base de datos** - Con SHA-256 y HMAC signature

**Permisos requeridos**: `MANAGE_DATABASE`

**Rate Limit**: 3 requests/hora (operación costosa)

**Request Body** (opcional):
```json
{
  "path": "/backups/backup_20251101_123045.sql"
}
```

**Response** (201 Created):
```json
{
  "id": 42,
  "file_path": "/backups/backup_20251101_123045.sql",
  "file_size": 104857600,
  "sha256_hash": "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855",
  "hmac_signature": "5d41402abc4b2a76b9719d911017c592",
  "status": "completed",
  "created_at": "2025-11-01T12:30:45Z"
}
```

**Características del Backup**:
- **SHA-256**: Hash criptográfico para verificar integridad
- **HMAC Signature**: Firma HMAC para detectar manipulación
- **Compresión**: Automática con gzip
- **Formato**: PostgreSQL dump (SQL)

**Error Responses**:
- `403 Forbidden`: Usuario sin permisos `MANAGE_DATABASE`
- `429 Too Many Requests`: Rate limit excedido (máx 3/hora)
- `500 Internal Server Error`: Error al crear backup

**Frontend Usage**:
```javascript
// Crear backup con ruta personalizada
const response = await fetch(`${BACKEND_URL}/api/v1/admin/database/backups`, {
  method: 'POST',
  headers: {
    'Authorization': `Bearer ${token}`,
    'Content-Type': 'application/json'
  },
  body: JSON.stringify({
    path: `/backups/backup_${new Date().toISOString().split('T')[0]}.sql`
  })
});

if (response.ok) {
  const backup = await response.json();
  console.log(`Backup creado: ${backup.file_path} (${(backup.file_size / 1024 / 1024).toFixed(2)} MB)`);
  console.log(`Hash: ${backup.sha256_hash.substring(0, 16)}...`);
}
```

---

### GET /api/v1/admin/database/backups

**Listar backups con verificación de integridad**

**Permisos requeridos**: `VIEW_SYSTEM_STATS`

**Query Parameters**:
- `limit` (int, default: 20, max: 100): Número de backups a listar

**Response** (200 OK):
```json
[
  {
    "id": 42,
    "file_path": "/backups/backup_20251101_123045.sql",
    "file_size": 104857600,
    "sha256_hash": "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855",
    "status": "completed",
    "created_at": "2025-11-01T12:30:45Z",
    "is_verified": true,
    "error_message": null
  },
  {
    "id": 41,
    "file_path": "/backups/backup_20251031_180000.sql",
    "file_size": 98304000,
    "sha256_hash": "d2d2d2d298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b123",
    "status": "completed",
    "created_at": "2025-10-31T18:00:00Z",
    "is_verified": true,
    "error_message": null
  },
  {
    "id": 40,
    "file_path": "/backups/backup_20251030_120000.sql",
    "file_size": 0,
    "sha256_hash": null,
    "status": "failed",
    "created_at": "2025-10-30T12:00:00Z",
    "is_verified": false,
    "error_message": "Disk full - backup aborted"
  }
]
```

**Status Values**:
- `completed`: Backup exitoso y verificado
- `in_progress`: Backup en proceso
- `failed`: Backup falló
- `corrupted`: Archivo corrupto (hash no coincide)

**Frontend Usage**:
```javascript
// Listar últimos 10 backups
const response = await fetch(`${BACKEND_URL}/api/v1/admin/database/backups?limit=10`, {
  headers: { 'Authorization': `Bearer ${token}` }
});

const backups = await response.json();
const verified = backups.filter(b => b.is_verified);
console.log(`${verified.length}/${backups.length} backups verificados`);
```

---

### POST /api/v1/admin/database/storage/refresh

**Actualizar vista materializada de almacenamiento**

**Permisos requeridos**: `MANAGE_DATABASE`

**Rate Limit**: 10 requests/hora

**Descripción**: Ejecuta `REFRESH MATERIALIZED VIEW pharmacy_storage_stats` para actualizar estadísticas de uso de almacenamiento por farmacia.

**Response** (204 No Content)

**Error Responses**:
- `403 Forbidden`: Usuario sin permisos `MANAGE_DATABASE`
- `429 Too Many Requests`: Rate limit excedido
- `500 Internal Server Error`: Error al actualizar vista

**Frontend Usage**:
```javascript
// Refrescar estadísticas de almacenamiento
const response = await fetch(`${BACKEND_URL}/api/v1/admin/database/storage/refresh`, {
  method: 'POST',
  headers: { 'Authorization': `Bearer ${token}` }
});

if (response.status === 204) {
  console.log('Estadísticas de almacenamiento actualizadas');
  // Recargar datos de almacenamiento
  loadStorageStats();
}
```

---

### GET /api/v1/admin/database/size ⭐

**Obtener tamaño de base de datos y tablas principales**

**Permisos requeridos**: `VIEW_SYSTEM_STATS`

**Response** (200 OK):
```json
{
  "database_size": 2147483648,
  "database_size_pretty": "2.00 GB",
  "top_tables": [
    {
      "table_name": "sales_data",
      "size_bytes": 1073741824,
      "size_mb": 1024.0,
      "size_pretty": "1.00 GB",
      "row_count": 500000,
      "percentage": 50.0
    },
    {
      "table_name": "product_catalog",
      "size_bytes": 536870912,
      "size_mb": 512.0,
      "size_pretty": "512.00 MB",
      "row_count": 67000,
      "percentage": 25.0
    },
    {
      "table_name": "sales_enrichment",
      "size_bytes": 268435456,
      "size_mb": 256.0,
      "size_pretty": "256.00 MB",
      "row_count": 450000,
      "percentage": 12.5
    },
    {
      "table_name": "nomenclator_local",
      "size_bytes": 134217728,
      "size_mb": 128.0,
      "size_pretty": "128.00 MB",
      "row_count": 67000,
      "percentage": 6.25
    }
  ]
}
```

**Frontend Usage**:
```javascript
// Obtener tamaño de base de datos
const response = await fetch(`${BACKEND_URL}/api/v1/admin/database/size`, {
  headers: { 'Authorization': `Bearer ${token}` }
});

const sizeInfo = await response.json();
console.log(`Tamaño total de BD: ${sizeInfo.database_size_pretty}`);

// Mostrar tabla más grande
const largestTable = sizeInfo.top_tables[0];
console.log(`Tabla más grande: ${largestTable.table_name} (${largestTable.size_pretty}, ${largestTable.row_count.toLocaleString('es-ES')} registros)`);
```

---

## 19. Subscription Management (NEW - Issue #444) ⭐

**Ruta base**: `/api/v1/subscription` (user) y `/api/v1/admin` (admin)

Endpoints para gestión de suscripciones y caducidad de planes PRO/MAX.

### GET /api/v1/subscription/status ⭐

**Obtener estado de suscripción del usuario actual**

**Response** (200 OK):
```json
{
  "plan": "pro",
  "is_active": true,
  "subscription_start": "2025-01-15T10:00:00Z",
  "subscription_end": "2026-03-31T23:59:59Z",
  "days_remaining": 111,
  "is_expiring_soon": false,
  "is_expired": false
}
```

**Campos importantes**:
- `days_remaining`: Días hasta expiración (null para FREE)
- `is_expiring_soon`: true si <= 7 días restantes
- `is_expired`: true si la suscripción ha expirado

**Frontend Usage**:
```javascript
// Obtener estado de suscripción
const response = await fetch(`${BACKEND_URL}/api/v1/subscription/status`, {
  headers: { 'Authorization': `Bearer ${token}` }
});

const status = await response.json();

// Mostrar banner de aviso si expira pronto
if (status.is_expiring_soon) {
  showExpirationBanner(status.days_remaining);
}
```

---

### PUT /api/v1/admin/users/{user_id}/subscription ⭐

**Actualizar suscripción de un usuario**

**Permisos requeridos**: `MANAGE_USERS`

**Rate Limit**: 10 requests/minuto

**Path Parameters**:
- `user_id` (UUID): ID del usuario

**Request Body**:
```json
{
  "plan": "pro",
  "subscription_end": "2026-03-31T23:59:59Z",
  "subscription_start": "2025-12-10T00:00:00Z"
}
```

**Campos**:
- `plan` (required): `free`, `pro` o `max`
- `subscription_end` (required for pro/max): Fecha de expiración
- `subscription_start` (optional): Fecha de inicio (default: now)

**Response** (200 OK):
```json
{
  "success": true,
  "message": "Subscription updated to pro",
  "user_id": "550e8400-e29b-41d4-a716-446655440000",
  "plan": "pro",
  "subscription_end": "2026-03-31T23:59:59Z"
}
```

**Error Responses**:
- `400 Bad Request`: Plan inválido o subscription_end faltante para PRO/MAX
- `403 Forbidden`: Usuario sin permisos `MANAGE_USERS`
- `404 Not Found`: Usuario no encontrado
- `429 Too Many Requests`: Rate limit excedido

**Frontend Usage**:
```javascript
// Actualizar suscripción a PRO con fecha de expiración
const response = await fetch(`${BACKEND_URL}/api/v1/admin/users/${userId}/subscription`, {
  method: 'PUT',
  headers: {
    'Authorization': `Bearer ${token}`,
    'Content-Type': 'application/json'
  },
  body: JSON.stringify({
    plan: 'pro',
    subscription_end: '2026-03-31T23:59:59Z'
  })
});

const result = await response.json();
console.log(`Suscripción actualizada: ${result.plan} hasta ${result.subscription_end}`);
```

---

### POST /api/v1/admin/subscriptions/check-expirations ⭐

**Ejecutar verificación manual de expiraciones**

**Permisos requeridos**: `MANAGE_USERS`

**Rate Limit**: 5 requests/minuto

**Descripción**: Ejecuta la verificación de suscripciones expiradas y degrada automáticamente a FREE los usuarios cuyo plan ha caducado.

**Response** (200 OK):
```json
{
  "total_checked": 15,
  "expired_count": 3,
  "downgraded_user_ids": [
    "550e8400-e29b-41d4-a716-446655440001",
    "550e8400-e29b-41d4-a716-446655440002",
    "550e8400-e29b-41d4-a716-446655440003"
  ],
  "errors": []
}
```

**Frontend Usage**:
```javascript
// Ejecutar verificación manual de expiraciones
const response = await fetch(`${BACKEND_URL}/api/v1/admin/subscriptions/check-expirations`, {
  method: 'POST',
  headers: { 'Authorization': `Bearer ${token}` }
});

const result = await response.json();
console.log(`${result.expired_count} suscripciones expiradas de ${result.total_checked}`);
```

---

### GET /api/v1/admin/subscriptions/stats ⭐

**Obtener estadísticas de suscripciones**

**Permisos requeridos**: `VIEW_SYSTEM_STATS`

**Response** (200 OK):
```json
{
  "by_plan": {
    "free": 45,
    "pro": 12,
    "max": 3
  },
  "expiring_soon_7_days": 2,
  "expired_not_downgraded": 0,
  "total_users": 60
}
```

**Frontend Usage**:
```javascript
// Obtener estadísticas de suscripciones
const response = await fetch(`${BACKEND_URL}/api/v1/admin/subscriptions/stats`, {
  headers: { 'Authorization': `Bearer ${token}` }
});

const stats = await response.json();
console.log(`PRO: ${stats.by_plan.pro}, MAX: ${stats.by_plan.max}, FREE: ${stats.by_plan.free}`);
console.log(`Expirando en 7 días: ${stats.expiring_soon_7_days}`);
```

---

### GET /api/v1/admin/subscriptions/expiring-soon

**Obtener usuarios con suscripciones próximas a expirar (paginado)**

**Permisos requeridos**: `VIEW_USERS`

**Query Parameters**:
- `days` (int, 1-30, default: 7): Días a futuro para buscar
- `skip` (int, ≥0, default: 0): Registros a omitir (paginación)
- `limit` (int, 1-100, default: 50): Máximo de registros a devolver

**Response** (200 OK):
```json
{
  "items": [
    {
      "user_id": "550e8400-e29b-41d4-a716-446655440001",
      "email": "farmacia@ejemplo.com",
      "full_name": "Dr. García López",
      "plan": "pro",
      "subscription_end": "2025-12-17T23:59:59Z",
      "days_remaining": 7,
      "is_expiring_soon": true
    }
  ],
  "total": 3,
  "skip": 0,
  "limit": 50
}
```

**Frontend Usage**:
```javascript
// Obtener usuarios con suscripciones próximas a expirar
const response = await fetch(
  `${BACKEND_URL}/api/v1/admin/subscriptions/expiring-soon?days=14&skip=0&limit=20`,
  { headers: { 'Authorization': `Bearer ${token}` } }
);

const data = await response.json();
console.log(`${data.total} usuarios expirando en los próximos 14 días`);
data.items.forEach(user => console.log(`${user.email}: ${user.days_remaining} días`));
```

---

## 20. Prescription Analytics (NEW - Issue #400) ⭐

**Ruta base**: `/api/v1/prescription`

Endpoints para análisis de ventas de prescripción con clasificación ATC y drill-down por categorías.

### GET /api/v1/prescription/{pharmacy_id}/overview

**Obtener resumen general de prescripción**

**Query Parameters**:
- `start_date` (date, required): Fecha inicio
- `end_date` (date, required): Fecha fin
- `employee_ids` (list[str], optional): Filtrar por empleados

**Response** (200 OK):
```json
{
  "total_units": 15000,
  "total_value": 125000.50,
  "prescription_percentage": 65.5,
  "otc_percentage": 34.5,
  "top_categories": ["Cardiovascular", "Sistema Nervioso", "Digestivo"]
}
```

### GET /api/v1/prescription/{pharmacy_id}/distribution-by-atc

**Distribución por código ATC**

**Query Parameters**:
- `start_date`, `end_date`, `employee_ids` (opcionales)

**Response** (200 OK):
```json
{
  "distribution": [
    {"atc_code": "C", "name": "Cardiovascular", "units": 3500, "value": 45000},
    {"atc_code": "N", "name": "Sistema Nervioso", "units": 2800, "value": 38000}
  ]
}
```

### GET /api/v1/prescription/{pharmacy_id}/waterfall-analysis

**Análisis waterfall YoY/MoM/QoQ** (Issue #458)

**Query Parameters**:
- `comparison_type` (string): `yoy`, `mom`, `qoq`
- `category` (string, optional): Filtrar por categoría

**Response** (200 OK):
```json
{
  "current_period": {"units": 5000, "value": 42000},
  "previous_period": {"units": 4500, "value": 38000},
  "variation_units": 500,
  "variation_percentage": 11.1
}
```

### GET /api/v1/prescription/{pharmacy_id}/top-contributors

**Top productos contributors**

**Response** (200 OK):
```json
{
  "contributors": [
    {"product_name": "Eutirox 50mcg", "units": 850, "value": 12500, "growth": 15.2}
  ]
}
```

### GET /api/v1/prescription/{pharmacy_id}/products-by-category

**Productos por categoría con drill-down**

### GET /api/v1/prescription/{pharmacy_id}/products-by-active-ingredient

**Productos por principio activo**

### GET /api/v1/prescription/{pharmacy_id}/seasonality/export (NEW - Issue #502) ⭐

**Exportación de forecast por categoría × mes para CSV**

**Query Parameters**:
- `date_from` (date, optional): Fecha inicio análisis
- `date_to` (date, optional): Fecha fin análisis
- `periods_ahead` (int, default=6): Meses a proyectar
- `top_categories` (int, default=10): Categorías principales a incluir

**Response** (200 OK):
```json
{
  "data": [
    {
      "category": "antigripales",
      "month": "Enero",
      "month_num": 1,
      "forecast": 1250.50,
      "lower_bound": 1100.00,
      "upper_bound": 1400.00,
      "seasonality_index": 1.35
    }
  ],
  "summary": {
    "total_forecast": 45000.00,
    "categories_count": 10,
    "months_count": 6
  },
  "export_info": {
    "generated_at": "2025-12-30T12:00:00Z",
    "pharmacy_id": "uuid",
    "date_range": "2024-01 a 2024-12"
  }
}
```

### GET /api/v1/prescription/{pharmacy_id}/seasonality/badges (NEW - Issue #502) ⭐

**Badges visuales para header del tab de estacionalidad**

**Response** (200 OK):
```json
{
  "anomalies_count": 3,
  "anomalies_badge_color": "danger",
  "next_peak_label": "Antigripales en 45d",
  "next_peak_badge_color": "info",
  "stockout_alerts": 2,
  "stockout_badge_color": "warning"
}
```

**Badge colors** (Bootstrap):
- `danger`: Urgente (anomalías severas, stockout crítico)
- `warning`: Atención (anomalías, stockout alto, pico cercano)
- `info`: Informativo (pico próximo)
- `secondary`: Normal (sin alertas)

---

## 20. Admin Prescription (NEW - Issue #408) ⭐

**Ruta base**: `/api/v1/admin/prescription`

Endpoints para gestión de clasificación de productos de prescripción.

### GET /api/v1/admin/prescription/stats

**Estadísticas de clasificación**

**Response** (200 OK):
```json
{
  "total_products": 67000,
  "classified": 65000,
  "unclassified": 2000,
  "classification_rate": 97.0,
  "active_categories": 12
}
```

### GET /api/v1/admin/prescription/unclassified-summary

**Resumen de productos sin clasificar**

### POST /api/v1/admin/prescription/classify

**Clasificar producto individual**

**Request Body**:
```json
{
  "product_id": "123456",
  "category": "Prescripción"
}
```

### POST /api/v1/admin/prescription/bulk-update

**Clasificación masiva desde Excel**

**Request**: multipart/form-data con archivo Excel

### POST /api/v1/admin/prescription/reference-list/upload (NEW - Issue #445) ⭐

**Cargar listados de referencia de prescripción**

Endpoint sostenible para cargar listados oficiales (dietas, tiras reactivas, efectos y accesorios).

**Query Parameters**:
- `truncate` (bool, default: false): Si True, elimina registros existentes antes de cargar

**Request**: multipart/form-data con archivo Excel (.xlsx)

**Excel Esperado** (hojas):
- 'Dietas': Productos dietoterapicos → DIETOTERAPICOS
- 'Tiras reactivas': Tiras glucosa → TIRAS_REACTIVAS_GLUCOSA
- 'Efectos y accesorios': Clasificados automaticamente en INCONTINENCIA, ORTOPEDIA, EFECTOS

**Response** (200 OK):
```json
{
  "total_loaded": 5793,
  "dietas_count": 1085,
  "tiras_count": 27,
  "incontinencia_count": 456,
  "ortopedia_count": 1409,
  "efectos_count": 2814,
  "special_codes_count": 2,
  "catalog_entries_created": 0,
  "truncated_before": true,
  "previous_count": 0,
  "execution_time_seconds": 13.13,
  "warnings": []
}
```

**Ejemplo curl**:
```bash
curl -X POST "http://localhost:8000/api/v1/admin/prescription/reference-list/upload?truncate=true" \
  -H "Authorization: Bearer {token}" \
  -F "file=@pf-fichero-1-noviembrev2xlsx.xlsx"
```

### GET /api/v1/admin/prescription/reference-list/stats (NEW - Issue #445) ⭐

**Estadísticas de listados de referencia**

**Response** (200 OK):
```json
{
  "total_entries": 5793,
  "category_breakdown": {
    "dietoterapicos": 1085,
    "tiras_reactivas_glucosa": 27,
    "incontinencia_financiada": 456,
    "ortopedia_financiada": 1409,
    "efectos_financiados": 2814
  },
  "source_breakdown": {
    "Dietas": 1085,
    "Tiras reactivas": 27,
    "Efectos y accesorios": 4679
  },
  "last_loaded_at": "2025-12-12T21:09:32Z",
  "catalog_coverage": 5000
}
```

---

## 21. Optimal Partners (NEW - Issue #415) ⭐

**Ruta base**: `/api/v1/optimal-partners`

### GET /api/v1/optimal-partners/{pharmacy_id}

**Calcular partners óptimos con algoritmo greedy**

**Query Parameters**:
- `max_partners` (int, default: 8): Número máximo de partners
- `min_coverage` (float, default: 0.8): Cobertura mínima objetivo

**Response** (200 OK):
```json
{
  "optimal_partners": [
    {"laboratory_code": "L001", "name": "Cinfa", "coverage": 0.25, "products_count": 1500},
    {"laboratory_code": "L002", "name": "Normon", "coverage": 0.20, "products_count": 1200}
  ],
  "total_coverage": 0.85,
  "optimization_score": 92.5
}
```

---

## 22. Intercambiable Groups (NEW - Issue #446) 🆕

**Ruta base**: `/api/v1/intercambiable-groups`

API para gestión de grupos intercambiables de productos OTC (venta libre).
Nivel 4 de la taxonomía jerárquica: NECESIDAD → Subcategoría → Marca/Línea → **Grupo Intercambiable**.

Los grupos intercambiables contienen productos de múltiples marcas que pueden sustituirse entre sí
(ej: fotoprotectores faciales SPF50+ de ISDIN, Heliocare, Avène).

### GET /api/v1/intercambiable-groups

**Listar todos los grupos intercambiables**

**Permisos requeridos**: Usuario autenticado

**Query Parameters**:
- `validated_only` (bool, default: false): Solo grupos validados por farmacéutico
- `necesidad` (string, optional): Filtrar por necesidad específica (ej: "proteccion_solar")

**Response** (200 OK):
```json
{
  "groups": [
    {
      "id": "550e8400-e29b-41d4-a716-446655440000",
      "name": "Productos para el sueño e insomnio",
      "slug": "productos-para-el-sueno-e-insomnio",
      "necesidad": "insomnio",
      "subcategory": "oral",
      "product_count": 15,
      "brand_count": 4,
      "total_sales_amount": 2450.50,
      "validated": true
    }
  ],
  "total": 8
}
```

---

### GET /api/v1/intercambiable-groups/summary

**Resumen estadístico de grupos intercambiables**

**Permisos requeridos**: Usuario autenticado

**Response** (200 OK):
```json
{
  "total_groups": 8,
  "validated_groups": 6,
  "total_products_in_groups": 120,
  "total_sales_in_groups": 15450.75,
  "by_necesidad": {
    "insomnio": {"count": 1, "products": 15},
    "vitaminas_minerales": {"count": 1, "products": 22},
    "digestivo": {"count": 1, "products": 12},
    "proteccion_solar": {"count": 2, "products": 35}
  }
}
```

---

### GET /api/v1/intercambiable-groups/slug/{slug}

**Obtener grupo por slug (URL-friendly identifier)**

**Permisos requeridos**: Usuario autenticado

**Path Parameters**:
- `slug` (string, required): Identificador URL-friendly del grupo

**Response** (200 OK):
```json
{
  "id": "550e8400-e29b-41d4-a716-446655440000",
  "name": "Productos para el sueño e insomnio",
  "slug": "productos-para-el-sueno-e-insomnio",
  "description": "Grupo de productos para conciliar el sueño...",
  "necesidad": "insomnio",
  "subcategory": "oral",
  "product_count": 15,
  "brand_count": 4,
  "total_sales_amount": 2450.50,
  "total_sales_count": 125,
  "validated": true,
  "validated_by": "farmaceutico@xfarma.es",
  "products": [
    {
      "product_name": "DORMIREL 25 MG",
      "brand": "DORMIREL",
      "brand_line": "classic",
      "total_ventas": 450.00,
      "num_ventas": 30
    }
  ]
}
```

**Error Responses**:
- `404 Not Found`: Grupo no encontrado

---

### GET /api/v1/intercambiable-groups/{group_id}

**Obtener grupo por UUID**

**Permisos requeridos**: Usuario autenticado

**Path Parameters**:
- `group_id` (UUID, required): ID único del grupo

**Response**: Mismo formato que `/slug/{slug}`

**Error Responses**:
- `404 Not Found`: Grupo no encontrado

---

### GET /api/v1/intercambiable-groups/{group_id}/products

**Obtener productos de un grupo**

**Permisos requeridos**: Usuario autenticado

**Path Parameters**:
- `group_id` (UUID, required): ID del grupo

**Query Parameters**:
- `limit` (int, default: 100, max: 1000): Número máximo de productos

**Response** (200 OK):
```json
{
  "group_id": "550e8400-e29b-41d4-a716-446655440000",
  "group_name": "Productos para el sueño e insomnio",
  "products": [
    {
      "product_name": "DORMIREL 25 MG",
      "brand": "DORMIREL",
      "brand_line": "classic",
      "total_ventas": 450.00,
      "num_ventas": 30
    },
    {
      "product_name": "Aquilea Sueño Instant",
      "brand": "AQUILEA",
      "brand_line": "sueno",
      "total_ventas": 380.00,
      "num_ventas": 25
    }
  ],
  "total": 15
}
```

**Error Responses**:
- `404 Not Found`: Grupo no encontrado

---

### POST /api/v1/intercambiable-groups/{group_id}/assign ⭐ ADMIN

**Asignar productos a un grupo (por necesidad)**

**Permisos requeridos**: Admin

Asigna productos venta_libre con la misma necesidad (`ml_category`) al grupo.
Usa bulk update optimizado para evitar timeouts en Render.

**Path Parameters**:
- `group_id` (UUID, required): ID del grupo objetivo

**Request Body**:
```json
{
  "pharmacy_id": "550e8400-e29b-41d4-a716-446655440001",  // opcional
  "limit": 1000
}
```

**Response** (200 OK):
```json
{
  "group_id": "550e8400-e29b-41d4-a716-446655440000",
  "group_name": "Productos para el sueño e insomnio",
  "necesidad": "insomnio",
  "assigned_count": 45
}
```

**Error Responses**:
- `403 Forbidden`: Usuario sin permisos de admin
- `404 Not Found`: Grupo no encontrado

---

### POST /api/v1/intercambiable-groups/assign-all ⭐ ADMIN

**Asignar productos a TODOS los grupos**

**Permisos requeridos**: Admin

Ejecuta asignación masiva en batch (optimizado para evitar N+1 queries).
Un solo commit para todas las asignaciones.

**Request Body**:
```json
{
  "pharmacy_id": null,        // opcional - todas las farmacias
  "validated_only": true      // solo grupos validados
}
```

**Response** (200 OK):
```json
{
  "groups_processed": 6,
  "total_assigned": 250,
  "details": [
    {
      "group_id": "550e8400-e29b-41d4-a716-446655440000",
      "group_name": "Productos para el sueño e insomnio",
      "necesidad": "insomnio",
      "assigned_count": 45
    },
    {
      "group_id": "550e8400-e29b-41d4-a716-446655440001",
      "group_name": "Vitaminas y minerales básicos",
      "necesidad": "vitaminas_minerales",
      "assigned_count": 62
    }
  ]
}
```

**Error Responses**:
- `403 Forbidden`: Usuario sin permisos de admin

**Notas de performance**:
- Usa batch commits (un commit para todas las asignaciones)
- Bulk update SQLAlchemy (una query SQL por grupo)
- Optimizado para Render (180s worker timeout)

---

## 23. Venta Libre / OTC Analysis (NEW - Issue #461) 🆕

Dashboard de análisis de productos venta libre/OTC por categoría NECESIDAD.

### 23.1 Treemap de Ventas por NECESIDAD

```
GET /api/v1/ventalibre/sales-by-necesidad
```

Datos agregados por categoría ml_category para treemap interactivo.

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| date_from | date | No | Fecha inicio (YYYY-MM-DD) |
| date_to | date | No | Fecha fin (YYYY-MM-DD) |
| employee_ids | List[str] | No | IDs de empleados para filtrar |

**Response:**
```json
{
  "nodes": [
    {
      "category": "proteccion_solar",
      "sales": 15000.0,
      "count": 234,
      "percentage": 25.3
    }
  ],
  "total_sales": 60000.0,
  "total_products": 1247
}
```

### 23.2 Lista de Productos

```
GET /api/v1/ventalibre/products
```

Lista paginada de productos OTC con filtros.

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| necesidad | str | No | Filtrar por categoría NECESIDAD |
| search | str | No | Buscar en nombre de producto |
| employee_ids | List[str] | No | IDs de empleados |
| date_from | date | No | Fecha inicio |
| date_to | date | No | Fecha fin |
| limit | int | No | Límite de resultados (default 50, max 100) |
| offset | int | No | Offset para paginación |

**Response:**
```json
{
  "products": [
    {
      "id": "uuid",
      "product_code": "PROT_001",
      "product_name": "Protector Solar FPS50",
      "ml_category": "proteccion_solar",
      "ml_confidence": 0.92,
      "detected_brand": "ISDIN",
      "total_sales": 1250.00,
      "total_units": 85
    }
  ],
  "total": 1247,
  "page": 1,
  "pages": 25
}
```

### 23.3 KPIs del Dashboard

```
GET /api/v1/ventalibre/kpis
```

Métricas agregadas para cabecera del dashboard.

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| date_from | date | No | Fecha inicio |
| date_to | date | No | Fecha fin |
| employee_ids | List[str] | No | IDs de empleados |

**Response:**
```json
{
  "total_sales": 45230.50,
  "total_products": 1247,
  "categories_count": 32,
  "coverage_percent": 94.2,
  "yoy_growth": 12.3
}
```

### 23.4 Categorías Disponibles

```
GET /api/v1/ventalibre/categories
```

Lista de categorías NECESIDAD con conteos para filtros.

**Response:**
```json
{
  "categories": [
    {"value": "proteccion_solar", "label": "Proteccion Solar", "count": 234},
    {"value": "higiene_bucal", "label": "Higiene Bucal", "count": 156}
  ]
}
```

---

### POST /api/v1/ventalibre/correct
```
Corrige la categoría NECESIDAD de un producto venta libre.

**Request Body:**
```json
{
  "product_id": "uuid-del-producto",
  "corrected_category": "proteccion_solar",
  "reviewer_notes": "Comentario opcional"
}
```

**Response:**
```json
{
  "success": true,
  "product_id": "uuid-del-producto",
  "old_category": "higiene_bucal",
  "new_category": "proteccion_solar",
  "message": "Categoría actualizada correctamente"
}
```

**Errores:**
- 400: Categoría no válida
- 404: Producto no encontrado
- 500: Error interno

---

### 23.5 Cluster Management - Split (Issue #464) 🆕

```
POST /api/v1/ventalibre/clusters/{category}/split
```

Divide un cluster moviendo productos seleccionados a una nueva categoría.
**Requiere rol admin.**

**Path Parameters:**
- `category`: Categoría origen de los productos

**Request Body:**
```json
{
  "product_ids": ["uuid-1", "uuid-2", "uuid-3"],
  "new_category": "higiene_corporal",
  "notes": "Separando productos de higiene corporal del cluster general"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Split completado: 3 productos movidos a 'higiene_corporal'",
  "source_category": "cuidado_personal",
  "new_category": "higiene_corporal",
  "products_moved": 3,
  "enrichments_synced": 45,
  "audit_log_id": "uuid-audit-log",
  "executed_at": "2025-12-21T10:30:00Z",
  "executed_by": "admin@farmacia.com",
  "source_category_remaining": 127,
  "new_category_total": 89
}
```

**Errores:**
- 400: Categoría destino inválida o sin productos
- 403: Usuario no es admin
- 500: Error interno

---

### 23.6 Cluster Management - Merge (Issue #464) 🆕

```
POST /api/v1/ventalibre/clusters/merge
```

Fusiona múltiples categorías en una categoría destino.
**Requiere rol admin.**

**Request Body:**
```json
{
  "source_categories": ["crema_solar", "protector_solar_facial"],
  "destination_category": "proteccion_solar",
  "notes": "Consolidando categorías de protección solar"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Merge completado: 156 productos movidos a 'proteccion_solar'",
  "source_categories": ["crema_solar", "protector_solar_facial"],
  "destination_category": "proteccion_solar",
  "total_products_merged": 156,
  "enrichments_synced": 1890,
  "products_per_source": {
    "crema_solar": 98,
    "protector_solar_facial": 58
  },
  "audit_log_id": "uuid-audit-log",
  "executed_at": "2025-12-21T10:35:00Z",
  "executed_by": "admin@farmacia.com",
  "destination_total": 245
}
```

**Errores:**
- 400: Categoría destino inválida o en las categorías origen
- 403: Usuario no es admin
- 404: Sin productos en categorías origen
- 500: Error interno

---

### 23.7 Preview Split (Issue #464) 🆕

```
GET /api/v1/ventalibre/clusters/{category}/preview-split
```

Previsualiza operación de split sin ejecutar (dry run).

**Path Parameters:**
- `category`: Categoría origen

**Query Parameters:**
- `product_ids`: UUIDs separados por coma
- `new_category`: Categoría destino propuesta

**Response:**
```json
{
  "source_category": "cuidado_personal",
  "new_category": "higiene_corporal",
  "is_valid_category": true,
  "products_to_move": [
    {
      "id": "uuid-1",
      "product_name": "Gel de Baño Dove 500ml",
      "current_category": "cuidado_personal",
      "detected_brand": "DOVE",
      "total_sales_count": 234,
      "human_verified": false
    }
  ],
  "products_count": 3,
  "source_category_remaining": 127,
  "total_sales_impact": 456,
  "warnings": []
}
```

---

### 23.8 Preview Merge (Issue #464) 🆕

```
GET /api/v1/ventalibre/clusters/preview-merge
```

Previsualiza operación de merge sin ejecutar (dry run).

**Query Parameters:**
- `source_categories`: Categorías origen separadas por coma
- `destination`: Categoría destino

**Response:**
```json
{
  "source_categories": ["crema_solar", "protector_solar_facial"],
  "destination_category": "proteccion_solar",
  "is_valid_destination": true,
  "source_impacts": [
    {
      "category": "crema_solar",
      "product_count": 98,
      "total_sales": 12450,
      "verified_count": 23
    },
    {
      "category": "protector_solar_facial",
      "product_count": 58,
      "total_sales": 8920,
      "verified_count": 12
    }
  ],
  "total_products_to_merge": 156,
  "total_sales_impact": 21370,
  "total_verified_products": 35,
  "destination_current_count": 89,
  "destination_after_merge": 245,
  "warnings": []
}
```

---

### 23.9 Cluster Stats (Issue #464) 🆕

```
GET /api/v1/ventalibre/clusters/{category}/stats
```

Obtiene estadísticas detalladas de un cluster.

**Path Parameters:**
- `category`: Nombre de la categoría

**Response:**
```json
{
  "category": "proteccion_solar",
  "display_name": "Protección Solar",
  "total_products": 245,
  "verified_products": 67,
  "pending_verification": 178,
  "total_sales_count": 34560,
  "pharmacies_count": 12,
  "top_brands": ["ISDIN", "HELIOCARE", "AVENE", "LA ROCHE POSAY", "VICHY"],
  "brand_count": 28,
  "avg_confidence": 0.87,
  "verification_rate": 27.3
}
```

---

### 23.10 Validate Category Name (Issue #464) 🆕

```
GET /api/v1/ventalibre/clusters/validate-name
```

Valida si un nombre de categoría es válido en NecesidadEspecifica.

**Query Parameters:**
- `name`: Nombre de categoría a validar

**Response (válida):**
```json
{
  "category": "proteccion_solar",
  "is_valid": true,
  "display_name": "Protección Solar",
  "parent_category": "solar",
  "reason": null
}
```

**Response (inválida):**
```json
{
  "category": "solar_cremas",
  "is_valid": false,
  "display_name": null,
  "parent_category": null,
  "reason": "'solar_cremas' no es una categoría válida en NecesidadEspecifica."
}
```

---

### 23.11 Brand Analysis by Category (Issue #493) 🆕

```
GET /api/v1/ventalibre/brands/{necesidad}
```

Obtiene análisis de marcas por categoría NECESIDAD incluyendo HHI (concentración de mercado).

**Path Parameters:**
- `necesidad`: Categoría NECESIDAD (ej: `dermocosmetica`, `higiene_bucal`)

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| pharmacy_id | UUID | Sí | ID de la farmacia |
| date_from | date | No | Fecha inicio (YYYY-MM-DD) |
| date_to | date | No | Fecha fin (YYYY-MM-DD) |
| employee_name | string | No | Filtrar por empleado |

**Response:**
```json
{
  "brands": [
    {
      "brand": "isdin",
      "sales": 12500.00,
      "share": 35.2,
      "units": 450,
      "avg_margin": 28.5
    }
  ],
  "total_sales": 35500.00,
  "hhi": 2150,
  "hhi_interpretation": {
    "level": "medium",
    "color": "warning",
    "title": "Concentración moderada",
    "message": "Evalúa alternativas. Algunas marcas dominan."
  },
  "coverage_percent": 87.4
}
```

---

### 23.12 Market Share Evolution (Issue #493) 🆕

```
GET /api/v1/ventalibre/market-share-evolution/{necesidad}
```

Obtiene evolución temporal de cuota de mercado por marca (para gráfico de áreas apiladas 100%).

**Path Parameters:**
- `necesidad`: Categoría NECESIDAD

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| pharmacy_id | UUID | Sí | ID de la farmacia |
| date_from | date | No | Fecha inicio |
| date_to | date | No | Fecha fin |
| top_n | int | No | Número de marcas top (default: 5) |

**Response:**
```json
{
  "time_series": [
    {
      "month": "2024-10",
      "isdin": 35.2,
      "vichy": 22.1,
      "avene": 15.8,
      "Otras": 26.9
    }
  ],
  "top_brands": ["isdin", "vichy", "avene", "bioderma", "eucerin"]
}
```

---

### 23.13 Value Quadrant (Issue #493) 🆕

```
GET /api/v1/ventalibre/value-quadrant/{necesidad}
```

Obtiene datos para scatter plot de Cuadrante de Valor (Margen vs Volumen).
Usa medianas dinámicas como thresholds para asignar cuadrantes.

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| pharmacy_id | UUID | Sí | ID de la farmacia |
| date_from | date | No | Fecha inicio |
| date_to | date | No | Fecha fin |

**Response:**
```json
{
  "brands": [
    {
      "brand": "isdin",
      "sales": 12500.00,
      "margin_pct": 28.5,
      "quadrant": "star"
    },
    {
      "brand": "vichy",
      "sales": 8200.00,
      "margin_pct": 18.2,
      "quadrant": "traffic"
    }
  ],
  "thresholds": {
    "median_sales": 5500.00,
    "median_margin": 22.0
  }
}
```

**Cuadrantes:**
- `star`: Alto volumen + Alto margen (⭐ Estrellas)
- `traffic`: Alto volumen + Bajo margen (🚶 Generadores de tráfico)
- `opportunity`: Bajo volumen + Alto margen (💡 Oportunidades)
- `review`: Bajo volumen + Bajo margen (❓ Revisar)

---

### 23.14 Brand Duel (Issue #493) 🆕

```
GET /api/v1/ventalibre/brand-duel
```

Compara dos marcas lado a lado en múltiples métricas.

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| pharmacy_id | UUID | Sí | ID de la farmacia |
| brand_a | string | Sí | Primera marca |
| brand_b | string | Sí | Segunda marca |
| necesidad | string | Sí | Categoría NECESIDAD |
| date_from | date | No | Fecha inicio |
| date_to | date | No | Fecha fin |

**Response:**
```json
{
  "brand_a": {
    "brand": "isdin",
    "share": 35.2,
    "margin_pct": 28.5,
    "avg_ticket": 25.50,
    "units": 450,
    "total_margin": 3562.50
  },
  "brand_b": {
    "brand": "vichy",
    "share": 22.1,
    "margin_pct": 18.2,
    "avg_ticket": 32.00,
    "units": 280,
    "total_margin": 1638.40
  },
  "winners": {
    "share": "a",
    "margin_pct": "a",
    "avg_ticket": "b",
    "units": "a",
    "total_margin": "a"
  }
}
```

**Nota:** El ganador se determina por valor mayor, excepto métricas inversas (rotación: menor = mejor).

---

### 23.15 Price Distribution (Issue #493) 🆕

```
GET /api/v1/ventalibre/price-distribution/{necesidad}
```

Obtiene distribución de precios por marca para boxplot (detección de canibalización).

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| pharmacy_id | UUID | Sí | ID de la farmacia |
| date_from | date | No | Fecha inicio |
| date_to | date | No | Fecha fin |
| top_n | int | No | Número de marcas top (default: 10) |

**Response:**
```json
{
  "brands": [
    {
      "brand": "isdin",
      "min": 8.50,
      "q1": 15.00,
      "median": 22.50,
      "q3": 32.00,
      "max": 55.00
    }
  ]
}
```

**Uso UX:** Solapamiento de cajas = posible canibalización. Hover muestra CN y nombre de producto.

---

### 23.16 Sales by L2 Subcategory (Issue #505) 🆕

```
GET /api/v1/ventalibre/sales-by-l2/{l1_category}
```

Obtiene ventas agregadas por subcategoría L2 para categorías con soporte L2.
Solo disponible para: `dermocosmetica`, `suplementos`, `higiene_bucal`, `infantil`, `sexual`, `control_peso`.

**Path Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| l1_category | L1WithL2Category | Sí | Categoría L1 con soporte L2 |

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| date_from | date | No | Fecha inicio |
| date_to | date | No | Fecha fin |
| employee_ids | List[UUID] | No | Filtro por empleados |

**Response:**
```json
{
  "nodes": [
    {
      "category": "solar_facial",
      "display_name": "Solar Facial",
      "archetype": "🎯 Estratégico",
      "sales": 5280.50,
      "count": 45,
      "units": 120,
      "percentage": 42.5
    }
  ],
  "total_sales": 12422.75,
  "total_products": 156,
  "l1_category": "dermocosmetica"
}
```

**Errores:**
- 422: Categoría L1 no tiene soporte L2

---

### 23.17 L2 Subcategories List (Issue #505) 🆕

```
GET /api/v1/ventalibre/subcategories/{l1_category}
```

Obtiene lista de subcategorías L2 disponibles para una categoría L1 con sus arquetipos.

**Path Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| l1_category | L1WithL2Category | Sí | Categoría L1 con soporte L2 |

**Response:**
```json
{
  "l1_category": "dermocosmetica",
  "l1_display_name": "Dermocosmética",
  "subcategories": [
    {
      "value": "solar_facial",
      "label": "Solar Facial",
      "archetype": "🎯 Estratégico",
      "archetype_description": "Alta facturación, alto margen"
    },
    {
      "value": "tratamiento_avanzado",
      "label": "Tratamiento Avanzado",
      "archetype": "💎 Premium",
      "archetype_description": "Ticket alto, asesoramiento"
    }
  ]
}
```

**Arquetipos definidos:**
- **Estratégico**: Alta facturación, alto margen
- **Premium**: Ticket alto, requiere asesoramiento
- **Impulso**: Compra no planificada
- **Mantenimiento**: Baja rotación, necesario en surtido
- **Prevención**: Estacional, anticipar demanda

---

### 23.18 L2 Coverage Stats (Issue #505) 🆕

```
GET /api/v1/ventalibre/l2-coverage
```

Obtiene estadísticas de cobertura de clasificación L2 por categoría L1.

**Response:**
```json
{
  "categories": [
    {
      "l1_category": "dermocosmetica",
      "total_products": 450,
      "with_l2": 380,
      "coverage_percent": 84.4
    },
    {
      "l1_category": "suplementos",
      "total_products": 280,
      "with_l2": 210,
      "coverage_percent": 75.0
    }
  ],
  "overall": {
    "total_products": 1200,
    "with_l2": 920,
    "coverage_percent": 76.7
  }
}
```

**Uso UX:** Mostrar tooltip de advertencia cuando coverage < 80%.

---

### 23.19 L2 Value Quadrant (Issue #505) 🆕

```
GET /api/v1/ventalibre/value-quadrant-l2/{l1_category}
```

Obtiene datos para scatter plot de Cuadrante de Valor a nivel L2 (Margen vs Ventas).

**Path Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| l1_category | L1WithL2Category | Sí | Categoría L1 con soporte L2 |

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| date_from | date | No | Fecha inicio |
| date_to | date | No | Fecha fin |
| employee_ids | List[UUID] | No | Filtro por empleados |

**Response:**
```json
{
  "subcategories": [
    {
      "l2": "solar_facial",
      "display_name": "Solar Facial",
      "archetype": "🎯 Estratégico",
      "sales": 5280.50,
      "margin_pct": 38.5,
      "transactions": 145,
      "quadrant": "star",
      "quadrant_label": "Estrellas"
    },
    {
      "l2": "hidratacion_basica",
      "display_name": "Hidratación Básica",
      "archetype": "🔄 Mantenimiento",
      "sales": 1200.00,
      "margin_pct": 22.5,
      "transactions": 89,
      "quadrant": "dog",
      "quadrant_label": "Perros"
    }
  ],
  "thresholds": {
    "median_sales": 2500.00,
    "median_margin": 30.0
  },
  "l1_category": "dermocosmetica"
}
```

**Cuadrantes:**
- **star**: Alto margen + Altas ventas (verde)
- **opportunity**: Bajo margen + Altas ventas (amarillo) - oportunidad de mejora
- **cash_cow**: Alto margen + Bajas ventas (azul) - vacas lecheras
- **dog**: Bajo margen + Bajas ventas (rojo) - perros

**Uso UX:** Scatter plot con líneas de mediana como ejes de cuadrante.

---

### 23.20 Time Series (Issue #491) 🆕

```
GET /api/v1/ventalibre/time-series
```

Serie temporal mensual de ventas por categoría NECESIDAD para gráfico de evolución.
Devuelve datos en formato long/flat agrupados por mes (YYYY-MM) y categoría.

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| date_from | date | No | Fecha inicio (default: period_months atrás) |
| date_to | date | No | Fecha fin (default: hoy) |
| employee_ids | List[UUID] | No | IDs de empleados para filtrar |
| period_months | int | No | Meses hacia atrás (default: 12) |

**Response:**
```json
{
  "time_series": [
    {"period": "2024-01", "category": "proteccion_solar", "sales": 1500.0, "units": 234},
    {"period": "2024-01", "category": "hidratacion", "sales": 2300.0, "units": 456},
    {"period": "2024-02", "category": "proteccion_solar", "sales": 1650.0, "units": 267}
  ],
  "category_summary": [
    {"category": "proteccion_solar", "total_sales": 18000.0, "total_units": 2800},
    {"category": "hidratacion", "total_sales": 27600.0, "total_units": 5472}
  ],
  "total_sales": 60000.0,
  "date_from": "2024-01-01",
  "date_to": "2024-12-31"
}
```

**Uso UX:** Gráfico de líneas con top 6 categorías visibles, resto en legendonly.

---

### 23.21 YoY Comparison (Issue #491) 🆕

```
GET /api/v1/ventalibre/yoy-comparison
```

Comparación Year-over-Year por categoría NECESIDAD (período actual vs año anterior).

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| date_from | date | No | Fecha inicio período actual |
| date_to | date | No | Fecha fin período actual |
| employee_ids | List[UUID] | No | IDs de empleados para filtrar |

**Response:**
```json
{
  "categories": [
    {
      "category": "proteccion_solar",
      "current_sales": 15000.0,
      "previous_sales": 12000.0,
      "variation_euros": 3000.0,
      "variation_percent": 25.0,
      "trend": "up",
      "sparkline_data": [1200, 1350, 1400, 1500, 1600, 1700]
    },
    {
      "category": "hidratacion",
      "current_sales": 8000.0,
      "previous_sales": 9500.0,
      "variation_euros": -1500.0,
      "variation_percent": -15.8,
      "trend": "down",
      "sparkline_data": [1200, 1100, 1000, 950, 900, 850]
    }
  ],
  "total_current": 60000.0,
  "total_previous": 50000.0,
  "total_variation_percent": 20.0,
  "current_period": {"from": "2024-01-01", "to": "2024-12-31"},
  "previous_period": {"from": "2023-01-01", "to": "2023-12-31"}
}
```

**Valores trend:** `"up"`, `"down"`, `"stable"`

**Campo sparkline_data:** Lista de valores mensuales (últimos 6 meses) para renderizar mini-gráfico SVG de tendencia.

**Uso UX:** Tabla ordenada por variación absoluta (mayor impacto primero), con sparklines de tendencia.

---

### 23.22 Top Contributors (Issue #491) 🆕

```
GET /api/v1/ventalibre/top-contributors
```

Top productos que más contribuyen al crecimiento/decrecimiento YoY.

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| date_from | date | No | Fecha inicio período actual |
| date_to | date | No | Fecha fin período actual |
| employee_ids | List[UUID] | No | IDs de empleados para filtrar |
| limit | int | No | Número de productos (default: 10, max: 50) |
| direction | string | No | Filtro: `"up"`, `"down"`, `"all"` (default) |

**Response:**
```json
{
  "contributors": [
    {
      "product_name": "ISDIN FOTOPROTECTOR SPF50+ 50ML",
      "category": "proteccion_solar",
      "current_sales": 2500.0,
      "previous_sales": 1500.0,
      "variation_euros": 1000.0,
      "variation_percent": 66.7,
      "impact_percent": 3.3
    },
    {
      "product_name": "AVENE HYDRANCE LIGERA 40ML",
      "category": "hidratacion",
      "current_sales": 800.0,
      "previous_sales": 1200.0,
      "variation_euros": -400.0,
      "variation_percent": -33.3,
      "impact_percent": -1.3
    }
  ],
  "total_change": 3000.0,
  "direction": "all"
}
```

**Uso UX:** Tabla con ranking (#1-#10), barras de impacto proporcionales.

---

### 23.23 Taxonomy Labeling (Issue #462) ⚠️ DEPRECATED

> **⚠️ DEPRECATED (ADR-004)**: Este endpoint está deprecado. Usar grupos curados en Admin → Grupos Curados.

```
POST /api/v1/clustering/label-taxonomy
```

Etiqueta clusters con taxonomía jerárquica (Tier 1 macro-categoría, Tier 2 categoría, Tier 3 nombre LLM).
Usa modelo híbrido: estructura determinística + naming creativo con LLM.
**Requiere rol admin.**

**Request Body:**
```json
{
  "pharmacy_id": "uuid-farmacia",
  "force": false,
  "cluster_ids": ["uuid-1", "uuid-2"]
}
```

| Param | Type | Required | Description |
|-------|------|----------|-------------|
| pharmacy_id | UUID | Sí | ID de la farmacia |
| force | bool | No | Forzar re-etiquetado (ignora threshold 10%) |
| cluster_ids | List[UUID] | No | Clusters específicos (si vacío, todos) |

**Response:**
```json
{
  "success": true,
  "message": "Etiquetado completado: 45 clusters procesados",
  "clusters_processed": 45,
  "clusters_skipped": 12,
  "clusters_updated": 33,
  "llm_calls_made": 33,
  "estimated_cost_usd": 0.12,
  "execution_time_seconds": 28.5,
  "details": [
    {
      "cluster_id": "uuid-1",
      "tier1": "higiene_bucal",
      "tier2": "sensibilidad_dental",
      "display_name": "Pastas Dientes Sensibles",
      "confidence": 0.92,
      "status": "updated"
    }
  ]
}
```

**Algoritmo Interno:**
1. Calcula `composition_hash` (SHA256 de product_ids)
2. Si cambio < 10% y no `force`, skip (ahorro costes)
3. Voto ponderado por ventas → Tier 2 (categoría ganadora)
4. Mapeo estático `TIER1_MAPPING` → Tier 1 (macro-categoría)
5. LLM genera `llm_generated_name` (Tier 3)

**Errores:**
- 400: Farmacia sin clusters o IDs inválidos
- 403: Usuario no es admin
- 429: Rate limit LLM
- 500: Error interno

---

```
GET /api/v1/clustering/label-taxonomy/status
```

> **⚠️ DEPRECATED (ADR-004)**: Este endpoint está deprecado.

Obtiene estado actual de etiquetado de clusters por farmacia.

**Query Parameters:**
| Param | Type | Required | Description |
|-------|------|----------|-------------|
| pharmacy_id | UUID | Sí | ID de la farmacia |

**Response:**
```json
{
  "pharmacy_id": "uuid-farmacia",
  "total_clusters": 150,
  "labeled_clusters": 145,
  "unlabeled_clusters": 5,
  "stale_clusters": 3,
  "last_labeled_at": "2025-12-24T10:30:00Z",
  "coverage_percent": 96.7,
  "tier1_distribution": {
    "higiene_bucal": 25,
    "dermocosmetica": 32,
    "cuidado_capilar": 18,
    "otros_parafarmacia": 8
  }
}
```

---

## 📝 Changelog de Documentación

### Versión 5.7.0 (2026-01-07)
**Evolution Analytics - Issue #491**

- 🆕 **Nuevos endpoints**: 3 evolution analytics endpoints
- ⭐ **Total endpoints**: 189 → 192 (+3 endpoints)
- 🔧 **Características**:
  - Time Series: Serie temporal mensual por categoría NECESIDAD
  - YoY Comparison: Comparación año vs año por categoría
  - Top Contributors: Productos con mayor impacto en variación YoY
- 📊 **Visualización**:
  - Gráfico de líneas con top 6 categorías visibles
  - Tabla YoY ordenada por variación absoluta
  - Ranking de productos con barras de impacto

**Endpoints documentados**:
- GET `/ventalibre/time-series` - Serie temporal por NECESIDAD
- GET `/ventalibre/yoy-comparison` - Comparación YoY
- GET `/ventalibre/top-contributors` - Top productos por impacto

### Versión 5.6.0 (2026-01-01)
**L2 Subcategory Analysis - Issue #505**

- 🆕 **Nuevos endpoints**: 4 L2 subcategory endpoints
- ⭐ **Total endpoints**: 185 → 189 (+4 endpoints)
- 🔧 **Características**:
  - Sales by L2: Ventas agregadas por subcategoría
  - Subcategories List: Lista de L2 disponibles con arquetipos
  - L2 Coverage: Estadísticas de cobertura de clasificación
  - Value Quadrant L2: Scatter plot margen vs ventas a nivel L2
- 📊 **Categorías L1 con soporte L2**:
  - dermocosmetica (7 subcategorías)
  - suplementos (6 subcategorías)
  - higiene_bucal (5 subcategorías)
  - infantil (5 subcategorías)
  - sexual (4 subcategorías)
  - control_peso (4 subcategorías)

**Endpoints documentados**:
- GET `/ventalibre/sales-by-l2/{l1_category}` - Ventas por L2
- GET `/ventalibre/subcategories/{l1_category}` - Lista subcategorías
- GET `/ventalibre/l2-coverage` - Cobertura clasificación L2
- GET `/ventalibre/value-quadrant-l2/{l1_category}` - Cuadrante de valor L2

### Versión 5.5.0 (2025-12-29)
**Brand Analysis - Issue #493**

- 🆕 **Nuevos endpoints**: 5 brand analysis endpoints
- ⭐ **Total endpoints**: 180 → 185 (+5 endpoints)
- 🔧 **Características**:
  - HHI (Herfindahl-Hirschman Index) para concentración de mercado
  - Cuadrante de Valor: Scatter Margen vs Volumen (dynamic thresholds)
  - Brand Duel: Comparación lado a lado de 2 marcas
  - Evolución de cuota: Áreas apiladas 100%
  - Boxplot de precios: Detección de canibalización

**Endpoints documentados**:
- GET `/ventalibre/brands/{necesidad}` - Marcas por categoría + HHI
- GET `/ventalibre/market-share-evolution/{necesidad}` - Evolución temporal
- GET `/ventalibre/value-quadrant/{necesidad}` - Cuadrante de valor
- GET `/ventalibre/brand-duel` - Comparación de marcas
- GET `/ventalibre/price-distribution/{necesidad}` - Distribución de precios

### Versión 5.4.0 (2025-12-24) ⚠️ DEPRECATED
**Taxonomy Labeling - Issue #462**

> **⚠️ DEPRECATED (ADR-004)**: Estos endpoints están deprecados. El clustering dinámico ha sido reemplazado por grupos curados. Ver Admin → Grupos Curados.

- 🆕 **Nuevos endpoints**: 2 taxonomy labeling endpoints
- ⭐ **Total endpoints**: 178 → 180 (+2 endpoints)
- 🔧 **Características**:
  - Etiquetado jerárquico: Tier 1 (macro) + Tier 2 (sub) + Tier 3 (LLM)
  - Modelo híbrido: estructura determinística + naming LLM
  - Control de estabilidad: threshold 10% cambio
  - Coste controlado: ~$0.15/farmacia

**Endpoints documentados** (deprecated):
- POST `/clustering/label-taxonomy` - Etiquetar clusters (admin) ⚠️
- GET `/clustering/label-taxonomy/status` - Estado de etiquetado ⚠️

### Versión 5.3.0 (2025-12-21) ⚠️ DEPRECATED
**Cluster Management - Issue #464**

> **⚠️ DEPRECATED (ADR-004)**: Estos endpoints están deprecados. Usar grupos curados en Admin → Grupos Curados.

- 🆕 **Nuevos endpoints**: 6 cluster management endpoints
- ⭐ **Total endpoints**: 172 → 178 (+6 endpoints)
- 🔧 **Características**:
  - Split: Dividir clusters moviendo productos seleccionados
  - Merge: Fusionar múltiples categorías en una
  - Preview: Dry-run para split y merge
  - Stats: Estadísticas detalladas de clusters
  - Validación: Verificar nombres de categoría

**Endpoints documentados** (deprecated):
- POST `/ventalibre/clusters/{category}/split` - Dividir cluster (admin) ⚠️
- POST `/ventalibre/clusters/merge` - Fusionar clusters (admin) ⚠️
- GET `/ventalibre/clusters/{category}/preview-split` - Preview split ⚠️
- GET `/ventalibre/clusters/preview-merge` - Preview merge ⚠️
- GET `/ventalibre/clusters/{category}/stats` - Estadísticas cluster ⚠️
- GET `/ventalibre/clusters/validate-name` - Validar nombre categoría ⚠️

### Versión 5.2.0 (2025-12-20)
**Venta Libre / OTC Analysis - Issue #461**

- 🆕 **Nueva sección**: 23. Venta Libre / OTC Analysis (5 endpoints)
- ⭐ **Total endpoints**: 168 → 173 (+5 endpoints)
- 🔧 **Características**:
  - Treemap de ventas por categoría NECESIDAD
  - Tabla de productos paginada con filtros
  - KPIs de dashboard (ventas, productos, cobertura)
  - Filtros reutilizables (fecha, empleados)

**Endpoints documentados**:
- GET `/ventalibre/sales-by-necesidad` - Treemap data
- GET `/ventalibre/products` - Lista paginada
- GET `/ventalibre/kpis` - Métricas agregadas
- GET `/ventalibre/categories` - Categorías disponibles
- POST `/ventalibre/correct` - Corregir clasificación producto

### Versión 5.1.0 (2025-12-17)
**Intercambiable Groups - Issue #446**

- 🆕 **Nueva sección**: 22. Intercambiable Groups (7 endpoints)
- ⭐ **Total endpoints**: 161 → 168 (+7 endpoints)
- 🔧 **Características**:
  - Taxonomía jerárquica nivel 4 para OTC
  - CRUD de grupos intercambiables
  - Asignación masiva optimizada (batch commits)
  - Filtros por necesidad y validación

**Endpoints documentados**:
- GET `/intercambiable-groups` - Listar grupos
- GET `/intercambiable-groups/summary` - Resumen estadístico
- GET `/intercambiable-groups/slug/{slug}` - Por slug
- GET `/intercambiable-groups/{group_id}` - Por ID
- GET `/intercambiable-groups/{group_id}/products` - Productos del grupo
- POST `/intercambiable-groups/{group_id}/assign` - Asignar (admin)
- POST `/intercambiable-groups/assign-all` - Asignar todos (admin)

### Versión 5.1.0 (2025-12-21) - Issue #458
**ML Classifier Monitoring**

- 🆕 **Nueva sección**: 22. Admin - ML Classifier Monitoring (5 endpoints)
- ⭐ **Total endpoints**: 156 → 161 (+5 endpoints)
- 📊 **KPIs**: Entropy, Outlier Rate, Confidence, Pending Validation, Coverage
- 🔔 **Alertas**: Sistema de alertas automáticas por thresholds
- 📈 **Histórico**: Tendencias de métricas últimos 30 días

**Endpoints documentados:**
- `GET /api/v1/admin/classifier/metrics` - Métricas actuales
- `GET /api/v1/admin/classifier/alerts` - Alertas activas
- `GET /api/v1/admin/classifier/distribution` - Distribución por categoría
- `GET /api/v1/admin/classifier/high-risk` - Productos alto riesgo
- `GET /api/v1/admin/classifier/metrics/history` - Histórico de tendencias

### Versión 5.0.0 (2025-12-08)
**Prescription Analytics & Admin Prescription**

- 🆕 **Nuevas secciones**: 19. Prescription Analytics (6 endpoints)
- 🆕 **Nuevas secciones**: 20. Admin Prescription (4 endpoints)
- 🆕 **Nuevas secciones**: 21. Optimal Partners (1 endpoint)
- ⭐ **Total endpoints**: 114 → 156 (+42 endpoints)

### Versión 4.2.0 (2025-11-01) - Issue #348 FASE 2
**Admin APIs - User Management & Database Management**

- 🆕 **Nuevas secciones**: 17. Admin - User Management (7 endpoints)
- 🆕 **Nuevas secciones**: 18. Admin - Database Management (4 endpoints)
- ⭐ **Total endpoints**: 101 → 112 (+11 endpoints)
- 🔐 **RBAC completo**: Permisos `MANAGE_USERS`, `VIEW_USERS`, `MANAGE_DATABASE`, `VIEW_SYSTEM_STATS`
- 📊 **Rate limiting**: Protección contra operaciones destructivas (3-10 req/hora)
- 🔧 **Características nuevas**:
  - User Management: CRUD completo, soft delete, restore, relación 1:1 con Pharmacy
  - Storage Usage: Vista materializada con estadísticas por farmacia
  - Backups: SHA-256 + HMAC signature para integridad
  - Database Size: Análisis de tamaño de BD y tablas principales

**Endpoints documentados**:

**Admin Users** (`/api/v1/admin/users`):
- POST `/` - Crear usuario con farmacia (transacción atómica)
- GET `/` - Listar usuarios con paginación y filtros
- GET `/{user_id}` - Obtener detalles de usuario
- PUT `/{user_id}` - Actualizar role/subscription/is_active
- DELETE `/{user_id}` - Soft delete GDPR compliant
- POST `/{user_id}/restore` - Restaurar desde soft delete
- GET `/storage/usage` - Estadísticas de almacenamiento por farmacia

**Admin Database** (`/api/v1/admin/database`):
- POST `/backups` - Crear backup con hash/signature
- GET `/backups` - Listar backups con verificación de integridad
- POST `/storage/refresh` - Actualizar vista materializada
- GET `/size` - Tamaño de BD y top tablas

**Ejemplos realistas**:
- Farmacias españolas con NIFs válidos (B12345678, B87654321)
- Direcciones y teléfonos de Madrid, Barcelona
- Storage limits por plan (free: 1GB, pro: 10GB, max: 100GB)
- Códigos HTTP correctos (201, 204, 403, 429)

### Versión 4.1.0 (2025-10-20) - REGLA #12
**Auto-Inicialización de Partners**

- 🆕 **Nuevo comportamiento**: Auto-inicialización transparente de partners en GET endpoints
- ⭐ **Actualizado**: `GET /api/v1/pharmacy-partners/{pharmacy_id}` con auto-init
- ⭐ **Actualizado**: `GET /api/v1/pharmacy-partners/{pharmacy_id}/selected` con auto-init
- 📋 **Documentado**: Comportamiento detallado para 3 casos (nueva farmacia, sin seleccionados, con seleccionados)
- 💡 **Criterios**: Top 8 laboratorios auto-seleccionados por ventas últimos 13 meses
- 🔍 **Logs**: Formato `[AUTO_INIT]` para identificar auto-inicializaciones

**Características nuevas**:
- Query param `force_refresh` para re-calcular sugerencias
- Response fields: `auto_initialized`, `initialization_timestamp`, `is_auto_suggested`, `suggestion_rank`
- Ejemplos frontend con async/await y manejo de estados

**Problema resuelto**:
- Farmacia nueva → `/generics` con lista vacía → Usuario confundido
- Ahora → Primera llamada GET auto-inicializa → Top 8 sugeridos visibles inmediatamente

### Versión 4.0.0 (2025-10-03) - Issue #108 / PR #107
**Actualización sistema de autenticación simplificado**

- ❌ **Eliminado**: Registro público (`POST /auth/register`)
- 🆕 **Nuevo**: Sistema de invitaciones (6 endpoints)
- 🔄 **Actualizado**: Roles simplificados (4 roles → 2 roles: admin/user)
- 📋 **Documentado**: Permisos detallados por rol
- ⚠️ **Ejemplos**: Respuestas de error 403 para usuarios no-admin
- 🔐 **Migración**: Roles manager/viewer → user (automático)

**Endpoints añadidos**:
- POST `/api/v1/invitations/create`
- GET `/api/v1/invitations/list/{pharmacy_id}`
- POST `/api/v1/invitations/validate`
- POST `/api/v1/invitations/accept`
- DELETE `/api/v1/invitations/cancel/{invitation_id}`
- POST `/api/v1/invitations/resend/{invitation_id}`

**Endpoints eliminados**:
- POST `/api/v1/register` (deprecado)

### Versión 3.1.0 (2025-09-21)
**Sistema Unificado y Herramientas Admin**

- **Sistema Unificado**: 2 endpoints para estado consolidado
- **Herramientas Admin**: 5 endpoints para operaciones centralizadas

### Versión 3.0.0 (2025-01-28)
**Partner Analysis y Laboratory Mapping**

- Partner Analysis Redesign (4 endpoints)
- Laboratory Mapping System (6 endpoints)

---

**Última actualización**: 2026-01-06
**Versión**: 5.2.0
**Total Endpoints**: 168
**Sistema de Roles**: Simplificado (admin/user)
**Onboarding**: Sistema de invitaciones exclusivamente
**Partners**: Auto-inicialización transparente (REGLA #12)
**Admin APIs**: User Management + Database Management (Issue #348 FASE 2)
**ML Monitoring**: Classifier health metrics + Coverage KPI (Issue #458)
