"""
Componente de tabla YoY para venta libre (Issue #491)

Tabla de comparación año vs año por categoría NECESIDAD.
Muestra variación en euros, porcentaje, tendencia visual y sparkline.
"""

import dash_bootstrap_components as dbc
from dash import html
from typing import Dict, List

from components.ventalibre.categories import format_necesidad_name
from styles.design_tokens import COLORS
from utils.helpers import format_currency


def create_sparkline_svg(
    data: List[float],
    width: int = 60,
    height: int = 20,
    trend: str = "stable"
) -> html.Div:
    """
    Crea un sparkline SVG inline basado en datos de serie temporal.

    Args:
        data: Lista de valores numéricos (ej: ventas mensuales)
        width: Ancho del SVG en pixels
        height: Alto del SVG en pixels
        trend: "up", "down", o "stable" para determinar color

    Returns:
        html.Div con SVG inline o placeholder si no hay datos
    """
    if not data or len(data) < 2:
        return html.Div(
            html.Small("--", className="text-muted"),
            style={"width": f"{width}px", "textAlign": "center"}
        )

    # Determinar color basado en tendencia (usando design tokens)
    if trend == "up":
        stroke_color = COLORS.get("success", "#28a745")
    elif trend == "down":
        stroke_color = COLORS.get("danger", "#dc3545")
    else:
        stroke_color = COLORS.get("secondary", "#6c757d")

    # Normalizar datos para que quepan en el SVG
    min_val = min(data)
    max_val = max(data)
    range_val = max_val - min_val if max_val != min_val else 1

    # Padding interno
    padding_x = 2
    padding_y = 3
    inner_width = width - 2 * padding_x
    inner_height = height - 2 * padding_y

    # Calcular puntos de la línea
    points = []
    for i, val in enumerate(data):
        x = padding_x + (i / (len(data) - 1)) * inner_width
        y = padding_y + inner_height - ((val - min_val) / range_val) * inner_height
        points.append(f"{x:.1f},{y:.1f}")

    polyline_points = " ".join(points)

    # Crear SVG como string embebido en Dash
    svg_content = f"""
    <svg width="{width}" height="{height}" viewBox="0 0 {width} {height}" xmlns="http://www.w3.org/2000/svg">
        <polyline
            fill="none"
            stroke="{stroke_color}"
            stroke-width="1.5"
            stroke-linecap="round"
            stroke-linejoin="round"
            points="{polyline_points}"
        />
        <circle cx="{points[-1].split(',')[0]}" cy="{points[-1].split(',')[1]}" r="2" fill="{stroke_color}" />
    </svg>
    """

    return html.Div(
        # Usar html.Iframe con srcdoc para embeber SVG de forma segura
        html.Iframe(
            srcDoc=svg_content.strip(),
            style={
                "width": f"{width}px",
                "height": f"{height}px",
                "border": "none",
                "overflow": "hidden",
                "display": "block"
            }
        ),
        style={"display": "inline-block", "verticalAlign": "middle"}
    )


def get_trend_icon(trend: str, variation_percent: float) -> html.Span:
    """
    Retorna icono de tendencia con color apropiado.

    Args:
        trend: "up", "down", o "stable"
        variation_percent: Porcentaje de variación

    Returns:
        html.Span con icono y estilo
    """
    if trend == "up":
        return html.Span([
            html.I(className="fas fa-arrow-up me-1"),
            f"+{abs(variation_percent):.1f}%"
        ], className="text-success fw-bold")
    elif trend == "down":
        return html.Span([
            html.I(className="fas fa-arrow-down me-1"),
            f"-{abs(variation_percent):.1f}%"
        ], className="text-danger fw-bold")
    else:
        return html.Span([
            html.I(className="fas fa-minus me-1"),
            "0%"
        ], className="text-secondary")


def create_yoy_table(yoy_data: Dict) -> html.Div:
    """
    Crea tabla de comparación YoY por categoría NECESIDAD con sparklines.

    Args:
        yoy_data: Dict con estructura:
            {
                "categories": [
                    {
                        "category": "piel",
                        "current_sales": 15000,
                        "previous_sales": 12000,
                        "variation_euros": 3000,
                        "variation_percent": 25.0,
                        "trend": "up",
                        "sparkline_data": [1200, 1300, 1400, 1500, 1600, 1700]
                    },
                    ...
                ],
                "total_current": 60000,
                "total_previous": 55000,
                "total_variation_percent": 9.1
            }

    Returns:
        html.Div con tabla Bootstrap incluyendo sparklines
    """
    categories = yoy_data.get("categories", [])

    if not categories:
        return html.Div([
            html.Div([
                html.I(className="fas fa-calendar-alt fa-3x text-muted mb-3"),
                html.P("No hay datos para comparación YoY", className="text-muted mb-1"),
                html.Small(
                    "Se requieren al menos 12 meses de datos",
                    className="text-muted"
                ),
            ], className="text-center py-4")
        ])

    # Ordenar por variación absoluta (mayor impacto primero)
    categories_sorted = sorted(
        categories,
        key=lambda x: abs(x.get("variation_euros", 0)),
        reverse=True
    )

    # Crear filas de tabla
    rows = []
    for cat in categories_sorted:
        category = cat.get("category", "")
        current_sales = cat.get("current_sales", 0)
        previous_sales = cat.get("previous_sales", 0)
        variation_euros = cat.get("variation_euros", 0)
        variation_percent = cat.get("variation_percent", 0)
        trend = cat.get("trend", "stable")
        sparkline_data = cat.get("sparkline_data", [])

        # Color de fondo basado en tendencia
        if trend == "up":
            bg_style = {"backgroundColor": "rgba(40, 167, 69, 0.05)"}
        elif trend == "down":
            bg_style = {"backgroundColor": "rgba(220, 53, 69, 0.05)"}
        else:
            bg_style = {}

        rows.append(
            html.Tr([
                html.Td(format_necesidad_name(category), className="align-middle"),
                html.Td(format_currency(current_sales), className="text-end align-middle"),
                html.Td(format_currency(previous_sales), className="text-end align-middle text-muted"),
                html.Td(
                    format_currency(variation_euros, show_sign=True),
                    className=f"text-end align-middle {'text-success' if variation_euros >= 0 else 'text-danger'}"
                ),
                html.Td(
                    get_trend_icon(trend, variation_percent),
                    className="text-end align-middle"
                ),
                html.Td(
                    create_sparkline_svg(sparkline_data, width=60, height=20, trend=trend),
                    className="align-middle text-center"
                ),
            ], style=bg_style)
        )

    # Fila de totales
    total_current = yoy_data.get("total_current", 0)
    total_previous = yoy_data.get("total_previous", 0)
    total_variation = total_current - total_previous
    total_variation_percent = yoy_data.get("total_variation_percent", 0)
    total_trend = "up" if total_variation > 0 else ("down" if total_variation < 0 else "stable")

    rows.append(
        html.Tr([
            html.Th("TOTAL", className="align-middle"),
            html.Th(format_currency(total_current), className="text-end align-middle"),
            html.Th(format_currency(total_previous), className="text-end align-middle text-muted"),
            html.Th(
                format_currency(total_variation, show_sign=True),
                className=f"text-end align-middle {'text-success' if total_variation >= 0 else 'text-danger'}"
            ),
            html.Th(
                get_trend_icon(total_trend, total_variation_percent),
                className="text-end align-middle"
            ),
            html.Th("", className="align-middle"),  # Sparkline vacío para totales
        ], className="table-light fw-bold")
    )

    return html.Div([
        dbc.Table([
            html.Thead([
                html.Tr([
                    html.Th("Categoría", style={"width": "25%"}),
                    html.Th("Actual", className="text-end", style={"width": "15%"}),
                    html.Th("Año Ant.", className="text-end", style={"width": "15%"}),
                    html.Th("Var €", className="text-end", style={"width": "15%"}),
                    html.Th("Var %", className="text-end", style={"width": "15%"}),
                    html.Th("Tendencia", className="text-center", style={"width": "15%"}),
                ])
            ]),
            html.Tbody(rows),
        ], striped=True, hover=True, responsive=True, size="sm", className="mb-0"),

        # Nota informativa
        html.Small([
            html.I(className="fas fa-info-circle me-1"),
            "Sparkline: últimos 6 meses | Comparación del período actual vs año anterior"
        ], className="text-muted d-block mt-2")
    ])
