"""
Componente de tabla Top Contributors para venta libre (Issue #491)

Tabla de productos que más contribuyen al crecimiento/decrecimiento YoY.
Muestra top 10 por defecto con impacto porcentual.
"""

import dash_bootstrap_components as dbc
from dash import html
from typing import Dict, List

from components.ventalibre.categories import PRINCIPAL_CATEGORIES, format_necesidad_name
from utils.helpers import format_currency, format_percentage


def truncate_product_name(name: str, max_length: int = 40) -> str:
    """Trunca nombre de producto si es muy largo."""
    if not name:
        return "Producto sin nombre"
    if len(name) <= max_length:
        return name
    return name[:max_length-3] + "..."


def get_impact_bar(impact_percent: float, direction: str) -> html.Div:
    """
    Crea barra visual de impacto proporcional.

    Args:
        impact_percent: Porcentaje de impacto sobre el total
        direction: "up" o "down"

    Returns:
        html.Div con barra de progreso
    """
    # Limitar a 100% para visualización
    bar_width = min(abs(impact_percent) * 10, 100)  # Escalar para mejor visibilidad

    color = "success" if direction == "up" else "danger"

    return html.Div([
        dbc.Progress(
            value=bar_width,
            color=color,
            style={"height": "8px"},
            className="mb-0"
        ),
        html.Small(f"{impact_percent:.1f}%", className=f"text-{color}")
    ], style={"minWidth": "60px"})


def create_top_contributors_table(
    contributors_data: Dict,
    direction_filter: str = "all"
) -> html.Div:
    """
    Crea tabla de productos que más contribuyen al cambio YoY.

    Args:
        contributors_data: Dict con estructura:
            {
                "contributors": [
                    {
                        "product_name": "IBUPROFENO 600MG 40 COMP",
                        "category": "dolor_fiebre",
                        "current_sales": 5000,
                        "previous_sales": 3500,
                        "variation_euros": 1500,
                        "variation_percent": 42.8,
                        "impact_percent": 2.5
                    },
                    ...
                ],
                "total_change": 60000,
                "direction": "all"
            }
        direction_filter: "all", "up", or "down"

    Returns:
        html.Div con tabla Bootstrap
    """
    contributors = contributors_data.get("contributors", [])
    total_change = contributors_data.get("total_change", 0)

    if not contributors:
        return html.Div([
            html.Div([
                html.I(className="fas fa-chart-line fa-3x text-muted mb-3"),
                html.P("No hay productos con cambios significativos", className="text-muted mb-1"),
                html.Small(
                    "Se requieren al menos 12 meses de datos",
                    className="text-muted"
                ),
            ], className="text-center py-4")
        ])

    # Filtrar por dirección si es necesario
    if direction_filter == "up":
        contributors = [c for c in contributors if c.get("variation_euros", 0) > 0]
    elif direction_filter == "down":
        contributors = [c for c in contributors if c.get("variation_euros", 0) < 0]

    if not contributors:
        direction_text = "crecimiento" if direction_filter == "up" else "decrecimiento"
        return html.Div([
            html.Div([
                html.I(className="fas fa-filter fa-2x text-muted mb-2"),
                html.P(f"No hay productos con {direction_text}", className="text-muted"),
            ], className="text-center py-3")
        ])

    # Crear filas de tabla
    rows = []
    for i, contrib in enumerate(contributors, 1):
        product_name = contrib.get("product_name", "")
        category = contrib.get("category", "")
        current_sales = contrib.get("current_sales", 0)
        previous_sales = contrib.get("previous_sales", 0)
        variation_euros = contrib.get("variation_euros", 0)
        variation_percent = contrib.get("variation_percent", 0)
        impact_percent = contrib.get("impact_percent", 0)

        # Determinar dirección para colores
        is_growth = variation_euros >= 0
        direction = "up" if is_growth else "down"
        var_class = "text-success" if is_growth else "text-danger"

        # Icono de ranking
        if i <= 3:
            rank_style = {"color": "#f39c12", "fontWeight": "bold"}  # Oro para top 3
        else:
            rank_style = {"color": "#6c757d"}

        rows.append(
            html.Tr([
                # Ranking
                html.Td(
                    html.Span(f"#{i}", style=rank_style),
                    className="align-middle text-center",
                    style={"width": "40px"}
                ),
                # Producto
                html.Td([
                    html.Div(
                        truncate_product_name(product_name),
                        className="fw-medium",
                        title=product_name  # Tooltip con nombre completo
                    ),
                    html.Small(
                        format_necesidad_name(category),
                        className="text-muted"
                    )
                ], className="align-middle"),
                # Variación €
                html.Td([
                    html.Span(
                        format_currency(variation_euros, show_sign=True),
                        className=var_class
                    ),
                    html.Br(),
                    html.Small(
                        f"({'+' if variation_percent >= 0 else ''}{variation_percent:.1f}%)",
                        className="text-muted"
                    )
                ], className="text-end align-middle"),
                # Impacto
                html.Td(
                    get_impact_bar(impact_percent, direction),
                    className="align-middle",
                    style={"minWidth": "80px"}
                ),
            ])
        )

    return html.Div([
        # Resumen superior
        html.Div([
            html.Small([
                html.Strong("Cambio total: "),
                html.Span(
                    format_currency(total_change, show_sign=True),
                    className="text-success" if total_change >= 0 else "text-danger"
                )
            ], className="me-3"),
            html.Small([
                html.Strong("Productos: "),
                str(len(contributors))
            ]),
        ], className="mb-2 d-flex flex-wrap"),

        # Tabla
        html.Div([
            dbc.Table([
                html.Thead([
                    html.Tr([
                        html.Th("#", className="text-center", style={"width": "40px"}),
                        html.Th("Producto"),
                        html.Th("Variación", className="text-end", style={"width": "100px"}),
                        html.Th("Impacto", style={"width": "100px"}),
                    ])
                ]),
                html.Tbody(rows),
            ], striped=True, hover=True, size="sm", className="mb-0"),
        ], style={"maxHeight": "400px", "overflowY": "auto"}),

        # Nota informativa
        html.Small([
            html.I(className="fas fa-info-circle me-1"),
            "Impacto = contribución del producto al cambio total YoY"
        ], className="text-muted d-block mt-2")
    ])
